import com.onionnetworks.fec.*;

import com.onionnetworks.util.*;

/*
 * Generates graph data and command lines for various ways of visualizing
 * FEC performance data.
 *
 * (c) Copyright 2000 Justin F. Chapweske
 *
 * @author Justin F. Chapweske (orasis@acm.org)
 */
public class GraphGenerator {

    public static FECMath fecMath = new FECMath(8);

    public static final Buffer[] buildData(int k, int size) {
        Buffer[] data = new Buffer[k];
        // build sample data
        for (int i = 0 ; i < k ; i++) {
            data[i] = new Buffer(size);
            for (int item=0; item < size; item++) {
                data[i].b[item+data[i].off] = 
                    (byte) (((item ^ i) + 3) & fecMath.gfSize);
            }
        }
        return data;
    }

    /**
     * This will test the time it takes to encode 100k of data by varying N
     * and overlaying graphs with a few different values of K.  Packet size
     * is 1k.
     */
    public static final void measureEncodeVaryNStepK() throws Exception {
        String title = "Encoding Speed Varying n: k=32,64,96,128";
        String x = "n";
        String y = "Time to Encode 100k (ms)";
        int[] kValues = new int[] {32,64,96,128};
        String[] graphNames = new String[kValues.length];
        int nMax = 256;
        Buffer tmp = new Buffer(1024); // we ignore this, but pass it in.
        int[][][] plots = new int[3][kValues.length][];
        int nStep = 8;

        // run each one 3 times and throw out the two extremes.
        for (int p=0;p<3;p++) {
            for (int i=0;i<kValues.length;i++) {
                int k = kValues[i];
                Buffer[] data = buildData(k,1024);
                plots[p][i] = new int[2*((nMax-k)/nStep)];
                for (int n=k+1,j=0;n<=nMax;n+=nStep,j++) {
                    System.err.println("k="+k+"n="+n);
                    // create the code and measure it.
                    FECCode code = FECCodeFactory.getDefault().
                        createFECCode(k,n);
                    long t1 = System.currentTimeMillis();
                    for (int a=0;a<100;a++) {
                        // do the mod to get a good sampling.
                        code.encode(data,tmp,k+((a * 19) % (n-k)));
                    }
                    long t2 = System.currentTimeMillis();
                    plots[p][i][2*j] = n;
                    plots[p][i][(2*j)+1] = (int) (t2 - t1);
                }
            }
        }

        int[][] result = new int[plots[0].length][];
        for (int i=0;i<result.length;i++) {
            result[i] = new int[plots[0][i].length];
            for (int j=0;j<result[i].length;j++) {
                result[i][j] = SimUtil.getMedian(new int[] {plots[0][i][j],
                                                        plots[1][i][j],
                                                        plots[2][i][j]});
            }
        }
        
        SimUtil.printGraphCommand(title,x,y,result,graphNames);
    }

    /**
     * This will test the time it takes to encode 100k of data by varying N
     * and overlaying graphs with a few different values of K.  Packet size
     * is 1k.
     */
    public static final void measureDecodeVaryNStepK() throws Exception {
        String title = "Worst Case Decode, Vary n: k=32,64,96,128";
        String x = "n";
        String y = "Time to Decode 100k (ms)";
        int[] kValues = new int[] {32};//,64,96,128};
        String[] graphNames = new String[kValues.length];
        int nMax = 256;
        int[][][] plots = new int[3][kValues.length][];
        int nStep = 8;

        // run each one 3 times and throw out the two extremes.
        for (int p=0;p<3;p++) {
            for (int i=0;i<kValues.length;i++) {
                int k = kValues[i];
                Buffer[] data = buildData(k,1024);
                Buffer fec = new Buffer(1024);
                int[] index = new int[k];
                for (int c=0;c<index.length;c++) {
                    index[c] = c;
                }
                plots[p][i] = new int[2*((nMax-k)/nStep)];
                for (int n=k+1,j=0;n<=nMax;n+=nStep,j++) {
                    System.err.println("k="+k+"n="+n);
                    // create the code and measure it.
                    FECCode code = FECCodeFactory.getDefault().
                        createFECCode(k,n);
                    code.encode(data,fec,k);
                    long t1 = System.currentTimeMillis();
                    for (int a=0;a<100;a++) {
                        int swap = (a * 19) % k;
                        index[swap] = k; 
                        data[swap] = fec;
                        code.decode(data,index);
                    }
                    long t2 = System.currentTimeMillis();
                    plots[p][i][2*j] = n;
                    plots[p][i][(2*j)+1] = (int) (t2 - t1);
                }
            }
        }

        int[][] result = new int[plots[0].length][];
        for (int i=0;i<result.length;i++) {
            result[i] = new int[plots[0][i].length];
            for (int j=0;j<result[i].length;j++) {
                result[i][j] = SimUtil.getMedian(new int[] {plots[0][i][j],
                                                        plots[1][i][j],
                                                        plots[2][i][j]});
            }
        }
        
        SimUtil.printGraphCommand(title,x,y,result,graphNames);
    }

    /**
     * Measures the time to set up the Code, this should increase
     * with both k and n as it must populate a k x n matrix.
     */
    public static final void measureSetupVaryNStepK() throws Exception {
        String title = "Setup Time Varying n: k=32,64,96,128";
        String x = "n";
        String y = "Time to Setup Code (ms)";
        int[] kValues = new int[] {32,64,96,128};
        String[] graphNames = new String[kValues.length];
        int nMax = 256;
        Buffer tmp = new Buffer(1024); // we ignore this, but pass it in.
        int[][][] plots = new int[3][kValues.length][];
        int nStep = 8;

        // run each one 3 times and throw out the two extremes.
        for (int p=0;p<3;p++) {
            for (int i=0;i<kValues.length;i++) {
                int k = kValues[i];
                Buffer[] data = buildData(k,1024);
                plots[p][i] = new int[2*((nMax-k)/nStep)];
                for (int n=k+1,j=0;n<=nMax;n+=nStep,j++) {
                    System.err.println("k="+k+"n="+n);
                    // create the code and measure it.
                    long t1 = System.currentTimeMillis();
                    FECCode code = FECCodeFactory.getDefault().
                        createFECCode(k,n);
                    long t2 = System.currentTimeMillis();
                    plots[p][i][2*j] = n;
                    plots[p][i][(2*j)+1] = (int) (t2 - t1);
                }
            }
        }

        int[][] result = new int[plots[0].length][];
        for (int i=0;i<result.length;i++) {
            result[i] = new int[plots[0][i].length];
            for (int j=0;j<result[i].length;j++) {
                result[i][j] = SimUtil.getMedian(new int[] {plots[0][i][j],
                                                         plots[1][i][j],
                                                         plots[2][i][j]});
            }
        }
        
        SimUtil.printGraphCommand(title,x,y,result,graphNames);
    }

    public static void main(String[] args) throws Exception {
        //measureEncodeVaryNStepK();
        measureDecodeVaryNStepK();
        //measureSetupVaryNStepK();
    }
}
