package expression

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/require"
	"google.golang.org/protobuf/types/known/structpb"
)

func TestExpandJobInputs(t *testing.T) {
	t.Run("returns expression string if it contains no expression", func(t *testing.T) {
		expanded, err := ExpandJobInputs([]JobInput{}, "hello world")
		require.NoError(t, err)
		require.Equal(t, "hello world", expanded.GetStringValue())
	})

	t.Run("expands job inputs", func(t *testing.T) {
		jobInputs := []JobInput{
			{Key: "msg", Value: structpb.NewStringValue("Bye")},
			{Key: "animals", Value: structpb.NewStructValue(
				&structpb.Struct{Fields: map[string]*structpb.Value{
					"cow_sound": structpb.NewStringValue("moo"),
					"sheepies":  structpb.NewStringValue("3"),
				}},
			)},
		}

		expr := "Cows go ${{job.inputs.animals.cow_sound}}, I have ${{job.inputs.animals.sheepies}} sheep. ${{job.inputs.msg}}!"
		expanded, err := ExpandJobInputs(jobInputs, expr)
		require.NoError(t, err)
		require.Equal(t, "Cows go moo, I have 3 sheep. Bye!", expanded.GetStringValue())
	})

	t.Run("ignores space", func(t *testing.T) {
		jobInputs := []JobInput{{Key: "msg", Value: structpb.NewStringValue("hi")}}

		expr := "${{      job.inputs.msg    }}"
		expanded, err := ExpandJobInputs(jobInputs, expr)
		require.NoError(t, err)
		require.Equal(t, "hi", expanded.GetStringValue())
	})

	t.Run("errors when can't access variable", func(t *testing.T) {
		expr := "Hello ${{job.inputs.world}}"
		_, err := ExpandJobInputs([]JobInput{}, expr)
		require.Error(t, err)
		require.Equal(t, `expanding expression: job.inputs.world: the "world" was not found`, err.Error())
	})

	t.Run("errors when expressions contain expressions", func(t *testing.T) {
		jobInputs := []JobInput{
			{Key: "foo", Value: structpb.NewStringValue("bar")},
			{Key: "bar", Value: structpb.NewStringValue("baz")},
		}

		_, err := ExpandJobInputs(jobInputs, "${{ job.inputs.${{ job.inputs.foo }} }}")
		require.Error(t, err)
	})

	t.Run("errors when string template expression does not return a string", func(t *testing.T) {
		jobInputs := []JobInput{{Key: "is_sunny", Value: structpb.NewBoolValue(false)}}

		_, err := ExpandJobInputs(jobInputs, "is it sunny today? ${{ job.inputs.is_sunny }}")
		require.Error(t, err)
		require.Equal(t, "expanding expression: job.inputs.is_sunny: must evaluate to a string when used in a string template", err.Error())
	})

	t.Run("expressions that are not string templates may return non-string values", func(t *testing.T) {
		jobInputs := []JobInput{
			{Key: "string", Value: structpb.NewStringValue("hi")},
			{Key: "object", Value: structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{}})},
			{Key: "bool", Value: structpb.NewBoolValue(false)},
			{Key: "number", Value: structpb.NewNumberValue(1)},
		}

		for _, test := range jobInputs {
			t.Run(test.Key, func(t *testing.T) {
				expanded, err := ExpandJobInputs(jobInputs, fmt.Sprintf("${{ job.inputs.%s }}", test.Key))
				require.NoError(t, err)
				require.Equal(t, test.Value, expanded)
			})
		}
	})
}
