function klu_make
%KLU_MAKE compiles the KLU mexFunctions
%
% Example:
%   klu_make
%
% KLU relies on AMD, COLAMD, and BTF for its ordering options, and can
% optionally use CHOLMOD, CCOLAMD, CAMD, and METIS as well.
%
% You must type the klu_make command while in the KLU/MATLAB directory.
%
% See also klu.

% KLU, Copyright (c) 2004-2022, University of Florida.  All Rights Reserved.
% Authors: Timothy A. Davis and Ekanathan Palamadai.
% SPDX-License-Identifier: LGPL-2.1+

metis_path = '../../CHOLMOD/SuiteSparse_metis' ;
with_cholmod = exist (metis_path, 'dir') ;

details = 0 ;       % if 1, print details of each command

d = '' ;
if (~isempty (strfind (computer, '64')))
    % 64-bit MATLAB
    d = '-largeArrayDims' ;
end

% MATLAB 8.3.0 now has a -silent option to keep 'mex' from burbling too much
if (~verLessThan ('matlab', '8.3.0'))
    d = ['-silent ' d] ;
end

fprintf ('Compiling KLU ') ;
kk = 0 ; 

include = '-I. -I../../AMD/Include -I../../COLAMD/Include -I../Include -I../../SuiteSparse_config -I../../BTF/Include' ;

if (with_cholmod)
    fprintf ('with CHOLMOD, CAMD, CCOLAMD, and METIS\n') ;
    include = [include ' -I../../CCOLAMD/Include -I../../CAMD/Include -I../../CHOLMOD/Include -I../User -I../../CHOLMOD'] ;
    include = [include ' -I' metis_path '/include'] ;
    include = [include ' -I' metis_path '/GKlib'] ;
    include = [include ' -I' metis_path '/libmetis'] ;
    include = ['-DNSUPERNODAL -DNMODIFY -DNMATRIXOPS -DNCHECK ' include] ;
else
    fprintf ('without CHOLMOD, CAMD, CCOLAMD, and METIS\n') ;
    include = ['-DNCHOLMOD ' include] ;
end

% do not attempt to compile CHOLMOD with large file support (not needed)
include = [include ' -DNLARGEFILE'] ;

suitesparse_src = { '../../SuiteSparse_config/SuiteSparse_config' } ;

amd_src = { ...
    '../../AMD/Source/amd_l1', ...
    '../../AMD/Source/amd_l2', ...
    '../../AMD/Source/amd_l_aat', ...
    '../../AMD/Source/amd_l_control', ...
    '../../AMD/Source/amd_l_defaults', ...
    '../../AMD/Source/amd_l_dump', ...
    '../../AMD/Source/amd_l_info', ...
    '../../AMD/Source/amd_l_order', ...
    '../../AMD/Source/amd_l_postorder', ...
    '../../AMD/Source/amd_l_post_tree', ...
    '../../AMD/Source/amd_l_preprocess', ...
    '../../AMD/Source/amd_l_valid' } ;

colamd_src = {
    '../../COLAMD/Source/colamd_l' } ;

if (with_cholmod)

    camd_src = { ...
        '../../CAMD/Source/camd_l1', ...
        '../../CAMD/Source/camd_l2', ...
        '../../CAMD/Source/camd_l_aat', ...
        '../../CAMD/Source/camd_l_control', ...
        '../../CAMD/Source/camd_l_defaults', ...
        '../../CAMD/Source/camd_l_dump', ...
        '../../CAMD/Source/camd_l_info', ...
        '../../CAMD/Source/camd_l_order', ...
        '../../CAMD/Source/camd_l_postorder', ...
        '../../CAMD/Source/camd_l_preprocess', ...
        '../../CAMD/Source/camd_l_valid' } ;

    ccolamd_src = {
        '../../CCOLAMD/Source/ccolamd_l' } ;

    cholmod_src = {
        '../../CHOLMOD/Core/cholmod_l_aat', ...
        '../../CHOLMOD/Core/cholmod_l_add', ...
        '../../CHOLMOD/Core/cholmod_l_band', ...
        '../../CHOLMOD/Core/cholmod_l_change_factor', ...
        '../../CHOLMOD/Core/cholmod_l_common', ...
        '../../CHOLMOD/Core/cholmod_l_complex', ...
        '../../CHOLMOD/Core/cholmod_l_copy', ...
        '../../CHOLMOD/Core/cholmod_l_dense', ...
        '../../CHOLMOD/Core/cholmod_l_error', ...
        '../../CHOLMOD/Core/cholmod_l_factor', ...
        '../../CHOLMOD/Core/cholmod_l_memory', ...
        '../../CHOLMOD/Core/cholmod_l_sparse', ...
        '../../CHOLMOD/Core/cholmod_l_transpose', ...
        '../../CHOLMOD/Core/cholmod_l_triplet', ...
        '../../CHOLMOD/Cholesky/cholmod_l_amd', ...
        '../../CHOLMOD/Cholesky/cholmod_l_analyze', ...
        '../../CHOLMOD/Cholesky/cholmod_l_colamd', ...
        '../../CHOLMOD/Cholesky/cholmod_l_etree', ...
        '../../CHOLMOD/Cholesky/cholmod_l_postorder', ...
        '../../CHOLMOD/Cholesky/cholmod_l_rowcolcounts', ...
        '../../CHOLMOD/Partition/cholmod_l_ccolamd', ...
        '../../CHOLMOD/Partition/cholmod_l_csymamd', ...
        '../../CHOLMOD/Partition/cholmod_l_camd', ...
        '../../CHOLMOD/Partition/cholmod_l_metis', ...
        '../../CHOLMOD/Partition/cholmod_metis_wrapper', ...
        '../../CHOLMOD/Partition/cholmod_l_nesdis' } ;

else
    camd_src = { } ;
    ccolamd_src = { } ;
    cholmod_src = { } ;
end

btf_src = {
    '../../BTF/Source/btf_l_maxtrans', ...
    '../../BTF/Source/btf_l_order', ...
    '../../BTF/Source/btf_l_strongcomp' } ;

klu_src = {
    '../Source/klu_l_free_symbolic', ...
    '../Source/klu_l_defaults', ...
    '../Source/klu_l_analyze_given', ...
    '../Source/klu_l_analyze', ...
    '../Source/klu_l_memory' } ;

if (with_cholmod)
    klu_src = [klu_src { '../User/klu_l_cholmod' }] ;                       %#ok
end

klu_zlsrc = {
    '../Source/klu_zl', ...
    '../Source/klu_zl_kernel', ...
    '../Source/klu_zl_dump', ...
    '../Source/klu_zl_factor', ...
    '../Source/klu_zl_free_numeric', ...
    '../Source/klu_zl_solve', ...
    '../Source/klu_zl_scale', ...
    '../Source/klu_zl_refactor', ...
    '../Source/klu_zl_tsolve', ...
    '../Source/klu_zl_diagnostics', ...
    '../Source/klu_zl_sort', ...
    '../Source/klu_zl_extract', ...
    } ;

klu_lsrc = {
    '../Source/klu_l', ...
    '../Source/klu_l_kernel', ...
    '../Source/klu_l_dump', ...
    '../Source/klu_l_factor', ...
    '../Source/klu_l_free_numeric', ...
    '../Source/klu_l_solve', ...
    '../Source/klu_l_scale', ...
    '../Source/klu_l_refactor', ...
    '../Source/klu_l_tsolve', ...
    '../Source/klu_l_diagnostics', ...
    '../Source/klu_l_sort', ...
    '../Source/klu_l_extract', ...
    } ;

try
    % ispc does not appear in MATLAB 5.3
    pc = ispc ;
catch
    % if ispc fails, assume we are on a Windows PC if it's not unix
    pc = ~isunix ;
end

if (pc)
    % Windows does not have drand48 and srand48, required by METIS.  Use
    % drand48 and srand48 in CHOLMOD/MATLAB/Windows/rand48.c instead.
    obj_extension = '.obj' ;
    cholmod_src = [cholmod_src {'../../CHOLMOD/MATLAB/Windows/rand48'}] ;
    include = [include ' -I../../CHOLMOD/MATLAB/Windows'] ;
else
    obj_extension = '.o' ;
end

% compile each library source file
obj = ' ' ;

source = [suitesparse_src amd_src btf_src klu_src colamd_src] ;
if (with_cholmod)
    source = [ccolamd_src camd_src cholmod_src source] ;
end
source = [source klu_zlsrc] ;
source = [source klu_lsrc] ;

for f = source
    ff = f {1} ;
    slash = strfind (ff, '/') ;
    if (isempty (slash))
        slash = 1 ;
    else
        slash = slash (end) + 1 ;
    end
    o = ff (slash:end) ;
    % fprintf ('%s\n', o) ;
    o = [o obj_extension] ;
    obj = [obj  ' ' o] ;					            %#ok
    s = sprintf ('mex %s -O %s -c %s.c', d, include, ff) ;
    kk = do_cmd (s, kk, details) ;
end

% compile the KLU mexFunction
s = sprintf ('mex %s -O %s -output klu klu_mex.c', d, include) ;
s = [s obj] ;                                                               %#ok

if (~(ispc || ismac))
    % for POSIX timing routine
    s = [s ' -lrt'] ;
end

kk = do_cmd (s, kk, details) ;

% clean up
s = ['delete ' obj] ;
do_cmd (s, kk, details) ;

fprintf ('\nKLU successfully compiled\n') ;

%-------------------------------------------------------------------------------

function rmfile (file)
% rmfile:  delete a file, but only if it exists
if (length (dir (file)) > 0)                                                %#ok
    delete (file) ;
end

%-------------------------------------------------------------------------------
function kk = do_cmd (s, kk, details)
%DO_CMD: evaluate a command, and either print it or print a "."
if (details)
    fprintf ('%s\n', s) ;
else
    if (mod (kk, 60) == 0)
        fprintf ('\n') ;
    end
    kk = kk + 1 ;
    fprintf ('.') ;
end
eval (s) ;

