pub const ext = @import("ext.zig");
const gcrui = @This();

const std = @import("std");
const compat = @import("compat");
const gtk = @import("gtk3");
const xlib = @import("xlib2");
const gdk = @import("gdk3");
const cairo = @import("cairo1");
const gobject = @import("gobject2");
const glib = @import("glib2");
const pango = @import("pango1");
const harfbuzz = @import("harfbuzz0");
const freetype2 = @import("freetype22");
const gio = @import("gio2");
const gmodule = @import("gmodule2");
const gdkpixbuf = @import("gdkpixbuf2");
const atk = @import("atk1");
const gcr = @import("gcr3");
const gck = @import("gck1");
/// An implementation of `gcrui.Renderer` which renders certificates.
pub const CertificateRenderer = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gcr.Certificate, gcr.Comparable, gcrui.Renderer };
    pub const Class = gcrui.CertificateRendererClass;
    f_parent: gobject.Object,
    f_pv: ?*gcrui.CertificateRendererPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The certificate attributes to display. One of the attributes must be
        /// a CKA_VALUE type attribute which contains a DER encoded certificate.
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*gck.Attributes;
        };

        /// The certificate to display. May be `NULL`.
        pub const certificate = struct {
            pub const name = "certificate";

            pub const Type = ?*gcr.Certificate;
        };

        /// The label to display.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Create a new certificate renderer to display the certificate.
    extern fn gcr_certificate_renderer_new(p_certificate: *gcr.Certificate) *gcrui.CertificateRenderer;
    pub const new = gcr_certificate_renderer_new;

    /// Create a new certificate renderer to display the label and attributes. One
    /// of the attributes should be a CKA_VALUE type attribute containing a DER
    /// encoded certificate.
    extern fn gcr_certificate_renderer_new_for_attributes(p_label: ?[*:0]const u8, p_attrs: ?*anyopaque) *gcrui.CertificateRenderer;
    pub const newForAttributes = gcr_certificate_renderer_new_for_attributes;

    /// Get the certificate displayed in the renderer. If no certificate was
    /// explicitly set, then the renderer will return itself since it acts as
    /// a valid certificate.
    extern fn gcr_certificate_renderer_get_certificate(p_self: *CertificateRenderer) *gcr.Certificate;
    pub const getCertificate = gcr_certificate_renderer_get_certificate;

    /// Set a certificate to display in the renderer.
    extern fn gcr_certificate_renderer_set_certificate(p_self: *CertificateRenderer, p_certificate: ?*gcr.Certificate) void;
    pub const setCertificate = gcr_certificate_renderer_set_certificate;

    extern fn gcr_certificate_renderer_get_type() usize;
    pub const getGObjectType = gcr_certificate_renderer_get_type;

    extern fn g_object_ref(p_self: *gcrui.CertificateRenderer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.CertificateRenderer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CertificateRenderer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A widget that can be used to display a certificate.
///
/// A certificate widget is normally in a collapsed state showing only
/// details, but can be expanded by the user.
///
/// Use `CertificateWidget.new` to create a new certificate widget. Only
/// one certificate can be displayed. It contains a `Viewer` internally
/// and `CertificateRenderer` is used to render the certificate to the
/// viewer.
///
/// To show more than one certificate in a view, create the viewer and
/// add renderers to it.
pub const CertificateWidget = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gcrui.CertificateWidgetClass;
    f_parent: gtk.Bin,
    f_pv: ?*gcrui.CertificateWidgetPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*gck.Attributes;
        };

        pub const certificate = struct {
            pub const name = "certificate";

            pub const Type = ?*gcr.Certificate;
        };
    };

    pub const signals = struct {};

    /// Create a new certificate widget which displays a given certificate.
    extern fn gcr_certificate_widget_new(p_certificate: ?*gcr.Certificate) *gcrui.CertificateWidget;
    pub const new = gcr_certificate_widget_new;

    /// Get the attributes displayed in the widget. The attributes should contain
    /// a certificate.
    extern fn gcr_certificate_widget_get_attributes(p_self: *CertificateWidget) ?*gck.Attributes;
    pub const getAttributes = gcr_certificate_widget_get_attributes;

    /// Get the certificate displayed in the widget.
    extern fn gcr_certificate_widget_get_certificate(p_self: *CertificateWidget) ?*gcr.Certificate;
    pub const getCertificate = gcr_certificate_widget_get_certificate;

    /// Set the attributes displayed in the widget. The attributes should contain
    /// a certificate.
    extern fn gcr_certificate_widget_set_attributes(p_self: *CertificateWidget, p_attrs: ?*gck.Attributes) void;
    pub const setAttributes = gcr_certificate_widget_set_attributes;

    /// Set the certificate displayed in the widget
    extern fn gcr_certificate_widget_set_certificate(p_self: *CertificateWidget, p_certificate: ?*gcr.Certificate) void;
    pub const setCertificate = gcr_certificate_widget_set_certificate;

    extern fn gcr_certificate_widget_get_type() usize;
    pub const getGObjectType = gcr_certificate_widget_get_type;

    extern fn g_object_ref(p_self: *gcrui.CertificateWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.CertificateWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CertificateWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Ain implementation of a `gtk.TreeModel` which contains a row for each
/// object in a `gcr.Collection`.
///
/// As objects are added or removed from the collection, rows are added and
/// removed from this model.
///
/// The row values come from the properties of the objects in the collection. Use
/// `CollectionModel.new` to create a new collection model. To have more
/// control over the values use a set of `gcr.Column` structures to
/// define the columns. This can be done with `CollectionModel.newFull` or
/// `CollectionModel.setColumns`.
///
/// Each row can have a selected state, which is represented by a boolean column.
/// The selected state can be toggled with `gcrui.CollectionModel.toggleSelected`
/// or set with `gcrui.CollectionModel.setSelectedObjects` and retrieved with
/// `CollectionModel.getSelectedObjects`.
///
/// To determine which object a row represents and vice versa, use the
/// `CollectionModel.iterForObject` or
/// `CollectionModel.objectForIter` functions.
pub const CollectionModel = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gtk.TreeModel, gtk.TreeSortable };
    pub const Class = gcrui.CollectionModelClass;
    f_parent: gobject.Object,
    f_pv: ?*gcrui.CollectionModelPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const collection = struct {
            pub const name = "collection";

            pub const Type = ?*gcr.Collection;
        };

        pub const columns = struct {
            pub const name = "columns";

            pub const Type = ?*anyopaque;
        };

        pub const mode = struct {
            pub const name = "mode";

            pub const Type = @compileError("no type information available");
        };
    };

    pub const signals = struct {};

    /// Create a new `gcrui.CollectionModel`. The variable argument list should contain
    /// pairs of property names, and `gobject.Type` values. The variable argument list should
    /// be terminated with `NULL`.
    extern fn gcr_collection_model_new(p_collection: *gcr.Collection, p_mode: gcrui.CollectionModelMode, ...) *gcrui.CollectionModel;
    pub const new = gcr_collection_model_new;

    /// Create a new `gcrui.CollectionModel`.
    extern fn gcr_collection_model_new_full(p_collection: *gcr.Collection, p_mode: gcrui.CollectionModelMode, p_columns: *const gcr.Column) *gcrui.CollectionModel;
    pub const newFull = gcr_collection_model_new_full;

    /// Set whether a given row is toggled selected or not.
    extern fn gcr_collection_model_change_selected(p_self: *CollectionModel, p_iter: *gtk.TreeIter, p_selected: c_int) void;
    pub const changeSelected = gcr_collection_model_change_selected;

    /// Get the column identifier for the column that contains the values
    /// of the selected state.
    extern fn gcr_collection_model_column_for_selected(p_self: *CollectionModel) c_int;
    pub const columnForSelected = gcr_collection_model_column_for_selected;

    /// Get the collection which this model represents
    extern fn gcr_collection_model_get_collection(p_self: *CollectionModel) *gcr.Collection;
    pub const getCollection = gcr_collection_model_get_collection;

    /// Get a list of checked/selected objects.
    extern fn gcr_collection_model_get_selected_objects(p_self: *CollectionModel) *glib.List;
    pub const getSelectedObjects = gcr_collection_model_get_selected_objects;

    /// Check whether a given row has been toggled as selected.
    extern fn gcr_collection_model_is_selected(p_self: *CollectionModel, p_iter: *gtk.TreeIter) c_int;
    pub const isSelected = gcr_collection_model_is_selected;

    /// Set `iter` to the row for the given object. If the object is not in this
    /// model, then `FALSE` will be returned.
    extern fn gcr_collection_model_iter_for_object(p_self: *CollectionModel, p_object: *gobject.Object, p_iter: *gtk.TreeIter) c_int;
    pub const iterForObject = gcr_collection_model_iter_for_object;

    /// Get the object that is represented by the given row in the model.
    extern fn gcr_collection_model_object_for_iter(p_self: *CollectionModel, p_iter: *const gtk.TreeIter) *gobject.Object;
    pub const objectForIter = gcr_collection_model_object_for_iter;

    /// Set the collection which this model represents
    extern fn gcr_collection_model_set_collection(p_self: *CollectionModel, p_collection: ?*gcr.Collection) void;
    pub const setCollection = gcr_collection_model_set_collection;

    /// Set the columns that the model should contain. `columns` is an array of
    /// `gcr.Column` structures, with the last one containing `NULL` for all values.
    ///
    /// This function can only be called once, and only if the model was not created
    /// without a set of columns. This function cannot be called after the model
    /// has been added to a view.
    ///
    /// The columns are accessed as static data. They should continue to remain
    /// in memory for longer than the GcrCollectionModel object.
    extern fn gcr_collection_model_set_columns(p_self: *CollectionModel, p_columns: *const gcr.Column) c_uint;
    pub const setColumns = gcr_collection_model_set_columns;

    /// Set the checked/selected objects.
    extern fn gcr_collection_model_set_selected_objects(p_self: *CollectionModel, p_selected: *glib.List) void;
    pub const setSelectedObjects = gcr_collection_model_set_selected_objects;

    /// Toggle the selected state of a given row.
    extern fn gcr_collection_model_toggle_selected(p_self: *CollectionModel, p_iter: *gtk.TreeIter) void;
    pub const toggleSelected = gcr_collection_model_toggle_selected;

    extern fn gcr_collection_model_get_type() usize;
    pub const getGObjectType = gcr_collection_model_get_type;

    extern fn g_object_ref(p_self: *gcrui.CollectionModel) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.CollectionModel) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CollectionModel, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A widget that can be used to select a certificate or key. It allows
/// the user to select one object from the selector at a time.
pub const ComboSelector = extern struct {
    pub const Parent = gtk.ComboBox;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.CellEditable, gtk.CellLayout };
    pub const Class = gcrui.ComboSelectorClass;
    f_parent: gtk.ComboBox,
    f_pv: ?*gcrui.ComboSelectorPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The collection which contains the objects to display in the selector.
        pub const collection = struct {
            pub const name = "collection";

            pub const Type = ?*gcr.Collection;
        };
    };

    pub const signals = struct {};

    /// Create a new `gcrui.TreeSelector`.
    extern fn gcr_combo_selector_new(p_collection: *gcr.Collection) *gcrui.ComboSelector;
    pub const new = gcr_combo_selector_new;

    /// Get the collection that this selector is displaying objects from.
    extern fn gcr_combo_selector_get_collection(p_self: *ComboSelector) *gcr.Collection;
    pub const getCollection = gcr_combo_selector_get_collection;

    /// Get the selected object in the selector, or `NULL` if nothing selected.
    extern fn gcr_combo_selector_get_selected(p_self: *ComboSelector) *gobject.Object;
    pub const getSelected = gcr_combo_selector_get_selected;

    /// Set the currently selected object in the selector, or clear the selection
    /// if selected is set to `NULL`.
    extern fn gcr_combo_selector_set_selected(p_self: *ComboSelector, p_selected: ?*gobject.Object) void;
    pub const setSelected = gcr_combo_selector_set_selected;

    extern fn gcr_combo_selector_get_type() usize;
    pub const getGObjectType = gcr_combo_selector_get_type;

    extern fn g_object_ref(p_self: *gcrui.ComboSelector) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.ComboSelector) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ComboSelector, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A renderer that can be used for unsupported data.
pub const FailureRenderer = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gcrui.Renderer};
    pub const Class = gcrui.FailureRendererClass;
    f_parent: gobject.Object,
    f_pv: ?*gcrui.FailureRendererPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*gck.Attributes;
        };

        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Create a new renderer for unsupported data.
    extern fn gcr_failure_renderer_new_unsupported(p_label: [*:0]const u8) *gcrui.Renderer;
    pub const newUnsupported = gcr_failure_renderer_new_unsupported;

    /// Create a new renderer for an error.
    extern fn gcr_failure_renderer_new(p_label: ?[*:0]const u8, p_error: *glib.Error) *gcrui.FailureRenderer;
    pub const new = gcr_failure_renderer_new;

    extern fn gcr_failure_renderer_get_type() usize;
    pub const getGObjectType = gcr_failure_renderer_get_type;

    extern fn g_object_ref(p_self: *gcrui.FailureRenderer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.FailureRenderer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FailureRenderer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A button which imports keys and certificates.
///
/// The import button shows a spinner when the button is activated. When more
/// than one importer is available, it shows a drop down to select which to
/// import to.
pub const ImportButton = extern struct {
    pub const Parent = gtk.Button;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Activatable, gtk.Buildable };
    pub const Class = gcrui.ImportButtonClass;
    f_parent: gtk.Button,
    f_pv: ?*gcrui.ImportButtonPrivate,

    pub const virtual_methods = struct {
        /// Emitted when the import completes, or fails.
        pub const imported = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_importer: *gcr.Importer, p_error: *glib.Error) void {
                return gobject.ext.as(ImportButton.Class, p_class).f_imported.?(gobject.ext.as(ImportButton, p_self), p_importer, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_importer: *gcr.Importer, p_error: *glib.Error) callconv(.C) void) void {
                gobject.ext.as(ImportButton.Class, p_class).f_imported = @ptrCast(p_implementation);
            }
        };

        /// Emitted when the import begins.
        pub const importing = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_importer: *gcr.Importer) void {
                return gobject.ext.as(ImportButton.Class, p_class).f_importing.?(gobject.ext.as(ImportButton, p_self), p_importer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_importer: *gcr.Importer) callconv(.C) void) void {
                gobject.ext.as(ImportButton.Class, p_class).f_importing = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Signal emitted when an import completes or fails.
        pub const imported = struct {
            pub const name = "imported";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_importer: *gobject.Object, p_error: *glib.Error, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ImportButton, p_instance))),
                    gobject.signalLookup("imported", ImportButton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Signal emitted when an import begins.
        pub const importing = struct {
            pub const name = "importing";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_importer: *gobject.Object, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ImportButton, p_instance))),
                    gobject.signalLookup("importing", ImportButton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new `gcrui.ImportButton`.
    extern fn gcr_import_button_new(p_label: ?[*:0]const u8) *gcrui.ImportButton;
    pub const new = gcr_import_button_new;

    /// Queue an item to import via the button
    extern fn gcr_import_button_add_parsed(p_self: *ImportButton, p_parsed: *gcr.Parsed) void;
    pub const addParsed = gcr_import_button_add_parsed;

    extern fn gcr_import_button_get_type() usize;
    pub const getGObjectType = gcr_import_button_get_type;

    extern fn g_object_ref(p_self: *gcrui.ImportButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.ImportButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ImportButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An implementation of `gcrui.Renderer` which renders keys.
pub const KeyRenderer = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gcrui.Renderer};
    pub const Class = gcrui.KeyRendererClass;
    f_parent: gobject.Object,
    f_pv: ?*gcrui.KeyRendererPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const object = struct {
            pub const name = "object";

            pub const Type = ?*gck.Object;
        };
    };

    pub const signals = struct {};

    /// Create a new key renderer which renders a given key in the attributes.
    extern fn gcr_key_renderer_new(p_label: ?[*:0]const u8, p_attrs: ?*gck.Attributes) *gcrui.KeyRenderer;
    pub const new = gcr_key_renderer_new;

    /// Get the attributes displayed in the renderer.
    extern fn gcr_key_renderer_get_attributes(p_self: *KeyRenderer) ?*gck.Attributes;
    pub const getAttributes = gcr_key_renderer_get_attributes;

    /// Get the attributes displayed in the renderer. The attributes should represent
    /// either an RSA, DSA, or EC key in PKCS`@"11"` style.
    extern fn gcr_key_renderer_set_attributes(p_self: *KeyRenderer, p_attrs: ?*gck.Attributes) void;
    pub const setAttributes = gcr_key_renderer_set_attributes;

    extern fn gcr_key_renderer_get_type() usize;
    pub const getGObjectType = gcr_key_renderer_get_type;

    extern fn g_object_ref(p_self: *gcrui.KeyRenderer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.KeyRenderer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *KeyRenderer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A key widget and renderer
///
/// A key widget can be used to display a RSA, DSA or EC key. The widget is
/// normally in a collapsed state showing only details, but can be expanded by
/// the user.
///
/// Use `KeyWidget.new` to create a new key widget. Only one key can be
/// displayed. A key widget contains a `Viewer` internally and
/// `KeyRenderer` is used to render the key to the viewer. To show more
/// than one key in a view, create the viewer and add renderers to it.
pub const KeyWidget = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gcrui.KeyWidgetClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*gck.Attributes;
        };
    };

    pub const signals = struct {};

    /// Create a new key widget which displays a given key in the attributes.
    extern fn gcr_key_widget_new(p_attrs: ?*gck.Attributes) *gcrui.KeyWidget;
    pub const new = gcr_key_widget_new;

    /// Get the attributes displayed in the widget.
    extern fn gcr_key_widget_get_attributes(p_self: *KeyWidget) ?*gck.Attributes;
    pub const getAttributes = gcr_key_widget_get_attributes;

    /// Get the attributes displayed in the widget. The attributes should represent
    /// either an RSA, DSA or EC key in PKCS`@"11"` style.
    extern fn gcr_key_widget_set_attributes(p_self: *KeyWidget, p_attrs: ?*gck.Attributes) void;
    pub const setAttributes = gcr_key_widget_set_attributes;

    extern fn gcr_key_widget_get_type() usize;
    pub const getGObjectType = gcr_key_widget_get_type;

    extern fn g_object_ref(p_self: *gcrui.KeyWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.KeyWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *KeyWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A selector widget to select 1 or more certificates or keys from a list.
///
/// Live search is available for quick filtering.
pub const ListSelector = extern struct {
    pub const Parent = gtk.TreeView;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Scrollable };
    pub const Class = gcrui.ListSelectorClass;
    f_parent: gtk.TreeView,
    f_pv: ?*gcrui.ListSelectorPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The collection which contains the objects to display in the selector.
        pub const collection = struct {
            pub const name = "collection";

            pub const Type = ?*gcr.Collection;
        };
    };

    pub const signals = struct {};

    /// Create a new `gcrui.ListSelector`.
    extern fn gcr_list_selector_new(p_collection: *gcr.Collection) *gcrui.ListSelector;
    pub const new = gcr_list_selector_new;

    /// Get the collection that this selector is displaying objects from.
    extern fn gcr_list_selector_get_collection(p_self: *ListSelector) *gcr.Collection;
    pub const getCollection = gcr_list_selector_get_collection;

    /// Get a list of selected objects.
    extern fn gcr_list_selector_get_selected(p_self: *ListSelector) *glib.List;
    pub const getSelected = gcr_list_selector_get_selected;

    /// Select certain objects in the selector.
    extern fn gcr_list_selector_set_selected(p_self: *ListSelector, p_selected: *glib.List) void;
    pub const setSelected = gcr_list_selector_set_selected;

    extern fn gcr_list_selector_get_type() usize;
    pub const getGObjectType = gcr_list_selector_get_type;

    extern fn g_object_ref(p_self: *gcrui.ListSelector) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.ListSelector) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ListSelector, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gcr.Prompt` implementation which shows a GTK dialog. The dialog
/// will remain visible (but insensitive) between prompts. If the user cancels
/// the dialog between prompts, then the dialog will be hidden.
pub const PromptDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gcr.Prompt, gtk.Buildable };
    pub const Class = gcrui.PromptDialogClass;
    f_parent: gtk.Dialog,
    f_pv: ?*gcrui.PromptDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the choice check box is visible or not.
        pub const choice_visible = struct {
            pub const name = "choice-visible";

            pub const Type = c_int;
        };

        /// Whether the password confirm entry is visible or not.
        pub const confirm_visible = struct {
            pub const name = "confirm-visible";

            pub const Type = c_int;
        };

        /// Whether the password entry is visible or not.
        pub const password_visible = struct {
            pub const name = "password-visible";

            pub const Type = c_int;
        };

        /// Whether the warning label is visible or not.
        pub const warning_visible = struct {
            pub const name = "warning-visible";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    extern fn gcr_prompt_dialog_get_type() usize;
    pub const getGObjectType = gcr_prompt_dialog_get_type;

    extern fn g_object_ref(p_self: *gcrui.PromptDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.PromptDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PromptDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.EntryBuffer` that uses non-pageable memory.
///
/// It's good practice to try to keep passwords or sensitive secrets out of
/// pageable memory whenever possible, so that they don't get written to disk.
///
/// A `SecureEntryBuffer` is a `gtk.EntryBuffer` to be used with
/// `gtk.Entry` which uses non-pageable memory to store a password placed
/// in the entry. In order to make any sense at all, the entry must have it's
/// visibility turned off, and just be displaying place holder characters for
/// the text. That is, a password style entry.
///
/// Use `gtk.Entry.newWithBuffer` or `gtk.Entry.setBuffer` to set this buffer
/// on an entry.
pub const SecureEntryBuffer = extern struct {
    pub const Parent = gtk.EntryBuffer;
    pub const Implements = [_]type{};
    pub const Class = gcrui.SecureEntryBufferClass;
    f_parent: gtk.EntryBuffer,
    f_pv: ?*gcrui.SecureEntryBufferPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Create a new secure entry buffer.
    extern fn gcr_secure_entry_buffer_new() *gcrui.SecureEntryBuffer;
    pub const new = gcr_secure_entry_buffer_new;

    extern fn gcr_secure_entry_buffer_get_type() usize;
    pub const getGObjectType = gcr_secure_entry_buffer_get_type;

    extern fn g_object_ref(p_self: *gcrui.SecureEntryBuffer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.SecureEntryBuffer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SecureEntryBuffer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A tree selector can be used to select certificates or keys. It allows
/// the user to select multiple objects from a tree.
pub const TreeSelector = extern struct {
    pub const Parent = gtk.TreeView;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Scrollable };
    pub const Class = gcrui.TreeSelectorClass;
    f_parent: gtk.TreeView,
    f_pv: ?*gcrui.TreeSelectorPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The collection which contains the objects to display in the selector.
        pub const collection = struct {
            pub const name = "collection";

            pub const Type = ?*gcr.Collection;
        };

        /// The columns to use to display the objects.
        pub const columns = struct {
            pub const name = "columns";

            pub const Type = ?*anyopaque;
        };
    };

    pub const signals = struct {};

    /// Create a new `gcrui.TreeSelector`.
    extern fn gcr_tree_selector_new(p_collection: *gcr.Collection, p_columns: *const gcr.Column) *gcrui.TreeSelector;
    pub const new = gcr_tree_selector_new;

    /// Get the collection that this selector is displaying objects from.
    extern fn gcr_tree_selector_get_collection(p_self: *TreeSelector) *gcr.Collection;
    pub const getCollection = gcr_tree_selector_get_collection;

    /// Get the columns displayed in a selector in multiple mode.
    extern fn gcr_tree_selector_get_columns(p_self: *TreeSelector) *const gcr.Column;
    pub const getColumns = gcr_tree_selector_get_columns;

    /// Get a list of selected objects.
    extern fn gcr_tree_selector_get_selected(p_self: *TreeSelector) *glib.List;
    pub const getSelected = gcr_tree_selector_get_selected;

    /// Select certain objects in the selector.
    extern fn gcr_tree_selector_set_selected(p_self: *TreeSelector, p_selected: *glib.List) void;
    pub const setSelected = gcr_tree_selector_set_selected;

    extern fn gcr_tree_selector_get_type() usize;
    pub const getGObjectType = gcr_tree_selector_get_type;

    extern fn g_object_ref(p_self: *gcrui.TreeSelector) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.TreeSelector) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *TreeSelector, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// This widget displays a set of unlock options for the user to select.
///
/// The user can choose between keeping caching the unlock indefinitely, or for
/// a given amount of time.
///
/// Each option has a different name, for example `GCR_UNLOCK_OPTION_ALWAYS`. These
/// names are used together with the various functions like
/// `UnlockOptionsWidget.getChoice`.
pub const UnlockOptionsWidget = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gcrui.UnlockOptionsWidgetClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const choice = struct {
            pub const name = "choice";

            pub const Type = ?[*:0]u8;
        };

        pub const ttl = struct {
            pub const name = "ttl";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Create a new `gcrui.UnlockOptionsWidget`.
    extern fn gcr_unlock_options_widget_new() *gcrui.UnlockOptionsWidget;
    pub const new = gcr_unlock_options_widget_new;

    /// Get the currently selected option, like `GCR_UNLOCK_OPTION_ALWAYS`.
    extern fn gcr_unlock_options_widget_get_choice(p_self: *UnlockOptionsWidget) [*:0]const u8;
    pub const getChoice = gcr_unlock_options_widget_get_choice;

    /// Get the label for one of the options. Use an option name like
    /// `GCR_UNLOCK_OPTION_ALWAYS`.
    extern fn gcr_unlock_options_widget_get_label(p_self: *UnlockOptionsWidget, p_option: [*:0]const u8) [*:0]const u8;
    pub const getLabel = gcr_unlock_options_widget_get_label;

    /// Get the sensitivity state for one of the options. Use an option name like
    /// `GCR_UNLOCK_OPTION_ALWAYS`.
    extern fn gcr_unlock_options_widget_get_sensitive(p_self: *UnlockOptionsWidget, p_option: [*:0]const u8) c_int;
    pub const getSensitive = gcr_unlock_options_widget_get_sensitive;

    /// Get the timeout setting set for unlock options that have a timeout.
    /// This will also return a valid value if the currently selected option
    /// does not have a timeout.
    extern fn gcr_unlock_options_widget_get_ttl(p_self: *UnlockOptionsWidget) c_uint;
    pub const getTtl = gcr_unlock_options_widget_get_ttl;

    /// Set the currently selected option. Use an option name like
    /// `GCR_UNLOCK_OPTION_ALWAYS`.
    extern fn gcr_unlock_options_widget_set_choice(p_self: *UnlockOptionsWidget, p_option: [*:0]const u8) void;
    pub const setChoice = gcr_unlock_options_widget_set_choice;

    /// Set the label for one of the options. Use an option name like
    /// `GCR_UNLOCK_OPTION_ALWAYS`.
    extern fn gcr_unlock_options_widget_set_label(p_self: *UnlockOptionsWidget, p_option: [*:0]const u8, p_text: [*:0]const u8) void;
    pub const setLabel = gcr_unlock_options_widget_set_label;

    /// Set the sensitivity state for one of the options. Use an option name like
    /// `GCR_UNLOCK_OPTION_ALWAYS`. The reason will be displayed as a tooltip.
    extern fn gcr_unlock_options_widget_set_sensitive(p_self: *UnlockOptionsWidget, p_option: [*:0]const u8, p_sensitive: c_int, p_reason: [*:0]const u8) void;
    pub const setSensitive = gcr_unlock_options_widget_set_sensitive;

    /// Set the current setting for the timeout. This can be set even when the
    /// currently selected option does not have a timeout.
    extern fn gcr_unlock_options_widget_set_ttl(p_self: *UnlockOptionsWidget, p_ttl: c_uint) void;
    pub const setTtl = gcr_unlock_options_widget_set_ttl;

    extern fn gcr_unlock_options_widget_get_type() usize;
    pub const getGObjectType = gcr_unlock_options_widget_get_type;

    extern fn g_object_ref(p_self: *gcrui.UnlockOptionsWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.UnlockOptionsWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *UnlockOptionsWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A viewer widget which can display certificates and keys that are
/// located in files.
pub const ViewerWidget = opaque {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gcrui.ViewerWidgetClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Display name for data being displayed. This is automatically
        /// calculated from a loaded file, or can be explicitly set.
        ///
        /// Used as a hint when displaying a title for the data, but may be
        /// overridden by the parsed data.
        pub const display_name = struct {
            pub const name = "display-name";

            pub const Type = ?[*:0]u8;
        };

        /// The parser used to parse loaded data into viewable items.
        pub const parser = struct {
            pub const name = "parser";

            pub const Type = ?*gcr.Parser;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when an item is added to the viewer widget.
        pub const added = struct {
            pub const name = "added";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_renderer: *gcrui.Renderer, p_parsed: *gcr.Parsed, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ViewerWidget, p_instance))),
                    gobject.signalLookup("added", ViewerWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new viewer widget.
    extern fn gcr_viewer_widget_new() *gcrui.ViewerWidget;
    pub const new = gcr_viewer_widget_new;

    /// Clear the error displayed on the viewer widget.
    extern fn gcr_viewer_widget_clear_error(p_self: *ViewerWidget) void;
    pub const clearError = gcr_viewer_widget_clear_error;

    /// Get the display name for data being displayed. This is automatically
    /// calculated from a loaded file, or can be explicitly set.
    ///
    /// Used as a hint when displaying a title for the data, but may be
    /// overridden by the parsed data.
    extern fn gcr_viewer_widget_get_display_name(p_self: *ViewerWidget) [*:0]const u8;
    pub const getDisplayName = gcr_viewer_widget_get_display_name;

    /// Get the parser used to parse loaded data into viewable items.
    extern fn gcr_viewer_widget_get_parser(p_self: *ViewerWidget) *gcr.Parser;
    pub const getParser = gcr_viewer_widget_get_parser;

    /// Get the viewer used to display the viewable items.
    extern fn gcr_viewer_widget_get_viewer(p_self: *ViewerWidget) *gcrui.Viewer;
    pub const getViewer = gcr_viewer_widget_get_viewer;

    /// Parse and load some data to be displayed into the viewer widgets. The data
    /// may contain multiple parseable items if the format can contain multiple
    /// items.
    extern fn gcr_viewer_widget_load_bytes(p_self: *ViewerWidget, p_display_name: ?[*:0]const u8, p_data: *glib.Bytes) void;
    pub const loadBytes = gcr_viewer_widget_load_bytes;

    /// Parse and load some data to be displayed into the viewer widgets. The data
    /// may contain multiple parseable items if the format can contain multiple
    /// items.
    ///
    /// This function will copy the data. Use `ViewerWidget.loadBytes` to avoid
    /// copying the data.
    extern fn gcr_viewer_widget_load_data(p_self: *ViewerWidget, p_display_name: ?[*:0]const u8, p_data: [*]const u8, p_n_data: usize) void;
    pub const loadData = gcr_viewer_widget_load_data;

    /// Display contents of a file in the viewer widget. Multiple files can
    /// be loaded.
    extern fn gcr_viewer_widget_load_file(p_self: *ViewerWidget, p_file: *gio.File) void;
    pub const loadFile = gcr_viewer_widget_load_file;

    /// Set the display name for data being displayed. Once explicitly
    /// set it will no longer be calculated automatically by loading data.
    ///
    /// Used as a hint when displaying a title for the data, but may be
    /// overridden by the parsed data.
    extern fn gcr_viewer_widget_set_display_name(p_self: *ViewerWidget, p_display_name: [*:0]const u8) void;
    pub const setDisplayName = gcr_viewer_widget_set_display_name;

    /// Show an error on the viewer widget. This is displayed on a info bar near
    /// the edge of the widget.
    extern fn gcr_viewer_widget_show_error(p_self: *ViewerWidget, p_message: [*:0]const u8, p_error: ?*glib.Error) void;
    pub const showError = gcr_viewer_widget_show_error;

    extern fn gcr_viewer_widget_get_type() usize;
    pub const getGObjectType = gcr_viewer_widget_get_type;

    extern fn g_object_ref(p_self: *gcrui.ViewerWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.ViewerWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ViewerWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An interface that's implemented by renderers which wish to render data to a
/// `Viewer`.
///
/// The interaction between `Renderer` and `Viewer` is not stable
/// yet, and so new renderers cannot be implemented outside the Gcr library at
/// this time.
///
/// To lookup a renderer for a given set of attributes, use the `gcrui.rendererCreate`
/// function. This will create and initialize a renderer that's capable of viewing
/// the data in those attributes.
pub const Renderer = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gcrui.RendererIface;
    pub const virtual_methods = struct {
        /// signal emitted when data being rendered changes
        pub const data_changed = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Renderer.Iface, p_class).f_data_changed.?(gobject.ext.as(Renderer, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Renderer.Iface, p_class).f_data_changed = @ptrCast(p_implementation);
            }
        };

        /// method invoked to populate a popup menu with additional
        ///                  renderer options
        pub const populate_popup = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_viewer: *gcrui.Viewer, p_menu: *gtk.Menu) void {
                return gobject.ext.as(Renderer.Iface, p_class).f_populate_popup.?(gobject.ext.as(Renderer, p_self), p_viewer, p_menu);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_viewer: *gcrui.Viewer, p_menu: *gtk.Menu) callconv(.C) void) void {
                gobject.ext.as(Renderer.Iface, p_class).f_populate_popup = @ptrCast(p_implementation);
            }
        };

        /// Render the contents of the renderer to the given viewer.
        pub const render_view = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_viewer: *gcrui.Viewer) void {
                return gobject.ext.as(Renderer.Iface, p_class).f_render_view.?(gobject.ext.as(Renderer, p_self), p_viewer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_viewer: *gcrui.Viewer) callconv(.C) void) void {
                gobject.ext.as(Renderer.Iface, p_class).f_render_view = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The attributes to display.
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*gck.Attributes;
        };

        /// The label to display.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// A signal that is emitted by the renderer when it's data
        /// changed and should be rerendered.
        pub const data_changed = struct {
            pub const name = "data-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Renderer, p_instance))),
                    gobject.signalLookup("data-changed", Renderer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create and initialize a renderer for the given attributes and label. These
    /// renderers should have been preregistered via `gcrui.rendererRegister`.
    extern fn gcr_renderer_create(p_label: ?[*:0]const u8, p_attrs: *gck.Attributes) ?*gcrui.Renderer;
    pub const create = gcr_renderer_create;

    /// Register a renderer to be created when matching attributes are passed to
    /// `gcrui.rendererCreate`.
    extern fn gcr_renderer_register(p_renderer_type: usize, p_attrs: *gck.Attributes) void;
    pub const register = gcr_renderer_register;

    /// Register all the well known renderers for certificates and keys known to the
    /// Gcr library.
    extern fn gcr_renderer_register_well_known() void;
    pub const registerWellKnown = gcr_renderer_register_well_known;

    /// Emit the `gcrui.Renderer.signals.data`-changed signal on the renderer. This is used by
    /// renderer implementations.
    extern fn gcr_renderer_emit_data_changed(p_self: *Renderer) void;
    pub const emitDataChanged = gcr_renderer_emit_data_changed;

    /// Get the PKCS`@"11"` attributes, if any, set for this renderer to display.
    extern fn gcr_renderer_get_attributes(p_self: *Renderer) ?*gck.Attributes;
    pub const getAttributes = gcr_renderer_get_attributes;

    /// Called by `gcrui.Viewer` when about to display a popup menu for the content
    /// displayed by the renderer. The renderer can add a menu item if desired.
    extern fn gcr_renderer_popuplate_popup(p_self: *Renderer, p_viewer: *gcrui.Viewer, p_menu: *gtk.Menu) void;
    pub const popuplatePopup = gcr_renderer_popuplate_popup;

    /// Render the contents of the renderer to the given viewer.
    extern fn gcr_renderer_render_view(p_self: *Renderer, p_viewer: *gcrui.Viewer) void;
    pub const renderView = gcr_renderer_render_view;

    /// Set the PKCS`@"11"` attributes for this renderer to display.
    extern fn gcr_renderer_set_attributes(p_self: *Renderer, p_attrs: ?*gck.Attributes) void;
    pub const setAttributes = gcr_renderer_set_attributes;

    extern fn gcr_renderer_get_type() usize;
    pub const getGObjectType = gcr_renderer_get_type;

    extern fn g_object_ref(p_self: *gcrui.Renderer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.Renderer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Renderer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An abstract interface that represents a widget that can hold
/// various renderers and display their contents.
///
/// The interaction between `Renderer` and `Viewer` is not stable
/// yet, and so viewers cannot be implemented outside the Gcr library at this
/// time.
///
/// Use the `Viewer.new` and `Viewer.newScrolled` to get default
/// implementations of viewers.
pub const Viewer = opaque {
    pub const Prerequisites = [_]type{gtk.Widget};
    pub const Iface = gcrui.ViewerIface;
    pub const virtual_methods = struct {
        /// Add a renderer to this viewer.
        pub const add_renderer = struct {
            pub fn call(p_class: anytype, p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gcrui.Renderer) void {
                return gobject.ext.as(Viewer.Iface, p_class).f_add_renderer.?(gobject.ext.as(Viewer, p_viewer), p_renderer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gcrui.Renderer) callconv(.C) void) void {
                gobject.ext.as(Viewer.Iface, p_class).f_add_renderer = @ptrCast(p_implementation);
            }
        };

        /// Get the number of renderers present in the viewer.
        pub const count_renderers = struct {
            pub fn call(p_class: anytype, p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_uint {
                return gobject.ext.as(Viewer.Iface, p_class).f_count_renderers.?(gobject.ext.as(Viewer, p_viewer));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_uint) void {
                gobject.ext.as(Viewer.Iface, p_class).f_count_renderers = @ptrCast(p_implementation);
            }
        };

        /// Get a pointer to the renderer at the given index. It is an error to request
        /// an index that is out of bounds.
        pub const get_renderer = struct {
            pub fn call(p_class: anytype, p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index_: c_uint) *gcrui.Renderer {
                return gobject.ext.as(Viewer.Iface, p_class).f_get_renderer.?(gobject.ext.as(Viewer, p_viewer), p_index_);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index_: c_uint) callconv(.C) *gcrui.Renderer) void {
                gobject.ext.as(Viewer.Iface, p_class).f_get_renderer = @ptrCast(p_implementation);
            }
        };

        /// Insert a renderer at a specific point in the viewer
        pub const insert_renderer = struct {
            pub fn call(p_class: anytype, p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gcrui.Renderer, p_before: ?*gcrui.Renderer) void {
                return gobject.ext.as(Viewer.Iface, p_class).f_insert_renderer.?(gobject.ext.as(Viewer, p_viewer), p_renderer, p_before);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gcrui.Renderer, p_before: ?*gcrui.Renderer) callconv(.C) void) void {
                gobject.ext.as(Viewer.Iface, p_class).f_insert_renderer = @ptrCast(p_implementation);
            }
        };

        /// Remove a renderer from this viewer.
        pub const remove_renderer = struct {
            pub fn call(p_class: anytype, p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gcrui.Renderer) void {
                return gobject.ext.as(Viewer.Iface, p_class).f_remove_renderer.?(gobject.ext.as(Viewer, p_viewer), p_renderer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_viewer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gcrui.Renderer) callconv(.C) void) void {
                gobject.ext.as(Viewer.Iface, p_class).f_remove_renderer = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Get an implementation of `gcrui.Viewer` that supports a view
    /// of multiple renderers.
    extern fn gcr_viewer_new() *gcrui.Viewer;
    pub const new = gcr_viewer_new;

    /// Get an implementation of `gcrui.Viewer` that supports a scrolled view
    /// of multiple renderers.
    extern fn gcr_viewer_new_scrolled() *gcrui.Viewer;
    pub const newScrolled = gcr_viewer_new_scrolled;

    /// Add a renderer to this viewer.
    extern fn gcr_viewer_add_renderer(p_viewer: *Viewer, p_renderer: *gcrui.Renderer) void;
    pub const addRenderer = gcr_viewer_add_renderer;

    /// Get the number of renderers present in the viewer.
    extern fn gcr_viewer_count_renderers(p_viewer: *Viewer) c_uint;
    pub const countRenderers = gcr_viewer_count_renderers;

    /// Get a pointer to the renderer at the given index. It is an error to request
    /// an index that is out of bounds.
    extern fn gcr_viewer_get_renderer(p_viewer: *Viewer, p_index_: c_uint) *gcrui.Renderer;
    pub const getRenderer = gcr_viewer_get_renderer;

    /// Insert a renderer at a specific point in the viewer
    extern fn gcr_viewer_insert_renderer(p_viewer: *Viewer, p_renderer: *gcrui.Renderer, p_before: ?*gcrui.Renderer) void;
    pub const insertRenderer = gcr_viewer_insert_renderer;

    /// Remove a renderer from this viewer.
    extern fn gcr_viewer_remove_renderer(p_viewer: *Viewer, p_renderer: *gcrui.Renderer) void;
    pub const removeRenderer = gcr_viewer_remove_renderer;

    extern fn gcr_viewer_get_type() usize;
    pub const getGObjectType = gcr_viewer_get_type;

    extern fn g_object_ref(p_self: *gcrui.Viewer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcrui.Viewer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Viewer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The class for `gcrui.CertificateRenderer`.
pub const CertificateRendererClass = extern struct {
    pub const Instance = gcrui.CertificateRenderer;

    /// The parent class.
    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CertificateRendererClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateRendererPrivate = opaque {};

pub const CertificateWidgetClass = extern struct {
    pub const Instance = gcrui.CertificateWidget;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *CertificateWidgetClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateWidgetPrivate = opaque {};

/// The class for `gcrui.CollectionModel`.
pub const CollectionModelClass = extern struct {
    pub const Instance = gcrui.CollectionModel;

    /// The parent class
    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CollectionModelClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CollectionModelPrivate = opaque {};

pub const ComboSelectorClass = extern struct {
    pub const Instance = gcrui.ComboSelector;

    f_parent_class: gtk.ComboBoxClass,

    pub fn as(p_instance: *ComboSelectorClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ComboSelectorPrivate = opaque {};

pub const FailureRendererClass = extern struct {
    pub const Instance = gcrui.FailureRenderer;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *FailureRendererClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FailureRendererPrivate = opaque {};

/// Class struct for `ImportButton`.
pub const ImportButtonClass = extern struct {
    pub const Instance = gcrui.ImportButton;

    /// The parent class
    f_parent_class: gtk.ButtonClass,
    /// Emitted when the import begins.
    f_importing: ?*const fn (p_self: *gcrui.ImportButton, p_importer: *gcr.Importer) callconv(.C) void,
    /// Emitted when the import completes, or fails.
    f_imported: ?*const fn (p_self: *gcrui.ImportButton, p_importer: *gcr.Importer, p_error: *glib.Error) callconv(.C) void,
    f_padding: [10]*anyopaque,

    pub fn as(p_instance: *ImportButtonClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ImportButtonPrivate = opaque {};

/// The class for `gcrui.KeyRenderer`.
pub const KeyRendererClass = extern struct {
    pub const Instance = gcrui.KeyRenderer;

    /// The parent class.
    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *KeyRendererClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const KeyRendererPrivate = opaque {};

pub const KeyWidgetClass = opaque {
    pub const Instance = gcrui.KeyWidget;

    pub fn as(p_instance: *KeyWidgetClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const KeyWidgetPrivate = opaque {};

pub const ListSelectorClass = extern struct {
    pub const Instance = gcrui.ListSelector;

    f_parent_class: gtk.TreeViewClass,

    pub fn as(p_instance: *ListSelectorClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ListSelectorPrivate = opaque {};

pub const PromptDialogClass = extern struct {
    pub const Instance = gcrui.PromptDialog;

    f_parent_class: gtk.DialogClass,

    pub fn as(p_instance: *PromptDialogClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PromptDialogPrivate = opaque {};

/// The interface for `gcrui.Renderer`
pub const RendererIface = extern struct {
    pub const Instance = gcrui.Renderer;

    /// the parent interface type
    f_parent: gobject.TypeInterface,
    /// signal emitted when data being rendered changes
    f_data_changed: ?*const fn (p_self: *gcrui.Renderer) callconv(.C) void,
    /// method invoked to render the data into a viewer
    f_render_view: ?*const fn (p_self: *gcrui.Renderer, p_viewer: *gcrui.Viewer) callconv(.C) void,
    /// method invoked to populate a popup menu with additional
    ///                  renderer options
    f_populate_popup: ?*const fn (p_self: *gcrui.Renderer, p_viewer: *gcrui.Viewer, p_menu: *gtk.Menu) callconv(.C) void,
    f_dummy1: ?*anyopaque,
    f_dummy2: ?*anyopaque,
    f_dummy3: ?*anyopaque,
    f_dummy4: ?*anyopaque,
    f_dummy5: ?*anyopaque,
    f_dummy6: ?*anyopaque,
    f_dummy7: ?*anyopaque,

    pub fn as(p_instance: *RendererIface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SecureEntryBufferClass = extern struct {
    pub const Instance = gcrui.SecureEntryBuffer;

    f_parent_class: gtk.EntryBufferClass,

    pub fn as(p_instance: *SecureEntryBufferClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SecureEntryBufferPrivate = opaque {};

pub const TreeSelectorClass = extern struct {
    pub const Instance = gcrui.TreeSelector;

    f_parent_class: gtk.TreeViewClass,

    pub fn as(p_instance: *TreeSelectorClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const TreeSelectorPrivate = opaque {};

pub const UnlockOptionsWidgetClass = opaque {
    pub const Instance = gcrui.UnlockOptionsWidget;

    pub fn as(p_instance: *UnlockOptionsWidgetClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const UnlockOptionsWidgetPrivate = opaque {};

/// The interface for `gcrui.Viewer`
pub const ViewerIface = extern struct {
    pub const Instance = gcrui.Viewer;

    /// The parent interface
    f_parent: gobject.TypeInterface,
    /// Virtual method to add a renderer
    f_add_renderer: ?*const fn (p_viewer: *gcrui.Viewer, p_renderer: *gcrui.Renderer) callconv(.C) void,
    /// Virtual method to insert a renderer
    f_insert_renderer: ?*const fn (p_viewer: *gcrui.Viewer, p_renderer: *gcrui.Renderer, p_before: ?*gcrui.Renderer) callconv(.C) void,
    /// Virtual method to remove a renderer
    f_remove_renderer: ?*const fn (p_viewer: *gcrui.Viewer, p_renderer: *gcrui.Renderer) callconv(.C) void,
    /// Virtual method to count renderers
    f_count_renderers: ?*const fn (p_viewer: *gcrui.Viewer) callconv(.C) c_uint,
    /// Virtual method to get a renderer
    f_get_renderer: ?*const fn (p_viewer: *gcrui.Viewer, p_index_: c_uint) callconv(.C) *gcrui.Renderer,
    f_dummy1: ?*anyopaque,
    f_dummy2: ?*anyopaque,
    f_dummy3: ?*anyopaque,
    f_dummy4: ?*anyopaque,

    pub fn as(p_instance: *ViewerIface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ViewerWidgetClass = extern struct {
    pub const Instance = gcrui.ViewerWidget;

    f_parent_class: gtk.BoxClass,

    pub fn as(p_instance: *ViewerWidgetClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// If a `CollectionModel` is created with a mode of
/// `GCR_COLLECTION_MODEL_TREE`, then any included objects that are themselves a
/// `gcr.Collection`, will have all child
/// objects include as child rows in a tree form.
pub const CollectionModelMode = enum(c_int) {
    list = 0,
    tree = 1,
    _,
};
