/* wmpasman
 * Copyright © 1999-2014  Brad Jorsch <anomie@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "wmpasman.h"
#include "die.h"
#include "dock.h"
#include "clipboard.h"
#include "keyring.h"
#include "popup_editpassword.h"

#include "letters.xpm"

/* Globals */
static GdkDisplay *display;
GtkWidget *mainwin, *iconwin;
static GdkPixbuf *letters_pixbuf;
static GtkIconTheme *theme;
static GdkRGBA fgcolor;
int clock_flags;
gboolean nonwmaker = FALSE;

static char weekdays[7][3] = { "SU", "MO", "TU", "WE", "TH", "FR", "SA" };
static char months[12][4] = { "JAN", "FEB", "MAR", "APR", "MAY", "JUN", "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" };
static char dweekdays[5][3] = { "SM", "BT", "PD", "PP", "SO" };
static char dmonths[5][4] = { "CHS", "DSC", "CFN", "BCY", "AFM" };

static GtkMenu *menu;
static GtkWidget *menu_newpass, *menu_delpass, *menu_editpass, *menu_lockkeyring, *menu_delkeyring;

/* Utility funcs */
static int draw_char(cairo_t *cr, char c, int x, int y) {
    int sx, w = 5, h = 7;
    switch (c) {
      case 127:
        return 5;

      case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G': case 'H': case 'I': case 'J': case 'K': case 'L': case 'M':
      case 'N': case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U': case 'V': case 'W': case 'X': case 'Y': case 'Z':
        sx = (c - 'A') * 6 + 102;
        break;

      case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7': case '8': case '9':
        sx = (c - '0') * 6 + 30;
        break;

      case ' ':
        sx = 24;
        break;

      case '@':
        sx = 90;
        break;

      case 'z':
        sx = 96; w = 3;
        break;

      case '\'':
        sx = 100; w = 1;
        break;

      case '-':
        sx = 16; w = 3;
        break;

      case ':':
        sx = 20; w = 1;
        break;

      case '.':
        sx = 22; w = 1;
        break;

      case 1: case 2: case 3: case 4:
        sx = (c - 1) * 4; w = 4; h = 4;
        break;

      case 126:
        sx = 258;
        break;

      default:
        warn(DEBUG_WARN, "Invalid character code %d passed to draw_char", c);
        return -1;
    }

    cairo_save(cr);
    cairo_rectangle(cr, x, y, w, h);
    cairo_clip(cr);
    gdk_cairo_set_source_pixbuf(cr, letters_pixbuf, x - sx, y);
    cairo_paint(cr);
    cairo_restore(cr);

    return w;
}

static void draw_window(cairo_t *cr, cairo_region_t *region, int x, int y, int w, int h) {
    cairo_rectangle_int_t rect = { x, y, w, h };
    cairo_region_union_rectangle(region, &rect);

    cairo_set_source_rgb(cr, 0, 0, 0);
    cairo_rectangle(cr, x, y, w, h);
    cairo_fill(cr);

    cairo_set_line_width(cr, 1);
    cairo_set_source_rgb(cr, 0.78, 0.78, 0.78);
    cairo_move_to(cr, x + w - .5, y + 1);
    cairo_line_to(cr, x + w - .5, y + h - .5);
    cairo_line_to(cr, x, y + h - .5);
    cairo_stroke(cr);

    cairo_set_source_rgb(cr, 0, 0, 0);
    cairo_move_to(cr, x + .5, y + h - 1);
    cairo_line_to(cr, x + .5, y + .5);
    cairo_line_to(cr, x + w, y + .5);
    cairo_stroke(cr);
}

static void draw_clock(cairo_t *cr) {
    struct tm *tm = NULL;
    struct timeval tv;
    char tmp[10];

    gettimeofday(&tv, NULL);
    if (clock_flags & CLOCK_BEATS) {
        if (gmtime(&tv.tv_sec)->tm_isdst <= 0) {
            tv.tv_sec+=3600;
        }
        tm = gmtime(&tv.tv_sec);

        float b = (tm->tm_hour*3600000 + tm->tm_min*60000 + tm->tm_sec*1000 + (float)tv.tv_usec/1000)/86400;
        g_snprintf(tmp, 10, "%07.3f ", b);
        draw_char(cr, '@', 6, 51);
        draw_char(cr, tmp[0], 12, 51);
        draw_char(cr, tmp[1], 18, 51);
        draw_char(cr, tmp[2], 24, 51);
        draw_char(cr, '.', 30, 51);
        draw_char(cr, tmp[4], 32, 51);
        draw_char(cr, tmp[5], 38, 51);
        draw_char(cr, tmp[6], 44, 51);
    } else {
        if (clock_flags & CLOCK_UTC) {
            tm = gmtime(&tv.tv_sec);
            draw_char(cr, 'z', 46, 51);
        } else {
            tm = localtime(&tv.tv_sec);
        }

        if (clock_flags & CLOCK_24HR) {
            draw_char(cr, 'H', 52, 51);
            g_snprintf(tmp, 10, "%02d%02d%02d", tm->tm_hour, tm->tm_min, tm->tm_sec);
        } else {
            if (tm->tm_hour > 11) {
                tm->tm_hour -= 12;
                draw_char(cr, 'P', 52, 51);
            } else {
                draw_char(cr, 'A', 52, 51);
            }
            if (tm->tm_hour == 0) {
                tm->tm_hour = 12;
            }
            g_snprintf(tmp, 10, "%2d%02d%02d", tm->tm_hour, tm->tm_min, tm->tm_sec);
        }
        draw_char(cr, tmp[0], 6, 51);
        draw_char(cr, tmp[1], 12, 51);
        draw_char(cr, ':', 18, 51);
        draw_char(cr, tmp[2], 20, 51);
        draw_char(cr, tmp[3], 26, 51);
        draw_char(cr, ':', 32, 51);
        draw_char(cr, tmp[4], 34, 51);
        draw_char(cr, tmp[5], 40, 51);
    }

    if (clock_flags & CLOCK_DISCORDIAN) {
        int i = tm->tm_yday;
        gboolean st_tib = FALSE;
        if (tm->tm_year % 4 == 0 && (tm->tm_year % 100 != 0 || tm->tm_year % 400 == 100)) {
            st_tib = (i == 59);
            if (i > 59) {
                --i;
            }
        }
        if (st_tib) {
            g_stpcpy(tmp, "ST.\x7fTIB'S");
        } else {
            g_snprintf(tmp, 10, "%.2s\x7f%02d-%.3s", dweekdays[i%5], (i%73) + 1, dmonths[i/73]);
        }
    } else {
        g_snprintf(tmp, 10, "%.2s\x7f%02d-%.3s", weekdays[tm->tm_wday], tm->tm_mday, months[tm->tm_mon]);
    }
    int x = 6;
    for (int i = 0; tmp[i]; i++) {
        x += draw_char(cr, tmp[i], x, 42) + 1;
    }
}

static gboolean clock_redraw(gpointer p G_GNUC_UNUSED) {
    static time_t t = 0;
    time_t cur_t = time(NULL);

    if (t != cur_t || (clock_flags & CLOCK_BEATS)) {
        // Schedule a redraw for the clock
        gtk_widget_queue_draw_area(iconwin, 5, 41, 54, 18);
        gtk_widget_queue_draw_area(mainwin, 5, 41, 54, 18);
        t = cur_t;
    }

    return TRUE;
}

/* Popup menu */
static void menu_newpass_cb(GtkMenuItem *i G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED) {
    app_deactivate();
    if (appstate == INACTIVE || appstate == NOPASS) {
        do_edit_password(GTK_WINDOW(iconwin), NULL, NULL, NULL, NULL);
    }
}

static void menu_delpass_cb2(GError *err) {
    if (err) {
        error_alert("Delete failed: %s", err);
    }
    app_finish_work();
}

static void menu_delpass_cb(GtkMenuItem *i G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED) {
    app_deactivate();
    if (appstate == INACTIVE) {
        GCancellable *cancel = app_start_work();
        password_delete(cancel, menu_delpass_cb2);
    }
}

static void menu_editpass_cb2(GObject *pw, const gchar *line1, const gchar *line2, SecretValue *password, GError *err) {
    app_finish_work();
    if (pw) {
        do_edit_password(GTK_WINDOW(iconwin), pw, line1, line2, password);
    } else {
        error_alert("Failed to fetch password for edit: %s", err);
    }
}

static void menu_editpass_cb(GtkMenuItem *i G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED) {
    app_deactivate();
    if (appstate == INACTIVE) {
        GCancellable *cancel = app_start_work();
        password_get_edit_data(cancel, menu_editpass_cb2);
    }
}

static void menu_lockkeyring_cb(GtkMenuItem *i G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED) {
    app_lock();
}

static void menu_delkeyring_confirm_cb(GtkWidget *dialog, gint response, gpointer d G_GNUC_UNUSED) {
    gtk_widget_destroy(dialog);

    if (response == GTK_RESPONSE_YES) {
        GCancellable *cancel = app_start_work();
        keyring_delete(cancel, menu_delpass_cb2);
    }
}

static void menu_delkeyring_cb(GtkMenuItem *i G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED) {
    if (appstate == STARTING || appstate == LOCKED) {
        return;
    }
    app_deactivate();
    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(iconwin),
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_WARNING,
        GTK_BUTTONS_YES_NO,
        "Deleting all passwords cannot be undone!"
    );
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_NO);
    gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog), "Really do this?");
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(menu_delkeyring_confirm_cb), dialog);
    gtk_widget_show_all(dialog);
}

void init_menu(void) {
    menu = GTK_MENU(gtk_menu_new());

    menu_newpass = gtk_menu_item_new_with_label("Add password");
    gtk_menu_shell_append((GtkMenuShell *)menu, menu_newpass);
    g_signal_connect(menu_newpass, "activate", G_CALLBACK(menu_newpass_cb), NULL);
    gtk_widget_set_sensitive(menu_newpass, FALSE);

    menu_delpass = gtk_menu_item_new_with_label("Delete current password");
    gtk_menu_shell_append((GtkMenuShell *)menu, menu_delpass);
    g_signal_connect(menu_delpass, "activate", G_CALLBACK(menu_delpass_cb), NULL);
    gtk_widget_set_sensitive(menu_delpass, FALSE);

    menu_editpass = gtk_menu_item_new_with_label("Edit current password");
    gtk_menu_shell_append((GtkMenuShell *)menu, menu_editpass);
    g_signal_connect(menu_editpass, "activate", G_CALLBACK(menu_editpass_cb), NULL);
    gtk_widget_set_sensitive(menu_editpass, FALSE);

    menu_lockkeyring = gtk_menu_item_new_with_label("Lock keyring");
    gtk_menu_shell_append((GtkMenuShell *)menu, menu_lockkeyring);
    g_signal_connect(menu_lockkeyring, "activate", G_CALLBACK(menu_lockkeyring_cb), NULL);
    gtk_widget_set_sensitive(menu_lockkeyring, FALSE);

    GtkWidget *sep = gtk_separator_menu_item_new();
    gtk_menu_shell_append((GtkMenuShell *)menu, sep);

    menu_delkeyring = gtk_menu_item_new_with_label("Delete keyring");
    gtk_menu_shell_append((GtkMenuShell *)menu, menu_delkeyring);
    g_signal_connect(menu_delkeyring, "activate", G_CALLBACK(menu_delkeyring_cb), NULL);
    gtk_widget_set_sensitive(menu_delkeyring, FALSE);

    gtk_widget_show_all((GtkWidget *)menu);
}

/* GTK event handlers */

static gboolean draw_handler(GtkWidget *widget, cairo_t *cr, gpointer d G_GNUC_UNUSED) {
    GdkRectangle rect;
    gdk_cairo_get_clip_rectangle(cr, &rect);
    if ( rect.x >= 5 && rect.y >= 40 && rect.x + rect.width <= 59 && rect.y + rect.height <= 59 ) {
        cairo_set_source_rgb(cr, 0, 0, 0);
        cairo_rectangle(cr, rect.x, rect.y, rect.width, rect.height);
        cairo_fill(cr);
        draw_clock(cr);
        return FALSE;
    }

    cairo_region_t *region = cairo_region_create();
    draw_window(cr, region, 4, 4, 40, 12);
    draw_window(cr, region, 46, 4, 14, 12);
    draw_window(cr, region, 4, 17, 56, 22);
    draw_window(cr, region, 4, 40, 56, 20);
    gtk_widget_shape_combine_region(widget, region);
    cairo_region_destroy(region);

    // Name indicator
    if (appstate != STARTING) {
        draw_char(cr, 'P', 6, 6);
        draw_char(cr, 'A', 12, 6);
        draw_char(cr, 'S', 18, 6);
        draw_char(cr, 'M', 24, 6);
        draw_char(cr, 'A', 30, 6);
        draw_char(cr, 'N', 36, 6);
    }

    // Application state
    char icon = 1;
    switch (appstate) {
      case STARTING: icon = 1; break;
      case LOCKED: icon = 1; break;
      case NOPASS: icon = 1; break;
      case INACTIVE: icon = 2; break;
      case WORKING: icon = 3; break;
      case ACTIVE: icon = 4; break;
    }
    draw_char(cr, icon, 51, 8);

    if (appstate == LOCKED) {
        draw_char(cr, 126, 29, 23);
    }

    // Draw current password name
    if (current_password || appstate == NOPASS) {
        int fw, fh;
        cairo_save(cr);
        cairo_rectangle(cr, 5, 18, 54, 20);
        cairo_clip(cr);

        PangoContext *ctx = pango_cairo_create_context(cr);
        static cairo_font_options_t *opt = NULL;
        if (!opt) {
            opt = cairo_font_options_create();
            cairo_font_options_set_antialias(opt, CAIRO_ANTIALIAS_GOOD);
            cairo_font_options_set_hint_style(opt, CAIRO_HINT_STYLE_FULL);
        }
        pango_cairo_context_set_font_options(ctx, opt);
        PangoLayout *layout = pango_layout_new(ctx);

        static PangoFontDescription *font = NULL;
        if (!font) {
            font = pango_font_description_new();
            pango_font_description_set_family_static(font, "Sans");
            pango_layout_set_text(layout, "Fy", -1);
            int min = 6 * PANGO_SCALE, max = 20 * PANGO_SCALE;
            while (min + 1 < max) {
                int avg = (min + max) / 2;
                int fh;
                pango_font_description_set_absolute_size(font, avg);
                pango_layout_set_font_description(layout, font);
                pango_layout_get_pixel_size(layout, NULL, &fh);
                if (fh <= 10) {
                    min = avg;
                } else {
                    max = avg;
                }
            }
            pango_font_description_set_absolute_size(font, min);
        }
        pango_layout_set_font_description(layout, font);

        if (current_password) {
            gdk_cairo_set_source_rgba(cr, &fgcolor);

            pango_layout_set_text(layout, current_password->line1, -1);
            pango_layout_get_pixel_size(layout, &fw, &fh);
            cairo_move_to(cr, MAX(5, 32 - fw / 2), 18);
            pango_cairo_show_layout(cr, layout);

            pango_layout_set_text(layout, current_password->line2, -1);
            pango_layout_get_pixel_size(layout, &fw, &fh);
            cairo_move_to(cr, MAX(5, 32 - fw / 2), 38 - fh);
            pango_cairo_show_layout(cr, layout);
        } else {
            cairo_set_source_rgb(cr, 1, 0.5, 0);

            pango_layout_set_text(layout, "Keyring", -1);
            pango_layout_get_pixel_size(layout, &fw, &fh);
            cairo_move_to(cr, MAX(5, 32 - fw / 2), 18);
            pango_cairo_show_layout(cr, layout);

            pango_layout_set_text(layout, "Empty", -1);
            pango_layout_get_pixel_size(layout, &fw, &fh);
            cairo_move_to(cr, MAX(5, 32 - fw / 2), 38 - fh);
            pango_cairo_show_layout(cr, layout);
        }

        g_object_unref(layout);
        g_object_unref(ctx);
        cairo_restore(cr);
    }

    draw_clock(cr);

    return FALSE;
}

static gboolean click_handler(GtkWidget *w G_GNUC_UNUSED, GdkEventButton *ev, gpointer d G_GNUC_UNUSED) {
    int x = ev->x, y = ev->y;

    if (x >= 4 && x <= 44 && y >= 4 && y <= 16) {
        if (appstate == STARTING) {
            return TRUE;
        }

        if (appstate == LOCKED) {
            app_unlock();
            return TRUE;
        }

        gtk_menu_attach_to_widget(menu, w, NULL);
        gtk_menu_popup(menu, NULL, NULL, NULL, NULL, ev->button, ev->time);
    } else if (x >= 46 && x <= 60 && y >= 4 && y <= 16 && ev->button <= 3) {
        if (appstate == STARTING) {
            return TRUE;
        }

        switch (ev->button) {
          case 1:
            if (appstate == WORKING) {
                app_cancel_work();
            } else if (appstate == LOCKED) {
                app_unlock();
            } else {
                app_activate();
            }
            break;
          case 2:
            if (appstate == LOCKED) {
                app_unlock();
            } else {
                app_deactivate();
            }
            break;
          case 3:
            app_lock();
            break;
        }
        return TRUE;
    } else if (x >= 4 && x <= 60 && y >= 18 && y <= 37) {
        if (appstate == STARTING) {
            return TRUE;
        }

        if (appstate == LOCKED) {
            app_unlock();
            return TRUE;
        }

        if (appstate == NOPASS) {
            do_edit_password(GTK_WINDOW(iconwin), NULL, NULL, NULL, NULL);
            return TRUE;
        }

        if (ev->button == 1) {
            password_next();
            redraw_dock();
        } else if (ev->button == 3) {
            password_prev();
            redraw_dock();
        }
        return TRUE;
    } else if (x >= 4 && x <= 60 && y >= 39 && y <= 60) {
        exec_command(ev->button);
        return TRUE;
    }

    return FALSE;
}

static gboolean scroll_handler(GtkWidget *w G_GNUC_UNUSED, GdkEventScroll *ev, gpointer d G_GNUC_UNUSED) {
    int x = ev->x, y = ev->y;

    if (ev->type != GDK_SCROLL) {
        return FALSE;
    }

    if (x >= 4 && x <= 60 && y >= 18 && y <= 37) {
        if (appstate == STARTING || appstate == LOCKED) {
            return FALSE;
        }

        switch (ev->direction) {
          case GDK_SCROLL_UP:
          case GDK_SCROLL_LEFT:
            password_prev();
            redraw_dock();
            break;
          case GDK_SCROLL_DOWN:
          case GDK_SCROLL_RIGHT:
            password_next();
            redraw_dock();
            break;
          default:
            break;
        }
    } else if (x >= 4 && x <= 60 && y >= 39 && y <= 60) {
        switch (ev->direction) {
          case GDK_SCROLL_UP:
            exec_command(4);
            break;
          case GDK_SCROLL_DOWN:
            exec_command(5);
            break;
          case GDK_SCROLL_LEFT:
            exec_command(6);
            break;
          case GDK_SCROLL_RIGHT:
            exec_command(7);
            break;
          default:
            break;
        }
    }

    return FALSE;
}


static gboolean query_tooltip_handler(GtkWidget *widget G_GNUC_UNUSED, gint x, gint y, gboolean keyboard_mode G_GNUC_UNUSED, GtkTooltip *tt, gpointer data G_GNUC_UNUSED) {
    static const GdkRectangle area = { .x = 4, .y = 18, .width = 56, .height = 19 };

    if (!current_password) {
        return FALSE;
    }

    if (x < area.x || x > area.x + area.width || y < area.y || y > area.y + area.height ) {
        return FALSE;
    }

    gtk_tooltip_set_tip_area(tt, &area);
    gchar *text = g_strjoin("\n", current_password->line1, current_password->line2, NULL);
    gtk_tooltip_set_text(tt, text);
    g_free(text);
    return TRUE;
}

/* Functions */

void redraw_dock(void) {
    gdk_window_invalidate_rect(gtk_widget_get_window(iconwin), NULL, FALSE);
    gdk_window_invalidate_rect(gtk_widget_get_window(mainwin), NULL, FALSE);
    gtk_tooltip_trigger_tooltip_query(display);
}

static void init_window(GtkWidget *win) {
    gtk_window_set_wmclass(GTK_WINDOW(win), g_get_prgname(), "DockApp");
    gtk_widget_set_size_request(win, 64, 64);
    gtk_window_set_resizable(GTK_WINDOW(win), FALSE);
    gtk_widget_set_app_paintable(win, TRUE);
    gtk_widget_add_events(win, GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_EXPOSURE_MASK | GDK_SCROLL_MASK);
    g_object_set(win, "has-tooltip", TRUE, NULL);
    g_signal_connect(G_OBJECT(win), "draw", G_CALLBACK(draw_handler), NULL);
    g_signal_connect(G_OBJECT(win), "button-release-event", G_CALLBACK(click_handler), NULL);
    g_signal_connect(G_OBJECT(win), "scroll-event", G_CALLBACK(scroll_handler), NULL);
    g_signal_connect(G_OBJECT(win), "query-tooltip", G_CALLBACK(query_tooltip_handler), NULL);
    g_signal_connect(G_OBJECT(win), "destroy", G_CALLBACK(gtk_main_quit), NULL);
}

void create_dock_icon(void) {
    warn(DEBUG_DEBUG, "Loading theme");
    theme = gtk_icon_theme_get_default();
    display = gdk_display_get_default();

    warn(DEBUG_DEBUG, "Loading pixbufs");
    letters_pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)letters_xpm);
    if (!letters_pixbuf) {
        die("Could not load letters pixbuf");
    }
    guchar *pixels = gdk_pixbuf_get_pixels(letters_pixbuf);
    pixels += 6 * gdk_pixbuf_get_rowstride(letters_pixbuf);
    fgcolor.red = pixels[0] / 255.0;
    fgcolor.green = pixels[1] / 255.0;
    fgcolor.blue = pixels[2] / 255.0;
    fgcolor.alpha = 1.0;

    warn(DEBUG_INFO, "Creating window");
    mainwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    if (nonwmaker) {
        iconwin = mainwin;
    } else {
        iconwin = gtk_window_new(GTK_WINDOW_POPUP);
    }
    if (mainwin == NULL || iconwin == NULL) {
        die("Couldn't create windows");
    }

    init_window(mainwin);
    if (!nonwmaker) {
        init_window(iconwin);
    }

    warn(DEBUG_DEBUG, "Creating popup menu");
    init_menu();

    warn(DEBUG_DEBUG, "Realizing windows");
    gtk_widget_realize(mainwin);
    gtk_widget_realize(iconwin);

    // GTK3 removed gdk_window_set_icon, sigh. 
    // https://bugzilla.gnome.org/show_bug.cgi?id=723437
    // So we need to set the icon window manually. But if we do it before
    // gtk_widget_show(), GTK will override it. So, return of the GTK hack:
    // reparent it to an unmapped window, show it, fix the hints, and then
    // reparent back to root. Sigh.
    /* HACK */ {
        Display *d = GDK_DISPLAY_XDISPLAY(display);
        Window mw = GDK_WINDOW_XID(gtk_widget_get_window(mainwin));
        Window iw = GDK_WINDOW_XID(gtk_widget_get_window(iconwin));
        Window p, dummy1, *dummy2;
        unsigned int dummy3;
        XQueryTree(d, mw, &dummy1, &p, &dummy2, &dummy3);
        if (dummy2) XFree(dummy2);
        Window w = XCreateSimpleWindow(d, p, 0, 0, 1, 1, 0, 0, 0);
        XReparentWindow(d, mw, w, 0, 0);
        gtk_widget_show(mainwin);
        gtk_widget_show(iconwin);
        XWMHints *wmHints = XGetWMHints(d, mw);
        if (!wmHints) {
            wmHints = XAllocWMHints();
        }
        wmHints->flags |= IconWindowHint;
        wmHints->icon_window = iw;
        XSetWMHints(d, mw, wmHints);
        XFree(wmHints);
        XReparentWindow(d, mw, p, 0, 0);
        XDestroyWindow(d, w);
    }

    warn(DEBUG_DEBUG, "Created window mainwin=%p (0x%lx) iconwin=%p (0x%lx)", gtk_widget_get_window(mainwin), GDK_WINDOW_XID(gtk_widget_get_window(mainwin)), gtk_widget_get_window(iconwin), GDK_WINDOW_XID(gtk_widget_get_window(iconwin)));

    g_timeout_add(250, clock_redraw, NULL);
}

void dock_state_changed(void) {
    if (appstate == STARTING || appstate == LOCKED) {
        gtk_menu_popdown(menu);
    }

    gtk_widget_set_sensitive(menu_newpass, appstate != WORKING);
    gtk_widget_set_sensitive(menu_delpass, appstate != WORKING && appstate != NOPASS);
    gtk_widget_set_sensitive(menu_editpass, appstate != WORKING && appstate != NOPASS);
    gtk_widget_set_sensitive(menu_lockkeyring, appstate != WORKING);
    gtk_widget_set_sensitive(menu_delkeyring, appstate != WORKING);

    redraw_dock();
}

void error_alert(const char *format, GError *err) {
    error_alert_with_parent(GTK_WINDOW(iconwin), format, err);
}

void error_alert_with_parent(GtkWindow *parent, const char *format, GError *err) {
    GtkWidget *dialog = gtk_message_dialog_new(parent,
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_ERROR,
        GTK_BUTTONS_CLOSE,
        format, err ? err->message : "<unknown error>"
    );
    gtk_window_set_modal(GTK_WINDOW(dialog), gtk_window_get_modal(parent));
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(gtk_widget_destroy), dialog);
    gtk_widget_show_all(dialog);
}
