/* File:      curr_sym.P
** Author(s): Sagonas/Swift/Warren/Moura
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: curr_sym.P,v 1.53 2010/01/25 15:58:13 pmoura Exp $
** 
*/


:- compiler_options([xpp_on,sysmod,optimize]).

/* TES: because of a 64-bit number used below, compilation may give
   the error: Overflow in radix notation, returning float */

#include "standard.h"
#include "flag_defs_xsb.h"
#include "heap_defs_xsb.h"
#include "psc_defs.h"
#include "extensions_xsb.h"
#include "cinterf_defs.h"
#include "incr_xsb_defs.h"
#include "token_defs_xsb.h"
#include "basicdefs.h"
#include "table_status_defs.h"
#include "psc_defs.h"
#include "io_defs_xsb.h"
#include "celltags_xsb.h"

/* ===== current_atom/1 ====================================== */

:- mode current_atom(?).
%current_atom(Atom) :- true.
current_atom(Module:Atom) :-
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	ModuleName = Module,
	psc_prop(ModulePSC, First_PSC_Pair),	% get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
	    warning(('Predicate current_atom/1 for module "',
		     ModuleName,
		     '" does not include usermod strings !')),
	    psc_scan_in_hash_table(0,PSC),
	    psc_arity(PSC, Arity), 
	    Arity =:= 0,
	    psc_name(PSC, Name),
	    Atom = Name
        ; psc_scan_in_module(First_PSC_Pair, PSC), 
	  psc_arity(PSC, Arity), 
	  0 = Arity,
	  psc_name(PSC, Name),
	  Atom = Name
	).

% Original Jiyang version of current_atom/1
%	psc_scan(PSC), psc_arity(PSC, N), N = 0, term_new(PSC, A0), A0 = A.


/* ===== current_functor/1 ===================================== */

%   current_functor(?Predicate_Indicator)
%   Succeeds iff Predicate_Indicator is a predicate indicator for one
%   of the procedures appearing in the database (both system and user 
%   defined but not necessarily loaded in the system). 
%   Predicate_Indicator can have the following forms:
%	 (i) Module:Functor/Arity.
%	(ii) Functor/Arity (for global modules)
%   A variable in place of Predicate_Indicator succeeds only for global
%   modules.

:- mode current_functor(?).
current_functor(Module:Functor/N) :-
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	ModuleName = Module,
	psc_prop(ModulePSC, First_PSC_Pair),	% get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
		current_functor(Functor/N)
        ; psc_scan_in_module(First_PSC_Pair, PSC),
	  psc_arity(PSC, Arity), 
	  psc_name(PSC, Name),
	  Functor = Name,
	  Arity = N
	).
current_functor(Functor/N) :-		% This takes care of global modules.
	!,				% i.e. "usermod" and "global".
	psc_scan_in_hash_table(0, PSC),
	psc_arity(PSC, Arity), 
	psc_name(PSC, Name),
	Functor = Name,
	Arity = N.

psc_scan_in_hash_table(Bucket_Index, PSC) :- 
	dereference_the_bucket(Bucket_Index, First_Pair_of_Bucket),
	psc_scan_in_bucket(First_Pair_of_Bucket, PSC).
psc_scan_in_hash_table(Bucket_Index, PSC) :- 
	next_bucket(Bucket_Index, Next_Bucket_Index),
	( Next_Bucket_Index =\= 0 ->		% At end of hash_table.
		psc_scan_in_hash_table(Next_Bucket_Index, PSC) ).

psc_scan_in_bucket(PSC_Pair, _) :-
	PSC_Pair =:= 0,		% PSC_Pair == NULL
	!,
	fail.
psc_scan_in_bucket(PSC_Pair, PSC) :-
	pair_psc(PSC_Pair, PSC).
psc_scan_in_bucket(PSC_Pair, Next_PSC) :- 
	pair_next(PSC_Pair, Next_PSC_Pair),
	psc_scan_in_bucket(Next_PSC_Pair, Next_PSC).

% Original Jiyang version of current_functor/2
%	psc_scan(PSC), psc_arity(PSC, N0), psc_name(PSC, Name),
%	psc_insert(Name, 0, PSC0), term_new(PSC0, F0), F0 = F, N0 = N.

/* ===== current_functor/2 ===================================== */

%   current_functor(?Name, ?Term_Indicator)
%   Succeeds iff Term_Indicator is the most general term corresponding to
%   one of the procedures appearing in the database (both system and user
%   defined but not necessarily loaded in the system) having functor Name.
%   Term_Indicator can have the following forms:
%	 (i) Module:Term.
%	(ii) Term (for global modules)
%   A variable in place of Term_Indicator succeeds only for global
%   modules. Note that the order of term generation is undetermined.

:- mode current_functor(?,?).
current_functor(Name, Term) :-		% This takes care of global modules.
					% i.e. "usermod" and "global".
	psc_scan_in_hash_table(0, PSC),
	psc_name(PSC, F),
	Name = F,
	psc_arity(PSC, Arity), 
	functor(Term, Name, Arity).
current_functor(Name, Term_Indicator) :-
	nonvar(Term_Indicator),
	Term_Indicator = :(Module, Term),
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	ModuleName = Module,
	psc_prop(ModulePSC, First_PSC_Pair),	% get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
		current_functor(Name, Term)
        ; psc_scan_in_module(First_PSC_Pair, PSC),
	  psc_name(PSC, F),
	  Name = F,
	  psc_arity(PSC, Arity),
	  functor(Term, Name, Arity)
	).

/* ===== current_index/2 =================================== */ 

/* Here is how I understand the index encoding to work.  Alternative
indices are put into a list of indices each of which may be simple or
composite.  Any of the 255 arguments of a term can be used as a single
index and is in the least significant byte.  Otherwise, the most
significant bit of a byte (the 128 bit) indicates * indexing.  A
single * index has a l.s. byte of 0 followed by a byte w. the 128-bit
set indicating the * index argument.

In a composite index each byte indicates the argument, and whether or
not the 128 bit is set indicates whether the index is a * index in
each byte.  Unlike simple * indices, the l.s. byte is not set to 0.

*/

:- mode current_index(+,?).
current_index(Term,Index):-
    check_nonvar(Term,current_index/2,1),
    current_index_1(Term,Index).

current_index_1(Mod:T,Index):- 
    (T = F/N -> functor(Term,F,N) ; T = Term),
    term_new_mod(Mod,Term,NewTerm),!,
    current_index_1(NewTerm,Index).
current_index_1(F/N,Index):-
	nonvar(F),nonvar(N),
	functor(Term,F,N),!,
	current_index_1(Term,Index).
current_index_1(Term,Index):- 
	'_$multifile'(Term),
	'_$multifile_comp'(Comp,Term),
	!,
	current_index_1(Comp,Index).
current_index_1(Term,Index):- 
	'_$index'(Term,Ind,_Size),!,
	decode_index(Ind,Index).
current_index_1(Term,trie):- 
	'_$trie_asserted'(Term),!.
current_index_1(_Term,1).

:- export decode_index/2.
decode_index(0,0) :- !.
decode_index([],[]):-!.
decode_index([Ind|Rest],[Index|R]):- !,
	decode_index_in_list(Ind,Index),
	decode_index(Rest,R).
decode_index(Ind,Out):- 
	decode_index([Ind],Out).

decode_index_in_list(Ind,Ind):- 
	0 < Ind,Ind <  256,!.
decode_index_in_list(Ind,IndOut):- 
	decode_star_index(Ind,_,IndOut).

decode_star_index(0,IndIn,IndIn):- !.
decode_star_index(Ind,IndIn,IndOut):- 
	Ind < 256,!,
	I is Ind - 128,
	(I == 0 -> 
	    IndOut = IndIn
	 ;  (var(IndIn) -> 
	        (I < 0 -> IndOut = Ind ; IndOut = *(I))
	      ; (I < 0 -> IndOut = IndIn + Ind ; IndOut = IndIn + *(I) ) ) ).
decode_star_index(Ind,IndIn,IndOut):- 
	PreInd1 is /\(Ind,255),
	decode_star_index(PreInd1,IndIn,IndMid),
	IndRest is Ind >> 8,
	decode_star_index(IndRest,IndMid,IndOut).

/* ===== current_input/1 ======================================= */
% TES: changed on 8/04 for new streams.

:- mode current_input(?).
current_input(Stream) :- 
	(	var(Stream) ->
		true
	;	integer(Stream),
		is_valid_stream(Stream,_) -> 
	    true
	;	domain_error(stream,Stream,current_input/1,1)
	),
	stat_flag(CURRENT_INPUT, S),
	S = Stream.

% Obsolescent...
:- mode current_input_port(?).
current_input_port(FileDes) :- stat_flag(CURRENT_INPUT, FileDes).


/* ===== current_op/3 ========================================= */
%   current_op(?Operator_Precedence, ?Operator_Specifier, ?Operator_Name)
%   Succeeds iff Operator_Name is an operator with type defined by
%   Operator_Specifier and precedence Operator_Precedence.
%   TES: added errors acc. to Prolog standard. 8/04

:- mode current_op(?,?,?).
current_op(Prec, Type, Name) :- 
	check_precedence(Prec),
	check_operator_type(Type),
	check_operator_name(Name),
	'_$op'(Prec, Type, Name).

check_precedence(Prec):-
	((var(Prec) ; (integer(Prec),Prec >= 1,Prec =< 1200)) -> 
	    true 
	  ; domain_error(operator_priority,Prec,current_op/3,1) ).

check_operator_type(Type):-
	(	var(Type) ->
		true
	;	is_operator_type(Type) -> 
	    true
	;	atom(Type) ->
		domain_error(operator_specifier,Type,current_op/3,2)
	;	type_error(atom,Type,current_op/3,2)
	).

is_operator_type(fx).
is_operator_type(fy).
is_operator_type(hx).
is_operator_type(hy).
is_operator_type(xf).
is_operator_type(yf).
is_operator_type(xfx).
is_operator_type(xfy).
is_operator_type(yfx).

check_operator_name(Name):-
	(	var(Name) ->
		true
	;	atom(Name) -> 
	    true 
	;	type_error(atom,Name,current_op/3,3)
	).


/* ===== current_output/1 ===================================== */
% TES: changed on 8/04 for new streams.

:- mode current_output(?).
current_output(Stream) :- 
	(	var(Stream) ->
		true
	;	integer(Stream),
		is_valid_stream(Stream,_) ->
		true
	;	domain_error(stream,Stream,current_output/1,1)	
	),
	stat_flag(CURRENT_OUTPUT, S),
	S = Stream.

% Obsolescent...
current_output_port(FileDes) :- stat_flag(CURRENT_OUTPUT, FileDes).


/* ===== current_predicate/1 ==================================== */

%   current_predicate(?Predicate_Indicator)
%   Succeeds iff Predicate_Indicator is a predicate indicator for one
%   of the loaded procedures in the database (both system and user defined).
%   Predicate_Indicator can have the following forms:
%	 (i) Module:Functor/Arity.
%	(ii) Functor/Arity (for global modules)
%   A variable in place of Predicate_Indicator succeeds only for global
%   modules.
%  TES: added ISO-style error handling 8/04.
% TES: redid so that we have a unified semantics for different forms
% of predicate indicators.  09/04

:- mode current_predicate(?).
current_predicate(Specifier) :-		% This takes care of global modules.
	check_current_predicate_argument(Specifier),
	(	Specifier = Module:Functor/N ->
		current_predicate_module(Module,Functor,N)
	;	Specifier = Functor/N,
		current_predicate_global(Functor,N)
	).

% PM: ISO oddly specifies only a type error for current_predicate/1,
% which is not consistent with other standard predicates
check_current_predicate_argument(Term) :-
	(	var(Term) ->
		true
	;	Term = Module:Functor/Arity ->
		(	nonvar(Module), \+ atom(Module), \+ structure(Module) ->
			type_error(predicate_indicator,Term,current_predicate/1,1)
		;	nonvar(Functor), \+ atom(Functor) ->
			type_error(predicate_indicator,Term,current_predicate/1,1)
		;	nonvar(Arity), (\+ integer(Arity); Arity < 0) ->
			type_error(predicate_indicator,Term,current_predicate/1,1)
		;	true
		)
	;	Term = Functor/Arity ->
		(	nonvar(Functor), \+ atom(Functor) ->
			type_error(predicate_indicator,Term,current_predicate/1,1)
		;	nonvar(Arity), (\+ integer(Arity); Arity < 0) ->
			type_error(predicate_indicator,Term,current_predicate/1,1)
		;	true
		)
	;	type_error(predicate_indicator,Term,current_predicate/1,1)
	).

/*
current_predicate(Specifier) :-		% This takes care of global modules.
	(Specifier = Functor/N -> 
	    current_predicate_global(Functor,N)
	  ; (Specifier = Module:Functor/N -> 
	        current_predicate_module(Module,Functor,N)
	      ; type_error(predicate_indicator,Specifier,current_predicate/1,1) ) ).
*/
current_predicate_global(Functor,N):- 
	!,				% i.e. "usermod" and "global".
	psc_scan_in_hash_table(0, PSC),
	psc_type(PSC, Type), 
	( Type =:= T_PRED -> true		% Regular loaded predicate.
	; Type =:= T_DYNA -> true		% Dynamic predicate.
	; Type =:= T_FORN			% Foreign predicate.
	),
	psc_arity(PSC, Arity), 
	psc_name(PSC, Name),
	Functor = Name,
	Arity = N.


current_predicate_module(Module,Functor,N) :-
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	atom_to_term(ModuleName,Module),
	psc_prop(ModulePSC, First_PSC_Pair), % get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
		current_predicate(Functor/N)
        ; psc_scan_in_module(First_PSC_Pair, PSC), 
          psc_type(PSC, Type), 
	  ( Type =:= T_PRED -> true		% Regular loaded predicate.
	  ; Type =:= T_DYNA -> true		% Dynamic predicate.
	  ; Type =:= T_FORN			% Foreign predicate.
	  ),
	  psc_arity(PSC, Arity), 
	  psc_name(PSC, Name),
	  Functor = Name, 
	  Arity = N
	).

module_scan(PSC_Pair, _, _) :-
	PSC_Pair =:= 0,		% PSC_Pair == NULL
	!,
	fail.
module_scan(PSC_Pair, ModuleName, ModulePSC) :- 
	pair_psc(PSC_Pair, ModulePSC),
	psc_name(ModulePSC, ModuleName).
module_scan(PSC_Pair, NextModuleName, ModulePSC) :- 
	pair_next(PSC_Pair, Next_PSC_Pair),
	module_scan(Next_PSC_Pair, NextModuleName, ModulePSC).

psc_scan_in_module(PSC_Pair, _) :-
	PSC_Pair =:= 0,		% PSC_Pair == NULL
	!,
	fail.
psc_scan_in_module(PSC_Pair, PSC) :-
	pair_psc(PSC_Pair, PSC).
psc_scan_in_module(PSC_Pair, Next_PSC) :- 
	pair_next(PSC_Pair, Next_PSC_Pair),
	psc_scan_in_module(Next_PSC_Pair, Next_PSC).


/* ===== current_predicate/2 =================================== */

%   current_predicate(?Name, ?Term_Indicator)
%   Succeeds iff Term_Indicator is the most general term corresponding to
%   one of the loaded procedures in the database (both system and user
%   defined) having functor Name.
%   Term_Indicator can have the following forms:
%	 (i) Module:Term.
%	(ii) Term (for global modules)
%   A variable in place of Term_Indicator succeeds only for global
%   modules. Note that the order of term generation is undetermined.

:- mode current_predicate(?,?).
current_predicate(Name, Term) :-	% This takes care of global modules.
					% i.e. "usermod" and "global".
	psc_scan_in_hash_table(0, PSC),
	psc_type(PSC, Type),
	( Type =:= T_PRED -> true		% Regular loaded predicate.
	; Type =:= T_DYNA -> true		% Dynamic predicate.
	; Type =:= T_FORN			% Foreign predicate.
	),
	psc_arity(PSC, Arity), 
	psc_name(PSC, F),
	Name = F,
	functor(Term, Name, Arity).
current_predicate(Name, Term_Indicator) :-
	nonvar(Term_Indicator),
	Term_Indicator = :(Module, Term),
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	ModuleName = Module,
	psc_prop(ModulePSC, First_PSC_Pair),	% get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
		current_predicate(Name, Term)
	; psc_scan_in_module(First_PSC_Pair, PSC), 
	  psc_type(PSC, Type), 
	  ( Type =:= T_PRED -> true		% Regular loaded predicate.
	  ; Type =:= T_DYNA -> true		% Dynamic predicate.
	  ; Type =:= T_FORN			% Foreign predicate.
	  ),
	  psc_arity(PSC, Arity), 
	  psc_name(PSC, F),
	  Name = F, 
	  functor(Term, Name, Arity)
	).


/* ===== current_module/1 ============================================= */

%   current_module(?Module)
%   Succeeds iff Module is one of the modules in the database. This 
%   includes both user modules and system modules. A module becomes 
%   "current" as soon as it is loaded in the system or when another 
%   module that is loaded in the system imports some predicates from 
%   that module. A module can never lose the property of being "current".

:- mode current_module(?).
current_module(Module) :- 
	get_first_psc_pair(PSC_Pair),
	scan_modules(PSC_Pair, ModuleName),
	atom_to_term(ModuleName,Module).

scan_modules(PSC_Pair, _) :-
	PSC_Pair =:= 0,		% PSC_Pair == NULL
	!,
	fail.
scan_modules(PSC_Pair, ModuleName) :- 
	pair_psc(PSC_Pair, ModulePSC),
	psc_name(ModulePSC, ModuleName).
scan_modules(PSC_Pair, NextModuleName) :- 
	pair_next(PSC_Pair, Next_PSC_Pair),
	scan_modules(Next_PSC_Pair, NextModuleName).


/* ===== current_module/2 ============================================= */

%   current_module(?Module, ?ObjectFile)
%   Gives the relationship between the modules and their associated 
%   module object file names. It is possible for a current module to
%   have no associated object file name or for the system to be unable 
%   to find out the object file name of a current module.

:- mode current_module(?,?).
current_module(Module, ObjectFile) :- 
	current_module(Module),
	module_file(Module,ObjectFile).

module_file('xsbrc', File) :-
	!,
	current_predicate('xsbrc':_),	% module ~/.xsb/xsbrc.P is indeed loaded
	!,
	xsb_configuration(user_auxdir, UsrAuxDir),
	slash(Slash),
	fmt_write_string(FileName, '%s%sxsbrc.%s',
			 arg(UsrAuxDir, Slash, XSB_OBJ_EXTENSION_ATOM)),
	FileName = File.
module_file(Module, File) :-
	psc_insertmod(Module,0,ModPsc),
	ModPsc \== 0,
	psc_ep(ModPsc,File),
	atom(File),		% has been loaded
	!.
module_file(Module, File) :-
	functor(Module,ModuleName,_),
	str_cat(ModuleName, XSB_OBJ_EXTENSION_ATOM, OFile),
	once((libpath(Lib),
	      str_cat(Lib, OFile, FileName),
	      machine_file_exists(FileName)
		% This once is needed so that we get only one filename
		% as an answer, exactly the one that corresponds to the
		% file that would be consulted when using the loader (ATBE).
	     )),
	FileName = File.  % maybe standardize filename?


/* ===== predicate_property/2 ========================================= */

%   predicate_property(?Predicate_Indicator, ?Property)
%   Succeeds iff predicate Predicate has the property Property.
%   Hacked by Kostis (10/27/92) to make it fast when searching 
%   in a global module with Predicate_Indicator ground.

:- mode predicate_property(?,?).
predicate_property(Term_Indicator, Property) :-
%   writeln(predicate_propertyc1(Term_Indicator, Property)),
	nonvar(Term_Indicator),
	Term_Indicator = :(Module, Term),
	nonvar(Term),		% send ':'(_,_) predicate to next clause
	!,
        functor(Term,F,A),functor(Skel,F,A),
        term_new_mod(Module,Skel,NewTerm),
%        writeln(term_new_mod(Module,Term,NewTerm)),
	pred_prop_compat(Property,Search_prop),
%	writeln(pred_prop_compat(_,Property,Search_prop)),
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	ModuleName = Module,
	psc_prop(ModulePSC, First_PSC_Pair),	% get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
		predicate_property(Skel, Search_prop)
        ; psc_scan_in_module(First_PSC_Pair, PSC),
	  psc_name(PSC, Name),
	  psc_arity(PSC, Arity),
	  functor(NewTerm, Name, Arity),
	  predicate_property_1(PSC,NewTerm, Search_prop)
	).
% TES -- I have no idea why T_FILE is used in the clause below: it
% seems to happen on 0-ary preds(?)
predicate_property(Term, Property) :-	% This takes care of global 
					% modules,"usermod" and "global".
%   writeln(predicate_propertyc2(Term, Property)),
	term_type(Term, Type),
	pred_prop_compat(Property,Search_prop),
	( Type =:= T_DYNA ->	% Hacked to make it fast when Term is given.
			% If Term is given and it is represented as a structure 
			% (has a PSC record) then using term_psc/2 we get to 
			% this PSC record directly without having to scan the 
			% whole PSC-Pair list.
%	 writeln(dyna),
		term_psc(Term, PSC),term_new(PSC,Skel),
		predicate_property_1(PSC,Skel, Search_prop)
	 ; Type =:= XSB_STRING ->  
%	        warning(('predicate_property/2 called on term of type T_FILE: ',Term)),
		conpsc(Term, PSC),
		predicate_property_1(PSC,Term, Search_prop)
	; psc_scan_in_hash_table(0, PSC),
	  psc_name(PSC, Name),
	  psc_arity(PSC, Arity), 
	  functor(Term, Name, Arity),
	  predicate_property_1(PSC,Term, Search_prop)
	).
   :- import term_new/2 from machine.
/* This is to ensure backward compatability with goals like
   predicate_property(X,tabled) */
pred_prop_compat(Propin,Prop):-
    nonvar(Propin),
    !,
    (Propin = tabled -> Prop = (table_reuse = _)
     ; Propin = incremental -> Prop = (updating = incremental)
     ; Propin = opaque -> Prop = (updating = opaque)
     ; Propin = Prop).
pred_prop_compat(Propin,Propin).

predicate_property_1(PSC,_Term, Property):- 
%   writeln(predicate_property_1_1(PSC,_Term, Property)),
    psc_properties(PSC, Property).
predicate_property_1(_PSC,Term, multifile):- 
%   writeln(predicate_property_1_2(PSC,_Term, Property)),
    standard:'_$multifile'(Term).
predicate_property_1(_PSC,Term, Property) :-
%   writeln(predicate_property_1_3(PSC,Term, Property)),
%   functor(Term,F,A),functor(Skel,F,A),
   get_tabling_options(Term,List),
   member(Property,List).

   :- import writeln/1 from standard.
%-----------------

psc_properties(PSC, Property) :- 
	psc_type(PSC, Type),
%	( Type =:= T_ORDI -> Property = unclassified  
        ( Type =:= T_ORDI -> psc_env(PSC,Env),Env = T_IMPORTED,Property = unloaded
	; Type =:= T_DYNA -> Property = (dynamic)
	; Type =:= T_PRED -> Property = static
	; Type =:= T_UDEF -> Property = unloaded
	; Type =:= T_FORN -> Property = foreign).
%	; Type =:= 14 -> Property = function ).
%psc_properties(PSC, static) :-
%	\+ (psc_type(PSC, Type), Type =:= T_DYNA).
psc_properties(PSC, Property) :-
	psc_env(PSC, Env),
 	( Env =:= T_EXPORTED -> psc_type(PSC,Type),Type \== T_ORDI,Property = exported	
	; Env =:= T_LOCAL -> Property = (local)
	; Env =:= T_IMPORTED -> psc_mod(PSC, ModulePSC), 
		       psc_name(ModulePSC, ModuleName), 
		       Property = imported_from(ModuleName)
	; Property = global ).
psc_properties(PSC, Property) :- 
	psc_spy(PSC, Spy),
	( Spy =\= 0 -> Property = spied ).

%psc_properties(PSC, Property) :- 
%	psc_tabled(PSC, Tabled),
%	(psc_type(PSC, Type),Type =:= T_DYNA -> 
%	    psc_get_incr(PSC,Incr),Incr == NONINCREMENTAL
%	  ; true),  
%	( (Tabled /\ 12 > 0),Property = tabled 
%        ; 
%	 ( Tabled == T_VARIANT -> Property = variant
%         ; Tabled == T_SUBSUMPTIVE -> Property = subsumptive ) ).
%%         ; Tabled == T_TABLED_UNDETERMINED -> Property = tabled(default)).
%psc_properties(PSC, Property) :- 
%	psc_get_incr(PSC, Incr),
%	( Incr == INCREMENTAL -> 
%	    Property = incremental 
%	  ; Incr == OPAQUE -> Property = opaque).
%psc_properties(PSC, Property) :- 
%	psc_shared(PSC, Shared),
%	( Shared =\= 0 -> Property = shared ).
psc_properties(PSC, Property) :-
	psc_name(PSC, Functor),		% Finding the Functor and the Arity
	psc_arity(PSC, Arity),		% twice is silly, isn''t it?
	(	standard_metapredicate_template(Functor,Arity,Template),
%		setof(Meta, standard_metapredicate(Functor, Arity, Meta), Metas),
%		psc_meta_predicate_property_args(Metas, 1, Arity, Args),
%		Template =.. [Functor| Args],
		Property = meta_predicate(Template)
	;	standard_symbol(Functor, Arity, _) ->
		( %	Property = xsb_standard_predicate ; 
			Property = built_in
		)
	 ).
	% Unfortunately this doesn''t take the Module into account.

/* PM: this code was used in the old implementation of the
       meta_predicate/1 predicate property:

psc_meta_predicate_property_args([], N, Arity, Args) :-
	(	N =< Arity ->
		N2 is N + 1,
		Args = [(*)| TArgs],
		psc_meta_predicate_property_args([], N2, Arity, TArgs)
	;	Args = []
	).

psc_meta_predicate_property_args([Meta| Metas], N, Arity, [Arg| Args]) :-
	N2 is N + 1,
	(	Meta =:= N ->
		Arg = (:),
		psc_meta_predicate_property_args(Metas, N2, Arity, Args)
	;	Arg = (*),
		psc_meta_predicate_property_args([Meta| Metas], N2, Arity, Args)
	).
*/

	 %%%
%------------------------------------
:- export set_predicate_property/2.
:- import is_tabled/1,set_psc_table_property/2,check_table_option/4,set_tif_property/3 from tables.

:- mode set_predicate_property(+,+).
set_predicate_property(TermSpec,Property):-
    (TermSpec = :(Module, T1) -> term_new_mod(Module, T1, Term) ; TermSpec = Term),
    is_tabled(Term),
    (Property = (P=V) -> Prop = P,Value = V ; Prop = Property),
    set_predicate_property_1(Term,Prop,Value).

translate_tabling_options([H],H1):-
    !,
    translate_tabling_option(H,H1).
translate_tabling_options([H|T],(H1,T1)):-
    translate_tabling_option(H,H1),
    translate_tabling_options(T,T1).

translate_tabling_option((table_reuse=M),M):- !.
translate_tabling_option((updating=M),M):- !.
translate_tabling_option((subgoal_abstract=M),subgoal_abstract(M)):- !.
translate_tabling_option((answer_abstract=M),answer_abstract(M)):- !.
translate_tabling_option((ground_term=M),M):- !.
translate_tabling_option((thread_sharing=M),M):- !.
translate_tabling_option((answer_subsumption=_),ans_subsumption):- !.




:- index(set_predicate_property_1/3,2).
set_predicate_property_1(Term,subgoal_abstract,N):-
    get_tabling_options(Term,OptionsL),translate_tabling_options(OptionsL,Options),
    check_table_option(subgoal_abstract(N),Options,Term,yes),
    check_integer(N,'set_predicate_property/3',1),
    set_tif_property(Term,subgoal_size,N).
set_predicate_property_1(Term,answer_abstract,N):-  
    %    get_tabling_options(Term,Options),  not needed, works with everythingn
    %    check_table_option(answer_abstract(_),Options,Term,yes), % should work with anything.
    check_integer(N,'set_predicate_property/3',1),
    set_tif_property(Term,answer_size,N).
set_predicate_property_1(Term,subsumptive,_):-
    get_tabling_options(Term,OptionsL),translate_tabling_options(OptionsL,Options),
    check_table_option(subsumptive,Options,Term,yes),
    set_psc_table_property(Term,subsumptive).
set_predicate_property_1(Term,variant,_):-
    get_tabling_options(Term,OptionsL),translate_tabling_options(OptionsL,Options),
    check_table_option(variant,Options,Term,yes),
    set_psc_table_property(Term,variant).
set_predicate_property_1(Term,incremental,_):-
    get_tabling_options(Term,OptionsL),translate_tabling_options(OptionsL,Options),
    check_table_option(incremental,Options,Term,yes),
    set_psc_table_property(Term,incremental).
set_predicate_property_1(Term,nonincremental,_):-
    get_tabling_options(Term,OptionsL),translate_tabling_options(OptionsL,Options),
    check_table_option(nonincremental,Options,Term,yes),
    set_psc_table_property(Term,nonincremental).
set_predicate_property_1(Term,opaque,_):-
    get_tabling_options(Term,OptionsL),translate_tabling_options(OptionsL,Options),
    check_table_option(opaque,Options,Term,yes),
    set_psc_table_property(Term,opaque).

/* ===== module_property/2 ============================================ */

%   module_property(?Module, ?Property)
%   Succeeds iff module Module has the property Property.

:- mode module_property(?,?).
module_property(Module, Property) :-
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	Module = ModuleName, 
	module_properties(ModulePSC, Property).

module_properties(PSC, Property) :- 
	psc_type(PSC, Type),
	( Type =:=  T_ORDI -> Property = unloaded
	; Type =:=  T_MODU -> Property = loaded ).

/* ===== subgoal_property/2 ================================== */
:- mode subgoal_property(?,?).
subgoal_property(Subgoal,Property):-
%   writeln(xbsubgoal_property(Subgoal,Property)),
   (var(Subgoal) -> 
	predicate_property(S,tabled),
%	writeln(predicate_property(S,tabled)),
	functor(S,F,A),current_predicate(M:F/A),Subgoal = M:S
      ;    (Subgoal = M:Goal -> term_new_mod(M,Goal,S) ; Subgoal = S)),
   Property = [Attribute,Value],
%   writeln(catch(get_calls(S,SF,_),_E,fail)),
   catch(get_calls(S,SF,_),_E,fail),
%   writeln('  subgoal_property_1'(S,SF,Property)),
   subgoal_property_1(S,SF,Attribute,Value).

:- mode variant_subgoal_property(?,?).
variant_subgoal_property(Subgoal,Property):- 
   get_call(Subgoal,SF,_),
   Property = [Attribute,Value],
   subgoal_property_1(Subgoal,SF,Attribute,Value).

%:- import incremental/1,invalid/1,calls_to/1,answers/1 from usermod.
:- index(subgoal_property_1(_,_,_),[3]).
subgoal_property_1(Subgoal,_SF,incremental,Value):-
 %   writeln(incremental_subgoal(Subgoal,N)),
    incremental_subgoal(Subgoal,N),
    (N == 0 -> Value = valid ; Value = invalid).
subgoal_property_1(Subgoal,_SF,evaluation_state,Value):-
	table_status(Subgoal, _PredType, CallType, AnsSetStatus,_Sf),
	subgoal_property_2(CallType,AnsSetStatus,Value).
subgoal_property_1(_Subgoal,SF,calls_to,Number):-
    get_callsto_number(SF,Number).
subgoal_property_1(_Subgoal,SF,answers,Number):-
    get_answer_number(SF,Number).

subgoal_property_2(PRODUCER_CALL,_AnsSetStatus,producer).
subgoal_property_2(SUBSUMED_CALL,_AnsSetStatus,subsumed).
%subgoal_property_2(INCR_NEEDS_REEVAL,_AnsSetStatus,incremental_needs_reeval).
subgoal_property_2(_CallType,INCOMPLETE_ANSWER_SET,incomplete).
subgoal_property_2(_CallType,COMPLETED_ANSWER_SET,completed).

/* ===== current_prolog_flag/2 ================================== */
       
user_dyn_clause_count:-
    setof(Mod,(current_module(Mod),\+ system_module(Mod)),ModList),
    member(Module,ModList),
    setof(F1/A1-P,(predicates_for_module(Module,P),predicate_property(P,dynamic),functor(P,F1,A1)),PredList),
    write(Module),writeln(':'),
    member(F/A-Pred,PredList),
    count(Module,Pred,Count),
    str_length(F,Flen),
    Tabbing is max(10,(40 - Flen)),
    number_codes(Count,NList),
    atom_codes(ACount,NList),
    str_length(ACount,ALen),
    NumTab is 10 - ALen,
    tab(4),write(F),write('/'),write(A),tab(Tabbing),tab(NumTab),writeln(Count),
    fail.
user_dyn_clause_count.

system_module(Module):- 
    xsb_configuration(srcdir,XSBroot),
    current_module(Module,File),
    str_sub(XSBroot,File).

predicates_for_module(Module,Pred):- 
    current_predicate(Module:F/A),
    functor(Pred,F,A),
    (Module = usermod -> \+ system_pred(Pred) ; true).

system_pred(cmp(_,_,_)).
system_pred(del(_,_,_)).
system_pred(default_user_error_handler(_)).
system_pred(interpreter_index_hook(_)).
system_pred(max_answers_for_subgoal_user_handler).
system_pred(max_incomplete_subgoals_user_handler).
system_pred(max_memory_user_handler).
system_pred(max_scc_subgoals_user_handler).
system_pred(max_table_answer_size_user_handler).
system_pred(max_table_subgoal_size_user_handler).
system_pred(invalidate_table_for(_,_)).
system_pred(library_directory(_)).
system_pred(na(_,_,_)).
system_pred(ns_smpl(_,_,_)).
system_pred(nf_smpl(_,_,_)).
system_pred(nc(_,_,_,_)).
system_pred(nda(_,_,_,_)).
system_pred(puc_smpl(_,_,_,_,_)).
system_pred(pus_smpl(_,_,_,_)).
system_pred(tc(_,_,_,_)).
system_pred('_$thread_exit_ball'(_,_)).

count(Module,Pred,Count):-
    findall(1,clause(Module:Pred,_),List),
    length(List,Count).

/* ===== current_prolog_flag/2 ================================== */
:- import tripwire/1 from usermod.

:- mode current_prolog_flag(+,?).
current_prolog_flag(Flag, Value) :- 
	(	var(Flag) ->
		true
	;	\+ atom(Flag),\+ (Flag = tripwire(_)) ->
		type_error(atom_or_variable,Flag,current_prolog_flag/2,1)
	;	\+ iso_flag(Flag),\+ (Flag = tripwire(_)),\+ xsb_flag(Flag),\+ prolog_commons_flag(Flag) ->
		domain_error(prolog_flag,Flag,current_prolog_flag/2,1)
	;	true
	),
	(	iso_flag(Flag,Value) 
	;	Flag = tripwire(F),
	        tripwire_1(F),
    	        (  Value = action(A),
   	           tripwire_action(F,FA),
  	           xsb_flag(FA,A)
  	         ; Value = limit(P),
  	           xsb_flag(F,P) )
         ;      \+compound(Flag),\+ compound(Value),xsb_flag(Flag,Value)
  	 ;	prolog_commons_flag(Flag,Value)
	).

settable_flag(unknown).
settable_flag(double_quotes).

settable_flag_value(unknown,error).
settable_flag_value(unknown,fail).
settable_flag_value(unknown,warning).
settable_flag_value(unknown,undefined).
settable_flag_value(unknown,user_hook).
% PM: avoid error when setting the double_quotes flag to codes
settable_flag_value(double_quotes,codes).

:- dynamic settable_prolog_flag/2.
:- asserta(settable_prolog_flag(unknown,error)).
:- asserta(settable_prolog_flag(double_quotes,codes)).

:- mode set_prolog_flag(+,+).
set_prolog_flag(Flag,Value) :-
	(	var(Flag) ->
		instantiation_error(set_prolog_flag/2,1,bound)
	;	var(Value) ->
		instantiation_error(set_prolog_flag/2,2,bound)
	; true),
% Don't check here -- want tripwires
%	;	atom(Flag) ->
%		true
%	;	type_error(atom,Flag,set_prolog_flag/2,1)
%	),
	(	settable_flag(Flag) ->
		(	settable_flag_value(Flag,Value) ->
			retractall(settable_prolog_flag(Flag,_)),
			asserta(settable_prolog_flag(Flag,Value))
		;	domain_error(flag_value,Value,set_prolog_flag/2,2)
		)
	;	set_xsb_flag(Flag,Value)
	).

%make_set_tail([Val],[Val,'}']) :- !.
%make_set_tail([Val|Vals],[Val,','|SetTail]) :- make_set_tail(Vals,SetTail).

% PM: list of ISO specified flags for error checking
iso_flag(bounded).
iso_flag(max_integer). 
iso_flag(min_integer).
iso_flag(max_arity).
iso_flag(integer_rounding_function).
iso_flag(debug).
iso_flag(unknown).
iso_flag(double_quotes).
iso_flag(char_conversion).


%% fix for 64-bit compilation?
iso_flag(bounded,true).

% TES: dont understand why we dont just say 7fffffff.
%iso_flag(old_max_integer,Maxint) :- Maxint is 14 * 16'08000000 + 16'0fffffff.
% Also, in 64-bits, we dont seem to get full wordsize.  We have 7 followed by 13 fs rather than by 15?
iso_flag(max_integer,Maxint):- 
	xsb_configuration(word_size,Size),
	(Size == '64' -> 
	    Maxint is (2**62 - 1) + 2**62
	;   Maxint is (2**30 - 1) + 2**30).
%'
%%iso_flag(min_integer,Minint) :- iso_flag(max_integer,Maxint), Minint is -Maxint - 1.
iso_flag(min_integer,Minint) :- iso_flag(max_integer,Maxint), Minint is -Maxint.
%iso_flag(max_arity,255).
iso_flag(max_arity,MAX_ARITY).
iso_flag(integer_rounding_function,toward_zero).
iso_flag(debug, Debug) :- xsb_flag(debugging, Debug).
iso_flag(unknown, State) :- settable_prolog_flag(unknown, State).
iso_flag(double_quotes, Meaning) :- settable_prolog_flag(double_quotes, Meaning).
iso_flag(char_conversion, off).

% TES: need to understand ISO char conversion
%iso_flag(char_conversion,{on,off})
% TES: we only do codes...
%iso_flag(double_quotes,{chars,codes,atom}).
% TES: shd. add
%iso_flag(unknown,{error,warning,fail})

/* ===== prolog_commons_flag/2 ========================================== */

% PM: list of Prolog Commons specified flags for error checking
prolog_commons_flag(dialect).
prolog_commons_flag(version_data).

prolog_commons_flag(dialect,xsb).
prolog_commons_flag(version_data, xsb(Major, Minor, Patch, _)):-
          xsb_configuration(major_version, MajorAtom), atom(MajorAtom),
          atom_codes(MajorAtom, MajorCodes),
          number_codes(Major, MajorCodes),
          xsb_configuration(minor_version, MinorAtom), atom(MinorAtom),
          atom_codes(MinorAtom, MinorCodes),
          number_codes(Minor, MinorCodes),
          xsb_configuration(patch_version, PatchAtom), atom(PatchAtom),
          atom_codes(PatchAtom, PatchCodes),
	  (PatchCodes = [] -> Patch = '' ; number_codes(Patch, PatchCodes)).

xsb_flag(Flag):- xsb_flag(Flag,_),!.

/* ===== xsb_flag/2 ========================================== */

%%   xsb_flag(?FlagName, ?Value)
%%   Succeeds if the XSB flag FlagName has the value Value.
%%
%%   IMPORTANT Convention: use xsb_flag for dynamic aspects of XSB setting,
%%   i.e., those that can change between sessions or within a session.
%%   In contrast, xsb_configuration is to be used for settings that are the
%%   same for different sessions and are fixed at XSB build time.
tripwire_1(max_table_subgoal_size).
tripwire_1(max_table_answer_size).
tripwire_1(max_incomplete_subgoals).
tripwire_1(max_scc_subgoals).
tripwire_1(max_memory).
tripwire_1(max_answers_for_subgoal). 

tripwire_action(max_table_subgoal_size,max_table_subgoal_size_action).
tripwire_action(max_table_answer_size,max_table_answer_size_action).
tripwire_action(max_incomplete_subgoals,max_incomplete_subgoals_action).
tripwire_action(max_scc_subgoals,max_scc_subgoals_action).
tripwire_action(max_memory,max_memory_action).
tripwire_action(max_answers_for_subgoal,max_answers_for_subgoal_action). 

:- mode xsb_flag(?,?).
xsb_flag(write_attributes,Value):- 
	stat_flag(WRITE_ATTRIBUTES, V), 
	( V =:= WA_IGNORE -> Value = ignore
	  ;  (V =:= WA_DOTS -> Value = dots
	      ;  (V =:= WA_PORTRAY -> Value = portray
  	           ;  V =:= WA_WRITE -> Value = write) ) ).
xsb_flag(write_depth,Depth):- 
	stat_flag(WRITE_DEPTH, Depth).
xsb_flag(backtrace_on_error, BT) :-
	stat_flag(BACKTRACE, Value), 
	( Value =:= 0 -> BT = off ; BT = on ).
xsb_flag(debugging, Debug) :-
	stat_flag(DEBUG_ON, Value), 
	( Value =:= 0 -> Debug = off ; Debug = on ).
xsb_flag(tracing, Trace) :-
	stat_flag(TRACE, Value), 
	( Value =:= 0 -> Trace = off ; Trace = on ).
xsb_flag(verboseness,Level) :-
	var(Level),
	stat_flag(VERBOSENESS_LEVEL,Level).
%xsb_flag(verboseness,Level) :-
%	number(Level),
%	stat_set_flag(VERBOSENESS_LEVEL,Level).
%% Goal passed on cmd line
xsb_flag(goal, Goal) :-
	stat_flag(CMD_LINE_GOAL, GoalUninterned),
	intern_string(GoalUninterned, Goal).
xsb_flag(dcg_style, DcgMode) :-
	stat_flag(DCG_MODE, Dcg_flag),
	( Dcg_flag =:= 0 -> DcgMode = xsb ; DcgMode = standard ).
xsb_flag(garbage_collection, GCmode) :-
	xsb_flag(heap_garbage_collection, GCmode).
xsb_flag(heap_garbage_collection, GCmode) :-
	stat_flag(GARBAGE_COLLECT, GCflag),
	( GCflag =:= NO_GC -> GCmode = none
	; GCflag =:= SLIDING_GC -> GCmode = sliding
	; GCflag =:= COPYING_GC -> GCmode = copying
	; GCflag =:= INDIRECTION_SLIDE_GC -> GCmode = indirection
	).
xsb_flag(clause_garbage_collection, OnOff) :-
	stat_flag(CLAUSE_GARBAGE_COLLECT, GCflag),
	( GCflag =:= 0 -> OnOff = off ; OnOff = on ).
xsb_flag(atom_garbage_collection, OnOff) :-
	stat_flag(STRING_GARBAGE_COLLECT, GCflag),
	( GCflag =:= 0 -> OnOff = off ; OnOff = on ).
xsb_flag(heap_margin, Margin) :-
	stat_flag(HEAP_GC_MARGIN, Margin).
xsb_flag(gc_verbose_level,Level) :-
	(   var(Level)
	->  stat_flag(VERBOSE_GC,VerboseGC),
	    stat_flag(COUNT_CHAINS,CountYN),
	    stat_flag(EXAMINE_DATA,Examine),
	    Level is VerboseGC + CountYN + Examine
	;   Level = 0
	->  stat_set_flag(VERBOSE_GC,0),
	    stat_set_flag(COUNT_CHAINS,0),
	    stat_set_flag(EXAMINE_DATA,0)
	;   Level = 1
	->  stat_set_flag(VERBOSE_GC,1),
	    stat_set_flag(COUNT_CHAINS,0),
	    stat_set_flag(EXAMINE_DATA,0)
	;   Level = 2
	->  stat_set_flag(VERBOSE_GC,1),
	    stat_set_flag(COUNT_CHAINS,0),
	    stat_set_flag(EXAMINE_DATA,1)
	;   Level = 3
	->  stat_set_flag(VERBOSE_GC,1),
	    stat_set_flag(COUNT_CHAINS,1),
	    stat_set_flag(EXAMINE_DATA,1)
	 ;  warning('xsb_flag(gc_verbose_level,X) -> X out of bounds.'),
	    check_atomic(Level,'xsb_flag/2',1) % ensure bound on return
	).
xsb_flag(verbose_gc, VerboseGC) :-
	(   var(VerboseGC)
	->  stat_flag(VERBOSE_GC,VerboseGC)
	;   VerboseGC = yes
	->  stat_set_flag(VERBOSE_GC, 1)
	;   VerboseGC = no
	->  stat_set_flag(VERBOSE_GC, 0)
	;   warning('xsb_flag(verbose_gc,X) -> X should be no, yes or unbound.'),
	    check_atomic(VerboseGC,'xsb_flag/2',2)
	).
xsb_flag(count_chains, CountYN) :-
	(   var(CountYN)
	->  stat_flag(COUNT_CHAINS,CountYN)
	;   CountYN = yes
	->  stat_set_flag(COUNT_CHAINS, 1)
	;   CountYN = no
	->  stat_set_flag(COUNT_CHAINS, 0)
	;   warning('xsb_flag(count_chains,X) -> X should be no, yes or unbound.'),
	    check_atomic(CountYN,'xsb_flag/2',2)
	).
xsb_flag(examine_data, ExamineYN) :-
	(var(ExamineYN)
	->  stat_flag(EXAMINE_DATA, ExamineYN)
	;   ExamineYN = yes
	->  stat_set_flag(EXAMINE_DATA, 1)
	;   ExamineYN = no
	->  stat_set_flag(EXAMINE_DATA,0)
	;   warning('xsb_flag(examine_data,X) -> X should be no, yes or unbound.'),
	    check_atomic(ExamineYN,'xsb_flag/2',2)
	).
xsb_flag(table_gc_action, Action) :-
	stat_flag(TABLE_GC_ACTION, ActionFlag),
	(ActionFlag == ABOLISH_TABLES_TRANSITIVELY -> 
	    Action = abolish_tables_transitively
	  ; Action = abolish_tables_singly).
xsb_flag(thread_pdlsize, Size) :-
	stat_flag(THREAD_PDLSIZE, Size).
xsb_flag(thread_glsize, Size) :-
	stat_flag(THREAD_GLSIZE, Size).
xsb_flag(thread_tcpsize, Size) :-
	stat_flag(THREAD_TCPSIZE, Size).
xsb_flag(thread_complsize, Size) :-
	stat_flag(THREAD_COMPLSIZE, Size).
xsb_flag(thread_detached, Bool) :-
	stat_flag(THREAD_DETACHED, B),
	(B == 0 -> Bool = false ; Bool = true).
/* Not changeable */
xsb_flag(max_threads, Max) :-
	stat_flag(MAX_THREAD_FLAG, Max).
xsb_flag(max_queue_terms, Max) :-
	stat_flag(MAX_QUEUE_TERMS, Max).
xsb_flag(shared_predicates,Bool):- 
	stat_flag(PRIVSHAR_DEFAULT,B),
	(B == 0 -> Bool = false ; Bool = true).
xsb_flag(warning_action,Action):- 
	stat_flag(WARNING_ACTION,A),
	(A == PRINT_WARNING -> 
	    Action = print_warning
	  ; (A == SILENT_WARNING -> 
	      Action = silent_warning
	     ; A == ERROR_WARNING,Action = error_warning) ).
xsb_flag(max_table_subgoal_var_num,Size):- 
	stat_flag(MAX_TABLE_SUBGOAL_VAR_NUM,Size).
xsb_flag(max_table_answer_var_num,Size):- 
	stat_flag(MAX_TABLE_ANSWER_VAR_NUM,Size).
xsb_flag(cyclic_check_size,Size):- 
	stat_flag(CYCLIC_CHECK_SIZE,Size).
xsb_flag(max_table_subgoal_size,Max):- 
	stat_flag(MAX_TABLE_SUBGOAL_SIZE,M),
	iso_flag(max_integer,Maxint),
	(M =:= Maxint -> Max = 0; M = Max).
xsb_flag(max_table_subgoal_size_action, Action) :-
	stat_flag(MAX_TABLE_SUBGOAL_ACTION, A),
	(A == XSB_ERROR -> Action = error 
         ; A == XSB_ABSTRACT -> Action = abstract 
         ; A == XSB_SUSPEND -> Action = suspend
         ; A == XSB_CUSTOM -> Action = custom).
xsb_flag(max_table_answer_size,Max):- 
	stat_flag(MAX_TABLE_ANSWER_METRIC,M),
	iso_flag(max_integer,Maxint),
	(M =:= Maxint -> Max = 0; M = Max).
xsb_flag(max_table_answer_size_action, Action) :-
	stat_flag(MAX_TABLE_ANSWER_ACTION, A),
	(A == XSB_ERROR -> Action = error 
         ; A == XSB_BRAT -> Action = abstract 
         ; A == XSB_SUSPEND -> Action = suspend
	 ; A == XSB_CUSTOM -> Action = custom).
%xsb_flag(max_table_answer_list_depth,Depth):- 
%	stat_flag(MAX_TABLE_ANSWER_LIST_DEPTH,Depth).
%xsb_flag(max_table_answer_list_action, Action) :-
%	stat_flag(MAX_TABLE_ANSWER_LIST_ACTION, A),
%	(A == XSB_ERROR -> Action = error 
%	; (A == XSB_FAILURE -> Action = failure 
%            ; A == XSB_WARNING, Action = warning)).
xsb_flag(max_incomplete_subgoals, Max) :-
	stat_flag(MAX_INCOMPLETE_SUBGOALS,M),
	iso_flag(max_integer,Maxint),
	(M =:= Maxint -> Max = 0; M = Max).
xsb_flag(max_incomplete_subgoals_action, Action) :-
	stat_flag(MAX_INCOMPLETE_SUBGOALS_ACTION,A),
	(A == XSB_ERROR -> Action = error 
	 ; A == XSB_SUSPEND -> Action = suspend 
	 ; A == XSB_CUSTOM -> Action = custom
         ; A == XSB_WARNING, Action = warning).
xsb_flag(max_scc_subgoals, Max) :-
	stat_flag(MAX_SCC_SUBGOALS,M),
	iso_flag(max_integer,Maxint),
	(M =:= Maxint -> Max = 0; M = Max).
xsb_flag(max_scc_subgoals_action, Action) :-
	stat_flag(MAX_SCC_SUBGOALS_ACTION,A),
	(A == XSB_ERROR -> Action = error 
	 ; A == XSB_SUSPEND -> Action = suspend 
	 ; A == XSB_CUSTOM -> Action = custom					    
         ; A == XSB_WARNING, Action = warning ).
xsb_flag(max_answers_for_subgoal, Max) :-
	stat_flag(MAX_ANSWERS_FOR_SUBGOAL,M),
	iso_flag(max_integer,Maxint),
	(M =:= Maxint -> Max = 0; M = Max).
xsb_flag(max_answers_for_subgoal_action, Action) :-
	stat_flag(MAX_ANSWERS_FOR_SUBGOAL_ACTION,A),
	(A == XSB_ABSTRACT -> Action = complete_soundly
	; A == XSB_ERROR -> Action = error 
	; A == XSB_SUSPEND -> Action = suspend
	; A == XSB_CUSTOM -> Action = custom ).
xsb_flag(unify_with_occurs_check, Mode) :-
	stat_flag(UNIFY_WITH_OCCURS_CHECK_FLAG, Flag),
	( Flag =:= 0 -> Mode = off ; Mode = on ).
xsb_flag(max_tab_usage, Mode) :-
	stat_flag(MAX_USAGE, Flag),
	( Flag =:= 0 -> Mode = off ; Mode = on ).
xsb_flag(max_memory, Max) :-
	stat_flag(MAX_MEMORY, Max).
xsb_flag(max_memory_action, Action) :-
	stat_flag(MAX_MEMORY_ACTION,A),
	(A == XSB_ERROR -> Action = error
	; (A == XSB_SUSPEND -> Action = suspend) ).
xsb_flag(exception_pre_action, Mode) :-
	stat_flag(EXCEPTION_PRE_ACTION, Flag),
	( Flag =:= PRINT_INCOMPLETE_ON_ABORT -> 
	  Mode = print_incomplete_tables 
	; Mode = none ).
xsb_flag(exception_action, Mode) :-
	stat_flag(EXCEPTION_ACTION, Flag),
	( Flag =:= UNDEFINED_TRUTH_VALUE -> 
	  Mode = undefined_truth_value
	; Mode = iso ).
xsb_flag(ctrace,Mode):- 
	stat_flag(CTRACE_CALLS, Flag),
	( Flag =:= 0 -> Mode = off
          ; (Flag =:= 1 -> Mode = calls_only 
           ; (Flag =:= 2 -> Mode = partial
             ; Mode = full) ) ).
xsb_flag(character_set,Mode):- 
	stat_flag(CHARACTER_SET, Flag),
	(Flag =:= LATIN_1
	 ->	Mode = latin_1
	 ; Flag =:= CP1252
	 ->	Mode = cp1252
	 ;	Mode = utf_8
	).
xsb_flag(alt_semantics,Mode):- 
	stat_flag(ALTERNATE_SEMANTICS, Flag),
	(Flag =:= NAF_CS
	 ->	Mode = cs
	 ; Flag =:= WEAK_CS
	 ->	Mode = weak_cs
	 ; Flag =:= GFP_SEMANTICS  % Not yet supported.
	 ->	Mode = gfp
         ;      Mode = off
	).
xsb_flag(errors_with_position,Mode):- 
	stat_flag(ERRORS_WITH_POSITION, Flag),
	( Flag =:= 0 -> Mode = off ; Mode = on ).
xsb_flag(float_display_format,Num):- 
    file_function(GET_FLOAT_DISPLAY_FORMAT,P,_),
    atom_codes(P,[_,_,_|R]),
    remove_last(R,NumL),
    atom_codes(Num,NumL).
xsb_flag(table_calling_method,Method):-
    stat_flag(TABLING_METHOD,C_Method),
    (C_Method =:= VARIANT_EVAL_METHOD ->
	 Method = variant
       ; Method = subsumptive).
xsb_flag(load_dync_error_limit,Num):-
    stat_flag(LOAD_DYNC_ERROR_LIMIT,Num).
	   
remove_last([_],[]):- !.
remove_last([H|T],[H|T1]):- 
    remove_last(T,T1).

%:- import writeln/1 from standard.
:- mode set_xsb_flag(+,+).
set_xsb_flag(Type,Val):- 
%	writeln(set_xsb_flag(Type,Val)),
	check_ground(Type,set_xsb_flag/2,1),
	check_ground(Val,set_xsb_flag/2,2),
	set_xsb_flag_1(Type,Val).

set_xsb_flag_1(tripwire(F),V):- 
	tripwire_1(F),
	(  V = action(A),
	   tripwire_action(F,FA),
  	   set_xsb_flag_1(FA,A)
	 ; V = limit(P),
	   set_xsb_flag_1(F,P) ),!.
%set_xsb_flag(tripwire(F),V):- 
%	tripwire_1(F),
%	set_xsb_flag_1(F,V).
set_xsb_flag_1(write_depth, Depth) :- !, 
	(integer(Depth) -> 
	    stat_set_flag(WRITE_DEPTH, Depth)
	;   type_error(integer,Depth,set_xsb_flag/2,2) ).
set_xsb_flag_1(backtrace_on_error, BT) :-  !, 
	(BT == off -> Value = 0 ; Value = 1),
	stat_set_flag(BACKTRACE, Value). 
set_xsb_flag_1(debugging, Debug) :-  !, 
	(Debug == off -> Value = 0 ; Value = 1),
	stat_set_flag(DEBUG_ON, Value). 
set_xsb_flag_1(tracing, Trace) :-  !, 
	(Trace == off -> Value = 0 ; Value = 1),
	stat_set_flag(TRACE, Value).
set_xsb_flag_1(verboseness,Level) :-  !, 
	stat_set_flag(VERBOSENESS_LEVEL,Level).
% TES: dont think setting makes sense here.
%% Goal passed on cmd line
%set_xsb_flag_1(goal, Goal) :-
%	stat_flag(CMD_LINE_GOAL, GoalUninterned),
%	intern_string(GoalUninterned, Goal).
set_xsb_flag_1(dcg_style, DcgMode) :-  !, 
	( DcgMode == xsb -> Value = 0 ; Value = 1 ),
	stat_set_flag(DCG_MODE, Value).
set_xsb_flag_1(garbage_collection, GCmode) :-  !, 
	set_xsb_flag_1(heap_garbage_collection, GCmode).
set_xsb_flag_1(heap_garbage_collection, GCmode) :-  !, 
	( GCmode = none -> GCflag = NO_GC 
	; GCmode = sliding -> GCflag = SLIDING_GC 
	; GCmode = copying -> GCflag = COPYING_GC
	; GCmode = indirection -> GCflag = INDIRECTION_SLIDE_GC
	; domain_error([none,sliding,copying,indirection],
                       GCmode,set_xsb_flag/2,2,'garbage_collection clause')
	),
	stat_set_flag(GARBAGE_COLLECT, GCflag).
set_xsb_flag_1(heap_margin, Margin) :-  !, 
	stat_set_flag(HEAP_GC_MARGIN, Margin).
set_xsb_flag_1(write_attributes,Code) :-  !, 
	( Code = ignore -> Flag = WA_IGNORE
	; Code = dots -> Flag = WA_DOTS
	; Code = portray -> Flag = WA_PORTRAY
	; Code = write -> Flag = WA_WRITE
	; domain_error([ignore,dots,portray,write],
                       Code,set_xsb_flag/2,2,'write_attributes clause')
	),
	stat_set_flag(WRITE_ATTRIBUTES, Flag).
set_xsb_flag_1(clause_garbage_collection, OnOff) :-  !, 
	( OnOff == off -> Value = 0 ; Value = 1 ),
	stat_set_flag(CLAUSE_GARBAGE_COLLECT, Value).
set_xsb_flag_1(atom_garbage_collection, OnOff) :-  !, 
	( OnOff == off -> Value = 0 ; Value = 1 ),
	stat_set_flag(STRING_GARBAGE_COLLECT, Value).
set_xsb_flag_1(gc_verbose_level,Level) :-  !, 
	(   Level = 0
	-> stat_set_flag(VERBOSE_GC,0),
	    stat_set_flag(COUNT_CHAINS,0),
	    stat_set_flag(EXAMINE_DATA,0)
	;   Level = 1
	->  stat_set_flag(VERBOSE_GC,1),
	    stat_set_flag(COUNT_CHAINS,0),
	    stat_set_flag(EXAMINE_DATA,0)
	;   Level = 2
	->  stat_set_flag(VERBOSE_GC,1),
	    stat_set_flag(COUNT_CHAINS,0),
	    stat_set_flag(EXAMINE_DATA,1)
	;   Level = 3
	->  stat_set_flag(VERBOSE_GC,1),
	    stat_set_flag(COUNT_CHAINS,1),
	    stat_set_flag(EXAMINE_DATA,1)
	;   domain_error([0,1,2,3],Level,set_xsb_flag/2,2,'gc_verbose_level clause')
	).
set_xsb_flag_1(verbose_gc, VerboseGC) :-  !, 
	(   VerboseGC = yes
	->  stat_set_flag(VERBOSE_GC, 1)
	;   VerboseGC = no
	->  stat_set_flag(VERBOSE_GC, 0)
	;     domain_error([0,1],VerboseGC,set_xsb_flag/2,2,'verbose_gc clause')
	).
set_xsb_flag_1(count_chains, CountYN) :-  !, 
	(   CountYN = yes
	->  stat_set_flag(COUNT_CHAINS, 1)
	;   CountYN = no
	->  stat_set_flag(COUNT_CHAINS, 0)
	;     domain_error([0,1],CountYN,set_xsb_flag/2,2,'count_chains clause')
	).
set_xsb_flag_1(examine_data, ExamineYN) :-  !, 
	(   ExamineYN = yes
	->  stat_set_flag(EXAMINE_DATA, 1)
	;   ExamineYN = no
	->  stat_set_flag(EXAMINE_DATA,0)
	;     domain_error([0,1],ExamineYN,set_xsb_flag/2,2,
	                  'examine_data clause')
	).
set_xsb_flag_1(table_gc_action, Action) :- !,
	(Action = abolish_tables_transitively -> 
	    stat_set_flag(TABLE_GC_ACTION, ABOLISH_TABLES_TRANSITIVELY)
	 ;  (Action = abolish_tables_singly -> 
	       stat_set_flag(TABLE_GC_ACTION, ABOLISH_TABLES_SINGLY)
	     ; domain_error([abolish_tables_transitively,', ',
	                     abolish_tables_singly],Action,set_xsb_flag/2,2,
			     'table_gc_action clause') ) ).
set_xsb_flag_1(thread_pdlsize, Size) :-!,
	stat_set_flag(THREAD_PDLSIZE, Size).
set_xsb_flag_1(thread_glsize, Size) :-!,
	stat_set_flag(THREAD_GLSIZE, Size).
set_xsb_flag_1(thread_tcpsize, Size) :-!,
	stat_set_flag(THREAD_TCPSIZE, Size).
set_xsb_flag_1(thread_complsize, Size) :-!,
	stat_set_flag(THREAD_COMPLSIZE, Size).
set_xsb_flag_1(thread_detached, Bool) :- !,
	(Bool == true -> B = 1 ; B = 0),
	stat_set_flag(THREAD_DETACHED, B).
set_xsb_flag_1(max_queue_terms, Num) :- !,
	(integer(Num) -> 
	    stat_set_flag(MAX_QUEUE_TERMS, Num)
	;   type_error(integer,Num,set_xsb_flag/2,2) ).
set_xsb_flag_1(warning_action,Action):- !,
	(Action == print_warning ->  A = PRINT_WARNING
	  ; (Action == silent_warning ->  A = SILENT_WARNING
	     ; (Action == error_warning ->  A = ERROR_WARNING
	        ; domain_error([print_warning,', ',silent_warning,', ',error_warning],
	                        Action,set_xsb_flag/2,2,'warning_action clause') ) ) ),
	 stat_set_flag(WARNING_ACTION,A).

% These next 3 are basically internal and should rarely need to be reset.
set_xsb_flag_1(cyclic_check_size, Size) :-!,
	stat_set_flag(CYCLIC_CHECK_SIZE,Size).
set_xsb_flag_1(max_table_subgoal_var_num, Size) :-!,
	( (integer(Size),Size > 0, Size =< 40000) -> 
	  stat_set_flag(MAX_TABLE_SUBGOAL_VAR_NUM, Size)
	 ; misc_error('Value must be an integer between 1 and 64000 for the flag max_table_subgoal_var_num') ).
set_xsb_flag_1(max_table_answer_var_num, Size) :-!,
	( (integer(Size),Size > 0, Size =< 40000) -> 
	  stat_set_flag(MAX_TABLE_ANSWER_VAR_NUM, Size)
	 ; misc_error('Value must be an integer between 1 and 40000 for the flag max_table_answer_var_num') ).

% The next set is for tripwires.
set_xsb_flag_1(max_table_subgoal_size, Size) :-!,
	(Size =:= 0 -> 
	  iso_flag(max_integer,MaxInt),
	  stat_set_flag(MAX_TABLE_SUBGOAL_SIZE, MaxInt)
	; stat_set_flag(MAX_TABLE_SUBGOAL_SIZE, Size)).
set_xsb_flag_1(max_table_subgoal_size_action, Action) :-!,
	(Action == error -> stat_set_flag(MAX_TABLE_SUBGOAL_ACTION, XSB_ERROR)
         ; Action == suspend ->  stat_set_flag(MAX_TABLE_SUBGOAL_ACTION, XSB_SUSPEND)
         ; Action == abstract ->  stat_set_flag(MAX_TABLE_SUBGOAL_ACTION, XSB_ABSTRACT)
         ; domain_error([error,', ',suspend,', ',abstract],
			Action,set_xsb_flag/2,2,'max_table_subgoal_action clause') ).
set_xsb_flag_1(max_table_answer_size, Size) :-!,
	(Size =:= 0 -> 
	  iso_flag(max_integer,MaxInt),
	  stat_set_flag(MAX_TABLE_ANSWER_METRIC, MaxInt)
	; stat_set_flag(MAX_TABLE_ANSWER_METRIC, Size) ).
set_xsb_flag_1(max_table_answer_size_action, Action) :-!,
	(Action == error -> stat_set_flag(MAX_TABLE_ANSWER_ACTION, XSB_ERROR)
         ; (Action == bounded_rationality ; Action == abstract) ->  
	       stat_set_flag(MAX_TABLE_ANSWER_ACTION, XSB_BRAT),
	       reinitialize_undefineds
         ; Action == suspend -> stat_set_flag(MAX_TABLE_ANSWER_ACTION, XSB_SUSPEND)
         ; Action == custom -> stat_set_flag(MAX_TABLE_ANSWER_ACTION, XSB_CUSTOM)
         ; domain_error([error,', ',suspend,', ',abstract],
                        Action,set_xsb_flag/2,2,'max_table_answer_action clause') ).
set_xsb_flag_1(max_incomplete_subgoals, Max) :-!,
	(Max =:= 0 -> 
	  iso_flag(max_integer,MaxInt),
	  stat_set_flag(MAX_INCOMPLETE_SUBGOALS, MaxInt)
	; stat_set_flag(MAX_INCOMPLETE_SUBGOALS, Max) ).
set_xsb_flag_1(max_incomplete_subgoals_action, Action) :-!,
	(Action == error -> stat_set_flag(MAX_INCOMPLETE_SUBGOALS_ACTION, XSB_ERROR)
         ; Action == warning -> stat_set_flag(MAX_INCOMPLETE_SUBGOALS_ACTION, XSB_WARNING)
         ; Action == suspend -> stat_set_flag(MAX_INCOMPLETE_SUBGOALS_ACTION, XSB_SUSPEND)
         ; Action == custom -> stat_set_flag(MAX_INCOMPLETE_SUBGOALS_ACTION, XSB_CUSTOM)
         ; domain_error([error,', ',warning,', ',suspend,', ',custom],
                        Action,set_prolog_flag/2,2,'max_incomplete_subgoals_action clause') ).
set_xsb_flag_1(max_scc_subgoals, Max) :-!,
	(Max =:= 0 -> 
	  iso_flag(max_integer,MaxInt),
	  stat_set_flag(MAX_SCC_SUBGOALS, MaxInt)
	; stat_set_flag(MAX_SCC_SUBGOALS, Max) ).
set_xsb_flag_1(max_scc_subgoals_action, Action) :-!,
	(Action == error -> stat_set_flag(MAX_SCC_SUBGOALS_ACTION, XSB_ERROR)
         ; Action == warning -> stat_set_flag(MAX_SCC_SUBGOALS_ACTION, XSB_WARNING)
         ; Action == suspend -> stat_set_flag(MAX_SCC_SUBGOALS_ACTION, XSB_SUSPEND)
         ; Action == custom -> stat_set_flag(MAX_SCC_SUBGOALS_ACTION, XSB_CUSTOM)
         ; domain_error([error,', ',warning,', ',suspend,', ',custom],
                        Action,set_prolog_flag/2,2,'max_scc_subgoals_action clause') ).
set_xsb_flag_1(max_answers_for_subgoal, Max) :-!,
	(Max =:= 0 -> 
	  iso_flag(max_integer,MaxInt),
	  stat_set_flag(MAX_ANSWERS_FOR_SUBGOAL, MaxInt)
	; stat_set_flag(MAX_ANSWERS_FOR_SUBGOAL, Max) ).
set_xsb_flag_1(max_answers_for_subgoal_action, Action) :-!,
	(Action == complete_soundly -> stat_set_flag(MAX_ANSWERS_FOR_SUBGOAL_ACTION, XSB_ABSTRACT)
	 ; Action == error ->     stat_set_flag(MAX_ANSWERS_FOR_SUBGOAL_ACTION, XSB_ERROR)
         ; Action == suspend ->   stat_set_flag(MAX_ANSWERS_FOR_SUBGOAL_ACTION, XSB_SUSPEND)
         ; Action == custom ->    stat_set_flag(MAX_ANSWERS_FOR_SUBGOAL_ACTION, XSB_CUSTOM)
         ; domain_error([error,', ',suspend,', ',custom,', ',custom,', ',complete_soundly],
                        Action,set_prolog_flag/2,2,'max_answers_for_subgoal_action clause') ).
set_xsb_flag_1(max_memory,Max):-!,
	(integer(Max),Max >= 0 -> 
	     stat_set_flag(MAX_MEMORY,Max)
	 ; real(Max),Max >= 0 -> 
	     sys_main_memory(Mem),
	     Ram1 is Max * Mem,
	     Ram is floor(Ram1/1024),
	     stat_set_flag(MAX_MEMORY,Ram)
	 ;  domain_error(positive_number,Max,set_prolog_flag/2,2,max_memory_clause) ).
set_xsb_flag_1(max_memory_action, Action) :-!,
	(Action == error -> 
	    stat_set_flag(MAX_MEMORY_ACTION, XSB_ERROR) ; 
	 (Action == suspend -> 
	    stat_set_flag(MAX_MEMORY_ACTION, XSB_SUSPEND)
          ; domain_error([error,suspend],Action,set_prolog_flag/2,2,'max_memory_action clause') ) ).
set_xsb_flag_1(alt_semantics,Mode):- 
        (Mode == cs -> 
   	    stat_set_flag(ALTERNATE_SEMANTICS, NAF_CS) ; 
        (Mode == weak_cs -> 
   	    stat_set_flag(ALTERNATE_SEMANTICS, NAF_CS),
            set_prolog_flag(unknown,undefined) ; 
        (Mode == gfp -> 
   	    stat_set_flag(ALTERNATE_SEMANTICS, GFP_SEMANTICS) ; 
        (Mode == off -> 
   	    stat_set_flag(ALTERNATE_SEMANTICS, WFS_SEMANTICS),
            set_prolog_flag(unknown,error) ; 
          domain_error([cs,weak_cs,gfp,off],Mode,set_prolog_flag/2,2,'alt_semantics clause') ) ) ) ).

set_xsb_flag_1(ctrace, OnOff) :-!,
	((OnOff == on ; OnOff == full) -> stat_set_flag(CTRACE_CALLS, 3)
         ; (OnOff == partial -> stat_set_flag(CTRACE_CALLS, 2)
            ; (OnOff == calls_only -> stat_set_flag(CTRACE_CALLS, 1)
               ;  stat_set_flag(CTRACE_CALLS, 0) ) ) ).
set_xsb_flag_1(unify_with_occurs_check, OnOff) :-!,
	(OnOff == on -> 
	    stat_set_flag(UNIFY_WITH_OCCURS_CHECK_FLAG, 1)
	 ;  stat_set_flag(UNIFY_WITH_OCCURS_CHECK_FLAG, 0)).
set_xsb_flag_1(exception_pre_action, Mode) :- !,
         (Mode = print_incomplete_tables-> 
	  stat_set_flag(EXCEPTION_PRE_ACTION, PRINT_INCOMPLETE_ON_ABORT)
	; (Mode = none ->
    	      stat_set_flag(EXCEPTION_PRE_ACTION, 0)
	    ; domain_error([print_incomplete_tables,none],
                        Mode,set_prolog_flag/2,2,'exception_pre_action clause') ) ).
set_xsb_flag_1(exception_action, Mode) :- !,
         (Mode = undefined_truth_value-> 
	  stat_set_flag(EXCEPTION_ACTION, UNDEFINED_TRUTH_VALUE),
	  reinitialize_undefineds
%	  ensure_loaded(tables)
	; (Mode = iso ->
    	      stat_set_flag(EXCEPTION_ACTION, 0)
	    ; domain_error([iso,undefined_truth_value],
                        Mode,set_prolog_flag/2,2,'exception_action clause') ) ).
set_xsb_flag_1(max_tab_usage, OnOff) :-!,
	(OnOff == on -> 
	    stat_set_flag(MAX_USAGE, 1)
	 ;  stat_set_flag(MAX_USAGE, 0)).
set_xsb_flag_1(character_set,Mode):- !,
	(Mode = latin_1
	 ->	stat_set_flag(CHARACTER_SET, LATIN_1)
	 ; Mode = cp1252
	 ->	stat_set_flag(CHARACTER_SET, CP1252)
	 ; Mode == utf_8
	 ->	stat_set_flag(CHARACTER_SET, UTF_8)
	 ;	domain_error([iso,undefined_character_set],
			     Mode,set_prolog_flag/2,2,'character_set clause')
	).
set_xsb_flag_1(errors_with_position,Mode):- !,
	( Mode = on -> 
	    stat_set_flag(ERRORS_WITH_POSITION, 1)
	 ;  (Mode = off, stat_set_flag(ERRORS_WITH_POSITION, 0))).
set_xsb_flag_1(float_display_format,P):- 
    integer(P),P >= 1,P =< 17,!,
    number_codes(P,PList),append([37,49,46],PList,L1),append(L1,[103],Flist),
    atom_codes(Format,Flist),
%    concat_atom(['%1.',P,g],Format),
    file_function(PUT_FLOAT_DISPLAY_FORMAT,Format,_).
set_xsb_flag_1(Type, _) :-
	(	iso_flag(Type)
	;	prolog_commons_flag(Type)
	),
	permission_error(modify,flag,Type,set_prolog_flag/2).
set_xsb_flag_1(table_calling_method,Method):-
    (Method = variant ->
	 stat_set_flag(TABLING_METHOD,VARIANT_EVAL_METHOD)
      ;  (Method = subsumptive -> 
	      stat_set_flag(TABLING_METHOD,SUBSUMPTIVE_EVAL_METHOD)
	 ;    domain_error([table_calling_method],
			     Method,set_prolog_flag/2,2,'table_calling_method clause')
	) ).
set_xsb_flag_1(load_dync_error_limit, Num) :-
    ((integer(Num),Num >= 0) ->
	 (Num > 0 -> SetNum = Num ; (iso_flag(max_integer,Maxint),SetNum = Maxint)),
    stat_set_flag(LOAD_DYNC_ERROR_LIMIT,SetNum)).
set_xsb_flag_1(Type, _) :-
	domain_error(prolog_flag,Type,set_prolog_flag/2,1).

:- export current_xsb_param/2.
:- import concat_atom/2 from string.
:- import atom_chars/2 from standard.

:- mode current_xsb_param(?,?).
current_xsb_param(Flag, Value) :- 
	(var(Flag) ->
	  true
	; \+ atom(Flag) ->
	  type_error(atom_or_variable,Flag,current_prolog_flag/2,1)
	 ; true ),
	(var(Value) ->
	   true
	 ; type_error(var,Value,current_xsb_param/2,2) ),
	current_xsb_param_1(Flag,Value).

current_xsb_param_1(float_display_format, Value) :- 
    file_function(GET_FLOAT_DISPLAY_FORMAT,Param,_),
    parse_float(Param,Width,Prec,Spec),
    Value = [width(Width),precision(Prec),specifier(Spec)].
current_xsb_param_1(max_table_subgoal_size, [limit(Max),action(Action)]) :- 
    xsb_flag(max_table_subgoal_size,Max),
    xsb_flag(max_table_subgoal_size_action, Action).
current_xsb_param_1(max_table_answer_size, [limit(Max),action(Action)]) :- 
    xsb_flag(max_table_answer_size,Max),
    xsb_flag(max_table_answer_size_action, Action).
current_xsb_param_1(max_incomplete_subgoals, [limit(Max),action(Action)]) :- 
    xsb_flag(max_incomplete_subgoals,Max),
    xsb_flag(max_incomplete_subgoals_action, Action).
current_xsb_param_1(max_scc_subgoals, [limit(Max),action(Action)]) :- 
    xsb_flag(max_scc_subgoals,Max),
    xsb_flag(max_scc_subgoals_action, Action).
current_xsb_param_1(max_memory, [limit(Max),action(Action)]) :- 
    xsb_flag(max_memory,Max),
    xsb_flag(max_memory_action, Action).
current_xsb_param_1(max_answers_for_subgoal, [limit(Max),action(Action)]) :- 
    xsb_flag(max_answers_for_subgoal,Max),
    xsb_flag(max_answers_for_subgoal_action, Action).

:- export set_xsb_param/2.
:- mode set_xsb_param(+,+).
set_xsb_param(Flag, Value) :- 
	(	var(Flag) ->
		true
	;	\+ atom(Flag) ->
		type_error(atom,Flag,current_xsb_param/2,1)
	;	set_xsb_param_1(Flag,Value)
	 ),!.
set_xsb_param(Type, _) :-
	domain_error(xsb_param,Type,set_xsb_param/2,1).

set_xsb_param_1(float_display_format, Value) :-
    file_function(GET_FLOAT_DISPLAY_FORMAT,Param,_),
    parse_float(Param,Width,Prec,Spec),
    parse_new_float_params(Value,VW,VP,VS),
    (var(VW) -> NewW = Width ; NewW = VW),
    (var(VP) -> NewP = Prec ; NewP = VP),
    (var(VS) -> NewS = Spec ; NewS = VS),
    concat_atom(['%',NewW,'.',NewP,NewS],NewFormat),
    file_function(PUT_FLOAT_DISPLAY_FORMAT,NewFormat,_).
set_xsb_param_1(max_table_subgoal_size, List) :-
    check_tripwire_param_list(List),
    (member(limit(L),List) -> set_prolog_flag(max_table_subgoal_size,L) ; true),
    (member(action(A),List) -> set_prolog_flag(max_table_subgoal_size_action,A) ; true).
set_xsb_param_1(max_table_answer_size, List) :-
    check_tripwire_param_list(List),
    (member(limit(L),List) -> set_prolog_flag(max_table_answer_size,L) ; true),
    (member(action(A),List) -> set_prolog_flag(max_table_answer_size_action,A) ; true).
set_xsb_param_1(max_scc_subgoals, List) :-
    check_tripwire_param_list(List),
    (member(limit(L),List) -> set_prolog_flag(max_scc_subgoals,L) ; true),
    (member(action(A),List) -> set_prolog_flag(max_scc_subgoals_action,A) ; true).
set_xsb_param_1(max_incomplete_subgoals, List) :-
    check_tripwire_param_list(List),
    (member(limit(L),List) -> set_prolog_flag(max_incomplete_subgoals,L) ; true),
    (member(action(A),List) -> set_prolog_flag(max_incomplete_subgoals_action,A) ; true).
set_xsb_param_1(max_memory, List) :-
    check_tripwire_param_list(List),
    (member(limit(L),List) -> set_prolog_flag(max_memory,L) ; true),
    (member(action(A),List) -> set_prolog_flag(max_memory_action,A) ; true).
set_xsb_param_1(max_answers_for_subgoal, List) :-
    check_tripwire_param_list(List),
    (member(limit(L),List) -> set_prolog_flag(max_answers_for_subgoal,L) ; true),
    (member(action(A),List) -> set_prolog_flag(max_answers_for_subgoal_action,A) ; true).

check_tripwire_param_list([]).
check_tripwire_param_list([H|T]):-
    (H = action(_) -> true
    ; H = limit(_) -> true
    ; domain_error(tripwire_param,H,set_xsb_param/2,1) ),
    check_tripwire_param_list(T).
    
parse_new_float_params([],_,_,_).
parse_new_float_params([H|T],W,P,S):- 
    parse_new_float_param(H,W,P,S),
    parse_new_float_params(T,W,P,S).

parse_new_float_param(width(W),W,_P,_S):-
    integer(W),W >= 1,W =< 30,!.
parse_new_float_param(precision(P),_W,P,_S):-
    integer(P),P >= 1,P =< 30,!.
parse_new_float_param(specifier(S),_W,_P,S):-
    member(S,[g,f,'G','F']),!.
parse_new_float_param(Param,_W,_P,_S):-
    domain_error(float_display_format_param,Param,set_xsb_param/2,2).

parse_float(Param,Width,Prec,Spec):-
    atom_chars(Param,Plist),
    get_float_width(Plist,Width,Plist1),
    get_float_precision(Plist1,Prec,Spec).
    
get_float_width(Plist,Width,Plist1):-
    (Plist = ['%',Width,'.'|Plist1] ->
       true
     ; Plist = [W1,W2,'.'|Plist1],
       concat_atom([W1,W2],Width) ).

get_float_precision(Plist,Prec,Spec):- 
    (Plist = [P1,P2,Spec] ->
       concat_atom([P1,P2],Prec)
     ; Plist = [Prec,Spec]).

/* ==================================================================== */

:- mode garbage_collection(+).
garbage_collection(GC) :- var(GC), !,
	abort('Uninstantiated argument 1 of garbage_collection/1').
garbage_collection(none) :- !, stat_set_flag(GARBAGE_COLLECT, NO_GC).
garbage_collection(sliding) :- !, stat_set_flag(GARBAGE_COLLECT, SLIDING_GC).
garbage_collection(copying) :- !, stat_set_flag(GARBAGE_COLLECT, COPYING_GC).
garbage_collection(indirection) :- !, stat_set_flag(GARBAGE_COLLECT, INDIRECTION_SLIDE_GC).
garbage_collection(_) :-
	abort('Wrong type in argument 1 of garbage_collection/1').


/* ===== hilog_symbol/1 =============================================== */

%   hilog_symbol(?Symbol)
%   Succeeds iff Symbol is one of the symbols that are declared as HiLog.

:- mode hilog_symbol(?).
hilog_symbol(HS) :- 
	'_$hilog_symbol'(HS).

/* ===== Auxilliary Predicates ======================================== */

get_first_psc_pair(PSC_Pair) :-
	stat_flag(MOD_LIST, PSC_Pair).

/* ===== predicate_property_20180213.P ======================================== */
:- export predicate_property_20180213/2.
:- import psc_shared/2,psc_get_incr/2,psc_tabled/2 from machine.

:- mode predicate_property_20180213(+,?).
predicate_property_20180213(Term_Indicator, Property) :-
	nonvar(Term_Indicator),
	Term_Indicator = :(Module, Term),
	nonvar(Term),		% send ':'(_,_) predicate to next clause
	!,
	get_first_psc_pair(PSC_Pair),
	module_scan(PSC_Pair, ModuleName, ModulePSC),
	ModuleName = Module,
	psc_prop(ModulePSC, First_PSC_Pair),	% get the first psc_pair 
						% of this module
	( First_PSC_Pair =:= 1 ->	% We are dealing with a global module!
		predicate_property_20180213(Term, Property)
        ; psc_scan_in_module(First_PSC_Pair, PSC),
	  psc_name(PSC, Name),
	  psc_arity(PSC, Arity),
	  functor(Term, Name, Arity),
	  psc_properties_20180213(PSC, Property)
	).
predicate_property_20180213(Term, Property) :-	% This takes care of global 
					% modules,"usermod" and "global".
	term_type(Term, Type),
	( Type =:= XSB_STRUCT -> % Hacked to make it fast when Term is given.
			% If Term is given and it is represented as a structure 
			% (has a PSC record) then using term_psc/2 we get to 
			% this PSC record directly without having to scan the 
			% whole PSC-Pair list.
		term_psc(Term, PSC),
		psc_properties_20180213(PSC, Property)
	; Type =:= XSB_STRING ->
		conpsc(Term, PSC),
		psc_properties_20180213(PSC, Property)
	; psc_scan_in_hash_table(0, PSC),
	  psc_name(PSC, Name),
	  psc_arity(PSC, Arity), 
	  functor(Term, Name, Arity),
	  psc_properties_20180213(PSC, Property)
	).

psc_properties_20180213(PSC, Property) :- 
	psc_type(PSC, Type),
%	( Type =:= T_ORDI -> Property = unclassified
        ( Type =:= T_ORDI -> psc_env(PSC,Env),Env = T_IMPORTED,Property = unloaded
	; Type =:= T_DYNA -> Property = (dynamic)
	; Type =:= T_PRED -> Property = static
	; Type =:= T_UDEF -> Property = unloaded
	; Type =:= T_FORN -> Property = foreign
	; Type =:= 14 -> Property = function ).
%psc_properties_20180213(PSC, static) :-
%	\+ (psc_type(PSC, Type), Type =:= T_DYNA).
psc_properties_20180213(PSC, Property) :-
	psc_env(PSC, Env),
 	( Env =:= T_EXPORTED -> psc_type(PSC,Type),Type \== T_ORDI,Property = exported	
	; Env =:= T_LOCAL -> Property = (local)
	; Env =:= T_IMPORTED -> psc_mod(PSC, ModulePSC), 
		       psc_name(ModulePSC, ModuleName), 
		       Property = imported_from(ModuleName)
	; Property = global ).
psc_properties_20180213(PSC, Property) :- 
	psc_spy(PSC, Spy),
	( Spy =\= 0 -> Property = spied ).
psc_properties_20180213(PSC, Property) :- 
	psc_tabled(PSC, Tabled),
	writeln(psc_tabled(PSC, Tabled)),
	(psc_type(PSC, Type),Type =:= T_DYNA -> 
	    psc_get_incr(PSC,Incr),Incr == NONINCREMENTAL
	  ; true),  
	( (Tabled /\ 12 > 0),Property = tabled 
        ; 
	 ( Tabled == T_VARIANT -> Property = variant
         ; Tabled == T_SUBSUMPTIVE -> Property = subsumptive ) ).
%         ; Tabled == T_TABLED_UNDETERMINED -> Property = tabled(default)).
psc_properties_20180213(PSC, Property) :- 
	psc_get_incr(PSC, Incr),
	( Incr == INCREMENTAL -> 
	    Property = incremental 
	  ; Incr == OPAQUE -> Property = opaque).
psc_properties_20180213(PSC, Property) :- 
	psc_shared(PSC, Shared),
	( Shared =\= 0 -> Property = shared ).
psc_properties_20180213(PSC, Property) :-
	psc_name(PSC, Functor),		% Finding the Functor and the Arity
	psc_arity(PSC, Arity),		% twice is silly, isn''t it?
	(	standard_metapredicate_template(Functor,Arity,Template),
%		setof(Meta, standard_metapredicate(Functor, Arity, Meta), Metas),
%		psc_meta_predicate_property_args(Metas, 1, Arity, Args),
%		Template =.. [Functor| Args],
		Property = meta_predicate(Template)
	;	standard_symbol(Functor, Arity, _) ->
		( %	Property = xsb_standard_predicate ; 
			Property = built_in
		)
	).
	% Unfortunately this doesn''t take the Module into account.

/* ----------------- end of file current_symbols.P -------------------- */

