/*
** File: packages/rdf/rewrite.P
** Author: Aneesh Ali
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C): 2010, University of Amsterdam
** Copyright (C) The Research Foundation of SUNY, 2011
** 
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**      http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
**
*/

:- module(rewrite,
	  [ rewrite/2,			% +Rule, +Input
	    rew_term_expansion/2,
	    rew_goal_expansion/2
	  ]).

:- op(1200, xfx, (::=)).

:- meta_predicate
	rewrite(1, +).

		 /*******************************
		 *	    COMPILATION		*
		 *******************************/

%%term_expansion(F, T) :- rew_term_expansion(F, T).

rew_term_expansion((Rule ::= RuleBody), (Head :- Body)) :-
	translate(RuleBody, Term, Body0),
	simplify(Body0, Body),
	Rule =.. [Name|List],
	Head =.. [Name,Term|List].

rew_goal_expansion(rewrite(To, From), Goal) :-
	nonvar(To),
	To = \Rule,
	callable(Rule),
	Rule =.. [Name|List],
	Goal =.. [Name,From|List].


		 /*******************************
		 *	      TOPLEVEL		*
		 *******************************/

%%	rewrite(:To, +From)
%
%	Invoke the term-rewriting system

rewrite(To, From) :-
	rew_goal_expansion(rewrite(To, From), Goal), 
	call(Goal).

rewrite(M:T, From) :-
	(   var(T)
	->  From = T
	;   T = \Rule
	->  Rule =.. [Name|List],
	    Goal =.. [Name,From|List],
	    M:Goal
	;   match(T, M, From)
	).

match(Rule, M, From) :-
	translate(Rule, From, Code),
	M:Code.

translate(Var, Var, true) :-
	var(Var), !.
translate((\Command, !), Var, (Goal, !)) :- !,
	(   callable(Command),
	    Command =.. [Name|List]
	->  Goal =.. [Name,Var|List]
	;   Goal = rewrite(\Command, Var)
	).
translate(\Command, Var, Goal) :- !,
	(   callable(Command),
	    Command =.. [Name|List]
	->  Goal =.. [Name,Var|List]
	;   Goal = rewrite(\Command, Var)
	).
translate(Atomic, Atomic, true) :-
	atomic(Atomic), !.
translate(C, _, Cmd) :-
	command(C, Cmd), !.
translate((A, B), T, Code) :-
	(   command(A, Cmd)
	->  !, translate(B, T, C),
	    Code = (Cmd, C)
	;   command(B, Cmd)
	->  !, translate(A, T, C),
	    Code = (C, Cmd)
	).
translate(Term0, Term, Command) :-
	functor(Term0, Name, Arity),
	functor(Term, Name, Arity),
	translate_args(0, Arity, Term0, Term, Command).

translate_args(N, N, _, _, true) :- !.
translate_args(I0, Arity, T0, T1, (C0,C)) :-
	I is I0 + 1,
	arg(I, T0, A0),
	arg(I, T1, A1),
	translate(A0, A1, C0),
	translate_args(I, Arity, T0, T1, C).

command(0, _) :- !,			% catch variables
	fail.
command({A}, A).
command(!, !).

		 /*******************************
		 *	      SIMPLIFY		*
		 *******************************/

%%	simplify(+Raw, -Simplified)
%
%	Get rid of redundant `true' goals generated by translate/3.

simplify(V, V) :-
	var(V), !.
simplify((A0,B), A) :-
	B == true, !,
	simplify(A0, A).
simplify((A,B0), B) :-
	A == true, !,
	simplify(B0, B).
simplify((A0, B0), C) :- !,
	simplify(A0, A),
	simplify(B0, B),
	(   (   A \== A0
	    ;	B \== B0
	    )
	->  simplify((A,B), C)
	;   C = (A,B)
	).
simplify(X, X).

