"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getInfraDeprecationsFactory = exports.configDeprecations = void 0;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deprecatedFieldMessage = (fieldName, defaultValue, configNames) => _i18n.i18n.translate('xpack.infra.deprecations.deprecatedFieldDescription', {
  defaultMessage: 'Configuring the "{fieldName}" field has been deprecated and will be removed in 8.0.0. This plugin is designed to work with ECS, and expects this field to have a value of `{defaultValue}`. It has a different value configured in Source {configCount, plural, one {Configuration} other {Configurations}}: {configNames}',
  values: {
    fieldName,
    defaultValue,
    configNames: configNames.join(', '),
    configCount: configNames.length
  }
});
const DEFAULT_VALUES = {
  timestamp: '@timestamp',
  tiebreaker: '_doc',
  container: 'container.id',
  host: 'host.name',
  pod: 'kubernetes.pod.uid'
};
const FIELD_DEPRECATION_FACTORIES = {
  timestamp: configNames => ({
    level: 'critical',
    title: _i18n.i18n.translate('xpack.infra.deprecations.timestampFieldTitle', {
      defaultMessage: 'Source configuration field "timestamp" is deprecated.'
    }),
    message: deprecatedFieldMessage(_i18n.i18n.translate('xpack.infra.deprecations.timestampFieldName', {
      defaultMessage: 'timestamp'
    }), DEFAULT_VALUES.timestamp, configNames),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.infra.deprecations.timestampAdjustIndexing', {
        defaultMessage: 'Adjust your indexing to use "{field}" as a timestamp.',
        values: {
          field: '@timestamp'
        }
      })]
    }
  }),
  tiebreaker: configNames => ({
    level: 'critical',
    title: _i18n.i18n.translate('xpack.infra.deprecations.tiebreakerFieldTitle', {
      defaultMessage: 'Source configuration field "tiebreaker" is deprecated.'
    }),
    message: deprecatedFieldMessage(_i18n.i18n.translate('xpack.infra.deprecations.tiebreakerFieldName', {
      defaultMessage: 'tiebreaker'
    }), DEFAULT_VALUES.tiebreaker, configNames),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.infra.deprecations.tiebreakerAdjustIndexing', {
        defaultMessage: 'Adjust your indexing to use "{field}" as a tiebreaker.',
        values: {
          field: '_doc'
        }
      })]
    }
  }),
  host: configNames => ({
    level: 'critical',
    title: _i18n.i18n.translate('xpack.infra.deprecations.hostnameFieldTitle', {
      defaultMessage: 'Source configuration field "host name" is deprecated.'
    }),
    message: deprecatedFieldMessage(_i18n.i18n.translate('xpack.infra.deprecations.hostnameFieldName', {
      defaultMessage: 'host name'
    }), DEFAULT_VALUES.host, configNames),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.infra.deprecations.hostAdjustIndexing', {
        defaultMessage: 'Adjust your indexing to identify hosts using "{field}"',
        values: {
          field: 'host.name'
        }
      })]
    }
  }),
  pod: configNames => ({
    level: 'critical',
    title: _i18n.i18n.translate('xpack.infra.deprecations.podIdFieldTitle', {
      defaultMessage: 'Source configuration field "pod ID" is deprecated.'
    }),
    message: deprecatedFieldMessage(_i18n.i18n.translate('xpack.infra.deprecations.podIdFieldName', {
      defaultMessage: 'pod ID'
    }), DEFAULT_VALUES.pod, configNames),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.infra.deprecations.podAdjustIndexing', {
        defaultMessage: 'Adjust your indexing to identify Kubernetes pods using "{field}"',
        values: {
          field: 'kubernetes.pod.uid'
        }
      })]
    }
  }),
  container: configNames => ({
    level: 'critical',
    title: _i18n.i18n.translate('xpack.infra.deprecations.containerIdFieldTitle', {
      defaultMessage: 'Source configuration field "container ID" is deprecated.'
    }),
    message: deprecatedFieldMessage(_i18n.i18n.translate('xpack.infra.deprecations.containerIdFieldName', {
      defaultMessage: 'container ID'
    }), DEFAULT_VALUES.container, configNames),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.infra.deprecations.containerAdjustIndexing', {
        defaultMessage: 'Adjust your indexing to identify Docker containers using "{field}"',
        values: {
          field: 'container.id'
        }
      })]
    }
  })
};
const configDeprecations = ({
  deprecate
}) => [...Object.keys(FIELD_DEPRECATION_FACTORIES).map(key => (completeConfig, _rootPath, addDeprecation) => {
  const configuredValue = (0, _lodash.get)(completeConfig, `xpack.infra.sources.default.fields.${key}`);
  if (typeof configuredValue === 'undefined') {
    return completeConfig;
  }
  addDeprecation({
    title: _i18n.i18n.translate('xpack.infra.deprecations.deprecatedFieldConfigTitle', {
      defaultMessage: '"{fieldKey}" is deprecated.',
      values: {
        fieldKey: key
      }
    }),
    message: _i18n.i18n.translate('xpack.infra.deprecations.deprecatedFieldConfigDescription', {
      defaultMessage: 'Configuring "xpack.infra.sources.default.fields.{fieldKey}" has been deprecated and will be removed in 8.0.0.',
      values: {
        fieldKey: key
      }
    }),
    level: 'warning',
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.infra.deprecations.removeConfigField', {
        defaultMessage: 'Remove "xpack.infra.sources.default.fields.{fieldKey}" from your Kibana configuration.',
        values: {
          fieldKey: key
        }
      })]
    }
  });
  return completeConfig;
}), deprecate('sources.default.logAlias', '8.0.0', {
  level: 'critical'
}), deprecate('sources.default.metricAlias', '8.0.0', {
  level: 'critical'
}), deprecate('enabled', '8.0.0', {
  level: 'critical'
})];
exports.configDeprecations = configDeprecations;
const getInfraDeprecationsFactory = sources => async ({
  savedObjectsClient
}) => {
  const deprecatedFieldsToSourceConfigMap = new Map();
  const sourceConfigurations = await sources.getAllSourceConfigurations(savedObjectsClient);
  for (const {
    configuration
  } of sourceConfigurations) {
    const {
      name,
      fields
    } = configuration;
    for (const [key, defaultValue] of Object.entries(DEFAULT_VALUES)) {
      const configuredValue = Reflect.get(fields, key);
      if (configuredValue !== defaultValue) {
        var _deprecatedFieldsToSo;
        const affectedConfigNames = (_deprecatedFieldsToSo = deprecatedFieldsToSourceConfigMap.get(key)) !== null && _deprecatedFieldsToSo !== void 0 ? _deprecatedFieldsToSo : [];
        affectedConfigNames.push(name);
        deprecatedFieldsToSourceConfigMap.set(key, affectedConfigNames);
      }
    }
  }
  const deprecations = [];
  if (deprecatedFieldsToSourceConfigMap.size > 0) {
    for (const [fieldName, affectedConfigNames] of deprecatedFieldsToSourceConfigMap.entries()) {
      const deprecationFactory = Reflect.get(FIELD_DEPRECATION_FACTORIES, fieldName);
      deprecations.push(deprecationFactory(affectedConfigNames));
    }
  }
  return deprecations;
};
exports.getInfraDeprecationsFactory = getInfraDeprecationsFactory;