"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeprecationsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _operators = require("rxjs/operators");
var _deprecations_factory = require("./deprecations_factory");
var _routes = require("./routes");
var _deprecation_config = require("./deprecation_config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * The deprecations service provides a way for the Kibana platform to communicate deprecated
 * features and configs with its users. These deprecations are only communicated
 * if the deployment is using these features. Allowing for a user tailored experience
 * for upgrading the stack version.
 *
 * The Deprecation service is consumed by the upgrade assistant to assist with the upgrade
 * experience.
 *
 * If a deprecated feature can be resolved without manual user intervention.
 * Using correctiveActions.api allows the Upgrade Assistant to use this api to correct the
 * deprecation upon a user trigger.
 *
 * @example
 * ```ts
 * import { DeprecationsDetails, GetDeprecationsContext, CoreSetup } from 'src/core/server';
 * import { i18n } from '@kbn/i18n';
 *
 * async function getDeprecations({ esClient, savedObjectsClient }: GetDeprecationsContext): Promise<DeprecationsDetails[]> {
 *   const deprecations: DeprecationsDetails[] = [];
 *   const count = await getFooCount(savedObjectsClient);
 *   if (count > 0) {
 *     deprecations.push({
 *       title: i18n.translate('xpack.foo.deprecations.title', {
 *         defaultMessage: `Foo's are deprecated`
 *       }),
 *       message: i18n.translate('xpack.foo.deprecations.message', {
 *         defaultMessage: `You have {count} Foo's. Migrate your Foo's to a dashboard to continue using them.`,
 *         values: { count },
 *       }),
 *       documentationUrl:
 *         'https://www.elastic.co/guide/en/kibana/current/foo.html',
 *       level: 'warning',
 *       correctiveActions: {
 *         manualSteps: [
 *            i18n.translate('xpack.foo.deprecations.manualStepOneMessage', {
 *              defaultMessage: 'Navigate to the Kibana Dashboard and click "Create dashboard".',
 *            }),
 *            i18n.translate('xpack.foo.deprecations.manualStepTwoMessage', {
 *              defaultMessage: 'Select Foo from the "New Visualization" window.',
 *            }),
 *         ],
 *         api: {
 *           path: '/internal/security/users/test_dashboard_user',
 *           method: 'POST',
 *           body: {
 *             username: 'test_dashboard_user',
 *             roles: [
 *               "machine_learning_user",
 *               "enrich_user",
 *               "kibana_admin"
 *             ],
 *             full_name: "Alison Goryachev",
 *             email: "alisongoryachev@gmail.com",
 *             metadata: {},
 *             enabled: true
 *           }
 *         },
 *       },
 *     });
 *   }
 *   return deprecations;
 * }
 *
 *
 * export class Plugin() {
 *   setup: (core: CoreSetup) => {
 *     core.deprecations.registerDeprecations({ getDeprecations });
 *   }
 * }
 * ```
 *
 * @public
 */

/**
 * Server-side client that provides access to fetch all Kibana deprecations
 *
 * @public
 */

/** @internal */

/** @internal */

/** @internal */
class DeprecationsService {
  constructor(coreContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "configService", void 0);
    (0, _defineProperty2.default)(this, "deprecationsFactory", void 0);
    this.logger = coreContext.logger.get('deprecations-service');
    this.configService = coreContext.configService;
  }
  async setup({
    http
  }) {
    this.logger.debug('Setting up Deprecations service');
    const config = await this.configService.atPath(_deprecation_config.config.path).pipe((0, _operators.take)(1)).toPromise();
    this.deprecationsFactory = new _deprecations_factory.DeprecationsFactory({
      logger: this.logger,
      config: {
        ignoredConfigDeprecations: config.skip_deprecated_settings
      }
    });
    (0, _routes.registerRoutes)({
      http
    });
    this.registerConfigDeprecationsInfo(this.deprecationsFactory);
    const deprecationsFactory = this.deprecationsFactory;
    return {
      getRegistry: domainId => {
        const registry = deprecationsFactory.getRegistry(domainId);
        return {
          registerDeprecations: registry.registerDeprecations
        };
      }
    };
  }
  start() {
    if (!this.deprecationsFactory) {
      throw new Error('`setup` must be called before `start`');
    }
    return {
      asScopedToClient: this.createScopedDeprecations()
    };
  }
  stop() {}
  createScopedDeprecations() {
    return (esClient, savedObjectsClient) => {
      return {
        getAllDeprecations: this.deprecationsFactory.getAllDeprecations.bind(null, {
          savedObjectsClient,
          esClient
        })
      };
    };
  }
  registerConfigDeprecationsInfo(deprecationsFactory) {
    const handledDeprecatedConfigs = this.configService.getHandledDeprecatedConfigs();
    for (const [domainId, deprecationsContexts] of handledDeprecatedConfigs) {
      const deprecationsRegistry = deprecationsFactory.getRegistry(domainId);
      deprecationsRegistry.registerDeprecations({
        getDeprecations: () => {
          return deprecationsContexts.map(({
            configPath,
            title = `${domainId} has a deprecated setting`,
            level = 'critical',
            message,
            correctiveActions,
            documentationUrl
          }) => ({
            configPath,
            title,
            level,
            message,
            correctiveActions,
            documentationUrl,
            deprecationType: 'config',
            requireRestart: true
          }));
        }
      });
    }
  }
}
exports.DeprecationsService = DeprecationsService;