"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asyncMap = asyncMap;
exports.asyncMapWithLimit = asyncMapWithLimit;
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _rxjs_ = require("../rxjs_7");
var _observable = require("./observable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getAllResults = input => (0, _rxjs_.lastValueFrom)((0, _rxjs.from)(input).pipe((0, _operators.toArray)()));

/**
 * Creates a promise whose values is the array of results produced by calling `fn` for
 * each item in `iterable`. `fn` can return either a Promise or Observable. If `fn`
 * returns observables then they will properly abort if an error occurs.
 *
 * The result array follows the order of the input iterable, even though the calls
 * to `fn` may not. (so avoid side effects)
 *
 * @param iterable Items to iterate
 * @param fn Function to call for each item. Result is added/concatenated into the result array in place of the input value
 */
async function asyncMap(iterable, fn) {
  return await asyncMapWithLimit(iterable, Infinity, fn);
}

/**
 * Creates a promise whose values is the array of results produced by calling `fn` for
 * each item in `iterable`. `fn` can return either a Promise or Observable. If `fn`
 * returns observables then they will properly abort if an error occurs.
 *
 * The number of concurrent executions of `fn` is limited by `limit`.
 *
 * The result array follows the order of the input iterable, even though the calls
 * to `fn` may not. (so avoid side effects)
 *
 * @param iterable Items to iterate
 * @param limit Maximum number of operations to run in parallel
 * @param fn Function to call for each item. Result is added/concatenated into the result array in place of the input value
 */
async function asyncMapWithLimit(iterable, limit, fn) {
  const results$ = (0, _observable.mapWithLimit$)(iterable, limit, async (item, i) => [i, await getAllResults(fn(item, i))]);
  const results = await getAllResults(results$);
  return results.sort(([a], [b]) => a - b).reduce((acc, [, result]) => acc.concat(result), []);
}