import _createForOfIteratorHelper from "@babel/runtime/helpers/createForOfIteratorHelper";
import _classCallCheck from "@babel/runtime/helpers/classCallCheck";
import _createClass from "@babel/runtime/helpers/createClass";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { createApplicationUsageMetric } from './metrics';
export var ApplicationUsageTracker = /*#__PURE__*/function () {
  function ApplicationUsageTracker(reporter) {
    _classCallCheck(this, ApplicationUsageTracker);
    _defineProperty(this, "trackedApplicationViews", {});
    _defineProperty(this, "reporter", void 0);
    _defineProperty(this, "currentAppId", void 0);
    _defineProperty(this, "currentApplicationKeys", []);
    _defineProperty(this, "beforeUnloadListener", void 0);
    _defineProperty(this, "onVisiblityChangeListener", void 0);
    this.reporter = reporter;
  }
  return _createClass(ApplicationUsageTracker, [{
    key: "createKey",
    value: function createKey(appId, viewId) {
      return {
        appId: appId,
        viewId: viewId
      };
    }
  }, {
    key: "trackApplications",
    value: function trackApplications(appKeys) {
      var _iterator = _createForOfIteratorHelper(appKeys.filter(Boolean)),
        _step;
      try {
        for (_iterator.s(); !(_step = _iterator.n()).done;) {
          var _step$value = _step.value,
            appId = _step$value.appId,
            viewId = _step$value.viewId;
          var serializedKey = ApplicationUsageTracker.serializeKey({
            appId: appId,
            viewId: viewId
          });
          if (typeof this.trackedApplicationViews[serializedKey] !== 'undefined') {
            continue;
          }
          var metric = createApplicationUsageMetric(appId, viewId);
          this.trackedApplicationViews[serializedKey] = metric;
        }
      } catch (err) {
        _iterator.e(err);
      } finally {
        _iterator.f();
      }
    }
  }, {
    key: "attachListeners",
    value: function attachListeners() {
      var _this = this;
      if (typeof window === 'undefined' || typeof document === 'undefined') {
        return;
      }
      this.beforeUnloadListener = function () {
        _this.flushTrackedViews();
      };
      this.onVisiblityChangeListener = function () {
        if (document.visibilityState === 'visible') {
          _this.resumeTrackingAll();
        } else if (document.visibilityState === 'hidden') {
          _this.pauseTrackingAll();
        }
      };

      // Before leaving the page, make sure we store the current usage
      window.addEventListener('beforeunload', this.beforeUnloadListener);

      // Monitoring dashboards might be open in background and we are fine with that
      // but we don't want to report hours if the user goes to another tab and Kibana is not shown
      document.addEventListener('visibilitychange', this.onVisiblityChangeListener);
    }
  }, {
    key: "detachListeners",
    value: function detachListeners() {
      if (typeof window === 'undefined' || typeof document === 'undefined') {
        return;
      }
      if (this.beforeUnloadListener) {
        window.removeEventListener('beforeunload', this.beforeUnloadListener);
      }
      if (this.onVisiblityChangeListener) {
        document.removeEventListener('visibilitychange', this.onVisiblityChangeListener);
      }
    }
  }, {
    key: "sendMetricsToReporter",
    value: function sendMetricsToReporter(metrics) {
      var _this2 = this;
      metrics.forEach(function (metric) {
        _this2.reporter.reportApplicationUsage(metric);
      });
    }
  }, {
    key: "updateViewClickCounter",
    value: function updateViewClickCounter(viewId) {
      if (!this.currentAppId) {
        return;
      }
      var appKey = ApplicationUsageTracker.serializeKey({
        appId: this.currentAppId,
        viewId: viewId
      });
      if (this.trackedApplicationViews[appKey]) {
        this.trackedApplicationViews[appKey].numberOfClicks++;
      }
    }
  }, {
    key: "flushTrackedViews",
    value: function flushTrackedViews() {
      var appViewMetrics = Object.values(this.trackedApplicationViews);
      this.sendMetricsToReporter(appViewMetrics);
      this.trackedApplicationViews = {};
    }
  }, {
    key: "start",
    value: function start() {
      this.attachListeners();
    }
  }, {
    key: "stop",
    value: function stop() {
      this.flushTrackedViews();
      this.detachListeners();
    }
  }, {
    key: "setCurrentAppId",
    value: function setCurrentAppId(appId) {
      // application change, flush current views first.
      this.flushTrackedViews();
      this.currentAppId = appId;
    }
  }, {
    key: "trackApplicationViewUsage",
    value: function trackApplicationViewUsage(viewId) {
      if (!this.currentAppId) {
        return;
      }
      var appKey = this.createKey(this.currentAppId, viewId);
      this.trackApplications([appKey]);
    }
  }, {
    key: "pauseTrackingAll",
    value: function pauseTrackingAll() {
      var _this3 = this;
      this.currentApplicationKeys = Object.values(this.trackedApplicationViews).map(function (_ref) {
        var appId = _ref.appId,
          viewId = _ref.viewId;
        return _this3.createKey(appId, viewId);
      });
      this.flushTrackedViews();
    }
  }, {
    key: "resumeTrackingAll",
    value: function resumeTrackingAll() {
      this.trackApplications(this.currentApplicationKeys);
      this.currentApplicationKeys = [];

      // We also want to send the report now because intervals and timeouts be stalled when too long in the "hidden" state
      // Note: it might be better to create a separate listener in the reporter for this.
      this.reporter.sendReports();
    }
  }, {
    key: "flushTrackedView",
    value: function flushTrackedView(viewId) {
      if (!this.currentAppId) {
        return;
      }
      var appKey = this.createKey(this.currentAppId, viewId);
      var serializedKey = ApplicationUsageTracker.serializeKey(appKey);
      var appViewMetric = this.trackedApplicationViews[serializedKey];
      if (appViewMetric) {
        this.sendMetricsToReporter([appViewMetric]);
        delete this.trackedApplicationViews[serializedKey];
      }
    }
  }], [{
    key: "serializeKey",
    value: function serializeKey(_ref2) {
      var appId = _ref2.appId,
        viewId = _ref2.viewId;
      return "".concat(appId, "-").concat(viewId);
    }
  }]);
}();