/*
 * Copyright (c) 2008, 2018, Oracle and/or its affiliates.
 * All rights reserved. Use is subject to license terms.
 *
 * This file is available and licensed under the following license:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the distribution.
 *  - Neither the name of Oracle Corporation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package ensemble.samples.media.alphamediaplayer;

import javafx.application.Application;
import javafx.beans.Observable;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.scene.Parent;
import javafx.scene.media.Media;
import javafx.scene.media.MediaPlayer;
import javafx.scene.Scene;
import javafx.stage.Stage;

/**
 * An alpha media player with 2 different media views and alpha channels.
 *
 * @sampleName Alpha Media Player
 * @preview preview.png
 * @docUrl http://docs.oracle.com/javase/8/javafx/media-tutorial/overview.htm#JFXMD101 Using JavaFX Media
 * @see javafx.scene.media.MediaPlayer
 * @see javafx.scene.media.Media
 * @playground arthPos (name="Arth Position", min=-100, max=100)
 * @playground fierPos (name="Fier Position", min=-100, max=100)
 * @playground arthRate (name="Arth Rate", min=0.1, max=1)
 * @playground fierRate (name="Fier Rate", min=0.1, max=1)
 * @conditionalFeatures WEB, MEDIA
 *
 * @related /Media/Advanced Media
 * @related /Media/Overlay Media Player
 * @related /Media/Streaming Media Player
 */
public class AlphaMediaPlayerApp extends Application {

    PlanetaryPlayerPane planetaryPlayerPane;
    private MediaPlayer arthPlayer;
    private MediaPlayer fierPlayer;

    private final DoubleProperty arthPos = new SimpleDoubleProperty(-90.0);
    public final DoubleProperty arthPosProperty() {
        return arthPos;
    }

    private final DoubleProperty fierPos = new SimpleDoubleProperty(50.0);
    public final DoubleProperty fierPosProperty() {
        return fierPos;
    }

    private final DoubleProperty arthRate = new SimpleDoubleProperty(1.0);
    public final DoubleProperty arthRateProperty() {
        return arthRate;
    }

    private final DoubleProperty fierRate = new SimpleDoubleProperty(1.0);
    public final DoubleProperty fierRateProperty() {
        return fierRate;
    }

    public Parent createContent() {
        String ARTH_URL =
            "https://download.oracle.com/otndocs/products/javafx/arth_512.flv";
        String FIER_URL =
            "https://download.oracle.com/otndocs/products/javafx/fier_512.flv";
        arthPlayer = new MediaPlayer(new Media(ARTH_URL));
        arthPlayer.setAutoPlay(true);
        fierPlayer = new MediaPlayer(new Media(FIER_URL));
        fierPlayer.setAutoPlay(true);

        arthPos.addListener((Observable observable) -> {
            planetaryPlayerPane.setTranslate1(arthPos.doubleValue());
        });

        fierPos.addListener((Observable observable) -> {
            planetaryPlayerPane.setTranslate2(fierPos.doubleValue());
        });

        arthRate.addListener((Observable observable) -> {
            arthPlayer.setRate(arthRate.doubleValue());
        });

        fierRate.addListener((Observable observable) -> {
            fierPlayer.setRate(fierRate.doubleValue());
        });

        String alphaMediaPlayerCss =
            getClass().getResource("AlphaMediaPlayer.css").toExternalForm();

        planetaryPlayerPane = new PlanetaryPlayerPane(arthPlayer, fierPlayer);
        planetaryPlayerPane.setMinSize(480, 320);
        planetaryPlayerPane.setPrefSize(480, 320);
        planetaryPlayerPane.setMaxSize(480, 320);
        planetaryPlayerPane.getStylesheets().add(alphaMediaPlayerCss);
        return planetaryPlayerPane;
    }

    public void play() {
        MediaPlayer.Status status = fierPlayer.getStatus();
        if (status == MediaPlayer.Status.UNKNOWN ||
                status == MediaPlayer.Status.HALTED) {
            return;
        }
        if (status == MediaPlayer.Status.PAUSED ||
                status == MediaPlayer.Status.STOPPED ||
                status == MediaPlayer.Status.READY) {
            fierPlayer.play();
            arthPlayer.play();
        }
    }

    @Override
    public void stop() {
        fierPlayer.stop();
        arthPlayer.stop();
    }

    @Override
    public void start(Stage primaryStage) throws Exception {
        primaryStage.setScene(new Scene(createContent()));
        primaryStage.show();
        play();
    }

    /**
     * Java main for when running without JavaFX launcher
     * @param args command line arguments
     */
    public static void main(String[] args) {
        launch(args);
    }
}
