import React, { createContext, useCallback, useContext, useEffect, useRef, useState } from 'react';
import { bind } from 'bind-event-listener';
import { fg } from '@atlaskit/platform-feature-flags';
import { getGlobalTheme, setGlobalTheme, ThemeMutationObserver } from '@atlaskit/tokens';
const defaultThemeSettings = () => ({
  dark: 'dark',
  light: 'light',
  spacing: 'spacing',
  typography: fg('platform-default-typography-modernized') ? 'typography-modernized' : undefined
});
const ColorModeContext = /*#__PURE__*/createContext(undefined);
const SetColorModeContext = /*#__PURE__*/createContext(undefined);
const ThemeContext = /*#__PURE__*/createContext(undefined);
const SetThemeContext = /*#__PURE__*/createContext(undefined);

/**
 * __UNSAFE_useColorModeForMigration()__
 *
 * Returns the current color mode when inside the app provider.
 * Unlike useColorMode, this utility returns undefined, instead of throwing an error, when the app provider is missing.
 * This allows it to be used by components that need to operate with and without an app provider.
 */
export function UNSAFE_useColorModeForMigration() {
  const value = useContext(ColorModeContext);
  return value;
}

/**
 * __useColorMode()__
 *
 * Returns the current color mode when inside the app provider.
 */
export function useColorMode() {
  const value = useContext(ColorModeContext);
  if (!value) {
    throw new Error('useColorMode must be used within AppProvider.');
  }
  return value;
}

/**
 * __useSetColorMode()__
 *
 * Returns the color mode setter when inside the app provider.
 */
export function useSetColorMode() {
  const value = useContext(SetColorModeContext);
  if (!value) {
    throw new Error('useSetColorMode must be used within AppProvider.');
  }
  return value;
}

/**
 * __useTheme()__
 *
 * Returns the current theme settings when inside the app provider.
 */
export function useTheme() {
  const theme = useContext(ThemeContext);
  const [resolvedTheme, setResolvedTheme] = useState(theme || getGlobalTheme());
  useEffect(() => {
    // We are using theme from context so no need to reference the DOM
    if (theme) {
      return;
    }
    const observer = new ThemeMutationObserver(setResolvedTheme);
    observer.observe();
    return () => observer.disconnect();
  }, [theme]);
  return resolvedTheme;
}

/**
 * __useSetTheme()__
 *
 * Returns the theme setter when inside the app provider.
 */
export function useSetTheme() {
  const value = useContext(SetThemeContext);
  if (!value) {
    throw new Error('useSetTheme must be used within AppProvider.');
  }
  return value;
}
const isMatchMediaAvailable = typeof window !== 'undefined' && 'matchMedia' in window;
const prefersDarkModeMql = isMatchMediaAvailable ? window.matchMedia('(prefers-color-scheme: dark)') : undefined;

// TODO: currently 'auto' color mode will always return 'light' in SSR.
// Additional work required: https://product-fabric.atlassian.net/browse/DSP-9781
function getReconciledColorMode(colorMode) {
  if (colorMode === 'auto') {
    return prefersDarkModeMql !== null && prefersDarkModeMql !== void 0 && prefersDarkModeMql.matches ? 'dark' : 'light';
  }
  return colorMode;
}
/**
 * __Theme provider__
 *
 * Provides global theming configuration.
 *
 * @internal
 */
function ThemeProvider({
  children,
  defaultColorMode,
  defaultTheme
}) {
  const [chosenColorMode, setChosenColorMode] = useState(defaultColorMode);
  const [reconciledColorMode, setReconciledColorMode] = useState(getReconciledColorMode(defaultColorMode));
  const [theme, setTheme] = useState(() => ({
    ...defaultThemeSettings(),
    ...defaultTheme
  }));
  const setColorMode = useCallback(colorMode => {
    setChosenColorMode(colorMode);
    setReconciledColorMode(getReconciledColorMode(colorMode));
  }, []);
  const setPartialTheme = useCallback(nextTheme => {
    setTheme(theme => ({
      ...theme,
      ...nextTheme
    }));
  }, []);
  const lastSetGlobalThemePromiseRef = useRef(null);
  useEffect(() => {
    /**
     * We need to wait for any previous `setGlobalTheme` calls to finish before calling it again.
     * This is to prevent race conditions as `setGlobalTheme` is async and mutates the DOM (e.g. sets the
     * `data-color-mode` attribute on the root element).
     *
     * Since we can't safely abort the `setGlobalTheme` execution, we need to wait for it to properly finish before
     * applying the new theme.
     *
     * Without this, we can end up in the following scenario:
     * 1. app loads with the default 'light' theme, kicking off `setGlobalTheme`
     * 2. app switches to 'dark' theme after retrieving value persisted in local storage, calling `setGlobalTheme` again
     * 3. `setGlobalTheme` function execution for `dark` finishes before the initial `light` execution
     * 4. `setGlobalTheme` function execution for `light` then finishes, resulting in the 'light' theme being applied.
     */
    const cleanupLastFnCall = async () => {
      if (lastSetGlobalThemePromiseRef.current) {
        const unbindFn = await lastSetGlobalThemePromiseRef.current;
        unbindFn();
        lastSetGlobalThemePromiseRef.current = null;
      }
    };
    const safelySetGlobalTheme = async () => {
      await cleanupLastFnCall();
      const promise = setGlobalTheme({
        ...theme,
        colorMode: reconciledColorMode
      });
      lastSetGlobalThemePromiseRef.current = promise;
    };
    safelySetGlobalTheme();
    return function cleanup() {
      cleanupLastFnCall();
    };
  }, [theme, reconciledColorMode]);
  useEffect(() => {
    if (!prefersDarkModeMql) {
      return;
    }
    const unbindListener = bind(prefersDarkModeMql, {
      type: 'change',
      listener: event => {
        if (chosenColorMode === 'auto') {
          setReconciledColorMode(event.matches ? 'dark' : 'light');
        }
      }
    });
    return unbindListener;
  }, [chosenColorMode]);
  return /*#__PURE__*/React.createElement(ColorModeContext.Provider, {
    value: reconciledColorMode
  }, /*#__PURE__*/React.createElement(SetColorModeContext.Provider, {
    value: setColorMode
  }, /*#__PURE__*/React.createElement(ThemeContext.Provider, {
    value: theme
  }, /*#__PURE__*/React.createElement(SetThemeContext.Provider, {
    value: setPartialTheme
  }, children))));
}
export default ThemeProvider;