/*  job_generateidentities.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_GENERATEIDENTITIES_H_
#define JOB_GENERATEIDENTITIES_H_

#include "job.h"
#include "identitygenerator.h"

namespace Reduze {

/// Job to create identities from an file with seed integrals
class GenerateIdentities: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("generate_identities");
		s.set_short_description(""//
					"Generates identities like IBPs for given seeds.");
		s.set_long_description(""//
					"Job to generate identities like IBPs for given seeds.");

		s.add_option("input_file", true, "string", "Input file name.");
		s.add_option("output_file", true, "string", "Output file name.");
		s.add_option("type", true, "string", ""//
					"Type of the identities. Valid types are:"
					" ibp, ibp_dim, ibp_dim_free, ibp_lee, lorentz,"
					" sector_relations, sector_symmetries."
					" For more information about the different types check the"
					" help for 'identity_generic_selection'.");
		s.add_option("options", true, "identity_generator_options", ""//
					"Options for generating the identities.");
		s.add_option("from", false, "integer", ""//
					"read from seed number 'from' (>=1) in the input file."
					" A value smaller than 1 means read from the beginning.");
		s.add_option("to", false, "integer", ""//
					"read until (and with) seed number 'to' (>=from) in the input file."
					" A value smaller than 1 means read until the end.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	GenerateIdentities() :
		from_(-1), to_(-1) {
		add_auto_io("input_file", input_file_);
		add_auto_io("output_file", output_file_);
		add_auto_io("type", type_);
		add_auto_io("options", options_);
		add_auto_io("from", from_);
		add_auto_io("to", to_);
	}
	GenerateIdentities(const std::string& input_file,
			const std::string& output_file, //
			const std::string& type, //
			const IdentityGeneratorOptions& options, //
			int from, int to) :
		input_file_(input_file), /**/
		output_file_(output_file), /**/
		type_(type), /**/
		options_(options), /**/
		from_(from), /**/
		to_(to) {
		add_auto_options();
	}
	virtual ~GenerateIdentities() {
	}

	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;

protected:
	virtual void add_auto_options() {
		add_auto_io("input_file", input_file_);
		add_auto_io("output_file", output_file_);
		add_auto_io("type", type_);
		add_auto_io("options", options_);
		add_auto_io("from", from_);
		add_auto_io("to", to_);
	}
	virtual void init();
	void compute(const std::set<INT>::const_iterator& it_from, //
			const std::set<INT>::const_iterator& it_to);

private:
	/// file with seed integrals
	std::string input_file_;
	/// file with equations
	std::string output_file_;
	/// type of the equations ibp, lorentz, etc.
	std::string type_;
	/// options for generating the identities
	IdentityGeneratorOptions options_;
	/// read the seed file from (and with) seed number 'from_' until (without) 'to_'
	int from_, to_;
};

} // namespace Reduze

#endif /* JOB_GENERATEIDENTITIES_H_ */
