/*  int.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "int.h"

#include <sstream>
#include <fstream>
#include <stdexcept>
#include <limits>
#include "kinematics.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "functions.h"
#include "files.h"
#include "rspoint.h"
#include "shiftoperators.h"
#include "equation.h"
#include "filedata.h" /* load_preferred() */

using namespace std;
using namespace GiNaC;

namespace Reduze {

// register type
namespace {
YAMLProxy<SeedGeneratorOptions> dummy;
}

// helper function
namespace {

template<class T>
T range_checked_cast(int i) {
  if (i < numeric_limits<T>::min() || i > numeric_limits<T>::max())
		throw runtime_error("range overflow: " + to_string(i)
				+ " is not in the bounds ["
				+ to_string(static_cast<int>(numeric_limits<T>::min())) + ", "
				+ to_string(static_cast<int>(numeric_limits<T>::max())) + "]");
  return static_cast<T>(i);
}

}

// static

bool (*INT::ordering)(const INT&, const INT&) = 0;

set<INT, INT::CompareNoPreferred> INT::preferred_integrals;

void INT::set_ordering(const string& type) {
	LOG("Setting integral ordering to " << type);
	if (type == "rs")
		ordering = &less_rs;
	else if (type == "sr")
		ordering = &less_sr;
	else if (type == "degrlex_r")
		ordering = &less_degrlex_r;
	else if (type == "degrlex_s")
		ordering = &less_degrlex_s;
	else
		ERROR("encountered unknown integral ordering \"" << type << "\"");
}

void INT::set_preferred(const INT& i) {
	const CrossedIntegralFamily* cc;
	cc = dynamic_cast<const CrossedIntegralFamily*> (i.integralfamily());
	if (cc != 0)
		preferred_integrals.insert(cc->crossing().uncross(i).first);
	else
		preferred_integrals.insert(i);
}

void INT::load_preferred(const string& filename) {
	if (filename.empty())
		return;
	LOG("Reading preferred masters from file \"" << filename << '\"');
	set<INT> masters;
	InFileINTs inpref(filename);
	inpref.get_all(masters);
	LOGX("  found " << masters.size() << " preferred integrals in file");
	for (set<INT>::iterator i = masters.begin(); i != masters.end(); ++i)
		INT::set_preferred(*i);
}

bool INT::is_preferred(const INT& i) {
	if (i.r() == 0 && i.s() == 0)
		return true;
	const CrossedIntegralFamily* cc;
	cc = dynamic_cast<const CrossedIntegralFamily*> (i.integralfamily());
	set<INT>::const_iterator it;
	if (cc != 0)
		it = preferred_integrals.find(cc->crossing().uncross(i).first);
	else
		it = preferred_integrals.find(i);
	return (it != preferred_integrals.end());
}

// Constructors of INT

INT::INT(const IntegralFamily* m) :
	integralfamily_(m), id_(0), t_(0), r_(0), s_(0), metric_mostly_plus_(false) {
	ASSERT(m != 0);
	v_.resize(m->num_propagators(), 0);
}

INT::INT(const IntegralFamily* m, int id) :
	integralfamily_(m), metric_mostly_plus_(false) {
	ASSERT(m != 0);
	v_.resize(m->num_propagators(), 0);
	int id_checked = id;
	if (id < 0 || id > integralfamily_->max_sector_id()) {
		id_checked = integralfamily_->max_sector_id();
		WARNING("ID " << id << " of " << integralfamily_->name() <<
				" is beyond allowed range. " <<
				"Minimal value is 0, maximal value is " <<
				integralfamily_->max_sector_id() << ".\n" <<
				"The ID number has been set to " << id_checked);
	}
	for (size_t i = 0; i < v_.size(); ++i)
		v_[i] = static_cast<int8> ((id_checked >> i) & 1);
	init();
}

INT::INT(const Sector& s) {
	INT i(s.integralfamily(), s.id());
	*this = i;
}

INT::INT(const IntegralFamily* ic, const vector<int8> & v) :
	integralfamily_(ic), v_(v), metric_mostly_plus_(false) {
	ASSERT(ic != 0);
	if (v.size() != integralfamily_->num_propagators())
		ABORT("Wrong number of propagators in integral");
	init();
}

INT::INT(const string & str) :
	metric_mostly_plus_(false) {
	// read from string
	istringstream ss(str);
	string integralfamily_id;
	ss >> integralfamily_id;
	if (integralfamily_id.empty())
		throw runtime_error("empty integral family in \"" + str + "\"");
	integralfamily_ = Files::instance()->integralfamily(integralfamily_id);

	list<int> nums;
	int n;
	while (!ss.eof() && ss >> n)
		nums.push_back(n);
	// don't allow any trailing stuff (which is not integers)
	if (ss.fail())
		throw runtime_error("invalid INT format in '" + str + "'");

	size_t num_props = integralfamily_->num_propagators();
	v_.resize(num_props);

	if (nums.size() == 4 + num_props) {
		list<int>::const_iterator ni = nums.begin();
		int t, id, r, s;
		t = range_checked_cast<int8>(*ni++);
		id = *ni++;
		r = range_checked_cast<int8>(*ni++);
		s = range_checked_cast<int8>(*ni++);
		for (vector<int8>::iterator vi = v_.begin(); ni != nums.end();)
			*vi++ = range_checked_cast<int8>(*ni++);
		init();
		if (t != t_ || id != id_ || r != r_ || s != s_)
			throw runtime_error("invalid r, s, t or id in INT spec '" + str
					+ "'");
	} else if (nums.size() == integralfamily_->num_propagators()) {
		// for compatibility with old format:
		list<int>::const_iterator ni = nums.begin();
		for (vector<int8>::iterator vi = v_.begin(); ni != nums.end();)
			*vi++ = range_checked_cast<int8>(*ni++);
		init();
	} else {
		throw runtime_error("invalid number of INT arguments in '" + str + "'");
	}
}

INT::INT(const YAML::Node& node) :
	metric_mostly_plus_(false) {
	string ic_str;
	node[0] >> ic_str;
	integralfamily_ = Files::instance()->integralfamily(ic_str);
	ASSERT(node.size() == integralfamily_->num_propagators() + 1);
	vector<int8> v_tmp;
	v_tmp.reserve(integralfamily_->num_propagators());
	int exp;
	for (size_t i = 1; i <= integralfamily_->num_propagators(); ++i) {
		node[i] >> exp;
		v_tmp.push_back(range_checked_cast<int8>(exp));
	}
	v_.swap(v_tmp);
	init();
}

INT::INT(const INTGeneric& in, const GiNaC::exmap& m) :
	integralfamily_(in.integralfamily()), metric_mostly_plus_(false) {
	size_t num_props = integralfamily_->num_propagators();
	v_.reserve(num_props);
	ASSERT(num_props == in.size());
	for (INTGeneric::const_iterator it = in.begin(); it != in.end(); ++it) {
		ex tmp = it->subs(m, subs_options::no_pattern);
		if (!is_a<numeric> (tmp))
			ABORT(tmp << " is not a numeric!\nCannot construct integral from " << in << " with the exmap " << m);
		numeric n = ex_to<numeric> (tmp);
		if (!n.is_integer()
				|| n < numeric_limits<int8>::min()
				|| n > numeric_limits<int8>::max())
			ABORT(n << " is not a valid integer!\nCannot construct integral from " << in << " with the exmap " << m);
		v_.push_back(static_cast<int8> (n.to_int()));
	}
	init();
}

INT INT::from_mma(const std::string& instr) {
	// replace MMA type symbols with spaces
	string str(instr);
	bool metricmp = false;
	size_t pos;
	if ((pos = str.find("INTE[")) != string::npos) {
		str.replace(pos, 5, "     ");
		metricmp = true;
	} else if ((pos = str.find("INT[")) != string::npos) {
		str.replace(pos, 4, "    ");
		metricmp = false;
	}
	INT res(chop_whitespaces(replace_chars(str, "{},;[]\"", ' ')));
	res.set_metric_mostly_plus(metricmp);
	return res;
}

void INT::init() {
	id_ = 0;
	t_ = r_ = s_ = 0;
	for (size_t i = 0; i < v_.size(); ++i)
		if (v_[i] > 0) {
			id_ += (1 << i);
			++t_;
			r_ += v_[i];
		} else {
			s_ += -v_[i];
		}
}

// GLOBAL FUNCTIONS


ostream& operator <<(ostream& strm, const INT& in) {
	if (in.metric_mostly_plus_)
		ABORT("No support for mostly plus metric yet.");
	strm << in.integralfamily_->name();
	strm << " " << static_cast<int> (in.t_)//
			<< " " << in.id_ << " "//
			<< " " << static_cast<int> (in.r_)//
			<< " " << static_cast<int> (in.s_) << " ";
	vector<int8>::const_iterator it = in.v_.begin();
	while (it != in.v_.end()) {
		strm << " " << static_cast<int> (*it);
		++it;
	}
	return strm;
}

void operator >>(const YAML::Node& node, INT& in) {
	string ic_str;
	node[0] >> ic_str;
	IntegralFamily* ic = Files::instance()->integralfamily(ic_str);
	in.integralfamily_ = ic;
	ASSERT(node.size() == ic->num_propagators() + 1);
	vector<int8> v_tmp;
	v_tmp.reserve(ic->num_propagators());
	int exp;
	for (size_t i = 1; i <= ic->num_propagators(); ++i) {
		node[i] >> exp;
		v_tmp.push_back(static_cast<int8> (exp));
	}
	in.v_.swap(v_tmp);
	in.init();
}

YAML::Emitter& operator <<(YAML::Emitter& os, const INT& in) {
	if (in.metric_mostly_plus_)
		ABORT("No support for mostly plus metric yet.");
	vector<int8>::const_iterator it;
	os << YAML::Flow << YAML::BeginSeq;
	os << in.integralfamily_->name();
	for (it = in.v_.begin(); it != in.v_.end(); ++it)
		os << static_cast<int> (*it);
	os << YAML::EndSeq;
	return os;
}

// member functions of INT

// public members

Sector INT::get_sector() const {
	return Sector(integralfamily_, id_);
}

INT INT::get_equivalent() const {
	const list<Permutation>& ps = integralfamily_->permutation_symmetries();
	if (ps.empty())
		return *this;
	INT best(*this);
	for (list<Permutation>::const_iterator p = ps.begin(); p != ps.end(); ++p) {
		INT candidate(*this);
		candidate.transform(*p);
		best = min(candidate, best);
	}
	return best;
}

INT& INT::transform(const Permutation& p) {
	const vector<vector<int> >& cycles = p.vector_rep();
	vector<vector<int> >::const_iterator it = cycles.begin();
	while (it != cycles.end()) {
		vector<int>::const_iterator vit = it->begin(), tmpvit;
		// applies the cycles on the propagator exponents
		if (vit != it->end()) {
			int8 tmp = v_[*vit];
			tmpvit = vit++;
			while (vit != it->end()) {
				v_[*tmpvit] = v_[*vit];
				tmpvit = vit++;
			}
			v_[*tmpvit] = tmp;
			++it;
		} else {
			++it;
		}
	}
	init();
	return *this;
}

bool INT::is_obviously_zero() const {
	return Sector(integralfamily_, id_).is_obviously_zero();
}

// should we go for a more uniform output ?
void INT::to_form_stream(ostream & strm) const {
	strm << (metric_mostly_plus_ ? "P" : "") << "INT("
			<< integralfamily_->name() << "," << static_cast<int> (t_) << ","//
			<< id_ << ","//
			<< static_cast<int> (r_) << ","//
			<< static_cast<int> (s_) << ","//
			<< "[],";
	for (vector<int8>::const_iterator it = v_.begin(); it != v_.end();) {
		strm << static_cast<int> (*it);
		if (++it != v_.end())
			strm << ",";
	}
	strm << ")";
}

void INT::to_mma_stream(ostream & strm) const {
	strm << (metric_mostly_plus_ ? "P" : "") << "INT" << "[" //
			<< "\"" << integralfamily_->name() << "\","//
			<< static_cast<int> (t_) << ","//
			<< id_ << ","//
			<< static_cast<int> (r_) << ","//
			<< static_cast<int> (s_) << ","//
			<< "{";
	for (vector<int8>::const_iterator it = v_.begin(); it != v_.end();) {
		strm << static_cast<int> (*it);
		if (++it != v_.end())
			strm << ",";
	}
	strm << "}]";
}

void INT::to_maple_stream(ostream & strm) const {
	strm << (metric_mostly_plus_ ? "P" : "") << "INT" << "(" //
			<< "\"" << integralfamily_->name() << "\","//
			<< static_cast<int> (t_) << ","//
			<< id_ << ","//
			<< static_cast<int> (r_) << ","//
			<< static_cast<int> (s_) << ","//
			<< "[";
	for (vector<int8>::const_iterator it = v_.begin(); it != v_.end();) {
		strm << static_cast<int> (*it);
		if (++it != v_.end())
			strm << ",";
	}
	strm << "])";
}

size_t INT::hash_value() const {
	unsigned long hash = integralfamily_->id();
	for (vector<int8>::const_iterator t = v_.begin(); t != v_.end(); ++t)
		hash = 5 * hash + *t;
	return size_t(hash);
}

// operators of INT

bool INT::operator ==(const INT & other) const {
	if (*integralfamily_ != *other.integralfamily_)
		return false;
	if (t_ != other.t_ || r_ != other.r_ || s_ != other.s_ || id_ != other.id_)
		return false;
	for (size_t i = 0; i < integralfamily_->num_propagators(); ++i)
		if (v_[i] != other.v_[i])
			return false;
	if (metric_mostly_plus_ != other.metric_mostly_plus_)
		return false;
	return true;
}

bool INT::operator !=(const INT & other) const {
	return !(*this == other);
}

// old implementation:

/*
 bool INT::operator <(const INT & other) const {
 if (t < other.t)
 return true;
 if (t > other.t)
 return false;
 if (r < other.r)
 return true;
 if (r > other.r)
 return false;
 if (s < other.s)
 return true;
 if (s > other.s)
 return false;
 // t, r, s of both INTs are now equal
 if (ID < other.ID)
 return true;
 if (ID > other.ID)
 return false;
 // t, r, s, ID of both INTs are now equal
 // v[i] > 0   <==>  other.v[i] > 0
 // v[i] <= 0  <==>  other.v[i] <= 0
 for (int i = 0; i < (int) integralfamily_->num_propagators(); ++i) {
 if (v[i] < other.v[i])
 return true;
 if (v[i] > other.v[i])
 return false;
 }
 // the INTs are equal now
 return false;
 }
 */

bool INT::less_rs(const INT& a, const INT& b) {
	Sector asec(a.integralfamily_, a.id_);
	Sector bsec(b.integralfamily_, b.id_);
	if (asec != bsec)
		return (asec < bsec);
	// sectors of both INTs are equal at this point
	if (a.r_ != b.r_)
		return (a.r_ < b.r_);
	if (a.s_ != b.s_)
		return (a.s_ < b.s_);
	// t, r, s, ID of both INTs are equal at this point
	// a.v[i] > 0   <==>  b.v[i] > 0
	// a.v[i] <= 0  <==>  b.v[i] <= 0
	int nprops = static_cast<int> (a.integralfamily_->num_propagators());
	for (int i = 0; i < nprops; ++i) {
		if (a.v_[i] != b.v_[i])
			return abs((int)a.v_[i]) > abs((int)b.v_[i]);
	}
	if (a.metric_mostly_plus_ != b.metric_mostly_plus_)
		return !a.metric_mostly_plus_;
	// the INTs are equal at this point
	return false;
}

bool INT::less_sr(const INT& a, const INT& b) {
	Sector asec(a.integralfamily_, a.id_);
	Sector bsec(b.integralfamily_, b.id_);
	if (asec != bsec)
		return (asec < bsec);
	// sectors of both INTs are equal at this point
	if (a.s_ != b.s_)
		return (a.s_ < b.s_);
	if (a.r_ != b.r_)
		return (a.r_ < b.r_);
	// t, r, s, ID of both INTs are equal at this point
	// a.v[i] > 0   <==>  b.v[i] > 0
	// a.v[i] <= 0  <==>  b.v[i] <= 0
	int nprops = static_cast<int> (a.integralfamily_->num_propagators());
	for (int i = 0; i < nprops; ++i) {
		if (a.v_[i] != b.v_[i])
			return abs((int)a.v_[i]) > abs((int)b.v_[i]);
	}
	if (a.metric_mostly_plus_ != b.metric_mostly_plus_)
		return !a.metric_mostly_plus_;
	// the INTs are equal at this point
	return false;
}

// new implementation: "degree reverse lexicographic ordering" (DegRevLex)
// Groebner basis computations perform typically well with a DegRevLex ordering;
// here, we define a variant of this type of ordering
//
// timing measurements for 2-loop planarbox (propagators ordered with increasing difficulty)
// indicate:
// 1. modified r,s,t,ID treatment wrt. above has only little impact on performance,
// 2. in general, exact treatment of elementwise v[i] comparison has severe impact on
// performance, DegRevLex reduces comp. time by 30% wrt. old in some tests
// hint: DegRevLex considers an INT to be easy if most of its "additional" nominators
// or denominators are concentrated at the beginning (easiest propagators)
//

bool INT::less_degrlex_r(const INT& a, const INT& b) {
	Sector asec(a.integralfamily_, a.id_);
	Sector bsec(b.integralfamily_, b.id_);
	if (asec != bsec)
		return (asec < bsec);
	// sectors of both INTs are equal at this point
	// note: sorting for r+s first achieves that for any integral only finitely
	//       many other integrals are lower
	if (a.r_ + a.s_ != b.r_ + b.s_)
		return (a.r_ + a.s_ < b.r_ + b.s_);
	if (a.r_ != b.r_)
		return a.r_ < b.r_;
	// t, r, s, ID of both INTs are equal at this point
	// a.v[i] > 0   <==>  b.v[i] > 0
	// a.v[i] <= 0  <==>  b.v[i] <= 0
	int nprops = static_cast<int> (a.integralfamily_->num_propagators());
	for (int i = 0; i < nprops; ++i) {
		if (a.v_[i] != b.v_[i])
			return abs((int)a.v_[i]) > abs((int)b.v_[i]);
	}
	if (a.metric_mostly_plus_ != b.metric_mostly_plus_)
		return !a.metric_mostly_plus_;
	// the INTs are equal at this point
	return false;
}

bool INT::less_degrlex_s(const INT& a, const INT& b) {
	Sector asec(a.integralfamily_, a.id_);
	Sector bsec(b.integralfamily_, b.id_);
	if (asec != bsec)
		return (asec < bsec);
	// sectors of both INTs are equal at this point
	// note: sorting for r+s first achieves that for any integral only finitely
	//       many other integrals are lower
	if (a.r_ + a.s_ != b.r_ + b.s_)
		return (a.r_ + a.s_ < b.r_ + b.s_);
	if (a.s_ != b.s_)
		return a.s_ < b.s_;
	// t, r, s, ID of both INTs are equal at this point
	// a.v[i] > 0   <==>  b.v[i] > 0
	// a.v[i] <= 0  <==>  b.v[i] <= 0
	int nprops = static_cast<int> (a.integralfamily_->num_propagators());
	for (int i = 0; i < nprops; ++i) {
		if (a.v_[i] != b.v_[i])
			return abs((int)a.v_[i]) > abs((int)b.v_[i]);
	}
	if (a.metric_mostly_plus_ != b.metric_mostly_plus_)
		return !a.metric_mostly_plus_;
	// the INTs are equal at this point
	return false;
}

/* this ordering performed really badly:
 bool INT::operator <(const INT & other) const {
 if (*integralfamily_ != *other.integralfamily_)
 return (*integralfamily_ < *other.integralfamily_);
 if (t != other.t)
 return (t < other.t);
 if (ID != other.ID)
 return (ID < other.ID);
 if (r + s != other.r + other.s)
 return (r + s < other.r + other.s);
 if (r != other.r)
 return r < other.r;
 //	if (s != other.s)
 //		return s < other.s;
 // t, r, s, ID of both INTs are now equal
 // v[i] > 0   <==>  other.v[i] > 0
 // v[i] <= 0  <==>  other.v[i] <= 0
 int nprops = static_cast<int> (integralfamily_->num_propagators());
 for (int i = nprops - 1; i >= 0; --i) {
 if (v[i] != other.v[i])
 return abs((int)v[i]) < abs((int)other.v[i]);
 }
 // the INTs are equal now
 return false;
 }
 */

GiNaC::ex INT::get_integrand() const {
	vector<Propagator> props = integralfamily_->propagators();
	ASSERT(props.size() == v_.size());
	GiNaC::ex result = 1;
	for (size_t i = 0; i < v_.size(); ++i)
		result *= pow(props[i], static_cast<int> (v_[i]));
	return result;
}

// class INTGeneric members

INTGeneric::INTGeneric(const IntegralFamily* m) :
	integralfamily_(m) {
	ASSERT(m != 0);
	v_ = m->propagator_exponents();
}

INTGeneric::INTGeneric(const std::string& str) {
	// read from string
	istringstream ss(str);
	string integralfamily_id;
	ss >> integralfamily_id;
	if (integralfamily_id.empty())
		throw runtime_error("empty integral family in '" + str + "'");
	integralfamily_ = Files::instance()->integralfamily(integralfamily_id);
	size_t num_props = integralfamily_->num_propagators();

	const lst& symbs = integralfamily_->propagator_exponent_symbols();
	vector<string> exps;
	exps.reserve(num_props);
	string n;
	while (!ss.eof() && ss >> n)
		exps.push_back(n);
	if (exps.size() != num_props)
		throw runtime_error(
				"invalid number of arguments of INTGeneric format in '" + str
						+ "'");
	v_.resize(num_props);
	// \todo allow symbol a_i in index j for i != j or only for i == j
	for (unsigned i = 0; i < num_props; ++i) {
		try {
			v_[i] = ex(exps[i], symbs);
		} catch (exception& e) {
			throw runtime_error("Failed to read an INTGeneric from '" + str
					+ "'\n" + e.what());
		}
	}
}

INTGeneric INTGeneric::shift(const OP& op) const {
	INTGeneric res(*this);
	int n = op.n();
	ASSERT(static_cast<size_t>(abs(n)) <= v_.size());
	if (n > 0)
		++res.v_[n - 1];
	if (n < 0)
		--res.v_[-n - 1];
	return res;
}

INTGeneric INTGeneric::shift(const OPPROD& op) const {
	INTGeneric res(*this);
	for (OPPROD::const_iterator it = op.begin(); it != op.end(); ++it)
		res = res.shift(*it);
	return res;
}

LinearCombinationGeneric INTGeneric::shift(const OPSUM& sum) const {
	LinearCombinationGeneric res;
	for (OPSUM::const_iterator it = sum.begin(); it != sum.end(); ++it)
		res.insert(shift(it->first), it->second);
	return res;
}

bool INTGeneric::operator ==(const INTGeneric& other) const {
	if (*integralfamily_ != *other.integralfamily_)
		return false;
	ASSERT(size() == other.size());
	const_iterator it1 = begin(), it2 = other.begin();
	for (; it1 != end(); ++it1, ++it2)
		if (it1->compare(*it2) != 0)
			return false;
	return true;
}
bool INTGeneric::operator !=(const INTGeneric& other) const {
	return !(*this == other);
}
bool INTGeneric::operator <(const INTGeneric& other) const {
	if (*integralfamily_ != *other.integralfamily_)
		return *integralfamily_ < *other.integralfamily_;
	ASSERT(size() == other.size());
	const_iterator it1 = begin(), it2 = other.begin();
	for (; it1 != end(); ++it1, ++it2) {
		int comp = reduze_compare(*it1, *it2);
		if (comp != 0)
			return comp < 0;
	}
	return false;
}

std::ostream& operator <<(std::ostream& os, const INTGeneric& in) {
	os << in.integralfamily()->name() << " ";
	for (INTGeneric::const_iterator it = in.begin(); it != in.end(); ++it)
		os << " " << *it;
	return os;
}

void INTGeneric::to_form_stream(std::ostream& os) const {
	os << "INT(" << integralfamily_->name() << ",[]";
	for (const_iterator it = begin(); it != end(); ++it)
		os << "," << *it;
	os << ")";
}

void INTGeneric::to_mma_stream(std::ostream& os) const {
	os << "INT" << "[\"" << integralfamily_->name() << "\",{";
	for (const_iterator it = begin(); it != end();) {
		os << *it;
		if (++it != end())
			os << ",";
	}
	os << "}]";
}

void INTGeneric::to_maple_stream(ostream& os) const {
	os << "INT" << "(\"" << integralfamily_->name() << "\",[";
	for (const_iterator it = begin(); it != end();) {
		os << *it;
		if (++it != end())
			os << ",";
	}
	os << "])";
}

// SeedGeneratorOptions


/// whether permutation symmetries of the integralfamily should be used
void SeedGeneratorOptions::set_reduce_to_equivalent(bool b) {
	reduce_to_equivalent_ = b;
}

void SeedGeneratorOptions::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;
	os << Key << "reduce_to_equivalent" << Value << reduce_to_equivalent_;
	os << EndMap;
}

void SeedGeneratorOptions::read(const YAML::Node& node) {
	verify_yaml_spec(node);
	const YAML::Node* n;
	if ((n = node.FindValue("reduce_to_equivalent")))
		*n >> reduce_to_equivalent_;
}

// SeedGenerator

void SeedGenerator::generate(int id, int r, int s, set<INT>& out) const {
	if (id == integralfamily_->max_sector_id() && s != 0) {
		LOG("integrals of sector " << id << " with s = " << s << " don't exist");
		return;
	}
	if (id == 0 && r != 0) {
		LOG("integrals of sector " << id << " with r = " << r << " don't exist");
		return;
	}
	size_t num_props = integralfamily_->num_propagators();
	ASSERT(num_props > 0);
	INT integral(integralfamily_, id);
	vector<int8> integral_v = integral.v();
	int t = static_cast<int> (integral.t());
	if (t > r)
		ABORT("r = " << r << " must be equal or bigger than t = " << t);
	size_t num_r = // special case for t<1
			t < 1 ? 1 : binom_int(r - 1, r - t);
	ASSERT(num_r > 0);
	vector<vector<int8> > arr1(num_r, vector<int8> (t, 0));
	partition(arr1, num_r, t, r, true);
	if (s < 0)
		ABORT("s = " << s << " must be equal or bigger than 0");
	size_t num_s = // special case for num_props - tmp_t < 1
			(num_props - t < 1) ? 1 : binom_int(s - 1 + num_props - t, s);
	ASSERT(num_s > 0);
	vector<vector<int8> > arr2(num_s, vector<int8> (num_props - t, 0));
	partition(arr2, num_s, num_props - t, s, false);

	vector<int8> vec(num_props);
	set<INT>::const_iterator pos = out.end();
	for (size_t i2 = 0; i2 < num_s; ++i2) {
		for (size_t i1 = 0; i1 < num_r; ++i1) {
			size_t k = 0, m = 0;
			for (size_t i = 0; i < num_props; ++i)
				vec[i] = (integral_v[i] == 1 ? arr1[i1][k++] : -arr2[i2][m++]);
			INT seed(integralfamily_, vec);
			if (options_.reduce_to_equivalent_)
				seed = seed.get_equivalent();
			pos = out.insert(pos, seed);
		}
	}
}

void SeedGenerator::generate(int id, int r_min, int r_max, int s_min,
		int s_max, set<INT>& out) const {
	for (int r = r_min; r <= r_max; ++r)
		for (int s = s_min; s <= s_max; ++s)
			generate(id, r, s, out);
}

void SeedGenerator::generate(int id,
		const RSFiniteGenericSelection& rs_generic, set<INT>& out) const {
	int t = integralfamily_->get_t(id);
	int r_min = rs_generic.r_min(t);
	int r_max = rs_generic.r_max(t);
	int s_min = rs_generic.s_min(t);
	int s_max = rs_generic.s_max(t);
	generate(id, r_min, r_max, s_min, s_max, out);
}

void SeedGenerator::generate(int id,
		const list<RSFiniteGenericSelection>& rs_generic, set<INT>& out) const {
	list<RSFiniteGenericSelection>::const_iterator it;
	for (it = rs_generic.begin(); it != rs_generic.end(); ++it)
		generate(id, *it, out);
}

} // namespace Reduze


