//go:build linux
// +build linux

/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package iptables

import (
	"testing"

	"github.com/lithammer/dedent"

	"k8s.io/apimachinery/pkg/util/sets"
)

func checkChains(t *testing.T, save []byte, expected sets.Set[Chain]) {
	chains := GetChainsFromTable(save)
	missing := expected.Difference(chains)
	if len(missing) != 0 {
		t.Errorf("GetChainsFromTable expected chains not present: %v", missing.UnsortedList())
	}
	extra := chains.Difference(expected)
	if len(extra) != 0 {
		t.Errorf("GetChainsFromTable expected chains unexpectedly present: %v", extra.UnsortedList())
	}
}

func TestGetChainsFromTable(t *testing.T) {
	iptablesSave := dedent.Dedent(`
		# Generated by iptables-save v1.4.21 on Fri Aug  7 14:47:37 2015
		*nat
		:PREROUTING ACCEPT [2:138]
		:INPUT ACCEPT [0:0]
		:OUTPUT ACCEPT [0:0]
		:POSTROUTING ACCEPT [0:0]
		:DOCKER - [0:0]
		:KUBE-NODEPORT-CONTAINER - [0:0]
		:KUBE-NODEPORT-HOST - [0:0]
		:KUBE-PORTALS-CONTAINER - [0:0]
		:KUBE-PORTALS-HOST - [0:0]
		:KUBE-SVC-1111111111111111 - [0:0]
		:KUBE-SVC-2222222222222222 - [0:0]
		:KUBE-SVC-3333333333333333 - [0:0]
		:KUBE-SVC-4444444444444444 - [0:0]
		:KUBE-SVC-5555555555555555 - [0:0]
		:KUBE-SVC-6666666666666666 - [0:0]
		-A PREROUTING -m comment --comment "handle ClusterIPs; NOTE: this must be before the NodePort rules" -j KUBE-PORTALS-CONTAINER
		-A PREROUTING -m addrtype --dst-type LOCAL -j DOCKER
		-A PREROUTING -m addrtype --dst-type LOCAL -m comment --comment "handle service NodePorts; NOTE: this must be the last rule in the chain" -j KUBE-NODEPORT-CONTAINER
		-A OUTPUT -m comment --comment "handle ClusterIPs; NOTE: this must be before the NodePort rules" -j KUBE-PORTALS-HOST
		-A OUTPUT ! -d 127.0.0.0/8 -m addrtype --dst-type LOCAL -j DOCKER
		-A OUTPUT -m addrtype --dst-type LOCAL -m comment --comment "handle service NodePorts; NOTE: this must be the last rule in the chain" -j KUBE-NODEPORT-HOST
		-A POSTROUTING -s 10.246.1.0/24 ! -o cbr0 -j MASQUERADE
		-A POSTROUTING -s 10.0.2.15 -d 10.0.2.15 -m comment --comment "handle pod connecting to self" -j MASQUERADE
		-A KUBE-PORTALS-CONTAINER -d 10.247.0.1 -p tcp -m comment --comment "portal for default/kubernetes:" -m state --state NEW -m tcp --dport 443 -j KUBE-SVC-5555555555555555
		-A KUBE-PORTALS-CONTAINER -d 10.247.0.10 -p udp -m comment --comment "portal for kube-system/kube-dns:dns" -m state --state NEW -m udp --dport 53 -j KUBE-SVC-6666666666666666
		-A KUBE-PORTALS-CONTAINER -d 10.247.0.10 -p tcp -m comment --comment "portal for kube-system/kube-dns:dns-tcp" -m state --state NEW -m tcp --dport 53 -j KUBE-SVC-2222222222222222
		-A KUBE-PORTALS-HOST -d 10.247.0.1 -p tcp -m comment --comment "portal for default/kubernetes:" -m state --state NEW -m tcp --dport 443 -j KUBE-SVC-5555555555555555
		-A KUBE-PORTALS-HOST -d 10.247.0.10 -p udp -m comment --comment "portal for kube-system/kube-dns:dns" -m state --state NEW -m udp --dport 53 -j KUBE-SVC-6666666666666666
		-A KUBE-PORTALS-HOST -d 10.247.0.10 -p tcp -m comment --comment "portal for kube-system/kube-dns:dns-tcp" -m state --state NEW -m tcp --dport 53 -j KUBE-SVC-2222222222222222
		-A KUBE-SVC-1111111111111111 -p udp -m comment --comment "kube-system/kube-dns:dns" -m recent --set --name KUBE-SVC-1111111111111111 --mask 255.255.255.255 --rsource -j DNAT --to-destination 10.246.1.2:53
		-A KUBE-SVC-2222222222222222 -m comment --comment "kube-system/kube-dns:dns-tcp" -j KUBE-SVC-3333333333333333
		-A KUBE-SVC-3333333333333333 -p tcp -m comment --comment "kube-system/kube-dns:dns-tcp" -m recent --set --name KUBE-SVC-3333333333333333 --mask 255.255.255.255 --rsource -j DNAT --to-destination 10.246.1.2:53
		-A KUBE-SVC-4444444444444444 -p tcp -m comment --comment "default/kubernetes:" -m recent --set --name KUBE-SVC-4444444444444444 --mask 255.255.255.255 --rsource -j DNAT --to-destination 10.245.1.2:443
		-A KUBE-SVC-5555555555555555 -m comment --comment "default/kubernetes:" -j KUBE-SVC-4444444444444444
		-A KUBE-SVC-6666666666666666 -m comment --comment "kube-system/kube-dns:dns" -j KUBE-SVC-1111111111111111
		COMMIT
		`)

	expected := sets.New(
		ChainPrerouting,
		Chain("INPUT"),
		Chain("OUTPUT"),
		ChainPostrouting,
		Chain("DOCKER"),
		Chain("KUBE-NODEPORT-CONTAINER"),
		Chain("KUBE-NODEPORT-HOST"),
		Chain("KUBE-PORTALS-CONTAINER"),
		Chain("KUBE-PORTALS-HOST"),
		Chain("KUBE-SVC-1111111111111111"),
		Chain("KUBE-SVC-2222222222222222"),
		Chain("KUBE-SVC-3333333333333333"),
		Chain("KUBE-SVC-4444444444444444"),
		Chain("KUBE-SVC-5555555555555555"),
		Chain("KUBE-SVC-6666666666666666"),
	)
	checkChains(t, []byte(iptablesSave), expected)
}
