#!/bin/bash
# Copyright 2024-2024 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2
#
# populate-election.sh: Create election files based on the election-details file.
#

ELECTION=${1%/}
fullpath=$(readlink -f "./${ELECTION}")

warn() {
	echo "$@" 1>&2
}

die() {
	echo "$@" 1>&2
	exit 1
}

if [[ -z "${ELECTION}" ]]; then
	die "usage: $(basename "$0") ELECTION-NAME"
fi

details="${fullpath}"/election-details

if ! test -f "${details}" ; then
	warn "Could not find $ELECTION at ${details}: missing file - generating from template"
	mkdir -p "${ELECTION}"/
	cat election-details.template >"${ELECTION}"/election-details
	_type=${ELECTION//*-}
	_yyyymm=${ELECTION//-*}
	sed -i \
		-e "s/__TYPE__/${_type}/g" \
		-e "s/__YYYYMM__/${_yyyymm}/g" \
		-e "s/__YYYY__/${_yyyymm:0:4}/g" \
		"${ELECTION}"/election-details
fi

if ! grep -sq -x -e "name: ${ELECTION}" "${details}" ; then
	die "Could not find $ELECTION at ${details}: bad content"
fi

for f in name startDate endDate officials voters ballot url ; do
	awk -v f=$f -F ': ' '($1==f){print $2}' "${details}" |grep -E -e '.{6,}' -sq || die "$ELECTION: missing field $f in $details"
done

for f in voters ballot officials ; do
	k1=${f/:*}
	k2=${f/*:}
	d=$ELECTION/${k1}-$ELECTION
	v=$(awk -v f="$k2" -F ': ' '($1==f){print $2}' "${details}")
	if [[ "$v" =~ "https://" ]]; then
		curl --fail -Lsq -o "${d}" "$v" || warn "Could not fetch ${f} from $v"
	else
		tr -s ', ' '\n' <<<"$v" |fmt -1 |sort >"$d"
	fi
done

for f in start:startDate stop:endDate ; do
	k1=${f/:*}
	k2=${f/*:}
	d=$ELECTION/${k1}-$ELECTION
	v=$(awk -v f="$k2" -F ': ' '($1==f){print $2}' "${details}")
	date +%s -d "$v" >"${d}"
done

if [ -f "${ELECTION}/ballot-${ELECTION}" ]; then
	overlap_candidate_official=$(grep -x -f "${ELECTION}/officials-${ELECTION}"  "${ELECTION}/ballot-${ELECTION}" -o)
	if [[ -n "${overlap_candidate_official}" ]]; then
		die "ERROR: One or more candidates are also election officials: ${overlap_candidate_official}"
	fi
fi

ln -sf "../Votify.pm" "${ELECTION}/"
