# frozen_string_literal: true

# Redmine - project management software
# Copyright (C) 2006-  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

require_relative '../test_helper'

class AttachmentTest < ActiveSupport::TestCase
  fixtures :users, :email_addresses, :projects, :roles, :members, :member_roles,
           :enabled_modules, :issues, :trackers, :attachments

  def setup
    User.current = nil
    set_tmp_attachments_directory
  end

  def test_container_for_new_attachment_should_be_nil
    assert_nil Attachment.new.container
  end

  def test_filename_should_remove_eols
    assert_equal "line_feed", Attachment.new(:filename => "line\nfeed").filename
    assert_equal "line_feed", Attachment.new(:filename => "some\npath/line\nfeed").filename
    assert_equal "carriage_return", Attachment.new(:filename => "carriage\rreturn").filename
    assert_equal "carriage_return", Attachment.new(:filename => "some\rpath/carriage\rreturn").filename
  end

  def test_create
    a = Attachment.new(:container => Issue.find(1),
                       :file => uploaded_test_file("testfile.txt", "text/plain"),
                       :author => User.find(1))
    assert a.save
    assert_equal 'testfile.txt', a.filename
    assert_equal 59, a.filesize
    assert_equal 'text/plain', a.content_type
    assert_equal 0, a.downloads
    assert_equal '6bc2eb7e87cfbf9145065689aaa8b5f513089ca0af68e2dc41f9cc025473d106', a.digest

    assert a.disk_directory
    assert_match %r{\A\d{4}/\d{2}\z}, a.disk_directory

    assert File.exist?(a.diskfile)
    assert_equal 59, File.size(a.diskfile)
  end

  def test_create_should_clear_content_type_if_too_long
    a = Attachment.new(:container => Issue.find(1),
                       :file => uploaded_test_file("testfile.txt", "text/plain"),
                       :author => User.find(1),
                       :content_type => 'a'*300)
    assert a.save
    a.reload
    assert_nil a.content_type
  end

  def test_shorted_filename_if_too_long
    file = mock_file_with_options(:original_filename => "#{'a'*251}.txt")

    a = Attachment.new(:container => Issue.find(1),
                       :file => file,
                       :author => User.find(1))
    assert a.save
    a.reload
    assert_equal 12 + 1 + 32 + 4, a.disk_filename.length
    assert_equal 255, a.filename.length
  end

  def test_copy_should_preserve_attributes
    # prevent re-use of data from other attachments with equal contents
    Attachment.where('id <> 1').destroy_all

    a = Attachment.find(1)
    copy = a.copy

    assert_save copy
    copy = Attachment.order('id DESC').first
    %w(filename filesize content_type author_id created_on description digest disk_filename disk_directory diskfile).each do |attribute|
      assert_equal a.send(attribute), copy.send(attribute), "#{attribute} was different"
    end
  end

  def test_size_should_be_validated_for_new_file
    with_settings :attachment_max_size => 0 do
      a = Attachment.new(:container => Issue.find(1),
                         :file => uploaded_test_file("testfile.txt", "text/plain"),
                         :author => User.find(1))
      assert !a.save
    end
  end

  def test_size_should_not_be_validated_when_copying
    a = Attachment.create!(:container => Issue.find(1),
                           :file => uploaded_test_file("testfile.txt", "text/plain"),
                           :author => User.find(1))
    with_settings :attachment_max_size => 0 do
      copy = a.copy
      assert copy.save
    end
  end

  def test_filesize_greater_than_2gb_should_be_supported
    with_settings :attachment_max_size => (50.gigabytes / 1024) do
      a = Attachment.create!(:container => Issue.find(1),
                             :file => uploaded_test_file("testfile.txt", "text/plain"),
                             :author => User.find(1))
      a.filesize = 20.gigabytes
      a.save!
      assert_equal 20.gigabytes, a.reload.filesize
    end
  end

  def test_extension_should_be_validated_against_allowed_extensions
    with_settings :attachment_extensions_allowed => "txt, png" do
      a = Attachment.new(:container => Issue.find(1),
                         :file => mock_file_with_options(:original_filename => "test.png"),
                         :author => User.find(1))
      assert_save a

      a = Attachment.new(:container => Issue.find(1),
                         :file => mock_file_with_options(:original_filename => "test.jpeg"),
                         :author => User.find(1))
      assert !a.save
    end
  end

  def test_extension_should_be_validated_against_denied_extensions
    with_settings :attachment_extensions_denied => "txt, png" do
      a = Attachment.new(:container => Issue.find(1),
                         :file => mock_file_with_options(:original_filename => "test.jpeg"),
                         :author => User.find(1))
      assert_save a

      a = Attachment.new(:container => Issue.find(1),
                         :file => mock_file_with_options(:original_filename => "test.png"),
                         :author => User.find(1))
      assert !a.save
    end
  end

  def test_extension_update_should_be_validated_against_denied_extensions
    with_settings :attachment_extensions_denied => "txt, png" do
      a = Attachment.new(:container => Issue.find(1),
                         :file => mock_file_with_options(:original_filename => "test.jpeg"),
                         :author => User.find(1))
      assert_save a

      b = Attachment.find(a.id)
      b.filename = "test.png"
      assert !b.save
    end
  end

  def test_valid_extension_should_be_case_insensitive
    with_settings :attachment_extensions_allowed => "txt, Png" do
      assert Attachment.valid_extension?(".pnG")
      assert !Attachment.valid_extension?(".jpeg")
    end
    with_settings :attachment_extensions_denied => "txt, Png" do
      assert !Attachment.valid_extension?(".pnG")
      assert Attachment.valid_extension?(".jpeg")
    end
  end

  def test_description_length_should_be_validated
    a = Attachment.new(:description => 'a' * 300)
    assert !a.save
    assert_not_equal [], a.errors[:description]
  end

  def test_destroy
    a = Attachment.new(:container => Issue.find(1),
                       :file => uploaded_test_file("testfile.txt", "text/plain"),
                       :author => User.find(1))
    assert a.save
    assert_equal 'testfile.txt', a.filename
    assert_equal 59, a.filesize
    assert_equal 'text/plain', a.content_type
    assert_equal 0, a.downloads
    assert_equal '6bc2eb7e87cfbf9145065689aaa8b5f513089ca0af68e2dc41f9cc025473d106', a.digest
    diskfile = a.diskfile
    assert File.exist?(diskfile)
    assert_equal 59, File.size(a.diskfile)
    assert a.destroy
    assert !File.exist?(diskfile)
  end

  def test_destroy_should_not_delete_file_referenced_by_other_attachment
    a = Attachment.create!(:container => Issue.find(1),
                           :file => uploaded_test_file("testfile.txt", "text/plain"),
                           :author => User.find(1))
    diskfile = a.diskfile

    copy = a.copy
    copy.save!

    assert File.exist?(diskfile)
    a.destroy
    assert File.exist?(diskfile)
    copy.destroy
    assert !File.exist?(diskfile)
  end

  def test_create_should_auto_assign_content_type
    a = Attachment.new(:container => Issue.find(1),
                       :file => uploaded_test_file("testfile.txt", ""),
                       :author => User.find(1))
    assert a.save
    assert_equal 'text/plain', a.content_type
  end

  def test_attachments_with_same_content_should_reuse_same_file
    a1 = Attachment.create!(:container => Issue.find(1), :author => User.find(1),
                            :file => mock_file(:filename => 'foo', :content => 'abcd'))
    a2 = Attachment.create!(:container => Issue.find(1), :author => User.find(1),
                            :file => mock_file(:filename => 'bar', :content => 'abcd'))
    assert_equal a1.diskfile, a2.diskfile
  end

  def test_attachments_with_same_content_should_not_reuse_same_file_if_deleted
    a1 = Attachment.create!(:container => Issue.find(1), :author => User.find(1),
                            :file => mock_file(:filename => 'foo', :content => 'abcd'))
    a1.delete_from_disk
    a2 = Attachment.create!(:container => Issue.find(1), :author => User.find(1),
                            :file => mock_file(:filename => 'bar', :content => 'abcd'))
    assert_not_equal a1.diskfile, a2.diskfile
  end

  def test_attachments_with_same_filename_at_the_same_time_should_not_overwrite
    a1 = Attachment.create!(:container => Issue.find(1), :author => User.find(1),
                            :file => mock_file(:filename => 'foo', :content => 'abcd'))
    a2 = Attachment.create!(:container => Issue.find(1), :author => User.find(1),
                            :file => mock_file(:filename => 'foo', :content => 'efgh'))
    assert_not_equal a1.diskfile, a2.diskfile
  end

  def test_identical_attachments_created_in_same_transaction_should_not_end_up_unreadable
    attachments = []
    Project.transaction do
      3.times do
        a = Attachment.create!(
          :container => Issue.find(1), :author => User.find(1),
          :file => mock_file(:filename => 'foo', :content => 'abcde')
        )
        attachments << a
      end
    end
    attachments.each do |a|
      assert a.readable?
    end
    assert_equal 1, attachments.map(&:diskfile).uniq.size
  end

  def test_filename_should_be_basenamed
    a = Attachment.new(:file => mock_file(:original_filename => "path/to/the/file"))
    assert_equal 'file', a.filename
  end

  def test_filename_should_be_sanitized
    a = Attachment.new(:file => mock_file(:original_filename => "valid:[] invalid:?%*|\"'<>chars"))
    assert_equal 'valid_[] invalid_chars', a.filename
  end

  def test_create_diskfile
    path = nil
    Attachment.create_diskfile("test_file.txt") do |f|
      path = f.path
      assert_match(/^\d{12}_test_file.txt$/, File.basename(path))
      assert_equal 'test_file.txt', File.basename(path)[13..-1]
    end
    File.unlink path

    Attachment.create_diskfile("test_accentué.txt") do |f|
      path = f.path
      assert_equal '770c509475505f37c2b8fb6030434d6b.txt', File.basename(f.path)[13..-1]
    end
    File.unlink path

    Attachment.create_diskfile("test_accentué") do |f|
      path = f.path
      assert_equal 'f8139524ebb8f32e51976982cd20a85d', File.basename(f.path)[13..-1]
    end
    File.unlink path

    Attachment.create_diskfile("test_accentué.ça") do |f|
      path = f.path
      assert_equal 'cbb5b0f30978ba03731d61f9f6d10011', File.basename(f.path)[13..-1]
    end
    File.unlink path
  end

  def test_title
    a = Attachment.new(:filename => "test.png")
    assert_equal "test.png", a.title

    a = Attachment.new(:filename => "test.png", :description => "Cool image")
    assert_equal "test.png (Cool image)", a.title
    assert_equal "test.png", a.filename
  end

  def test_new_attachment_should_be_editable_by_author
    user = User.find(1)
    a = Attachment.new(:author => user)
    assert_equal true, a.editable?(user)
  end

  def test_prune_should_destroy_old_unattached_attachments
    Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1, :created_on => 2.days.ago)
    Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1, :created_on => 2.days.ago)
    Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1)

    assert_difference 'Attachment.count', -2 do
      Attachment.prune
    end
  end

  def test_archive_attachments
    attachment = Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1)
    zip_data = Attachment.archive_attachments([attachment])
    file_names = []
    Zip::InputStream.open(StringIO.new(zip_data)) do |io|
      while (entry = io.get_next_entry)
        file_names << entry.name
      end
    end
    assert_equal ['testfile.txt'], file_names
  end

  def test_archive_attachments_without_attachments
    zip_data = Attachment.archive_attachments([])
    assert_nil zip_data
  end

  def test_archive_attachments_should_rename_duplicate_file_names
    attachment1 = Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1)
    attachment2 = Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1)
    zip_data = Attachment.archive_attachments([attachment1, attachment2])
    file_names = []
    Zip::InputStream.open(StringIO.new(zip_data)) do |io|
      while (entry = io.get_next_entry)
        file_names << entry.name
      end
    end
    assert_equal ['testfile.txt', 'testfile(1).txt'], file_names
  end

  def test_move_from_root_to_target_directory_should_move_root_files
    a = Attachment.find(20)
    assert a.disk_directory.blank?
    # Create a real file for this fixture
    File.write(a.diskfile, 'test file at the root of files directory')
    assert a.readable?
    Attachment.move_from_root_to_target_directory

    a.reload
    assert_equal '2012/05', a.disk_directory
    assert a.readable?
  end

  test "Attachmnet.attach_files should attach the file" do
    issue = Issue.first
    assert_difference 'Attachment.count' do
      Attachment.attach_files(
        issue,
        '1' => {
          'file' => uploaded_test_file('testfile.txt', 'text/plain'),
          'description' => 'test'
        })
    end
    attachment = Attachment.order('id DESC').first
    assert_equal issue, attachment.container
    assert_equal 'testfile.txt', attachment.filename
    assert_equal 59, attachment.filesize
    assert_equal 'test', attachment.description
    assert_equal 'text/plain', attachment.content_type
    assert File.exist?(attachment.diskfile)
    assert_equal 59, File.size(attachment.diskfile)
  end

  test "Attachmnet.attach_files should add unsaved files to the object as unsaved attachments" do
    # Max size of 0 to force Attachment creation failures
    with_settings(:attachment_max_size => 0) do
      @project = Project.find(1)
      response = Attachment.attach_files(@project, {
                                           '1' => {'file' => mock_file, 'description' => 'test'},
                                           '2' => {'file' => mock_file, 'description' => 'test'}
                                         })

      assert response[:unsaved].present?
      assert_equal 2, response[:unsaved].length
      assert response[:unsaved].first.new_record?
      assert response[:unsaved].second.new_record?
      assert_equal response[:unsaved], @project.unsaved_attachments
    end
  end

  test "Attachment.attach_files should preserve the content_type of attachments added by token" do
    @project = Project.find(1)
    attachment = Attachment.create!(:file => uploaded_test_file("testfile.txt", ""), :author_id => 1, :created_on => 2.days.ago)
    assert_equal 'text/plain', attachment.content_type
    Attachment.attach_files(@project, {'1' => {'token' => attachment.token}})
    attachment.reload
    assert_equal 'text/plain', attachment.content_type
  end

  def test_update_digest_to_sha256_should_update_digest
    set_fixtures_attachments_directory
    attachment = Attachment.find 6
    assert attachment.readable?
    attachment.update_digest_to_sha256!
    assert_equal 'ac5c6e99a21ae74b2e3f5b8e5b568be1b9107cd7153d139e822b9fe5caf50938', attachment.digest
  ensure
    set_tmp_attachments_directory
  end

  def test_update_attachments
    attachments = Attachment.where(:id => [2, 3]).to_a
    assert(
      Attachment.update_attachments(
        attachments,
        {
          '2' => {:filename => 'newname.txt', :description => 'New description'},
          3 => {:filename => 'othername.txt'}
        }
      )
    )
    attachment = Attachment.find(2)
    assert_equal 'newname.txt', attachment.filename
    assert_equal 'New description', attachment.description

    attachment = Attachment.find(3)
    assert_equal 'othername.txt', attachment.filename
  end

  def test_update_attachments_with_failure
    attachments = Attachment.where(:id => [2, 3]).to_a
    assert(
      !Attachment.update_attachments(
        attachments,
        {
          '2' => {
            :filename => '', :description => 'New description'
          },
          3 => {:filename => 'othername.txt'}
        }
      )
    )
    attachment = Attachment.find(3)
    assert_equal 'logo.gif', attachment.filename
  end

  def test_update_attachments_should_sanitize_filename
    attachments = Attachment.where(:id => 2).to_a
    assert(
      Attachment.update_attachments(
        attachments,
        {2 => {:filename => 'newname?.txt'}}
      )
    )
    attachment = Attachment.find(2)
    assert_equal 'newname_.txt', attachment.filename
  end

  def test_latest_attach
    set_fixtures_attachments_directory
    a1 = Attachment.find(16)
    assert_equal "testfile.png", a1.filename
    assert a1.readable?
    assert (! a1.visible?(User.anonymous))
    assert a1.visible?(User.find(2))
    a2 = Attachment.find(17)
    assert_equal "testfile.PNG", a2.filename
    assert a2.readable?
    assert (! a2.visible?(User.anonymous))
    assert a2.visible?(User.find(2))
    assert a1.created_on < a2.created_on

    la1 = Attachment.latest_attach([a1, a2], "testfile.png")
    assert_equal 17, la1.id
    la2 = Attachment.latest_attach([a1, a2], "Testfile.PNG")
    assert_equal 17, la2.id
  ensure
    set_tmp_attachments_directory
  end

  def test_latest_attach_should_not_error_with_string_with_invalid_encoding
    string = "width:50\xFE-Image.jpg"
    assert_equal false, string.valid_encoding?

    Attachment.latest_attach(Attachment.limit(2).to_a, string)
  end

  def test_latest_attach_should_support_unicode_case_folding
    a_capital = Attachment.create!(
      :author => User.find(1),
      :file => mock_file(:filename => 'Ā.TXT')
    )
    a_small = Attachment.create!(
      :author => User.find(1),
      :file => mock_file(:filename => 'ā.txt')
    )

    assert_equal(a_small, Attachment.latest_attach([a_capital, a_small], 'Ā.TXT'))
  end

  def test_thumbnailable_should_be_true_for_images
    skip unless convert_installed?
    assert_equal true, Attachment.new(:filename => 'test.jpg').thumbnailable?
    assert_equal true, Attachment.new(:filename => 'test.webp').thumbnailable?
  end

  def test_thumbnailable_should_be_false_for_images_if_convert_is_unavailable
    Redmine::Thumbnail.stubs(:convert_available?).returns(false)
    assert_equal false, Attachment.new(:filename => 'test.jpg').thumbnailable?
  end

  def test_thumbnailable_should_be_false_for_non_images
    assert_equal false, Attachment.new(:filename => 'test.txt').thumbnailable?
  end

  if convert_installed?
    def test_thumbnail_should_generate_the_thumbnail
      set_fixtures_attachments_directory
      Attachment.clear_thumbnails
      to_test = []
      # image/png
      to_test << Attachment.find(16)
      # application/pdf
      if Redmine::Thumbnail.gs_available?
        to_test << Attachment.find(23)
      else
        puts '(Ghostscript not available)'
      end

      assert_difference "Dir.glob(File.join(Attachment.thumbnails_storage_path, '*.thumb')).size", to_test.size do
        to_test.each do |attachment|
          thumbnail = attachment.thumbnail
          thumbnail_name = "#{attachment.digest}_#{attachment.filesize}_#{Setting.thumbnails_size}.thumb"
          assert_equal thumbnail_name, File.basename(thumbnail)
          assert File.exist?(thumbnail)
        end
      end
    ensure
      set_tmp_attachments_directory
    end

    def test_should_reuse_thumbnail
      Attachment.clear_thumbnails

      a = Attachment.create!(
        :container => Issue.find(1),
        :file => uploaded_test_file("2010/11/101123161450_testfile_1.png", "image/png"),
        :author => User.find(1)
      )
      a_thumb = b_thumb = nil
      assert_difference "Dir.glob(File.join(Attachment.thumbnails_storage_path, '*.thumb')).size" do
        a_thumb = a.thumbnail
      end

      b = Attachment.create!(
        :container => Issue.find(2),
        :file => uploaded_test_file("2010/11/101123161450_testfile_1.png", "image/png"),
        :author => User.find(1)
      )
      assert_no_difference "Dir.glob(File.join(Attachment.thumbnails_storage_path, '*.thumb')).size" do
        b_thumb = b.thumbnail
      end
      assert_equal a_thumb, b_thumb
    end

    def test_destroy_should_destroy_thumbnails
      a = Attachment.create!(
        :container => Issue.find(1),
        :file => uploaded_test_file("2010/11/101123161450_testfile_1.png", "image/png"),
        :author => User.find(1)
      )
      diskfile  = a.diskfile
      thumbnail = a.thumbnail
      assert File.exist?(diskfile)
      assert File.exist?(thumbnail)
      assert a.destroy
      refute File.exist?(diskfile)
      refute File.exist?(thumbnail)
    end

    def test_thumbnail_should_return_nil_if_generation_fails
      Redmine::Thumbnail.expects(:generate).raises(SystemCallError, 'Something went wrong')
      set_fixtures_attachments_directory
      attachment = Attachment.find(16)
      assert_nil attachment.thumbnail
    ensure
      set_tmp_attachments_directory
    end

    def test_thumbnail_should_be_at_least_of_requested_size
      set_fixtures_attachments_directory
      attachment = Attachment.find(16)
      Attachment.clear_thumbnails
      [
        [0, 100],
        [49, 50],
        [50, 50],
        [51, 100],
        [100, 100],
        [101, 150],
      ].each do |size, generated_size|
        thumbnail = attachment.thumbnail(size: size)
        assert_equal(
          "8e0294de2441577c529f170b6fb8f638_2654_#{generated_size}.thumb",
          File.basename(thumbnail))
      end
    ensure
      set_tmp_attachments_directory
    end

    def test_thumbnail_should_timeout
      dummy_pid = 37530
      Process.stubs(:spawn).returns(dummy_pid)
      Process.stubs(:wait2).raises(Timeout::Error)
      Process.stubs(:kill).returns(1)
      Process.stubs(:wait).returns(dummy_pid)
      Rails.logger.expects(:error).with(regexp_matches(/Creating thumbnail timed out/))

      set_fixtures_attachments_directory
      Attachment.clear_thumbnails

      attachment = Attachment.find(16)
      thumbnail = attachment.thumbnail

      assert_nil thumbnail
    ensure
      set_tmp_attachments_directory
    end
  else
    puts '(ImageMagick convert not available)'
  end

  def test_is_text
    js_attachment = Attachment.new(
      :container => Issue.find(1),
      :file => uploaded_test_file('hello.js', 'application/javascript'),
      :author => User.find(1))

    to_test = {
      js_attachment => true,               # hello.js (application/javascript)
      attachments(:attachments_003) => false, # logo.gif (image/gif)
      attachments(:attachments_004) => true,  # source.rb (application/x-ruby)
      attachments(:attachments_015) => true,  # private.diff (text/x-diff)
      attachments(:attachments_016) => false, # testfile.png (image/png)
    }
    to_test.each do |attachment, expected|
      assert_equal expected, attachment.is_text?, attachment.inspect
    end
  end
end
