/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.root;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.time.Instant;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code started} element of the events namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public final class Started extends Event {

	/**
	 * Qualified name of the {@code started} element.
	 */
	public static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_EVENTS, "started");

	/**
	 * Qualified name of the {@code name} attribute of a started event.
	 */
	private static final QualifiedName NAME = QualifiedName.of(Namespace.REPORTING_EVENTS, "name");

	/**
	 * Qualified name of the {@code parentId} attribute of a started event.
	 */
	public static final QualifiedName PARENT_ID = QualifiedName.of(Namespace.REPORTING_EVENTS, "parentId");

	Started(Context context) {
		super(context, ELEMENT);
	}

	@Override
	Started withId(String id) {
		super.withId(id);
		return this;
	}

	@Override
	Started withTime(Instant time) {
		super.withTime(time);
		return this;
	}

	Started withName(String name) {
		withAttribute(NAME, name);
		return this;
	}

	/**
	 * Set the {@code parentId} attribute of this started event.
	 *
	 * @param parentId the parent id
	 * @return this event
	 */
	@SuppressWarnings("UnusedReturnValue")
	public Started withParentId(String parentId) {
		withAttribute(PARENT_ID, parentId);
		return this;
	}
}
