/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.api;

import org.apiguardian.api.API;
import org.opentest4j.reporting.schema.QualifiedName;
import org.w3c.dom.Document;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * Context encapsulating a single XML document along with its namespace configuration.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class Context {

	static Context create(QualifiedName rootElement, NamespaceRegistry namespaceRegistry) {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.newDocument();
			Context context = new Context(document, namespaceRegistry);
			document.appendChild(createRoot(rootElement, namespaceRegistry, context));
			return context;
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private static org.w3c.dom.Element createRoot(QualifiedName rootElement, NamespaceRegistry namespaceRegistry,
			Context context) {
		org.w3c.dom.Element root = context.createElement(rootElement);
		namespaceRegistry.getAdditionalNamespaces() //
				.forEach(namespace -> {
					String prefix = namespaceRegistry.getPrefix(namespace).orElseThrow(IllegalStateException::new);
					root.setAttribute("xmlns:" + prefix, namespace.getUri());
				});
		return root;
	}

	private final ConcurrentMap<QualifiedName, String> prefixedNames = new ConcurrentHashMap<>();
	private final Document document;
	private final NamespaceRegistry namespaceRegistry;

	private Context(Document document, NamespaceRegistry namespaceRegistry) {
		this.document = document;
		this.namespaceRegistry = namespaceRegistry;
	}

	Document getDocument() {
		return document;
	}

	org.w3c.dom.Element createElement(QualifiedName qualifiedName) {
		return namespaceRegistry.getPrefix(qualifiedName.getNamespace()).isPresent()
				? document.createElementNS(qualifiedName.getNamespace().getUri(), prefixed(qualifiedName))
				: document.createElement(qualifiedName.getSimpleName());
	}

	String prefixed(QualifiedName qualifiedName) {
		Optional<String> prefix = namespaceRegistry.getPrefix(qualifiedName.getNamespace());
		return prefixedNames.computeIfAbsent(qualifiedName,
			__ -> prefix.map(it -> it + ":" + qualifiedName.getSimpleName()).orElse(qualifiedName.getSimpleName()));
	}
}
