// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Lambda_b and B0 production at 7 and 8 TeV
  class LHCB_2016_I1391317 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2016_I1391317);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      declare(UnstableParticles(), "UFS");

      for (double eVal : allowedEnergies()) {
        const int en = round(eVal);
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        int ih(en==8000);
        // histograms
        book(_h_Lambda[en],{2.0,2.5,3.0,3.5,4.0,4.5});
        book(_h_B[en]     ,{2.0,2.5,3.0,3.5,4.0,4.5});
        for (size_t iy=0; iy < _h_Lambda[en]->numBins(); ++iy) {
          book(_h_Lambda[en]->bin(iy+1), 1+ih, 1, 1+iy);
          book(_h_B[en]     ->bin(iy+1), 3+ih, 1, 1+iy);
        }
        for (size_t ix=0; ix<2; ++ix) {
          book(_h_pT[en+ix],"TMP/h_pT_"+toString(en+ix),refData(5,1,1+ih));
          book(_h_y [en+ix],"TMP/h_y_" +toString(en+ix),refData(6,1,1+ih));
        }
      }
      raiseBeamErrorIf(_sqs==0);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Final state of unstable particles to get particle spectra
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      // loop over onium states
      for (const Particle& p : ufs.particles(Cuts::abspid==5122 || Cuts::abspid==511)) {
        // skip copies due mixing
        if (p.children().size()==1 && p.children()[0].abspid()==p.abspid()) continue;
        const double rap = p.rap();
        if (rap<2. || rap>4.5) continue;
        const double pT = p.pT();
        if (p.abspid()==5122) {
          _h_Lambda[_sqs]->fill(rap, pT/GeV);
          if (p.pid()>0) {
            _h_pT[_sqs+0]->fill(pT/GeV);
            if (pT<20*GeV) _h_y[_sqs+0]->fill(rap);
          }
          else {
            _h_pT[_sqs+1]->fill(pT);
            if (pT<20*GeV) _h_y[_sqs+1]->fill(rap);
          }
        }
        else {
          _h_B[_sqs]->fill(rap, pT/GeV);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // branching ratios
      const vector<double> br={3.2e-4,1.27e-3};
      // 0.5 particle/antiparticle
      const double factor = 0.5*crossSection()/picobarn/sumOfWeights();

      scale(_h_Lambda, br[0]*factor);
      divByGroupWidth(_h_Lambda);
      scale(_h_B, br[1]*factor);
      divByGroupWidth(_h_B);

      for (double eVal : allowedEnergies()) {
        const int en = int(eVal+0.5);
        int ih(en==8000);
        Estimate1DPtr tmp;
        book(tmp, 5, 1, 1+ih);
        asymm(_h_pT[en+0], _h_pT[en+1], tmp);
        tmp->scale(100.);
        book(tmp, 6, 1, 1+ih);
        asymm(_h_y[en+0], _h_y[en+1], tmp);
        tmp->scale(100.);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<int,Histo1DGroupPtr> _h_Lambda, _h_B;
    map<int,Histo1DPtr> _h_pT, _h_y;
    int _sqs = 0;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2016_I1391317);

}
