/*
*  RAL -- Rubrica Addressbook Library
*  file: card.h
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef _R_CARD_H
#define _R_CARD_H

#include <time.h>
#include <glib.h>
#include <glib-object.h>
#include <libxml/tree.h>

#include "types.h"
#include "group_box.h"
#include "group.h"
#include "address.h"
#include "net.h"
#include "telephone.h"
#include "ref.h"



#define R_CARD_TYPE            (r_card_get_type())

#define R_CARD(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj),   \
                                R_CARD_TYPE, RCard))

#define R_CARD_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass),    \
                                R_CARD_TYPE, RCardClass))

#define IS_R_CARD(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj),   \
                                R_CARD_TYPE))

#define IS_R_CARD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass),    \
                                R_CARD_TYPE))

#define R_CARD_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj),    \
                                R_CARD_TYPE, RCardClass))


typedef enum {
  R_RATE_NONE = 0,         /* no importance         */
  R_RATE_FEW,              /* few importance        */
  R_RATE_NORMAL,           /* normal                */
  R_RATE_HIGHT,            /* hight importance      */
  R_RATE_VERY_HIGHT        /* very hight importance */
} RRate;


enum {
  UNKNOW_CARD,
  PERSONAL_CARD,
  COMPANY_CARD
};


typedef struct _RCard          RCard;
typedef struct _RCardClass     RCardClass;
typedef struct _RCardPrivate   RCardPrivate;


struct _RCard 
{
  GObject parent;
  
  RCardPrivate* priv;
};


struct _RCardClass
{
  GObjectClass parent;
  
  /* methods */
  void     (*free)       (RCard* card);

  gpointer (*copy)       (RCard* card); 
  RInfos*  (*get_infos)  (RCard* card); 
  gboolean (*search)     (RCard* card, const gchar* str);

  void     (*print_card) (RCard* card); 

  /* signals */
};



GType  r_card_get_type (void);

RCard* r_card_new                (void);
void   r_card_free               (RCard* card);
RCard* r_card_copy               (RCard* card);
void   r_card_reassign_id        (RCard* card, glong id);

gboolean r_card_is_personal      (RCard* card);
gboolean r_card_is_deleted       (RCard* card);
gboolean r_card_search           (RCard* card, const gchar* str);

/*  card infos
*/
RInfos*  r_card_get_infos        (RCard* card);
void     r_card_free_infos       (RInfos *infos);


/*  group
*/
void     r_card_add_group        (RCard* card, RGroup* group);
gboolean r_card_delete_group     (RCard* card, gchar* name);
gboolean r_card_rename_group     (RCard* card, gchar* oldname, gchar* newname);
gboolean r_card_has_groups       (RCard* card);

gpointer r_card_find_group          (RCard* card, gchar* name);
gboolean r_card_belong_to_group     (RCard* card, const gchar* group_name);
GList*   r_card_get_groups_owned_by (RCard* card, const gchar* owner);

RGroup*  r_card_get_group        (RCard* card);
RGroup*  r_card_get_next_group   (RCard* card);
RGroup*  r_card_get_prev_group   (RCard* card);
void     r_card_reset_group      (RCard* card);
void     r_card_foreach_group    (RCard* card, RFunc func, gpointer user_data);


/*  address
*/
void     r_card_add_address      (RCard* card, RAddress* address);
gboolean r_card_delete_address   (RCard* card, RAddress* address);
gboolean r_card_replace_address  (RCard* card, RAddress* old, RAddress* new);
gpointer r_card_get_address      (RCard* card);
gpointer r_card_get_next_address (RCard* card);
gpointer r_card_get_prev_address (RCard* card);
void     r_card_reset_address    (RCard* card);
gpointer r_card_find_address     (RCard* card, RAddressType type);
void     r_card_foreach_address  (RCard* card, RFunc func, gpointer user_data);

/*  net address
*/
void     r_card_add_net_address      (RCard* card, RNetAddress* net);
gboolean r_card_delete_net_address   (RCard* card, RNetAddress* address);
gboolean r_card_replace_net_address  (RCard* card, RNetAddress* old,
				      RNetAddress* new);
gpointer r_card_get_net_address      (RCard* card);
gpointer r_card_get_next_net_address (RCard* card);
gpointer r_card_get_prev_net_address (RCard* card);
void     r_card_reset_net_address    (RCard* card);
gpointer r_card_find_net_address     (RCard* card, RNetAddressType type);
void     r_card_foreach_net_address  (RCard* card, RFunc func, 
				      gpointer user_data);


/*  telephone
*/
void     r_card_add_telephone      (RCard* card, RTelephone* tel);
gboolean r_card_delete_telephone   (RCard* card, RTelephone* phone);
gboolean r_card_replace_telephone  (RCard* card, RTelephone* old,
				    RTelephone* new);
gpointer r_card_get_telephone      (RCard* card);
gpointer r_card_get_next_telephone (RCard* card);
gpointer r_card_get_prev_telephone (RCard* card);
void     r_card_reset_telephone    (RCard* card);
gpointer r_card_find_telephone     (RCard* card, RTelephoneType type);
void     r_card_foreach_telephone  (RCard* card, RFunc func, 
				    gpointer user_data);


/*  references
 */
void     r_card_add_ref      (RCard* card, RRef* ref);
gboolean r_card_del_ref      (RCard* card, RRef* ref);
gpointer r_card_get_ref      (RCard* card);
gpointer r_card_get_next_ref (RCard* card);
gpointer r_card_get_prev_ref (RCard* card);
void     r_card_reset_ref    (RCard* card);
void     r_card_foreach_ref  (RCard* card, RFunc func, gpointer user_data);
gboolean r_card_has_refs     (RCard* card);


/*  utils
*/
gchar*   r_card_get_group_owner    (RCard* card, RGroup* group);
gchar*   r_card_get_home_page      (RCard* card);
gchar*   r_card_get_email          (RCard* card);
gchar*   r_card_get_irc            (RCard* card);



/* just for tests
 */
void r_card_print     (RCard* card);

#endif
