/* RandomTreeGenerator.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.generators.graphs.standard;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import org.grinvin.factories.FactoryParameterException;
import org.grinvin.generators.graphs.AbstractGraphGenerator;
import org.grinvin.generators.graphs.AbstractGraphGeneratorInstance;
import org.grinvin.generators.graphs.GraphGeneratorInstance;
import org.grinvin.generators.graphs.GraphGeneratorSink;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.GraphBundle;

/**
 * Generator for random trees.
 */
public class RandomTreeGenerator extends AbstractGraphGenerator {
    
    //
    @Override protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int n0 = (Integer)values[0];
        int n1 = (Integer)values[1];
        int d = (Integer)values[2];
        int count = (Integer)values[3];
        if (n0 > n1 || n0 <= 0)
            throw new FactoryParameterException("Invalid order range");
        if (d <= 1)
            throw new FactoryParameterException("Invalid maximum degree");
        if (count < 1)
            throw new FactoryParameterException("Graph count must be at least one");
    }
    
    //
    public GraphGeneratorInstance createInstance() {
        return new RandomTreeGeneratorInstance(this,
                ((Integer)values[0]).intValue(),
                ((Integer)values[1]).intValue(),
                ((Integer)values[2]).intValue(),
                ((Integer)values[3]).intValue()
                );
    }

    //
    private static final Random RG = new Random();
        
    
    //
    private class RandomTreeGeneratorInstance extends AbstractGraphGeneratorInstance {
        
        // lower order
        private final int n0;
        
        // higher order
        private final int n1;

        // maximum degree
        private final int d;
        
        // count
        private final int count;
        
        //
        public RandomTreeGeneratorInstance(AbstractGraphGenerator agg,
                int n0, int n1, int d, int count) {
            super(agg);
            this.n0 = n0;
            this.n1 = n1;
            this.d = d;
            this.count = count;
        }
        
        /**
         * Generate random trees.
         */
        public void generate(GraphGeneratorSink sink) {
            for (int c=0; c < count; c++) {
                GraphBundle gb = sink.createGraphBundle();
                Graph graph = gb.createGraph();
                int n = n0 + RG.nextInt(n1+1-n0);
                int[] degrees = new int[n];
                for (int i=0; i < n; i++)
                    graph.addNewVertex();

                List<Integer> possibleParents = new ArrayList<Integer>(n);
                possibleParents.add(0);
                for (int i=1; i < n; i++) {
                    int parent = possibleParents.get(RG.nextInt(possibleParents.size()));
                    graph.addNewEdge(graph.getVertex(parent), graph.getVertex(i));
                    degrees[i]=1;
                    degrees[parent]++;
                    if(degrees[parent]==d)
                        possibleParents.remove(new Integer(parent));
                    possibleParents.add(i);
                }
                createTreeEmbedding(gb);
                sink.receiveGraphBundle(gb);
            }
        }
        
        public void createTreeEmbedding(GraphBundle bundle){
            
            // TODO: should go into a different class?
            
            Graph graph = bundle.getGraph();
            int order = graph.getNumberOfVertices();
            Embedding embedding = bundle.createEmbedding();
            if (embedding != null) {
                embedding.setDimension(2);
                int[] ecc = bundle.eccentricityList();
                int root = 0;
                int radius = ecc[0];
                for(int i = 1; i < ecc.length; i++)
                    if(ecc[i]<radius){
                        radius = ecc[i];
                        root = i;
                    }
                int[][] adjList = bundle.adjacencyList();
                
                embedTreeVertex(bundle, embedding, root, 2.0/radius, 1.0, -1.0, 1.0, -1);
            }

        }
        
        private void embedTreeVertex(GraphBundle bundle, Embedding embedding, int root, double verticalStep, double y, double minX, double maxX, int parent){
            double[] coords = new double[2];
            coords[0] = (maxX + minX)/2;
            coords[1] = y;
            embedding.setCoordinates(bundle.getGraph().getVertex(root), coords);
            
            int[][] adjList = bundle.adjacencyList();
            int nrOfChildren = adjList[root].length;
            if(parent!=-1)
                nrOfChildren--;
            double width = (maxX - minX)/nrOfChildren;
            int childRank = 0;
            for(int vertex : adjList[root]){
                if(vertex != parent){
                    embedTreeVertex(bundle, embedding, vertex, verticalStep, y - verticalStep, minX + childRank*width, minX + (childRank+1)*width, root);
                    childRank++;
                }
            }
        }
    }
    
}
