// Copyright 2008 The Closure Library Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS-IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

goog.provide('goog.math.Coordinate3Test');
goog.setTestOnly('goog.math.Coordinate3Test');

goog.require('goog.math.Coordinate3');
goog.require('goog.testing.jsunit');

function assertCoordinate3Equals(a, b) {
  assertTrue(b + ' should be equal to ' + a,
             goog.math.Coordinate3.equals(a, b));
}


function testCoordinate3MissingXYZ() {
  var noXYZ = new goog.math.Coordinate3();
  assertEquals(0, noXYZ.x);
  assertEquals(0, noXYZ.y);
  assertEquals(0, noXYZ.z);
  assertCoordinate3Equals(noXYZ, new goog.math.Coordinate3());
}


function testCoordinate3MissingYZ() {
  var noYZ = new goog.math.Coordinate3(10);
  assertEquals(10, noYZ.x);
  assertEquals(0, noYZ.y);
  assertEquals(0, noYZ.z);
  assertCoordinate3Equals(noYZ, new goog.math.Coordinate3(10));
}


function testCoordinate3MissingZ() {
  var noZ = new goog.math.Coordinate3(10, 20);
  assertEquals(10, noZ.x);
  assertEquals(20, noZ.y);
  assertEquals(0, noZ.z);
  assertCoordinate3Equals(noZ, new goog.math.Coordinate3(10, 20));
}


function testCoordinate3IntegerValues() {
  var intCoord = new goog.math.Coordinate3(10, 20, -19);
  assertEquals(10, intCoord.x);
  assertEquals(20, intCoord.y);
  assertEquals(-19, intCoord.z);
  assertCoordinate3Equals(intCoord, new goog.math.Coordinate3(10, 20, -19));
}


function testCoordinate3FloatValues() {
  var floatCoord = new goog.math.Coordinate3(10.5, 20.897, -71.385);
  assertEquals(10.5, floatCoord.x);
  assertEquals(20.897, floatCoord.y);
  assertEquals(-71.385, floatCoord.z);
  assertCoordinate3Equals(floatCoord,
      new goog.math.Coordinate3(10.5, 20.897, -71.385));
}


function testCoordinate3OneNonNumericValue() {
  var dim5 = new goog.math.Coordinate3('ten', 1000, 85);
  assertTrue(isNaN(dim5.x));
  assertEquals(1000, dim5.y);
  assertEquals(85, dim5.z);
}


function testCoordinate3AllNonNumericValues() {
  var nonNumeric = new goog.math.Coordinate3('ten',
                                             {woop: 'test'},
                                             Math.sqrt(-1));
  assertTrue(isNaN(nonNumeric.x));
  assertTrue(isNaN(nonNumeric.y));
  assertTrue(isNaN(nonNumeric.z));
}


function testCoordinate3Origin() {
  var origin = new goog.math.Coordinate3(0, 0, 0);
  assertEquals(0, origin.x);
  assertEquals(0, origin.y);
  assertEquals(0, origin.z);
  assertCoordinate3Equals(origin, new goog.math.Coordinate3(0, 0, 0));
}


function testCoordinate3Clone() {
  var c = new goog.math.Coordinate3();
  assertCoordinate3Equals(c, c.clone());
  c.x = -12;
  c.y = 13;
  c.z = 5;
  assertCoordinate3Equals(c, c.clone());
}


function testToString() {
  assertEquals('(0, 0, 0)', new
               goog.math.Coordinate3().toString());
  assertEquals('(1, 0, 0)', new
               goog.math.Coordinate3(1).toString());
  assertEquals('(1, 2, 0)', new
               goog.math.Coordinate3(1, 2).toString());
  assertEquals('(0, 0, 0)', new goog.math.Coordinate3(0, 0, 0).toString());
  assertEquals('(1, 2, 3)', new goog.math.Coordinate3(1, 2, 3).toString());
  assertEquals('(-4, 5, -3)', new goog.math.Coordinate3(-4, 5, -3).toString());
  assertEquals('(11.25, -71.935, 2.8)',
               new goog.math.Coordinate3(11.25, -71.935, 2.8).toString());
}


function testEquals() {
  var a = new goog.math.Coordinate3(3, 4, 5);
  var b = new goog.math.Coordinate3(3, 4, 5);
  var c = new goog.math.Coordinate3(-3, 4, -5);

  assertTrue(goog.math.Coordinate3.equals(null, null));
  assertFalse(goog.math.Coordinate3.equals(a, null));
  assertTrue(goog.math.Coordinate3.equals(a, a));
  assertTrue(goog.math.Coordinate3.equals(a, b));
  assertFalse(goog.math.Coordinate3.equals(a, c));
}


function testCoordinate3Distance() {
  var a = new goog.math.Coordinate3(-2, -3, 1);
  var b = new goog.math.Coordinate3(2, 0, 1);
  assertEquals(5, goog.math.Coordinate3.distance(a, b));
}


function testCoordinate3SquaredDistance() {
  var a = new goog.math.Coordinate3(7, 11, 1);
  var b = new goog.math.Coordinate3(3, -1, 1);
  assertEquals(160, goog.math.Coordinate3.squaredDistance(a, b));
}


function testCoordinate3Difference() {
  var a = new goog.math.Coordinate3(7, 11, 1);
  var b = new goog.math.Coordinate3(3, -1, 1);
  assertCoordinate3Equals(goog.math.Coordinate3.difference(a, b),
                          new goog.math.Coordinate3(4, 12, 0));
}


function testToArray() {
  var a = new goog.math.Coordinate3(7, 11, 1);
  var b = a.toArray();
  assertEquals(b.length, 3);
  assertEquals(b[0], 7);
  assertEquals(b[1], 11);
  assertEquals(b[2], 1);

  var c = new goog.math.Coordinate3('abc', 'def', 'xyz');
  var result = c.toArray();
  assertTrue(isNaN(result[0]));
  assertTrue(isNaN(result[1]));
  assertTrue(isNaN(result[2]));
}


function testFromArray() {
  var a = [1, 2, 3];
  var b = goog.math.Coordinate3.fromArray(a);
  assertEquals('(1, 2, 3)', b.toString());

  var c = [1, 2];
  var d = goog.math.Coordinate3.fromArray(c);
  assertEquals('(1, 2, 0)', d.toString());

  var e = [1];
  var f = goog.math.Coordinate3.fromArray(e);
  assertEquals('(1, 0, 0)', f.toString());

  var g = [];
  var h = goog.math.Coordinate3.fromArray(g);
  assertEquals('(0, 0, 0)', h.toString());

  var tooLong = [1, 2, 3, 4, 5, 6];
  assertThrows('Error should be thrown attempting to convert an invalid type.',
      goog.partial(goog.math.Coordinate3.fromArray, tooLong));
}
