/* shishi.h	Header file for Shishi library.                       -*- c -*-
 * Copyright (C) 2002, 2003  Simon Josefsson
 *
 * This file is part of Shishi.
 *
 * Shishi is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Shishi is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Shishi; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef SHISHI_H
#define SHISHI_H

#include <stddef.h>		/* size_t */
#include <stdio.h>		/* FILE */
#include <time.h>		/* time_t */
#include <shishi-int.h>		/* uint32_t */

#define SHISHI_VERSION "0.0.3"

/* Error codes */
enum
{
  SHISHI_OK = 0,
  SHISHI_ASN1_ERROR = 1,
  SHISHI_FOPEN_ERROR = 2,
  SHISHI_FCLOSE_ERROR = 3,
  SHISHI_MALLOC_ERROR = 4,
  SHISHI_BASE64_ERROR = 5,
  SHISHI_REALM_MISMATCH = 6,
  SHISHI_CNAME_MISMATCH = 7,
  SHISHI_NONCE_MISMATCH = 8,
  SHISHI_TGSREP_BAD_KEYTYPE = 9,
  SHISHI_KDCREP_BAD_KEYTYPE = 10,
  SHISHI_APREP_BAD_KEYTYPE = 11,
  SHISHI_APREP_VERIFY_FAILED = 12,
  SHISHI_APREQ_BAD_KEYTYPE = 13,
  SHISHI_TOO_SMALL_BUFFER = 14,
  SHISHI_DERIVEDKEY_TOO_SMALL = 15,
  SHISHI_KEY_TOO_LARGE = 16,
  SHISHI_CRYPTO_ERROR = 17,
  SHISHI_CRYPTO_INTERNAL_ERROR = 18,
  SHISHI_SOCKET_ERROR = 19,
  SHISHI_BIND_ERROR = 20,
  SHISHI_SENDTO_ERROR = 21,
  SHISHI_RECVFROM_ERROR = 22,
  SHISHI_CLOSE_ERROR = 23,
  SHISHI_KDC_TIMEOUT = 24,
  SHISHI_KDC_NOT_KNOWN_FOR_REALM = 25,
  SHISHI_TTY_ERROR = 26,
  SHISHI_GOT_KRBERROR = 27,
  SHISHI_HANDLE_ERROR = 28,
  SHISHI_INVALID_TKTS = 29,
  SHISHI_TICKET_BAD_KEYTYPE = 30,
  SHISHI_INVALID_KEY = 31,
  SHISHI_APREQ_DECRYPT_FAILED = 32,
  SHISHI_TICKET_DECRYPT_FAILED = 33,
  SHISHI_INVALID_TICKET = 34,
  SHISHI_OUT_OF_RANGE = 35,
  SHISHI_ASN1_NO_ELEMENT = 36,
  SHISHI_SAFE_BAD_KEYTYPE = 37,
  SHISHI_SAFE_VERIFY_FAILED = 38,
  SHISHI_PKCS5_INVALID_PRF = 39,
  SHISHI_PKCS5_INVALID_ITERATION_COUNT = 40,
  SHISHI_PKCS5_INVALID_DERIVED_KEY_LENGTH = 41,
  SHISHI_PKCS5_DERIVED_KEY_TOO_LONG = 42,
  SHISHI_INVALID_PRINCIPAL_NAME = 43,
  SHISHI_INVALID_ARGUMENT = 44
};

/* This is not specified in the ASN.1 syntax for some reason. */
typedef enum
{
  SHISHI_NT_UNKNOWN = 0,	/* Name type not known */
  SHISHI_NT_PRINCIPAL,		/* Just the name of the principal as in */
  /* DCE, or for users */
  SHISHI_NT_SRV_INST,		/* Service and other unique instance (krbtgt) */
  SHISHI_NT_SRV_HST,		/* Service with host name as instance */
  /* (telnet, rcommands) */
  SHISHI_NT_SRV_XHST,		/* Service with host as remaining components */
  SHISHI_NT_UID			/* Unique ID */
}
Shishi_name_type;

/* This is not specified in the ASN.1 syntax for some reason. */
typedef enum
{
  SHISHI_PA_TGS_REQ = 1,	/* padata is APREQ */
  SHISHI_PA_ENC_TIMESTAMP,	/* timestamp pre authentication */
  SHISHI_PA_PW_SALT,		/* salt for string-to-key */
  SHISHI_PA_ETYPE_INFO = 10,	/* in krb-error, request for preauth */
  SHISHI_PA_USE_SPECIFIED_KVNO = 20	/* use specified kvno */
}
Shishi_padata_type;

/* This is not specified in the ASN.1 syntax for some reason. */
typedef enum
{
  SHISHI_TR_DOMAIN_X500_COMPRESS = 1
}
Shishi_tr_type;

typedef enum
{
  SHISHI_APOPTIONS_RESERVED = 0x1,	/* bit 0 */
  SHISHI_APOPTIONS_USE_SESSION_KEY = 0x2,	/* bit 1 */
  SHISHI_APOPTIONS_MUTUAL_REQUIRED = 0x4	/* bit 2 */
}
Shishi_apoptions;

typedef enum
{
  SHISHI_TICKETFLAGS_RESERVED = 0x1,	/* bit 0 */
  SHISHI_TICKETFLAGS_FORWARDABLE = 0x2,	/* bit 1 */
  SHISHI_TICKETFLAGS_FORWARDED = 0x4,	/* bit 2 */
  SHISHI_TICKETFLAGS_PROXIABLE = 0x8,	/* bit 3 */
  SHISHI_TICKETFLAGS_PROXY = 0x10,	/* bit 4 */
  SHISHI_TICKETFLAGS_MAY_POSTDATE = 0x20,	/* bit 5 */
  SHISHI_TICKETFLAGS_POSTDATED = 0x40,	/* bit 6 */
  SHISHI_TICKETFLAGS_INVALID = 0x80,	/* bit 7 */
  SHISHI_TICKETFLAGS_RENEWABLE = 0x100,	/* bit 8 */
  SHISHI_TICKETFLAGS_INITIAL = 0x200,	/* bit 9 */
  SHISHI_TICKETFLAGS_PRE_AUTHENT = 0x400,	/* bit 10 */
  SHISHI_TICKETFLAGS_HW_AUTHENT = 0x800,	/* bit 11 */
  SHISHI_TICKETFLAGS_TRANSITED_POLICY_CHECKED = 0x1000,	/* bit 12 */
  SHISHI_TICKETFLAGS_OK_AS_DELEGATE = 0x2000	/* bit 13 */
}
Shishi_ticketflags;

typedef enum
{
  SHISHI_KDCOPTIONS_RESERVED = 0x1,	/* bit 0 */
  SHISHI_KDCOPTIONS_FORWARDABLE = 0x2,	/* bit 1 */
  SHISHI_KDCOPTIONS_FORWARDED = 0x4,	/* bit 2 */
  SHISHI_KDCOPTIONS_PROXIABLE = 0x8,	/* bit 3 */
  SHISHI_KDCOPTIONS_PROXY = 0x10,	/* bit 4 */
  SHISHI_KDCOPTIONS_ALLOW_POSTDATE = 0x20,	/* bit 5 */
  SHISHI_KDCOPTIONS_POSTDATED = 0x40,	/* bit 6 */
  SHISHI_KDCOPTIONS_UNUSED7 = 0x80,	/* bit 7 */
  SHISHI_KDCOPTIONS_RENEWABLE = 0x100,	/* bit 8 */
  SHISHI_KDCOPTIONS_UNUSED9 = 0x200,	/* bit 9 */
  SHISHI_KDCOPTIONS_UNUSED10 = 0x400,	/* bit 10 */
  SHISHI_KDCOPTIONS_UNUSED11 = 0x800,	/* bit 11 */
  SHISHI_KDCOPTIONS_DISABLE_TRANSITED_CHECK = 0x4000000,	/* bit 26 */
  SHISHI_KDCOPTIONS_RENEWABLE_OK = 0x8000000,	/* bit 27 */
  SHISHI_KDCOPTIONS_ENC_TKT_IN_SKEY = 0x10000000,	/* bit 28 */
  SHISHI_KDCOPTIONS_RENEW = 0x40000000,	/* bit 30 */
  SHISHI_KDCOPTIONS_VALIDATE = 0x80000000	/* bit 31 */
}
Shishi_KDCOptions;

typedef enum
{
  SHISHI_MSGTYPE_AS_REQ = 0x10,
  SHISHI_MSGTYPE_AS_REP = 0x11,
  SHISHI_MSGTYPE_TGS_REQ = 0x12,
  SHISHI_MSGTYPE_TGS_REP = 0x13
}
Shishi_msgtype;

typedef enum
{
  SHISHI_LRTYPE_LAST_INITIAL_TGT_REQUEST = 1,
  SHISHI_LRTYPE_LAST_INITIAL_REQUEST = 2,
  SHISHI_LRTYPE_NEWEST_TGT_ISSUE = 3,
  SHISHI_LRTYPE_LAST_RENEWAL = 4,
  SHISHI_LRTYPE_LAST_REQUEST = 5
}
Shishi_lrtype;

typedef enum
{
  SHISHI_NULL = 0,
  SHISHI_DES_CBC_CRC = 1,
  SHISHI_DES_CBC_MD4 = 2,
  SHISHI_DES_CBC_MD5 = 3,
  SHISHI_DES_CBC_NONE = 4,
  SHISHI_DES3_CBC_NONE = 6,
  SHISHI_DES3_CBC_HMAC_SHA1_KD = 16,
  SHISHI_AES128_CTS_HMAC_SHA1_96 = 17,
  SHISHI_AES256_CTS_HMAC_SHA1_96 = 18
}
Shishi_etype;

typedef enum
{
  SHISHI_CRC32 = 1,
  SHISHI_RSA_MD4 = 2,
  SHISHI_RSA_MD4_DES = 3,
  SHISHI_DES_MAC = 4,
  SHISHI_DES_MAC_K = 5,
  SHISHI_RSA_MD4_DES_K = 6,
  SHISHI_RSA_MD5 = 7,
  SHISHI_RSA_MD5_DES = 8,
  SHISHI_RSA_MD5_DES_GSS = 9,  /* XXX */
  SHISHI_HMAC_SHA1_96_AES128 = 10,
  SHISHI_HMAC_SHA1_96_AES256 = 11,
  SHISHI_HMAC_SHA1_DES3_KD = 12
}
Shishi_cksumtype;

typedef enum
{
  SHISHI_FILETYPE_TEXT = 0,
  SHISHI_FILETYPE_DER,
  SHISHI_FILETYPE_HEX,
  SHISHI_FILETYPE_BASE64,
  SHISHI_FILETYPE_BINARY
}
Shishi_filetype;

typedef enum
{
  /* 1. AS-REQ PA-ENC-TIMESTAMP padata timestamp, encrypted with the
     client key */
  SHISHI_KEYUSAGE_ASREQ_PA_ENC_TIMESTAMP = 1,
  /* 2. AS-REP Ticket and TGS-REP Ticket (includes TGS session key or
     application session key), encrypted with the service key  */
  SHISHI_KEYUSAGE_ENCTICKETPART = 2,
  /* 3. AS-REP encrypted part (includes TGS session key or application
     session key), encrypted with the client key */
  SHISHI_KEYUSAGE_ENCASREPPART = 3,
  /* 4. TGS-REQ KDC-REQ-BODY AuthorizationData, encrypted with the TGS
     session key  */
  SHISHI_KEYUSAGE_TGSREQ_AUTHORIZATIONDATA_TGS_SESSION_KEY = 4,
  /* 5. TGS-REQ KDC-REQ-BODY AuthorizationData, encrypted with the TGS
     authenticator subkey (section 5.4.1) */
  SHISHI_KEYUSAGE_TGSREQ_AUTHORIZATIONDATA_TGS_AUTHENTICATOR_KEY = 5,
  /* 6. TGS-REQ PA-TGS-REQ padata AP-REQ Authenticator cksum, keyed with the
     TGS session key  */
  SHISHI_KEYUSAGE_TGSREQ_APREQ_AUTHENTICATOR_CKSUM = 6,
  /* 7. TGS-REQ PA-TGS-REQ padata AP-REQ Authenticator (includes TGS
     authenticator subkey), encrypted with the TGS session key */
  SHISHI_KEYUSAGE_TGSREQ_APREQ_AUTHENTICATOR = 7,
  /* 8. TGS-REP encrypted part (includes application session key), encrypted
     with the TGS session key */
  SHISHI_KEYUSAGE_ENCTGSREPPART_SESSION_KEY = 8,
  /* 9. TGS-REP encrypted part (includes application session key), encrypted
     with the TGS authenticator subkey */
  SHISHI_KEYUSAGE_ENCTGSREPPART_AUTHENTICATOR_KEY = 9,
  /* 10. AP-REQ Authenticator cksum, keyed with the application
     session key */
  SHISHI_KEYUSAGE_APREQ_AUTHENTICATOR_CKSUM = 10,
  /* 11. AP-REQ Authenticator (includes application authenticator subkey),
     encrypted with the application session key */
  SHISHI_KEYUSAGE_APREQ_AUTHENTICATOR = 11,
  /* 12. AP-REP encrypted part (includes application session subkey),
     encrypted with the application session key */
  SHISHI_KEYUSAGE_ENCAPREPPART = 12,
  /* 13. KRB-PRIV encrypted part, encrypted with a key chosen by the
     application */
  SHISHI_KEYUSAGE_KRB_PRIV = 13,
  /* 14. KRB-CRED encrypted part, encrypted with a key chosen by the
     application */
  SHISHI_KEYUSAGE_KRB_CRED = 14,
  /* 15. KRB-SAFE cksum, keyed with a key chosen by the application */
  SHISHI_KEYUSAGE_KRB_SAFE = 15,
  /* 18. KRB-ERROR checksum (e-cksum) */
  SHISHI_KEYUSAGE_KRB_ERROR = 18,
  /* 19. AD-KDCIssued checksum (ad-checksum) */
  SHISHI_KEYUSAGE_AD_KDCISSUED = 19,
  /* 20. Checksum for Mandatory Ticket Extensions */
  SHISHI_KEYUSAGE_TICKET_EXTENSION = 20,
  /* 21. Checksum in Authorization Data in Ticket Extensions */
  SHISHI_KEYUSAGE_TICKET_EXTENSION_AUTHORIZATION = 21,
  /* 22-24. Reserved for use in GSSAPI mechanisms derived from RFC 1964.
     (raeburn/MIT) */
  SHISHI_KEYUSAGE_GSS_R1 = 22,
  SHISHI_KEYUSAGE_GSS_R2 = 23,
  SHISHI_KEYUSAGE_GSS_R3 = 24
    /* 25-511. Reserved for future use in Kerberos and related protocols. */
    /* 512-1023. Reserved for uses internal to a Kerberos implementation. */
}
Shishi_keyusage;

typedef enum
{
  SHISHI_TKTSHINTFLAGS_ACCEPT_EXPIRED = 1
}
Shishi_tkts_hintflags;

struct Shishi_tkts_hint
{
  int startpos;
  char *server;
  char *serverrealm;
  char *client;
  char *clientrealm;
  int flags;
  int32_t etype;
  char *passwd;
  time_t starttime;
  time_t endtime;
  time_t renew_till;
  int renewable;
};
typedef struct Shishi_tkts_hint Shishi_tkts_hint;

typedef struct Shishi Shishi;
typedef struct Shishi_tkt Shishi_tkt;
typedef struct Shishi_tkts Shishi_tkts;
typedef struct Shishi_as Shishi_as;
typedef struct Shishi_tgs Shishi_tgs;
typedef struct Shishi_ap Shishi_ap;
typedef struct Shishi_key Shishi_key;
typedef struct Shishi_safe Shishi_safe;
#ifndef _SHISHI_HAS_LIBTASN1_H
typedef struct node_asn_struct *ASN1_TYPE;
#endif
typedef ASN1_TYPE Shishi_asn1;

/* init.c */
extern Shishi *shishi (void);
extern int shishi_init (Shishi ** handle);
extern int shishi_init_with_paths (Shishi ** handle,
				   const char *tktsfile,
				   const char *systemcfgfile,
				   const char *usercfgfile);
extern int shishi_init_server (Shishi ** handle);
extern int shishi_init_server_with_paths (Shishi ** handle,
					  const char *systemcfgfile);
extern void shishi_info (Shishi * handle, const char *fmt, ...);
extern void shishi_warn (Shishi * handle, const char *fmt, ...);

/* done.c */
extern void shishi_done (Shishi * handle);

/* cfg.c */
extern int shishi_cfg (Shishi * handle, char *option);
extern int shishi_cfg_from_file (Shishi * handle, const char *cfg);
extern int shishi_cfg_print (Shishi * handle, FILE * fh);
extern const char *shishi_cfg_default_systemfile (Shishi * handle);
extern const char *shishi_cfg_default_userdirectory (Shishi * handle);
extern const char *shishi_cfg_default_userfile (Shishi * handle);
extern int shishi_cfg_clientkdcetype (Shishi * handle, int32_t ** etypes);
extern int shishi_cfg_clientkdcetype_set (Shishi * handle, char *value);

/* error.c */
extern const char *shishi_strerror (int err);
extern const char *shishi_strerror_details (Shishi * handle);
extern void shishi_error_set (Shishi * handle, const char *error);
extern void shishi_error_printf (Shishi * handle, char *format, ...);
extern void shishi_error_clear (Shishi * handle);

/* realm.c */
extern char *shishi_realm_default_guess (void);
extern const char *shishi_realm_default (Shishi * handle);
extern void shishi_realm_default_set (Shishi * handle, const char *realm);
extern const char *shishi_realm_for_server_file (Shishi * handle,
						 char *server);
extern const char *shishi_realm_for_server_dns (Shishi * handle,
						char *server);
extern const char *shishi_realm_for_server (Shishi * handle, char *server);

/* principal.c */
extern char *shishi_principal_default_guess (void);
extern const char *shishi_principal_default (Shishi * handle);
extern void shishi_principal_default_set (Shishi * handle,
					  const char *principal);
extern int shishi_principal_name_get (Shishi * handle, Shishi_asn1 namenode,
				      const char *namefield, char *out,
				      size_t * outlen);
extern int shishi_principal_name_realm_get (Shishi * handle,
					    Shishi_asn1 namenode,
					    const char *namefield,
					    Shishi_asn1 realmnode,
					    const char *realmfield,
					    char *out, size_t * outlen);
extern int shishi_principal_name_set (Shishi * handle,
				      Shishi_asn1 namenode,
				      const char *namefield,
				      Shishi_name_type name_type,
				      const char *name[]);
extern int shishi_principal_set (Shishi * handle,
				 Shishi_asn1 namenode,
				 const char *namefield, const char *name);
extern int shishi_parse_name (Shishi * handle, const char *name,
			      char **principal, char **realm);
extern char *shishi_server_for_local_service (Shishi * handle,
					      const char *service);

/* ticket.c */
extern int shishi_ticket_sname_get (Shishi * handle,
				    Shishi_asn1 ticket,
				    char *server, size_t * serverlen);
extern int shishi_ticket_sname_set (Shishi * handle,
				    Shishi_asn1 ticket,
				    Shishi_name_type name_type,
				    char *sname[]);
extern int shishi_ticket_snamerealm_get (Shishi * handle, Shishi_asn1 ticket,
					 char *serverrealm,
					 int *serverrealmlen);
extern int shishi_ticket_srealmserver_set (Shishi * handle,
					   Shishi_asn1 ticket, char *realm,
					   char *server);
extern int shishi_ticket_set_server (Shishi * handle, Shishi_asn1 ticket,
				     const char *server);
extern int shishi_ticket_realm_get (Shishi * handle, Shishi_asn1 ticket,
				    char *realm, int *realmlen);
extern int shishi_ticket_realm_set (Shishi * handle, Shishi_asn1 ticket,
				    const char *realm);
extern int shishi_ticket_get_enc_part_etype (Shishi * handle,
					     Shishi_asn1 ticket,
					     int32_t * etype);
extern int shishi_ticket_set_enc_part (Shishi * handle, Shishi_asn1 ticket,
				       int etype, int kvno,
				       char *buf, size_t buflen);
extern int shishi_ticket_add_enc_part (Shishi * handle, Shishi_asn1 ticket,
				       Shishi_key * key,
				       Shishi_asn1 encticketpart);
extern int shishi_enckdcreppart_key_set (Shishi * handle,
					 Shishi_asn1 enckdcreppart,
					 Shishi_key * key);
extern int shishi_ticket_decrypt (Shishi * handle, Shishi_asn1 ticket,
				  Shishi_key * key,
				  Shishi_asn1 * encticketpart);

/* tkt.c */
extern Shishi_asn1 shishi_tkt_ticket (Shishi_tkt * tkt);
extern Shishi_asn1 shishi_tkt_kdcrep (Shishi_tkt * tkt);
extern Shishi_asn1 shishi_tkt_enckdcreppart (Shishi_tkt * tkt);
extern void shishi_tkt_enckdcreppart_set (Shishi_tkt * tkt,
					  Shishi_asn1 enckdcreppart);
extern Shishi_asn1 shishi_tkt_encticketpart (Shishi_tkt * tkt);
extern void shishi_tkt_encticketpart_set (Shishi_tkt * tkt,
					  Shishi_asn1 encticketpart);
extern Shishi_key *shishi_tkt_key (Shishi_tkt * tkt);
extern int shishi_tkt_key_set (Shishi_tkt * tkt, Shishi_key * key);
extern int shishi_tkt (Shishi * handle, Shishi_tkt ** tkt);
extern Shishi_tkt *shishi_tkt2 (Shishi * handle,
				Shishi_asn1 ticket,
				Shishi_asn1 enckdcreppart,
				Shishi_asn1 kdcrep);
extern int shishi_tkt_pretty_print (Shishi_tkt * tkt, FILE * fh);
extern int shishi_tkt_realm (Shishi_tkt * tkt, char *realm, int *realmlen);
extern int shishi_tkt_cnamerealm (Shishi_tkt * tkt,
				  char *cnamerealm, int *cnamerealmlen);
extern int shishi_tkt_cnamerealm_p (Shishi_tkt * tkt, const char *client);
extern int shishi_tkt_client (Shishi_tkt * tkt, char *client, int *clientlen);
extern int shishi_tkt_client_p (Shishi_tkt * tkt, const char *client);
extern int shishi_tkt_clientrealm_set (Shishi_tkt * tkt,
				       char *realm, char *client);
extern int shishi_tkt_serverrealm_set (Shishi_tkt * tkt,
				       char *realm, char *server);
extern int shishi_tkt_build (Shishi_tkt * tkt, Shishi_key * key);
extern int shishi_tkt_lastreq (Shishi_tkt * tkt,
			       char *lrtime, int *lrtimelen, int lrtype);
extern time_t shishi_tkt_lastreqc (Shishi_tkt * tkt, Shishi_lrtype lrtype);
extern int shishi_tkt_lastreq_pretty_print (Shishi_tkt * tkt, FILE * fh);
extern int shishi_tkt_authtime (Shishi_tkt * tkt,
				char *authtime, int *authtimelen);
extern time_t shishi_tkt_authctime (Shishi_tkt * tkt);
extern int shishi_tkt_starttime (Shishi_tkt * tkt,
				 char *starttime, int *starttimelen);
extern time_t shishi_tkt_startctime (Shishi_tkt * tkt);
extern int shishi_tkt_endtime (Shishi_tkt * tkt,
			       char *endtime, int *endtimelen);
extern time_t shishi_tkt_endctime (Shishi_tkt * tkt);
extern int shishi_tkt_renew_till (Shishi_tkt * tkt,
				  char *renewtilltime, int *renewtilllen);
extern time_t shishi_tkt_renew_tillc (Shishi_tkt * tkt);
extern int shishi_tkt_keytype (Shishi_tkt * tkt, int32_t * etype);
extern int shishi_tkt_keytype_p (Shishi_tkt * tkt, int32_t etype);
extern int shishi_tkt_server (Shishi_tkt * tkt,
			      char *service, int *servicelen);
extern int shishi_tkt_server_p (Shishi_tkt * tkt, const char *service);
extern int shishi_tkt_server_realm (Shishi_tkt * tkt,
				    char *serverrealm, int *serverrealmlen);
extern int shishi_tkt_valid_at_time_p (Shishi_tkt * tkt, time_t now);
extern int shishi_tkt_valid_now_p (Shishi_tkt * tkt);
extern int shishi_tkt_decrypt (Shishi_tkt * tkt, Shishi_key * key);
extern void shishi_tkt_done (Shishi_tkt * tkt);
extern int shishi_tkt_flags (Shishi_tkt * tkt, int *flags);
extern int shishi_tkt_flags_set (Shishi_tkt * tkt, int flags);
extern int shishi_tkt_forwardable_p (Shishi_tkt * tkt);
extern int shishi_tkt_forwarded_p (Shishi_tkt * tkt);
extern int shishi_tkt_proxiable_p (Shishi_tkt * tkt);
extern int shishi_tkt_proxy_p (Shishi_tkt * tkt);
extern int shishi_tkt_may_postdate_p (Shishi_tkt * tkt);
extern int shishi_tkt_postdated_p (Shishi_tkt * tkt);
extern int shishi_tkt_invalid_p (Shishi_tkt * tkt);
extern int shishi_tkt_renewable_p (Shishi_tkt * tkt);
extern int shishi_tkt_initial_p (Shishi_tkt * tkt);
extern int shishi_tkt_pre_authent_p (Shishi_tkt * tkt);
extern int shishi_tkt_hw_authent_p (Shishi_tkt * tkt);
extern int shishi_tkt_transited_policy_checked_p (Shishi_tkt * tkt);
extern int shishi_tkt_ok_as_delegate_p (Shishi_tkt * tkt);

/* tkts.c */
extern char *shishi_tkts_default_file_guess (void);
extern const char *shishi_tkts_default_file (Shishi * handle);
extern void shishi_tkts_default_file_set (Shishi * handle,
					  const char *tktsfile);
extern Shishi_tkts *shishi_tkts_default (Shishi * handle);
extern int shishi_tkts_default_to_file (Shishi_tkts * tkts);
extern int shishi_tkts (Shishi * handle, Shishi_tkts ** tkts);
extern Shishi_tkt *shishi_tkts_nth (Shishi_tkts * tkts, int ticketno);
extern int shishi_tkts_size (Shishi_tkts * tkts);
extern int shishi_tkts_add (Shishi_tkts * tkts, Shishi_tkt * tkt);
extern int shishi_tkts_new (Shishi_tkts * tkts,
			    Shishi_asn1 ticket,
			    Shishi_asn1 enckdcreppart, Shishi_asn1 kdcrep);
extern int shishi_tkts_remove (Shishi_tkts * tkts, int ticketno);
extern int shishi_tkts_expire (Shishi_tkts * tkts);
extern int shishi_tkts_print_for_service (Shishi_tkts * tkts,
					  FILE * fh, const char *service);
extern int shishi_tkts_print (Shishi_tkts * tkts, FILE * fh);
extern int shishi_tkts_write (Shishi_tkts * tkts, FILE * fh);
extern int shishi_tkts_to_file (Shishi_tkts * tkts, const char *filename);
extern int shishi_tkts_read (Shishi_tkts * tkts, FILE * fh);
extern int shishi_tkts_from_file (Shishi_tkts * tkts, const char *filename);
extern void shishi_tkts_done (Shishi_tkts ** tkts);
extern int shishi_tkt_match_p (Shishi_tkt * tkt, Shishi_tkts_hint * hint);
extern Shishi_tkt *shishi_tkts_find (Shishi_tkts * tkts,
				     Shishi_tkts_hint * hint);
extern Shishi_tkt *shishi_tkts_find_for_clientserver (Shishi_tkts * tkts,
						      const char *client,
						      const char *server);
extern Shishi_tkt *shishi_tkts_find_for_server (Shishi_tkts * tkts,
						const char *server);
extern Shishi_tkt *shishi_tkts_get (Shishi_tkts * tkts,
				    Shishi_tkts_hint * hint);
extern Shishi_tkt *shishi_tkts_get_for_clientserver (Shishi_tkts * tkts,
						     const char *client,
						     const char *server);
extern Shishi_tkt *shishi_tkts_get_for_server (Shishi_tkts * tkts,
					       const char *server);
extern Shishi_tkt *shishi_tkts_get_for_localservicepasswd (Shishi_tkts * tkts,
							   const char
							   *service,
							   const char
							   *passwd);

/* diskio.c */
extern int
shishi_enckdcreppart_print (Shishi * handle,
			    FILE * fh, Shishi_asn1 enckdcreppart);
extern int
shishi_enckdcreppart_save (Shishi * handle,
			   FILE * fh, Shishi_asn1 enckdcreppart);
extern int
shishi_enckdcreppart_parse (Shishi * handle,
			    FILE * fh, Shishi_asn1 * enckdcreppart);
extern int
shishi_enckdcreppart_read (Shishi * handle,
			   FILE * fh, Shishi_asn1 * enckdcreppart);
extern int shishi_ticket_save (Shishi * handle, FILE * fh,
			       Shishi_asn1 ticket);
extern int shishi_ticket_print (Shishi * handle, FILE * fh,
				Shishi_asn1 ticket);
extern int shishi_kdc_req_print (Shishi * handle, FILE * fh,
				 Shishi_asn1 asreq);
extern int shishi_kdc_rep_print (Shishi * handle, FILE * fh,
				 Shishi_asn1 asreq);
extern int shishi_kdc_print (Shishi * handle, FILE * fh, Shishi_asn1 asreq,
			     Shishi_asn1 asrep, Shishi_asn1 encasreppart);
extern int shishi_kdc_req_parse (Shishi * handle, FILE * fh,
				 Shishi_asn1 * asreq);
extern int shishi_kdc_rep_parse (Shishi * handle, FILE * fh,
				 Shishi_asn1 * asrep);
extern int shishi_ticket_parse (Shishi * handle, FILE * fh,
				Shishi_asn1 * ticket);
extern int shishi_ticket_read (Shishi * handle, FILE * fh,
			       Shishi_asn1 * ticket);

/* authenticator.c */
extern Shishi_asn1 shishi_authenticator (Shishi * handle);
extern int shishi_authenticator_set_crealm (Shishi * handle,
					    Shishi_asn1 authenticator,
					    const char *crealm);
extern int shishi_authenticator_set_cname (Shishi * handle,
					   Shishi_asn1 authenticator,
					   Shishi_name_type name_type,
					   const char *cname[]);
extern int shishi_authenticator_client_set (Shishi * handle,
					    Shishi_asn1 authenticator,
					    const char *client);
extern int shishi_authenticator_ctime_get (Shishi * handle,
					   Shishi_asn1 authenticator,
					   char *ctime);
extern int shishi_authenticator_ctime_set (Shishi * handle,
					   Shishi_asn1 authenticator,
					   char *ctime);
extern int shishi_authenticator_cusec_get (Shishi * handle,
					   Shishi_asn1 authenticator,
					   int *cusec);
extern int shishi_authenticator_cusec_set (Shishi * handle,
					   Shishi_asn1 authenticator,
					   int cusec);
extern int shishi_authenticator_cname_get (Shishi * handle,
					   Shishi_asn1 authenticator,
					   char *cname, int *cnamelen);
extern int shishi_authenticator_cnamerealm_get (Shishi * handle,
						Shishi_asn1 authenticator,
						char *cnamerealm,
						int *cnamerealmlen);
extern int shishi_authenticator_remove_cksum (Shishi * handle,
					      Shishi_asn1 authenticator);
extern int shishi_authenticator_cksum (Shishi * handle,
				       Shishi_asn1 authenticator,
				       int32_t * cksumtype,
				       char *cksum, size_t * cksumlen);
extern int shishi_authenticator_set_cksum (Shishi * handle,
					   Shishi_asn1 authenticator,
					   int cksumtype,
					   char *cksum, size_t cksumlen);
extern int shishi_authenticator_add_cksum (Shishi * handle,
					   Shishi_asn1 authenticator,
					   Shishi_key * key,
					   int keyusage,
					   char *data, int datalen);
extern int
shishi_authenticator_clear_authorizationdata (Shishi * handle,
					      Shishi_asn1 authenticator);
extern int
shishi_authenticator_add_authorizationdata (Shishi * handle,
					    Shishi_asn1 authenticator,
					    int adtype,
					    char *addata, int addatalen);
extern int
shishi_authenticator_authorizationdata (Shishi * handle,
					Shishi_asn1 authenticator,
					int *adtype,
					char *addata, int *addatalen,
					int nth);
extern int shishi_authenticator_read (Shishi * handle, FILE * fh,
				      Shishi_asn1 * authenticator);
extern int shishi_authenticator_parse (Shishi * handle, FILE * fh,
				       Shishi_asn1 * authenticator);
extern int shishi_authenticator_from_file (Shishi * handle,
					   Shishi_asn1 * authenticator,
					   int filetype, char *filename);
extern int shishi_authenticator_print (Shishi * handle, FILE * fh,
				       Shishi_asn1 authenticator);
extern int shishi_authenticator_to_file (Shishi * handle,
					 Shishi_asn1 authenticator,
					 int filetype, char *filename);
extern int shishi_authenticator_save (Shishi * handle, FILE * fh,
				      Shishi_asn1 authenticator);

/* as.c */
extern Shishi_asn1 shishi_as_req (Shishi_as * as);
extern int shishi_as_req_build (Shishi_as * as);
extern void shishi_as_req_set (Shishi_as * as, Shishi_asn1 asreq);
extern int shishi_as_req_der (Shishi_as * as, char *out, int *outlen);
extern int shishi_as_req_der_set (Shishi_as * as, char *der, size_t derlen);
extern Shishi_asn1 shishi_as_rep (Shishi_as * as);
extern void shishi_as_rep_set (Shishi_as * as, Shishi_asn1 asrep);
extern int shishi_as_rep_build (Shishi_as * as, Shishi_key * key);
extern int shishi_as_rep_der (Shishi_as * as, char *out, int *outlen);
extern int shishi_as_rep_der_set (Shishi_as * as, char *der, size_t derlen);
extern Shishi_asn1 shishi_as_encasreppart (Shishi_as * as);
extern void shishi_as_encasreppart_set (Shishi_as * as,
					Shishi_asn1 encasreppart);
extern Shishi_asn1 shishi_as_krberror (Shishi_as * as);
extern int shishi_as_krberror_der (Shishi_as * as, char *out, int *outlen);
extern void shishi_as_krberror_set (Shishi_as * as, Shishi_asn1 krberror);
extern Shishi_tkt *shishi_as_tkt (Shishi_as * as);
extern void shishi_as_tkt_set (Shishi_as * as, Shishi_tkt * tkt);
extern int shishi_as (Shishi * handle, Shishi_as ** as);
extern int shishi_as_password_cnamerealmsname (Shishi * handle,
					       char *password,
					       Shishi_as ** as,
					       char *cname,
					       char *realm, char *sname);
extern int shishi_as_key_cnamerealmsname (Shishi * handle,
					  Shishi_key * key,
					  Shishi_as ** as,
					  char *cname,
					  char *realm, char *sname);
extern int shishi_as_cnamerealmsname (Shishi * handle,
				      char *password,
				      Shishi_key * key,
				      Shishi_as ** as,
				      char *cname, char *realm, char *sname);
extern int shishi_as_sendrecv (Shishi_as * as);
extern int shishi_as_rep_process (Shishi_as * as,
				  Shishi_key * key, const char *password);

/* tgs.c */
extern int shishi_tgs (Shishi * handle, Shishi_tgs ** tgs);
extern Shishi_tkt *shishi_tgs_tgtkt (Shishi_tgs * tgs);
extern void shishi_tgs_tgtkt_set (Shishi_tgs * tgs, Shishi_tkt * tgtkt);
extern Shishi_ap *shishi_tgs_ap (Shishi_tgs * tgs);
extern Shishi_asn1 shishi_tgs_req (Shishi_tgs * tgs);
extern int shishi_tgs_req_build (Shishi_tgs * tgs);
extern Shishi_asn1 shishi_tgs_rep (Shishi_tgs * tgs);
extern int shishi_tgs_rep_process (Shishi_tgs * tgs);
extern Shishi_asn1 shishi_tgs_krberror (Shishi_tgs * tgs);
extern Shishi_tkt *shishi_tgs_tkt (Shishi_tgs * tgs);
extern void shishi_tgs_tkt_set (Shishi_tgs * tgs, Shishi_tkt * tkt);
extern int shishi_tgs_sendrecv (Shishi_tgs * tgs);
extern int shishi_tgs_set_server (Shishi_tgs * tgs, const char *server);
extern int shishi_tgs_set_realm (Shishi_tgs * tgs, const char *realm);
extern int shishi_tgs_set_realmserver (Shishi_tgs * tgs,
				       const char *realm, const char *server);

/* kdcreq.c */
extern int shishi_kdcreq (Shishi * handle, char *realm,
			  char *service, Shishi_asn1 * req);
extern Shishi_asn1 shishi_asreq (Shishi * handle);
extern Shishi_asn1 shishi_asreq_rsc (Shishi * handle, char *realm,
				     char *server, char *client);
extern Shishi_asn1 shishi_tgsreq (Shishi * handle);
extern Shishi_asn1 shishi_tgsreq_rst (Shishi * handle, char *realm,
				      char *server, Shishi_tkt * tkt);
extern int shishi_kdcreq_save (Shishi * handle, FILE * fh,
			       Shishi_asn1 kdcreq);
extern int shishi_kdcreq_print (Shishi * handle, FILE * fh,
				Shishi_asn1 kdcreq);
extern int shishi_kdcreq_to_file (Shishi * handle, Shishi_asn1 kdcreq,
				  int filetype, char *filename);
extern int shishi_kdcreq_parse (Shishi * handle, FILE * fh,
				Shishi_asn1 * kdcreq);
extern int shishi_kdcreq_read (Shishi * handle, FILE * fh,
			       Shishi_asn1 * kdcreq);
extern int shishi_kdcreq_from_file (Shishi * handle, Shishi_asn1 * kdcreq,
				    int filetype, char *filename);
extern int shishi_asreq_cnamerealm_get (Shishi * handle, Shishi_asn1 kdcreq,
					char *cnamerealm,
					size_t * cnamerealmlen);
extern int shishi_kdcreq_nonce (Shishi * handle, Shishi_asn1 kdcreq,
				uint32_t * nonce);
extern int shishi_kdcreq_cname_get (Shishi * handle,
				    Shishi_asn1 kdcreq,
				    char *cname, size_t * cnamelen);
extern int shishi_kdcreq_set_cname (Shishi * handle, Shishi_asn1 kdcreq,
				    Shishi_name_type name_type,
				    const char *principal);
extern int shishi_kdcreq_sname_get (Shishi * handle, Shishi_asn1 kdcreq,
				    char *sname, size_t * snamelen);
extern int shishi_kdcreq_snamerealm_get (Shishi * handle, Shishi_asn1 kdcreq,
					 char *snamerealm,
					 size_t * snamerealmlen);
extern int shishi_kdcreq_set_sname (Shishi * handle, Shishi_asn1 kdcreq,
				    Shishi_name_type name_type,
				    const char *sname[]);
extern int shishi_kdcreq_realm_get (Shishi * handle, Shishi_asn1 kdcreq,
				    char *realm, int *realmlen);
extern int shishi_kdcreq_set_realm (Shishi * handle, Shishi_asn1 kdcreq,
				    const char *realm);
extern int shishi_kdcreq_set_server (Shishi * handle, Shishi_asn1 req,
				     const char *service);
extern int shishi_kdcreq_set_realmserver (Shishi * handle, Shishi_asn1 req,
					  char *realm, char *service);
extern int shishi_kdcreq_etype (Shishi * handle, Shishi_asn1 kdcreq,
				int32_t * etype, int netype);
extern int shishi_kdcreq_set_etype (Shishi * handle, Shishi_asn1 kdcreq,
				    int32_t * etype, int netype);
extern int shishi_kdcreq_options (Shishi * handle, Shishi_asn1 kdcreq,
				  int *flags);
extern int shishi_kdcreq_renewable_p (Shishi * handle, Shishi_asn1 kdcreq);
extern int shishi_kdcreq_options_set (Shishi * handle, Shishi_asn1 kdcreq,
				      int options);
extern int shishi_kdcreq_options_add (Shishi * handle, Shishi_asn1 kdcreq,
				      int option);
extern int shishi_kdcreq_clear_padata (Shishi * handle, Shishi_asn1 kdcreq);
extern int shishi_kdcreq_add_padata (Shishi * handle,
				     Shishi_asn1 kdcreq,
				     int padatatype, char *data, int datalen);
extern int shishi_kdcreq_add_padata_tgs (Shishi * handle,
					 Shishi_asn1 kdcreq,
					 Shishi_asn1 apreq);
extern int shishi_kdcreq_build (Shishi * handle, Shishi_asn1 kdcreq);

/* enckdcreppart.c */
extern Shishi_asn1 shishi_enckdcreppart (Shishi * handle);
extern Shishi_asn1 shishi_encasreppart (Shishi * handle);
extern int shishi_enckdcreppart_srealmserver_set (Shishi * handle,
						  Shishi_asn1 enckdcreppart,
						  const char *srealm,
						  const char *server);
extern int shishi_enckdcreppart_nonce_set (Shishi * handle,
					   Shishi_asn1 enckdcreppart,
					   uint32_t nonce);
extern int
shishi_enckdcreppart_populate_encticketpart (Shishi * handle,
					     Shishi_asn1 enckdcreppart,
					     Shishi_asn1 encticketpart);
extern int shishi_enckdcreppart_flags_set (Shishi * handle,
					   Shishi_asn1 enckdcreppart,
					   int flags);

/* kdc.c */
extern int shishi_as_derive_salt (Shishi * handle,
				  Shishi_asn1 asreq,
				  Shishi_asn1 asrep, char *salt,
				  size_t * saltlen);
extern int shishi_tgs_process (Shishi * handle, Shishi_asn1 tgsreq,
			       Shishi_asn1 tgsrep,
			       Shishi_asn1 oldenckdcreppart,
			       Shishi_asn1 * enckdcreppart);
extern int shishi_as_process (Shishi * handle, Shishi_asn1 asreq,
			      Shishi_asn1 asrep,
			      const char *string,
			      Shishi_asn1 * enckdcreppart);
extern int shishi_kdc_process (Shishi * handle, Shishi_asn1 kdcreq,
			       Shishi_asn1 kdcrep, Shishi_key * key,
			       int keyusage, Shishi_asn1 * enckdcreppart);
extern int shishi_kdcreq_sendrecv (Shishi * handle, Shishi_asn1 kdcreq,
				   Shishi_asn1 * kdcrep);
extern int shishi_kdc_copy_crealm (Shishi * handle, Shishi_asn1 kdcrep,
				   Shishi_asn1 encticketpart);
extern int shishi_as_check_crealm (Shishi * handle, Shishi_asn1 asreq,
				   Shishi_asn1 asrep);
extern int shishi_kdc_copy_cname (Shishi * handle, Shishi_asn1 kdcrep,
				  Shishi_asn1 encticketpart);
extern int shishi_as_check_cname (Shishi * handle, Shishi_asn1 asreq,
				  Shishi_asn1 asrep);
extern int shishi_kdc_copy_nonce (Shishi * handle, Shishi_asn1 kdcreq,
				  Shishi_asn1 enckdcreppart);
extern int shishi_kdc_check_nonce (Shishi * handle, Shishi_asn1 kdcreq,
				   Shishi_asn1 enckdcreppart);

/* kdcrep.c */
extern Shishi_asn1 shishi_asrep (Shishi * handle);
extern Shishi_asn1 shishi_tgsrep (Shishi * handle);
extern int shishi_kdcrep_save (Shishi * handle, FILE * fh,
			       Shishi_asn1 kdcrep);
extern int shishi_kdcrep_print (Shishi * handle, FILE * fh,
				Shishi_asn1 kdcrep);
extern int shishi_kdcrep_to_file (Shishi * handle, Shishi_asn1 kdcrep,
				  int filetype, char *filename);
extern int shishi_kdcrep_parse (Shishi * handle, FILE * fh,
				Shishi_asn1 * kdcrep);
extern int shishi_kdcrep_read (Shishi * handle, FILE * fh,
			       Shishi_asn1 * kdcrep);
extern int shishi_kdcrep_from_file (Shishi * handle, Shishi_asn1 * kdcrep,
				    int filetype, char *filename);
extern int shishi_kdcrep_clear_padata (Shishi * handle, Shishi_asn1 kdcrep);
extern int shishi_kdcrep_get_enc_part_etype (Shishi * handle,
					     Shishi_asn1 kdcrep,
					     int32_t * etype);
extern int shishi_kdcrep_add_enc_part (Shishi * handle,
				       Shishi_asn1 kdcrep,
				       Shishi_key * key,
				       int keyusage,
				       Shishi_asn1 enckdcreppart);
extern int shishi_kdcrep_get_ticket (Shishi * handle,
				     Shishi_asn1 kdcrep,
				     Shishi_asn1 * ticket);
extern int shishi_kdcrep_set_ticket (Shishi * handle, Shishi_asn1 kdcrep,
				     Shishi_asn1 ticket);
extern int shishi_kdcrep_crealm_set (Shishi * handle,
				     Shishi_asn1 kdcrep, const char *crealm);
extern int shishi_kdcrep_cname_set (Shishi * handle,
				    Shishi_asn1 kdcrep,
				    Shishi_name_type name_type,
				    const char *cname[]);
extern int shishi_kdcrep_client_set (Shishi * handle, Shishi_asn1 kdcrep,
				     const char *client);
extern int shishi_kdcrep_crealmserver_set (Shishi * handle,
					   Shishi_asn1 kdcrep,
					   const char *crealm,
					   const char *client);
extern int shishi_kdcrep_set_enc_part (Shishi * handle, Shishi_asn1 kdcrep,
				       int etype, int kvno, char *buf,
				       int buflen);

/* krberror.c */
extern Shishi_asn1 shishi_krberror (Shishi * handle);
extern int shishi_krberror_print (Shishi * handle, FILE * fh,
				  Shishi_asn1 krberror);
extern int shishi_krberror_save (Shishi * handle, FILE * fh,
				 Shishi_asn1 krberror);
extern int shishi_krberror_to_file (Shishi * handle, Shishi_asn1 krberror,
				    int filetype, char *filename);
extern int shishi_krberror_parse (Shishi * handle, FILE * fh,
				  Shishi_asn1 * krberror);
extern int shishi_krberror_read (Shishi * handle, FILE * fh,
				 Shishi_asn1 * krberror);
extern int shishi_krberror_from_file (Shishi * handle, Shishi_asn1 * krberror,
				      int filetype, char *filename);
extern const char *shishi_krberror_errorcode_message (Shishi * handle,
						      int errorcode);
extern const char *shishi_krberror_message (Shishi * handle,
					    Shishi_asn1 krberror);
extern int shishi_krberror_etext (Shishi * handle, Shishi_asn1 krberror,
				  char *etext, size_t * etextlen);
extern int shishi_krberror_errorcode (Shishi * handle,
				      Shishi_asn1 krberror, int *errorcode);
extern int shishi_krberror_errorcode_fast (Shishi * handle,
					   Shishi_asn1 krberror);
extern int shishi_krberror_pretty_print (Shishi * handle,
					 FILE * fh, Shishi_asn1 krberror);

/* base64.c */
extern void shishi_to_base64 (unsigned char *out, const unsigned char *in,
			      int len, int olen);
extern int shishi_from_base64 (unsigned char *out, const unsigned char *in);

/* vasprintf.c */
extern int shishi_asprintf (char **result, char *format, ...);

/* gztime.c */
extern const char *shishi_generalize_time (Shishi * handle, time_t t);
extern time_t shishi_generalize_ctime (Shishi * handle, const char *t);

/* crypto.c */
extern int shishi_cipher_supported_p (int type);
extern const char *shishi_cipher_name (int type);
extern int shishi_cipher_blocksize (int type);
extern int shishi_cipher_minpadsize (int type);
extern int shishi_cipher_confoundersize (int type);
extern size_t shishi_cipher_keylen (int type);
extern size_t shishi_cipher_randomlen (int type);
extern int shishi_cipher_defaultcksumtype (int32_t type);
extern int shishi_cipher_parse (const char *cipher);
extern int shishi_checksum_supported_p (int32_t type);
extern const char *shishi_checksum_name (int32_t type);
extern size_t shishi_checksum_cksumlen (int32_t type);
extern int shishi_checksum_parse (const char *checksum);
extern int shishi_string_to_key (Shishi * handle,
				 int32_t keytype,
				 const char *password, size_t passwordlen,
				 const char *salt, size_t saltlen,
				 const char *parameter, Shishi_key * outkey);
extern int shishi_random_to_key (Shishi * handle,
				 int32_t keytype,
				 char *random, size_t randomlen,
				 Shishi_key * outkey);
extern int shishi_encrypt (Shishi * handle,
			   Shishi_key * key,
			   int keyusage,
			   char *in, size_t inlen,
			   char **out, size_t * outlen);
extern int shishi_encrypt_iv (Shishi * handle,
			      Shishi_key * key,
			      int keyusage,
			      char *iv, size_t ivlen,
			      char *in, size_t inlen,
			      char **out, size_t * outlen);
extern int shishi_encrypt_iv_etype (Shishi * handle,
				    Shishi_key * key,
				    int keyusage,
				    int32_t etype,
				    char *iv, size_t ivlen,
				    char *in, size_t inlen,
				    char **out, size_t * outlen);
extern int shishi_decrypt (Shishi * handle,
			   Shishi_key * key,
			   int keyusage,
			   char *in, size_t inlen,
			   char **out, size_t * outlen);
extern int shishi_decrypt_iv (Shishi * handle,
			      Shishi_key * key,
			      int keyusage,
			      char *iv, size_t ivlen,
			      char *in, size_t inlen,
			      char **out, size_t * outlen);
extern int shishi_decrypt_iv_etype (Shishi * handle,
				    Shishi_key * key,
				    int keyusage,
				    int32_t etype,
				    char *iv, size_t ivlen,
				    char *in, size_t inlen,
				    char **out, size_t * outlen);
extern int shishi_checksum (Shishi * handle,
			    Shishi_key * key,
			    int keyusage,
			    int32_t cksumtype,
			    char *in, size_t inlen,
			    char **out, size_t * outlen);
extern int shishi_randomize (Shishi * handle, char *data, size_t datalen);
extern int shishi_dk (Shishi * handle,
		      Shishi_key * key,
		      char *constant, int constantlen,
		      Shishi_key * derivedkey);
extern int shishi_dr (Shishi * handle,
		      Shishi_key * key,
		      char *constant, size_t constantlen,
		      char *derivedrandom, size_t derivedrandomlen);
extern int shishi_n_fold (Shishi * handle, char *in, size_t inlen,
			  char *out, size_t outlen);
extern int shishi_pbkdf2_sha1 (const char *P, size_t Plen,
			       const char *S, size_t Slen,
			       unsigned int c,
			       unsigned int dkLen,
			       char *DK);

/* version.c */
extern const char *shishi_check_version (const char *req_version);

/* password.c */
extern int shishi_read_password (FILE * fh, char *s, int size);
extern int
shishi_prompt_password_raw (FILE * in, char *s, int size,
			    FILE * out, char *format, ...);
extern int
shishi_prompt_password (Shishi * handle,
			FILE * in, char *s, int size,
			FILE * out, char *format, ...);

/* asn1.c */
extern int shishi_a2d_field (Shishi * handle,
			     Shishi_asn1 node,
			     const char *field, char *der, int *len);
extern int shishi_a2d (Shishi * handle, Shishi_asn1 node,
		       char *der, int *len);
extern int shishi_a2d_new_field (Shishi * handle, Shishi_asn1 node,
				 const char *field, char **der, int *len);
extern int shishi_new_a2d (Shishi * handle, Shishi_asn1 node,
			   char **der, int *len);
extern int shishi_asn1_done (Shishi * handle, Shishi_asn1 node);
extern int shishi_asn1_read (Shishi * handle, Shishi_asn1 node,
			     const char *field, char *data, size_t * datalen);
extern int shishi_asn1_read2 (Shishi * handle, Shishi_asn1 node,
			      const char *field,
			      char **data, size_t * datalen);
extern int shishi_asn1_read_integer (Shishi * handle, Shishi_asn1 node,
				     const char *field, int *i);
extern int shishi_asn1_read_int32 (Shishi * handle, Shishi_asn1 node,
				   const char *field, int32_t * i);
extern int shishi_asn1_read_uint32 (Shishi * handle, Shishi_asn1 node,
				    const char *field, uint32_t * i);
extern int shishi_asn1_read_bitstring (Shishi * handle, Shishi_asn1 node,
				       const char *field, int *flags);
extern int shishi_asn1_read_optional (Shishi * handle,
				      Shishi_asn1 node, const char *field,
				      char *data, size_t * datalen);
extern int shishi_asn1_read_empty_p (Shishi * handle,
				     Shishi_asn1 node, const char *field);
extern int shishi_asn1_write (Shishi * handle, Shishi_asn1 node,
			      const char *field,
			      const char *data, size_t datalen);
extern int shishi_asn1_write_integer (Shishi * handle, Shishi_asn1 node,
				      const char *field, int n);
extern int shishi_asn1_write_int32 (Shishi * handle, Shishi_asn1 node,
				    const char *field, int32_t n);
extern int shishi_asn1_write_uint32 (Shishi * handle, Shishi_asn1 node,
				     const char *field, uint32_t n);
extern int shishi_asn1_write_bitstring (Shishi * handle, Shishi_asn1 node,
					const char *field, int flags);
extern int shishi_asn1_number_of_elements (Shishi * handle,
					   Shishi_asn1 node,
					   const char *field, int *n);
extern Shishi_asn1 shishi_asn1_asreq (Shishi * handle);
extern Shishi_asn1 shishi_asn1_asrep (Shishi * handle);
extern Shishi_asn1 shishi_asn1_tgsreq (Shishi * handle);
extern Shishi_asn1 shishi_asn1_tgsrep (Shishi * handle);
extern Shishi_asn1 shishi_asn1_apreq (Shishi * handle);
extern Shishi_asn1 shishi_asn1_aprep (Shishi * handle);
extern Shishi_asn1 shishi_asn1_ticket (Shishi * handle);
extern Shishi_asn1 shishi_asn1_encapreppart (Shishi * handle);
extern Shishi_asn1 shishi_asn1_encticketpart (Shishi * handle);
extern Shishi_asn1 shishi_asn1_authenticator (Shishi * handle);
extern Shishi_asn1 shishi_asn1_enckdcreppart (Shishi * handle);
extern Shishi_asn1 shishi_asn1_encasreppart (Shishi * handle);
extern Shishi_asn1 shishi_asn1_krberror (Shishi * handle);
extern Shishi_asn1 shishi_asn1_krbsafe (Shishi * handle);
extern Shishi_asn1 shishi_der2asn1 (Shishi * handle,
				    const char *fieldname,
				    const char *nodename,
				    const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_ticket (Shishi * handle,
					   const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_encticketpart (Shishi * handle,
						  const char *der,
						  size_t derlen);
extern Shishi_asn1 shishi_der2asn1_asreq (Shishi * handle,
					  const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_tgsreq (Shishi * handle,
					   const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_asrep (Shishi * handle,
					  const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_tgsrep (Shishi * handle,
					   const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_kdcrep (Shishi * handle,
					   const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_kdcreq (Shishi * handle,
					   const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_encasreppart (Shishi * handle,
						 const char *der,
						 size_t derlen);
extern Shishi_asn1 shishi_der2asn1_enctgsreppart (Shishi * handle,
						  const char *der,
						  size_t derlen);
extern Shishi_asn1 shishi_der2asn1_enckdcreppart (Shishi * handle,
						  const char *der,
						  size_t derlen);
extern Shishi_asn1 shishi_der2asn1_authenticator (Shishi * handle,
						  const char *der,
						  size_t derlen);
extern Shishi_asn1 shishi_der2asn1_krberror (Shishi * handle,
					     const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_krbsafe (Shishi * handle,
					    const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_apreq (Shishi * handle,
					  const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_aprep (Shishi * handle,
					  const char *der, size_t derlen);
extern Shishi_asn1 shishi_der2asn1_encapreppart (Shishi * handle,
						 const char *der,
						 size_t derlen);

/* ap.c */
extern const char *shishi_ap_option2string (int option);
extern int shishi_ap_string2option (const char *str);

extern int shishi_ap (Shishi * handle, Shishi_ap ** ap);
extern int shishi_ap_set_tktoptions (Shishi_ap * ap,
				     Shishi_tkt * tkt, int options);
extern int shishi_ap_tktoptions (Shishi * handle,
				 Shishi_ap ** ap,
				 Shishi_tkt * tkt, int options);
extern int shishi_ap_set_tktoptionsdata (Shishi_ap * ap,
					 Shishi_tkt * tkt,
					 int options, char *data, int len);
extern int shishi_ap_tktoptionsdata (Shishi * handle,
				     Shishi_ap ** ap,
				     Shishi_tkt * tkt,
				     int options, char *data, int len);
extern int shishi_ap_set_tktoptionsasn1usage (Shishi_ap * ap,
					      Shishi_tkt * tkt,
					      int options,
					      Shishi_asn1 node,
					      char *field,
					      int authenticatorcksumkeyusage,
					      int authenticatorkeyusage);
extern int shishi_ap_tktoptionsasn1usage (Shishi * handle,
					  Shishi_ap ** ap,
					  Shishi_tkt * tkt,
					  int options,
					  Shishi_asn1 node,
					  char *field,
					  int authenticatorcksumkeyusage,
					  int authenticatorkeyusage);

extern Shishi_tkt *shishi_ap_tkt (Shishi_ap * ap);
extern void shishi_ap_tkt_set (Shishi_ap * ap, Shishi_tkt * tkt);

extern int shishi_ap_authenticator_cksumdata (Shishi_ap * ap,
					      char *out, int *len);
extern void
shishi_ap_authenticator_cksumdata_set (Shishi_ap * ap,
				       char *authenticatorcksumdata,
				       int authenticatorcksumdatalen);

extern Shishi_asn1 shishi_ap_authenticator (Shishi_ap * ap);
extern void shishi_ap_authenticator_set (Shishi_ap * ap,
					 Shishi_asn1 authenticator);

extern Shishi_asn1 shishi_ap_req (Shishi_ap * ap);
extern void shishi_ap_req_set (Shishi_ap * ap, Shishi_asn1 apreq);
extern int shishi_ap_req_der (Shishi_ap * ap, char *out, int *outlen);
extern int shishi_ap_req_der_set (Shishi_ap * ap, char *der, size_t derlen);
extern int shishi_ap_req_der_new (Shishi_ap * ap, char **out, int *outlen);
extern int shishi_ap_req_build (Shishi_ap * ap);
extern int shishi_ap_req_asn1 (Shishi_ap * ap, Shishi_asn1 * apreq);
extern int shishi_ap_req_process (Shishi_ap * ap, Shishi_key * key);
extern int shishi_ap_req_build (Shishi_ap * ap);

extern Shishi_asn1 shishi_ap_rep (Shishi_ap * ap);
extern void shishi_ap_rep_set (Shishi_ap * ap, Shishi_asn1 aprep);
extern int shishi_ap_rep_der (Shishi_ap * ap, char *out, size_t * outlen);
extern int shishi_ap_rep_der_set (Shishi_ap * ap, char *der, size_t derlen);
extern int shishi_ap_rep_verify (Shishi_ap * ap);
extern int shishi_ap_rep_verify_der (Shishi_ap * ap, char *der,
				     size_t derlen);
extern int shishi_ap_rep_verify_asn1 (Shishi_ap * ap, Shishi_asn1 aprep);
extern int shishi_ap_rep_asn1 (Shishi_ap * ap, Shishi_asn1 * aprep);
extern int shishi_ap_rep_build (Shishi_ap * ap);

extern Shishi_asn1 shishi_ap_encapreppart (Shishi_ap * ap);
extern void shishi_ap_encapreppart_set (Shishi_ap * ap,
					Shishi_asn1 encapreppart);

/* key.c */
extern const char *shishi_key_principal (Shishi_key * key);
extern void shishi_key_principal_set (Shishi_key * key,
				      const char *principal);
extern const char *shishi_key_realm (Shishi_key * key);
extern void shishi_key_realm_set (Shishi_key * key, const char *realm);
extern int shishi_key_type (Shishi_key * key);
extern void shishi_key_type_set (Shishi_key * key, int32_t type);
extern char *shishi_key_value (Shishi_key * key);
extern void shishi_key_value_set (Shishi_key * key, const char *value);
extern int shishi_key_value_set_random (Shishi_key * key,
					char *random, size_t randomlen);
extern const char *shishi_key_name (Shishi_key * key);
extern size_t shishi_key_length (Shishi_key * key);
extern int shishi_key_version (Shishi_key * key);
extern void shishi_key_version_set (Shishi_key * key, int version);
extern int shishi_key (Shishi * handle, Shishi_key ** key);
extern void shishi_key_done (Shishi_key ** key);
extern void shishi_key_copy (Shishi_key * dstkey, Shishi_key * srckey);
extern int shishi_key_print (Shishi * handle, FILE * fh, Shishi_key * key);
extern int shishi_key_to_file (Shishi * handle,
			       const char *filename, Shishi_key * key);
extern int shishi_key_parse (Shishi * handle, FILE * fh, Shishi_key ** key);
extern int shishi_key_random (Shishi * handle,
			      int32_t type, Shishi_key ** key);
extern int shishi_key_from_value (Shishi * handle,
				  int32_t type,
				  char *value, Shishi_key ** key);
extern int shishi_key_from_base64 (Shishi * handle,
				   int32_t type,
				   char *value, Shishi_key ** key);
extern int shishi_key_from_random (Shishi * handle,
				   int32_t type,
				   char *random, size_t randomlen,
				   Shishi_key ** outkey);
extern int shishi_key_from_string (Shishi * handle,
				   int32_t type,
				   const char *password, size_t passwordlen,
				   const char *salt, size_t saltlen,
				   const char *parameter,
				   Shishi_key ** outkey);

/* keys.c */
extern Shishi_key *
shishi_keys_for_serverrealm_in_file (Shishi * handle,
				     const char *filename,
				     const char *server,
				     const char *realm);
extern Shishi_key *
shishi_keys_for_server_in_file (Shishi * handle,
				const char *filename,
				const char *server);
extern Shishi_key *
shishi_keys_for_localservicerealm_in_file (Shishi * handle,
					   const char *filename,
					   const char *service,
					   const char *realm);

/* hostkeys.c */
extern const char *shishi_hostkeys_default_file (Shishi * handle);
extern void shishi_hostkeys_default_file_set (Shishi * handle,
					      const char *hostkeysfile);
extern Shishi_key *shishi_hostkeys_for_server (Shishi * handle,
					       const char *server);
extern Shishi_key *shishi_hostkeys_for_serverrealm (Shishi * handle,
						    const char *server,
						    const char *realm);
extern Shishi_key *shishi_hostkeys_for_localservicerealm (Shishi * handle,
							  const char *service,
							  const char *realm);
extern Shishi_key *shishi_hostkeys_for_localservice (Shishi * handle,
						     const char *service);

/* encapreppart.c */
extern Shishi_asn1 shishi_encapreppart (Shishi * handle);
extern int shishi_encapreppart_time_copy (Shishi * handle,
					  Shishi_asn1 encapreppart,
					  Shishi_asn1 authenticator);
extern int shishi_encapreppart_ctime_get (Shishi * handle,
					  Shishi_asn1 encapreppart,
					  char *ctime);
extern int shishi_encapreppart_ctime_set (Shishi * handle,
					  Shishi_asn1 encapreppart,
					  char *ctime);
extern int shishi_encapreppart_cusec_get (Shishi * handle,
					  Shishi_asn1 encapreppart,
					  int *cusec);
extern int shishi_encapreppart_cusec_set (Shishi * handle,
					  Shishi_asn1 encapreppart,
					  int cusec);
extern int shishi_encapreppart_print (Shishi * handle, FILE * fh,
				      Shishi_asn1 encapreppart);
extern int shishi_encapreppart_save (Shishi * handle, FILE * fh,
				     Shishi_asn1 encapreppart);
extern int shishi_encapreppart_to_file (Shishi * handle,
					Shishi_asn1 encapreppart,
					int filetype, char *filename);
extern int shishi_encapreppart_read (Shishi * handle, FILE * fh,
				     Shishi_asn1 * encapreppart);
extern int shishi_encapreppart_parse (Shishi * handle, FILE * fh,
				      Shishi_asn1 * encapreppart);
extern int shishi_encapreppart_from_file (Shishi * handle,
					  Shishi_asn1 * encapreppart,
					  int filetype, char *filename);
extern int shishi_encapreppart_get_key (Shishi * handle,
					Shishi_asn1 encapreppart,
					int32_t * keytype,
					char *keyvalue,
					size_t * keyvalue_len);
extern int shishi_encapreppart_seqnumber_get (Shishi * handle,
					      Shishi_asn1 encapreppart,
					      uint32_t *seqnumber);

/* apreq.c */
extern Shishi_asn1 shishi_apreq (Shishi * handle);
extern int shishi_apreq_parse (Shishi * handle, FILE * fh,
			       Shishi_asn1 * apreq);
extern int shishi_apreq_from_file (Shishi * handle, Shishi_asn1 * apreq,
				   int filetype, char *filename);
extern int shishi_apreq_print (Shishi * handle, FILE * fh, Shishi_asn1 apreq);
extern int shishi_apreq_to_file (Shishi * handle, Shishi_asn1 apreq,
				 int filetype, char *filename);
extern int shishi_apreq_read (Shishi * handle, FILE * fh,
			      Shishi_asn1 * apreq);
extern int shishi_apreq_save (Shishi * handle, FILE * fh, Shishi_asn1 apreq);
extern int shishi_apreq_options (Shishi * handle, Shishi_asn1 apreq,
				 int *flags);
extern int shishi_apreq_options_set (Shishi * handle, Shishi_asn1 apreq,
				     int options);
extern int shishi_apreq_set_ticket (Shishi * handle, Shishi_asn1 apreq,
				    Shishi_asn1 ticket);
extern int shishi_apreq_set_authenticator (Shishi * handle, Shishi_asn1 apreq,
					   int32_t etype,
					   char *buf, int buflen);
extern int shishi_apreq_add_authenticator (Shishi * handle, Shishi_asn1 apreq,
					   Shishi_key * key, int keyusage,
					   Shishi_asn1 authenticator);
extern int shishi_apreq_options (Shishi * handle, Shishi_asn1 apreq,
				 int *flags);
extern int shishi_apreq_use_session_key_p (Shishi * handle,
					   Shishi_asn1 apreq);
extern int shishi_apreq_mutual_required_p (Shishi * handle,
					   Shishi_asn1 apreq);
extern int shishi_apreq_options_set (Shishi * handle, Shishi_asn1 apreq,
				     int options);
extern int shishi_apreq_options_add (Shishi * handle, Shishi_asn1 apreq,
				     int option);
extern int shishi_apreq_get_ticket (Shishi * handle, Shishi_asn1 apreq,
				    Shishi_asn1 * ticket);
extern int shishi_apreq_get_authenticator_etype (Shishi * handle,
						 Shishi_asn1 apreq,
						 int32_t * etype);
extern int shishi_apreq_decrypt (Shishi * handle, Shishi_asn1 apreq,
				 Shishi_key * key, int keyusage,
				 Shishi_asn1 * authenticator);

/* kdcrep.c */
extern int shishi_kdcrep_decrypt (Shishi * handle,
				  Shishi_asn1 kdcrep,
				  Shishi_key * key,
				  int keyusage, Shishi_asn1 * enckdcreppart);
/* enckdcreppart.c */
extern int shishi_enckdcreppart_get_key (Shishi * handle,
					 Shishi_asn1 enckdcreppart,
					 Shishi_key ** key);
extern int shishi_enckdcreppart_srealm_set (Shishi * handle,
					    Shishi_asn1 enckdcreppart,
					    const char *srealm);
extern int shishi_enckdcreppart_sname_set (Shishi * handle,
					   Shishi_asn1 enckdcreppart,
					   Shishi_name_type name_type,
					   char *sname[]);
extern int shishi_enckdcreppart_server_set (Shishi * handle,
					    Shishi_asn1 enckdcreppart,
					    const char *server);

/* aprep.c */
extern Shishi_asn1 shishi_aprep (Shishi * handle);
extern int shishi_aprep_print (Shishi * handle, FILE * fh, Shishi_asn1 aprep);
extern int shishi_aprep_save (Shishi * handle, FILE * fh, Shishi_asn1 aprep);
extern int shishi_aprep_to_file (Shishi * handle, Shishi_asn1 aprep,
				 int filetype, char *filename);
extern int shishi_aprep_read (Shishi * handle, FILE * fh,
			      Shishi_asn1 * aprep);
extern int shishi_aprep_parse (Shishi * handle, FILE * fh,
			       Shishi_asn1 * aprep);
extern int shishi_aprep_from_file (Shishi * handle, Shishi_asn1 * aprep,
				   int filetype, char *filename);
extern int shishi_aprep_decrypt (Shishi * handle, Shishi_asn1 aprep,
				 Shishi_key * key, int keyusage,
				 Shishi_asn1 * encapreppart);
extern int shishi_aprep_verify (Shishi * handle, Shishi_asn1 authenticator,
				Shishi_asn1 encapreppart);
extern int shishi_aprep_enc_part_set (Shishi * handle, Shishi_asn1 aprep,
				      int etype, const char *buf, int buflen);
extern int shishi_aprep_enc_part_add (Shishi * handle, Shishi_asn1 aprep,
				      Shishi_asn1 encticketpart,
				      Shishi_asn1 encapreppart);
extern int shishi_aprep_enc_part_make (Shishi * handle, Shishi_asn1 aprep,
				       Shishi_asn1 authenticator,
				       Shishi_asn1 encticketpart);
extern int shishi_aprep_get_enc_part_etype (Shishi * handle,
					    Shishi_asn1 aprep,
					    int32_t * etype);

/* netio.c */
extern int shishi_kdc_sendrecv (Shishi * handle, char *realm,
				const char *indata, size_t inlen,
				char **outdata, size_t * outlen);

/* encticketpart.c */
extern Shishi_asn1 shishi_encticketpart (Shishi * handle);
extern int shishi_encticketpart_key_set (Shishi * handle,
					 Shishi_asn1 encticketpart,
					 Shishi_key * key);
extern int shishi_encticketpart_get_enc_part_etype (Shishi * handle,
						    Shishi_asn1 encticketpart,
						    int32_t * etype);
extern int shishi_encticketpart_get_key (Shishi * handle,
					 Shishi_asn1 encticketpart,
					 Shishi_key ** key);
extern int shishi_encticketpart_crealm_set (Shishi * handle,
					    Shishi_asn1 encticketpart,
					    const char *realm);
extern int shishi_encticketpart_cname_get (Shishi * handle,
					   Shishi_asn1 encticketpart,
					   char *cname, int *cnamelen);
extern int shishi_encticketpart_cname_set (Shishi * handle,
					   Shishi_asn1 encticketpart,
					   Shishi_name_type name_type,
					   const char *principal);
extern int shishi_encticketpart_cnamerealm_get (Shishi * handle,
						Shishi_asn1 encticketpart,
						char *cnamerealm,
						int *cnamerealmlen);
extern int shishi_encticketpart_print (Shishi * handle, FILE * fh,
				       Shishi_asn1 encticketpart);
extern int shishi_encticketpart_flags_set (Shishi * handle,
					   Shishi_asn1 encticketpart,
					   int flags);
extern int shishi_encticketpart_transited_set (Shishi * handle,
					       Shishi_asn1 encticketpart,
					       int trtype,
					       char *trdata,
					       size_t trdatalen);
extern int shishi_encticketpart_authtime_set (Shishi * handle,
					      Shishi_asn1 encticketpart,
					      const char *authtime);
extern int shishi_encticketpart_endtime_set (Shishi * handle,
					     Shishi_asn1 encticketpart,
					     const char *endtime);
extern int shishi_encticketpart_authtime (Shishi * handle,
					  Shishi_asn1 encticketpart,
					  char *authtime, int *authtimelen);
extern time_t shishi_encticketpart_authctime (Shishi * handle,
					      Shishi_asn1 encticketpart);

/* safe.c */
extern int shishi_safe (Shishi * handle, Shishi_safe ** safe);
extern Shishi_key *shishi_safe_key (Shishi_safe * safe);
extern void shishi_safe_key_set (Shishi_safe * safe, Shishi_key * key);
extern Shishi_asn1 shishi_safe_safe (Shishi_safe * safe);
extern void shishi_safe_safe_set (Shishi_safe * safe, Shishi_asn1 asn1safe);
extern int shishi_safe_safe_der (Shishi_safe * safe, char *out, int *outlen);
extern int shishi_safe_safe_der_set (Shishi_safe * safe,
				     char *der, size_t derlen);
extern int shishi_safe_print (Shishi * handle, FILE * fh, Shishi_asn1 safe);
extern int shishi_safe_save (Shishi * handle, FILE * fh, Shishi_asn1 safe);
extern int shishi_safe_to_file (Shishi * handle, Shishi_asn1 safe,
				int filetype, char *filename);
extern int shishi_safe_parse (Shishi * handle, FILE * fh, Shishi_asn1 * safe);
extern int shishi_safe_read (Shishi * handle, FILE * fh, Shishi_asn1 * safe);
extern int shishi_safe_from_file (Shishi * handle, Shishi_asn1 * safe,
				  int filetype, char *filename);
extern int shishi_safe_cksum (Shishi * handle,
			      Shishi_asn1 safe,
			      int32_t * cksumtype,
			      char *cksum, size_t * cksumlen);
extern int shishi_safe_set_cksum (Shishi * handle,
				  Shishi_asn1 safe,
				  int32_t cksumtype,
				  char *cksum, size_t cksumlen);
extern int shishi_safe_user_data (Shishi * handle,
				  Shishi_asn1 safe,
				  char *userdata, size_t * userdatalen);
extern int shishi_safe_set_user_data (Shishi * handle,
				      Shishi_asn1 safe,
				      char *userdata, size_t userdatalen);
extern int shishi_safe_build (Shishi_safe * safe, Shishi_key * key);
extern int shishi_safe_verify (Shishi_safe * safe, Shishi_key * key);

#endif
