/* nettle.c   shishi crypto wrappers around nettle.
 * Copyright (C) 2002, 2003, 2004  Simon Josefsson
 *
 * This file is part of Shishi.
 *
 * Shishi is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Shishi is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Shishi; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/* Note: This file is only built if Shishi uses Nettle. */

#include "internal.h"
#include "crypto.h"

#include "hmac.h"
#include "arcfour.h"
#include "des.h"
#include "aes.h"
#include "cbc.h"
#include "cbc-cts.h"
#include "cbc-mac.h"

int
_shishi_crypto_init (Shishi * handle)
{
  return SHISHI_OK;
}

/**
 * shishi_randomize:
 * @handle: shishi handle as allocated by shishi_init().
 * @strong: 0 iff operation should not block, non-0 for very strong randomness.
 * @data: output array to be filled with random data.
 * @datalen: size of output array.
 *
 * Store cryptographically random data of given size in the provided
 * buffer.
 *
 * Return value: Returns %SHISHI_OK iff successful.
 **/
int
shishi_randomize (Shishi * handle, int strong, void *data, size_t datalen)
{
  int fd;
  char *device;
  size_t len = 0;
  int rc;

  if (strong)
    device = "/dev/random";
  else
    device = "/dev/urandom";

  fd = open (device, O_RDONLY);
  if (fd < 0)
    {
      shishi_error_printf (handle, "Could not open %s: %s", device,
			   strerror (errno));
      return SHISHI_FILE_ERROR;
    }

  do
    {
      ssize_t tmp;

      tmp = read (fd, data, datalen);

      if (tmp < 0)
	{
	  shishi_error_printf (handle, "Could not read from %s: %s",
			       device, strerror (errno));
	  return SHISHI_FILE_ERROR;
	}

      len += tmp;

      if (len < datalen)
	shishi_error_printf (handle, "Short read from %s: %d < %d",
			     device, len, datalen);
    }
  while (len < datalen);

  rc = close (fd);
  if (rc < 0)
    shishi_warn (handle, "Could not close %s: %s", device, strerror (errno));

  return SHISHI_OK;
}

/* Table of CRCs of all 8-bit messages.  Generated by running code
   from RFC 1952 modified to print out the table. */
static uint32_t crc32_table[256] = {
  0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f,
  0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
  0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2,
  0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
  0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
  0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
  0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c,
  0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
  0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423,
  0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
  0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106,
  0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
  0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d,
  0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
  0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
  0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
  0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7,
  0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
  0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa,
  0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
  0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81,
  0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
  0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84,
  0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
  0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
  0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
  0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e,
  0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
  0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55,
  0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
  0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28,
  0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
  0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f,
  0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
  0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
  0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
  0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69,
  0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
  0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc,
  0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
  0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693,
  0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
  0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

/*
 * The following function was extracted from RFC 1952 by Simon
 * Josefsson, and modified to be compatible with the modified CRC-32
 * used by RFC 1510, for the Shishi project.  Original copyright and
 * notice from the document follows:
 *
 *    Copyright (c) 1996 L. Peter Deutsch
 *
 *    Permission is granted to copy and distribute this document for
 *    any purpose and without charge, including translations into
 *    other languages and incorporation into compilations, provided
 *    that the copyright notice and this notice are preserved, and
 *    that any substantive changes or deletions from the original are
 *    clearly marked.
 *
 * The copyright on RFCs, and consequently the function below, are
 * supposedly also retroactively claimed by the Internet Society
 * (according to rfc-editor@rfc-editor.org), with the following
 * copyright notice:
 *
 *    Copyright (C) The Internet Society.  All Rights Reserved.
 *
 *    This document and translations of it may be copied and furnished
 *    to others, and derivative works that comment on or otherwise
 *    explain it or assist in its implementation may be prepared,
 *    copied, published and distributed, in whole or in part, without
 *    restriction of any kind, provided that the above copyright
 *    notice and this paragraph are included on all such copies and
 *    derivative works.  However, this document itself may not be
 *    modified in any way, such as by removing the copyright notice or
 *    references to the Internet Society or other Internet
 *    organizations, except as needed for the purpose of developing
 *    Internet standards in which case the procedures for copyrights
 *    defined in the Internet Standards process must be followed, or
 *    as required to translate it into languages other than English.
 *
 *    The limited permissions granted above are perpetual and will not be
 *    revoked by the Internet Society or its successors or assigns.
 *
 *    This document and the information contained herein is provided
 *    on an "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET
 *    ENGINEERING TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR
 *    IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY THAT THE USE
 *    OF THE INFORMATION HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY
 *    IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *    PARTICULAR PURPOSE.
 *
 */
static uint32_t
update_crc32 (uint32_t crc, const char *buf, size_t len)
{
  size_t n;

  for (n = 0; n < len; n++)
    crc = crc32_table[(crc ^ buf[n]) & 0xff] ^ (crc >> 8);

  return crc;
}

/**
 * shishi_crc:
 * @handle: shishi handle as allocated by shishi_init().
 * @in: input character array of data to checksum.
 * @inlen: length of input character array of data to checksum.
 * @out: newly allocated character array with checksum of data.
 *
 * Compute checksum of data using CRC32 modified according to RFC
 * 1510.  The @out buffer must be deallocated by the caller.
 *
 * The modifications compared to standard CRC32 is that no initial and
 * final XOR is performed, and that the output is returned in
 * LSB-first order.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_crc (Shishi * handle, const char *in, size_t inlen, char *out[4])
{
  uint32_t crc = update_crc32 (0, in, inlen);

  *out = xmalloc (4);
  (*out)[0] = crc & 0xFF;
  (*out)[1] = (crc >> 8) & 0xFF;
  (*out)[2] = (crc >> 16) & 0xFF;
  (*out)[3] = (crc >> 24) & 0xFF;

  return SHISHI_OK;
}

/**
 * shishi_md4:
 * @handle: shishi handle as allocated by shishi_init().
 * @in: input character array of data to hash.
 * @inlen: length of input character array of data to hash.
 * @out: newly allocated character array with hash of data.
 *
 * Compute hash of data using MD4.  The @out buffer must be
 * deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_md4 (Shishi * handle,
	    const char *in, size_t inlen, char *out[MD4_DIGEST_SIZE])
{
  struct md4_ctx md4;

  md4_init (&md4);
  md4_update (&md4, inlen, in);
  *out = xmalloc (MD4_DIGEST_SIZE);
  md4_digest (&md4, MD4_DIGEST_SIZE, *out);

  return SHISHI_OK;
}

/**
 * shishi_md5:
 * @handle: shishi handle as allocated by shishi_init().
 * @in: input character array of data to hash.
 * @inlen: length of input character array of data to hash.
 * @out: newly allocated character array with hash of data.
 *
 * Compute hash of data using MD5.  The @out buffer must be
 * deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_md5 (Shishi * handle,
	    const char *in, size_t inlen, char *out[MD5_DIGEST_SIZE])
{
  struct md5_ctx md5;

  md5_init (&md5);
  md5_update (&md5, inlen, in);
  *out = xmalloc (MD5_DIGEST_SIZE);
  md5_digest (&md5, MD5_DIGEST_SIZE, *out);

  return SHISHI_OK;
}

/**
 * shishi_hmac_md5:
 * @handle: shishi handle as allocated by shishi_init().
 * @key: input character array with key to use.
 * @keylen: length of input character array with key to use.
 * @in: input character array of data to hash.
 * @inlen: length of input character array of data to hash.
 * @outhash: newly allocated character array with keyed hash of data.
 *
 * Compute keyed checksum of data using HMAC-MD5.  The @outhash buffer
 * must be deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_hmac_md5 (Shishi * handle,
		 const char *key, size_t keylen,
		 const char *in, size_t inlen, char *outhash[MD5_DIGEST_SIZE])
{
  struct hmac_md5_ctx ctx;

  hmac_md5_set_key (&ctx, keylen, key);
  hmac_md5_update (&ctx, inlen, in);
  *outhash = xmalloc (MD5_DIGEST_SIZE);
  hmac_md5_digest (&ctx, MD5_DIGEST_SIZE, *outhash);

  return SHISHI_OK;
}

/**
 * shishi_hmac_sha1:
 * @handle: shishi handle as allocated by shishi_init().
 * @key: input character array with key to use.
 * @keylen: length of input character array with key to use.
 * @in: input character array of data to hash.
 * @inlen: length of input character array of data to hash.
 * @outhash: newly allocated character array with keyed hash of data.
 *
 * Compute keyed checksum of data using HMAC-SHA1.  The @outhash
 * buffer must be deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_hmac_sha1 (Shishi * handle,
		  const char *key, size_t keylen,
		  const char *in, size_t inlen,
		  char *outhash[SHA1_DIGEST_SIZE])
{
  struct hmac_sha1_ctx ctx;

  hmac_sha1_set_key (&ctx, keylen, key);
  hmac_sha1_update (&ctx, inlen, in);
  *outhash = xmalloc (SHA1_DIGEST_SIZE);
  hmac_sha1_digest (&ctx, SHA1_DIGEST_SIZE, *outhash);

  return SHISHI_OK;
}

/**
 * shishi_des_cbc_mac:
 * @handle: shishi handle as allocated by shishi_init().
 * @key: input character array with key to use.
 * @iv: input character array with initialization vector to use, can be NULL.
 * @in: input character array of data to hash.
 * @inlen: length of input character array of data to hash.
 * @out: newly allocated character array with keyed hash of data.
 *
 * Computed keyed checksum of data using DES-CBC-MAC.  The @out buffer
 * must be deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_des_cbc_mac (Shishi * handle,
		    const char key[DES_KEY_SIZE],
		    const char iv[DES_BLOCK_SIZE],
		    const char *in, size_t inlen, char *out[DES_BLOCK_SIZE])
{
  struct CBC_MAC_CTX (struct des_ctx, DES_BLOCK_SIZE) des;
  int rc;

  rc = des_set_key (&des.ctx, key);
  if (!rc)
    {
      shishi_error_printf (handle, "Nettle des_set_key() failed (%d)", rc);
      return SHISHI_CRYPTO_INTERNAL_ERROR;
    }

  if (iv)
    CBC_SET_IV (&des, iv);
  else
    memset (des.iv, 0, DES_BLOCK_SIZE);

  *out = xmalloc (DES_BLOCK_SIZE);

  CBC_MAC (&des, des_encrypt, inlen, *out, in);

  return SHISHI_OK;
}

/**
 * shishi_arcfour:
 * @handle: shishi handle as allocated by shishi_init().
 * @decryptp: 0 to indicate encryption, non-0 to indicate decryption.
 * @key: input character array with key to use.
 * @keylen: length of input key array.
 * @iv: input character array with initialization vector to use, or NULL.
 * @ivout: output character array with updated initialization vector, or NULL.
 * @in: input character array of data to encrypt/decrypt.
 * @inlen: length of input character array of data to encrypt/decrypt.
 * @out: newly allocated character array with encrypted/decrypted data.
 *
 * Encrypt or decrypt data (depending on @decryptp) using ARCFOUR.
 * The @out buffer must be deallocated by the caller.
 *
 * The "initialization vector" used here is the concatenation of the
 * sbox and i and j, and is thus always of size 256 + 1 + 1.  This is
 * a slight abuse of terminology, and assumes you know what you are
 * doing.  Don't use it if you can avoid to.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_arcfour (Shishi * handle, int decryptp,
		const char *key, size_t keylen,
		const char iv[258], char *ivout[258],
		const char *in, size_t inlen, char **out)
{
  struct arcfour_ctx ctx;

  *out = xmalloc (inlen);

  if (iv)
    memcpy (&ctx, iv, sizeof (ctx));
  else
    arcfour_set_key (&ctx, keylen, key);

  arcfour_crypt (&ctx, inlen, *out, in);

  if (ivout)
    {
      *ivout = xmalloc (sizeof (ctx));
      memcpy (*ivout, &ctx, sizeof (ctx));
    }

  return SHISHI_OK;
}

/**
 * shishi_des:
 * @handle: shishi handle as allocated by shishi_init().
 * @decryptp: 0 to indicate encryption, non-0 to indicate decryption.
 * @key: input character array with key to use.
 * @iv: input character array with initialization vector to use, or NULL.
 * @ivout: output character array with updated initialization vector, or NULL.
 * @in: input character array of data to encrypt/decrypt.
 * @inlen: length of input character array of data to encrypt/decrypt.
 * @out: newly allocated character array with encrypted/decrypted data.
 *
 * Encrypt or decrypt data (depending on @decryptp) using DES in CBC
 * mode.  The @out buffer must be deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_des (Shishi * handle, int decryptp,
	    const char key[DES_KEY_SIZE],
	    const char iv[DES_BLOCK_SIZE],
	    char *ivout[DES_BLOCK_SIZE],
	    const char *in, size_t inlen, char **out)
{
  struct CBC_CTX (struct des_ctx, DES_BLOCK_SIZE) des;
  int rc;

  *out = xmalloc (inlen);

  rc = des_set_key (&des.ctx, key);
  if (!rc)
    {
      shishi_error_printf (handle, "Nettle setkey failed");
      return SHISHI_CRYPTO_INTERNAL_ERROR;
    }

  if (iv)
    CBC_SET_IV (&des, iv);
  else
    memset (des.iv, 0, sizeof (des.iv));

  if (decryptp)
    CBC_DECRYPT (&des, des_decrypt, inlen, *out, in);
  else
    CBC_ENCRYPT (&des, des_encrypt, inlen, *out, in);

  if (ivout)
    *ivout = xclone (des.iv, DES_BLOCK_SIZE);

  return SHISHI_OK;
}

/**
 * shishi_3des:
 * @handle: shishi handle as allocated by shishi_init().
 * @decryptp: 0 to indicate encryption, non-0 to indicate decryption.
 * @key: input character array with key to use.
 * @iv: input character array with initialization vector to use, or NULL.
 * @ivout: output character array with updated initialization vector, or NULL.
 * @in: input character array of data to encrypt/decrypt.
 * @inlen: length of input character array of data to encrypt/decrypt.
 * @out: newly allocated character array with encrypted/decrypted data.
 *
 * Encrypt or decrypt data (depending on @decryptp) using 3DES in CBC
 * mode.  The @out buffer must be deallocated by the caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_3des (Shishi * handle, int decryptp,
	     const char key[DES3_KEY_SIZE],
	     const char iv[DES3_BLOCK_SIZE],
	     char *ivout[DES3_BLOCK_SIZE],
	     const char *in, size_t inlen, char **out)
{
  struct CBC_CTX (struct des3_ctx, DES3_BLOCK_SIZE) des3;
  int rc;

  *out = xmalloc (inlen);

  rc = des3_set_key (&des3.ctx, key);
  if (!rc)
    {
      shishi_error_printf (handle, "Nettle setkey failed");
      return SHISHI_CRYPTO_INTERNAL_ERROR;
    }

  if (iv)
    CBC_SET_IV (&des3, iv);
  else
    memset (des3.iv, 0, sizeof (des3.iv));

  if (decryptp)
    CBC_DECRYPT (&des3, des3_decrypt, inlen, *out, in);
  else
    CBC_ENCRYPT (&des3, des3_encrypt, inlen, *out, in);

  if (ivout)
    *ivout = xclone (des3.iv, DES3_BLOCK_SIZE);

  return SHISHI_OK;
}

/**
 * shishi_aes_cts:
 * @handle: shishi handle as allocated by shishi_init().
 * @decryptp: 0 to indicate encryption, non-0 to indicate decryption.
 * @key: input character array with key to use.
 * @keylen: length of input character array with key to use.
 * @iv: input character array with initialization vector to use, or NULL.
 * @ivout: output character array with updated initialization vector, or NULL.
 * @in: input character array of data to encrypt/decrypt.
 * @inlen: length of input character array of data to encrypt/decrypt.
 * @out: newly allocated character array with encrypted/decrypted data.
 *
 * Encrypt or decrypt data (depending on @decryptp) using AES in
 * CBC-CTS mode.  The length of the key, @keylen, decide if AES 128 or
 * AES 256 should be used.  The @out buffer must be deallocated by the
 * caller.
 *
 * Return value: Returns SHISHI_OK iff successful.
 **/
int
shishi_aes_cts (Shishi * handle, int decryptp,
		const char *key, size_t keylen,
		const char iv[AES_BLOCK_SIZE],
		char *ivout[AES_BLOCK_SIZE],
		const char *in, size_t inlen, char **out)
{
  struct CBC_CTS_CTX (struct aes_ctx, AES_BLOCK_SIZE) aes;

  *out = xmalloc (inlen);

  if (iv)
    CBC_SET_IV (&aes, iv);
  else
    memset (aes.iv, 0, sizeof (aes.iv));

  if (decryptp)
    {
      aes_set_decrypt_key (&aes.ctx, keylen, key);
      CBC_CTS_DECRYPT (&aes, aes_decrypt, inlen, *out, in);
    }
  else
    {
      aes_set_encrypt_key (&aes.ctx, keylen, key);
      CBC_CTS_ENCRYPT (&aes, aes_encrypt, inlen, *out, in);
    }

  if (ivout)
    /* XXX what is the output iv for CBC-CTS mode?
       but is this value useful at all for that mode anyway?
       Mostly it is DES apps that want the updated iv, so this is ok. */
    *ivout = xclone (aes.iv, AES_BLOCK_SIZE);

  return SHISHI_OK;
}
