/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "lexer.h"
#include "str.h"
#include "var.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

#if DEBUGGING
static void debug_print ();
#endif

/* Variables on MIS VAL. */
static variable **v;
static int nv;

/* Type of the variables on MIS VAL. */
static int type;

/* Width of string variables on MIS VAL. */
static int width;

/* Items to fill-in var structs with. */
static int miss_type;
static value missing[3];

static int parse_varnames (void);
static int parse_numeric (void);
static int parse_alpha (void);

int
cmd_missing_values (void)
{
  int i;

  match_id (MISSING);
  match_id (VALUES);
  while (token != '.')
    {
#if __CHECKER__
      memset (missing, 0, sizeof (missing));
#endif

      if (!parse_varnames ())
	goto fail;

      if (token != ')')
	{
	  if ((type == NUMERIC && !parse_numeric ())
	      || (type == ALPHA && !parse_alpha ()))
	    goto fail;
	}
      else
	miss_type = MISSING_NONE;

      if (!match_tok (')'))
	{
	  msg (SE, _("`)' expected after value specification."));
	  goto fail;
	}

      for (i = 0; i < nv; i++)
	{
	  v[i]->miss_type = miss_type;
	  memcpy (v[i]->missing, missing, sizeof (value[3]));
	}

      match_tok ('/');
      free (v);
    }
  if (token != '.')
    return syntax_error (_("expecting end of command"));
#if DEBUGGING
  debug_print ();
#endif
  return 1;

fail:
  free (v);
  return -1;
}

static int
parse_varnames ()
{
  int i;

  if (!parse_variables (NULL, &v, &nv, PV_SAME_TYPE))
    return 0;
  if (!match_tok ('('))
    return msg (SE, _("`(' expected after variable name%s."), nv > 1 ? "s" : "");

  type = v[0]->type;
  if (type == NUMERIC)
    return 1;

  width = v[0]->width;
  for (i = 1; i < nv; i++)
    if (v[i]->type == ALPHA && v[i]->nv != 1)
      return msg (SE, _("Long string value specified."));
    else if (v[i]->type == ALPHA && width != v[i]->width)
      return msg (SE, _("Short strings must be of equal width."));
  return 1;
}

/* Number or range? */
enum
  {
    MV_NOR_NOTHING,		/* Empty. */
    MV_NOR_NUMBER,		/* Single number. */
    MV_NOR_RANGE		/* Range. */
  };

/* A single value or a range. */
typedef struct
  {
    int type;			/* One of NOR_*. */
    double d[2];		/* d[0]=lower bound or value, d[1]=upper bound. */
  }
num_or_range;

/* Parses something of the form <num>, or LO[WEST] THRU <num>, or
   <num> THRU HI[GHEST], or <num> THRU <num>, and sets the appropriate
   members of NOR.  Returns success. */
static int
parse_num_or_range (num_or_range * nor)
{
  if (match_id (LO) || match_id (LOWEST))
    {
      nor->type = MV_NOR_RANGE;
      force_match_id (THRU);
      force_num ();
      nor->d[0] = LOWEST;
      nor->d[1] = tokval;
    }
  else if (token == NUM)
    {
      nor->d[0] = tokval;
      get_token ();

      if (match_id (THRU))
	{
	  nor->type = MV_NOR_RANGE;
	  if (match_id (HI) || match_id (HIGHEST))
	    nor->d[1] = HIGHEST;
	  else
	    {
	      force_num ();
	      nor->d[1] = tokval;
	      get_token ();

	      if (nor->d[0] > nor->d[1])
		return msg (SE, _("Range %g THRU %g is not valid because %g is "
			"greater than %g."), nor->d[0], nor->d[1], nor->d[0],
			    nor->d[1]);
	    }
	}
      else
	nor->type = MV_NOR_NUMBER;
    }
  else
    return -1;

  return 1;
}

/* Parses a set of numeric missing values and stores them into
   `missing[]' and `miss_type' global variables. */
static int
parse_numeric (void)
{
  num_or_range set[3], t;
  int r;

  set[1].type = set[2].type = MV_NOR_NOTHING;

  /* Get first number or range. */
  r = parse_num_or_range (&set[0]);
  if (r < 1)
    {
      if (r == -1)
	msg (SE, _("Number or range expected."));
      return 0;
    }

  /* Get second and third optional number or range. */
  match_tok (',');
  r = parse_num_or_range (&set[1]);
  if (r == 1)
    {
      match_tok (',');
      r = parse_num_or_range (&set[2]);
    }
  if (r == 0)
    return 0;

  /* Force range, if present, into set[0]. */
  if (set[1].type == MV_NOR_RANGE)
    t = set[1], set[1] = set[0], set[0] = t;
  if (set[2].type == MV_NOR_RANGE)
    t = set[2], set[2] = set[0], set[0] = t;

  /* Ensure there's not more than one range, or one range
     plus one value. */
  if (set[1].type == MV_NOR_RANGE || set[2].type == MV_NOR_RANGE)
    return msg (SE, _("At most one range can exist in the missing values "
		"for any one variable."));
  if (set[0].type == MV_NOR_RANGE && set[2].type != MV_NOR_NOTHING)
    return msg (SE, _("At most one individual value can be missing along "
		"with one range."));

  /* Set missing[] from set[]. */
  if (set[0].type == MV_NOR_RANGE)
    {
      int x = 0;

      if (set[0].d[0] == LOWEST)
	{
	  miss_type = MISSING_LOW;
	  missing[x++].f = set[0].d[1];
	}
      else if (set[0].d[1] == HIGHEST)
	{
	  miss_type = MISSING_HIGH;
	  missing[x++].f = set[0].d[0];
	}
      else
	{
	  miss_type = MISSING_RANGE;
	  missing[x++].f = set[0].d[0];
	  missing[x++].f = set[0].d[1];
	}

      if (set[1].type == MV_NOR_NUMBER)
	{
	  miss_type += 3;
	  missing[x].f = set[1].d[0];
	}
    }
  else
    {
      if (set[0].type == MV_NOR_NUMBER)
	{
	  miss_type = MISSING_1;
	  missing[0].f = set[0].d[0];
	}
      if (set[1].type == MV_NOR_NUMBER)
	{
	  miss_type = MISSING_2;
	  missing[1].f = set[1].d[0];
	}
      if (set[2].type == MV_NOR_NUMBER)
	{
	  miss_type = MISSING_3;
	  missing[2].f = set[2].d[0];
	}
    }

  return 1;
}

static int
parse_alpha (void)
{
  for (miss_type = 0; token == STRING && miss_type < 3; miss_type++)
    {
      if ((int) strlen (tokstr) != width)
	return msg (SE, _("String is not of proper length."));
      strncpy (missing[miss_type].s, tokstr, 8);
      get_token ();
      match_tok (',');
    }
  if (miss_type < 1)
    return msg (SE, _("String expected."));

  return 1;
}

/* Copy the missing values from variable SRC to variable DEST. */
void
copy_missing_values (variable *dest, const variable *src)
{
  static const int n_values[MISSING_COUNT] = 
    {
      0, 1, 2, 3, 2, 1, 1, 3, 2, 2,
    };
    
  assert (dest->width == src->width);
  assert (src->miss_type >= 0 && src->miss_type < MISSING_COUNT);
  
  {
    int i;

    dest->miss_type = src->miss_type;
    for (i = 0; i < n_values[src->miss_type]; i++)
      if (src->type == NUMERIC)
	dest->missing[i].f = src->missing[i].f;
      else
	memcpy (dest->missing[i].s, src->missing[i].s, src->width);
  }
}


/* Debug output. */

#if DEBUGGING
static void
debug_print ()
{
  int i, j;

  puts (_("Missing value:"));
  for (i = 0; i < nvar; i++)
    {
      printf ("	 %8s: ", var[i]->name);
      if (var[i]->type == ALPHA && var[i]->nv > 1)
	puts (_("(long string variable)"));
      else
	switch (var[i]->miss_type)
	  {
	  case MISSING_NONE:
	    printf (_("(no missing values)\n"));
	    break;
	  case MISSING_1:
	  case MISSING_2:
	  case MISSING_3:
	    printf ("(MISSING_%d)", var[i]->miss_type);
	    for (j = 0; j < var[i]->miss_type; j++)
	      if (var[i]->type == ALPHA)
		printf ("  \"%.*s\"", var[i]->width, var[i]->missing[j].s);
	      else
		printf ("  %.2g", var[i]->missing[j].f);
	    printf ("\n");
	    break;
	  case MISSING_RANGE:
	    printf ("(MISSING_RANGE)  %.2g THRU %.2g\n",
		    var[i]->missing[0].f, var[i]->missing[1].f);
	    break;
	  case MISSING_RANGE_1:
	    printf ("(MISSING_RANGE_1)	%.2g THRU %.2g, %.2g\n",
		    var[i]->missing[0].f, var[i]->missing[1].f,
		    var[i]->missing[2].f);
	    break;
	  default:
	    printf (_("(!!!INTERNAL ERROR--%d!!!)\n"), var[i]->miss_type);
	  }
    }
}
#endif /* DEBUGGING */
