;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2015 Sou Bunnbu <iyzsong@gmail.com>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages ninja)
  #:use-module ((guix licenses) #:select (asl2.0))
  #:use-module (guix packages)
  #:use-module (guix download)
  #:use-module (guix build-system gnu)
  #:use-module (gnu packages)
  #:use-module (gnu packages python))

(define-public ninja
  (package
    (name "ninja")
    (version "1.5.3")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/martine/ninja/"
                                  "archive/v" version ".tar.gz"))
              (sha256
               (base32
                "1h3yfwcfl61v493vna6jia2fizh8rpig7qw2504cvkr6gid3p5bw"))
              (patches (list (search-patch "ninja-tests.patch")))))
    (build-system gnu-build-system)
    (arguments
     '(#:phases
       (alist-replace
        'configure
        (lambda _
          (substitute* "src/subprocess-posix.cc"
            (("/bin/sh") (which "sh"))))
        (alist-replace
         'build
         (lambda _
           (zero? (system* "./configure.py" "--bootstrap")))
         (alist-replace
          'check
          (lambda _
            (and (zero? (system* "./configure.py"))
                 (zero? (system* "./ninja" "ninja_test"))
                 (zero? (system* "./ninja_test"))))
          (alist-replace
           'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (doc (string-append out "/share/doc/ninja")))
               (mkdir-p bin)
               (copy-file "ninja" (string-append bin "/ninja"))
               (mkdir-p doc)
               (copy-file "doc/manual.asciidoc"
                          (string-append doc "/manual.asciidoc"))))
           %standard-phases))))))
    (native-inputs `(("python" ,python-2)))
    (home-page "http://martine.github.io/ninja/")
    (synopsis "Small build system")
    (description
     "Ninja is a small build system with a focus on speed.  It differs from
other build systems in two major respects: it is designed to have its input
files generated by a higher-level build system, and it is designed to run
builds as fast as possible.")
    (license asl2.0)))
