/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  The konversation DCOP interface class
  begin:     Mar 7 2003
  copyright: (C) 2003 by Alex Zepeda
  email:     zipzippy@sonic.net
*/

#include "konvdcop.h"
#include "konversationapplication.h"
#include "connectionmanager.h"
#include "awaymanager.h"
#include "channel.h"
#include "identity.h"
#include "server.h"

#include <tdeapplication.h>
#include <kdebug.h>
#include <dcopclient.h>
#include <tdelocale.h>


KonvDCOP::KonvDCOP() : DCOPObject("irc"), TQObject(0, "irc")
{
    connectDCOPSignal("kdesktop", "KScreensaverIface", "KDE_start_screensaver()", "setScreenSaverStarted()", false);
    connectDCOPSignal("kdesktop", "KScreensaverIface", "KDE_stop_screensaver()", "setScreenSaverStopped()", false);
}

void KonvDCOP::raw(const TQString& server,const TQString& command)
{
    kdDebug() << "KonvDCOP::raw()" << endl;
    // send raw IRC protocol data
    emit dcopRaw(server,command);
}

TQStringList KonvDCOP::listServers()
{
    KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);

    TQStringList hosts;
    TQPtrList<Server> serverList = konvApp->getConnectionManager()->getServerList();
    Server* server;

    for (server = serverList.first(); server; server = serverList.next())
        if (server) hosts << server->getServerName();

    return hosts;
}

TQStringList KonvDCOP::listConnectedServers()
{
    KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);

    TQStringList connectedHosts;
    TQPtrList<Server> serverList = konvApp->getConnectionManager()->getServerList();
    Server* server;

    for (server = serverList.first(); server; server = serverList.next())
        if (server && server->isConnected()) connectedHosts << server->getServerName();

    return connectedHosts;
}

void KonvDCOP::setAway(const TQString& awaymessage)
{
    static_cast<KonversationApplication*>(kapp)->getAwayManager()->requestAllAway(awaymessage);
}

void KonvDCOP::setBack()
{
    static_cast<KonversationApplication*>(kapp)->getAwayManager()->requestAllUnaway();
}

void KonvDCOP::setScreenSaverStarted()
{
    static_cast<KonversationApplication*>(kapp)->getAwayManager()->setManagedIdentitiesAway();
}

void KonvDCOP::setScreenSaverStopped()
{
    static_cast<KonversationApplication*>(kapp)->getAwayManager()->setManagedIdentitiesUnaway();
}

void KonvDCOP::sayToAll(const TQString &message)
{
    emit dcopMultiServerRaw("msg " + message);
}

void KonvDCOP::actionToAll(const TQString &message)
{
    emit dcopMultiServerRaw("me " + message);
}

void KonvDCOP::say(const TQString& _server,const TQString& _target,const TQString& _command)
{
    //Sadly, copy on write doesn't exist with TQString::replace
    TQString server(_server), target(_target), command(_command);

    // TODO: this just masks a greater problem - Server::addQuery will return a query for '' --argonel
    // TODO: other DCOP calls need argument checking too --argonel
    if (server.isEmpty() || target.isEmpty() || command.isEmpty())
        kdDebug() <<  "KonvDCOP::say() requires 3 arguments." << endl;
    else
    {
        command.replace('\n',"\\n");
        command.replace('\r',"\\r");
        target.remove('\n');
        target.remove('\r');
        server.remove('\n');
        server.remove('\r');
        // Act as if the user typed it
        emit dcopSay(server,target,command);
    }
}

void KonvDCOP::info(const TQString& string)
{
    kdDebug() << "KonvDCOP::info()" << endl;
    emit dcopInfo(string);
}

void KonvDCOP::debug(const TQString& string)
{
    kdDebug() << "KonvDCOP::debug()" << endl;
    emit dcopInfo(TQString("Debug: %1").arg(string));
}

void KonvDCOP::error(const TQString& string)
{
    kdDebug() << "KonvDCOP::error()" << endl;
    emit dcopInfo(TQString("Error: %1").arg(string));
}

void KonvDCOP::insertMarkerLine()
{
    emit dcopInsertMarkerLine();
}

void KonvDCOP::connectToServer(const TQString& address, int port, const TQString& channel, const TQString& password)
{
    emit connectTo(Konversation::SilentlyReuseConnection, address, TQString::number(port), password, "", channel);
}

TQString KonvDCOP::getNickname(const TQString& serverName)
{
    Server* server = KonversationApplication::instance()->getConnectionManager()->getServerByName(serverName);

    if (!server)
    {
        error( i18n( "getNickname: Server %1 is not found." ).arg( serverName ) );
        return TQString();
    }

    return server->getNickname();
}

TQString KonvDCOP::getAnyNickname()
{
    KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);

    Server* server = konvApp->getConnectionManager()->getAnyServer();

    if (server) return server->getNickname();

    return TQString();
}

TQString KonvDCOP::getChannelEncoding(const TQString& server, const TQString& channel)
{
    return Preferences::channelEncoding(server,channel);
}

// Identity stuff
KonvIdentDCOP::KonvIdentDCOP()
: DCOPObject("identity"),
TQObject(0, "identity")
{
}

TQStringList KonvIdentDCOP::listIdentities()
{
    TQStringList identities;
    IdentityList ids = Preferences::identityList();
    for(IdentityList::ConstIterator it = ids.begin(); it != ids.end(); ++it)
    {
        identities.append((*it)->getName());
    }
    return identities;
}

void KonvIdentDCOP::setrealName(const TQString &id_name, const TQString& name)
{
    IdentityList ids = Preferences::identityList();

    for(IdentityList::iterator it = ids.begin(); it != ids.end(); ++it)
    {
        if ((*it)->getName() == id_name)
        {
            (*it)->setRealName(name);
            return;
        }
    }

}

TQString KonvIdentDCOP::getrealName(const TQString &id_name)
{
    IdentityList ids = Preferences::identityList();

    for(IdentityList::ConstIterator it = ids.begin(); it != ids.end(); ++it)
    {
        if ((*it)->getName() == id_name)
        {
            return (*it)->getRealName();
        }
    }

    return TQString();
}

void KonvIdentDCOP::setIdent(const TQString &/*identity*/, const TQString& /*ident*/)
{
    //Preferences::identityByName(identity)->.setIdent(;
}

TQString KonvIdentDCOP::getIdent(const TQString &identity)
{
    return Preferences::identityByName(identity)->getIdent();
}

void KonvIdentDCOP::setNickname(const TQString &identity, int index,const TQString& nick)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setNickname(index, nick);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getNickname(const TQString &identity, int index)
{
    return Preferences::identityByName(identity)->getNickname(index);
}

void KonvIdentDCOP::setBot(const TQString &identity, const TQString& bot)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setBot(bot);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getBot(const TQString &identity)
{
    return Preferences::identityByName(identity)->getBot();
}

void KonvIdentDCOP::setPassword(const TQString &identity, const TQString& password)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setPassword(password);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getPassword(const TQString &identity)
{
    return Preferences::identityByName(identity)->getPassword();
}

void KonvIdentDCOP::setNicknameList(const TQString &identity, const TQStringList& newList)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setNicknameList(newList);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQStringList KonvIdentDCOP::getNicknameList(const TQString &identity)
{
    return Preferences::identityByName(identity)->getNicknameList();
}

void KonvIdentDCOP::setQuitReason(const TQString &identity, const TQString& reason)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setQuitReason(reason);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getQuitReason(const TQString &identity)
{
    return Preferences::identityByName(identity)->getQuitReason();
}


void KonvIdentDCOP::setPartReason(const TQString &identity, const TQString& reason)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setPartReason(reason);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getPartReason(const TQString &identity)
{
    return Preferences::identityByName(identity)->getPartReason();
}

void KonvIdentDCOP::setKickReason(const TQString &identity, const TQString& reason)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setKickReason(reason);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getKickReason(const TQString &identity)
{
    return Preferences::identityByName(identity)->getKickReason();
}

void KonvIdentDCOP::setShowAwayMessage(const TQString &identity, bool state)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setShowAwayMessage(state);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

bool KonvIdentDCOP::getShowAwayMessage(const TQString &identity)
{
    return Preferences::identityByName(identity)->getShowAwayMessage();
}

void KonvIdentDCOP::setAwayMessage(const TQString &identity, const TQString& message)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setAwayMessage(message);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getAwayMessage(const TQString &identity)
{
    const TQString f = Preferences::identityByName(identity)->getAwayMessage();
    return f;
}

void KonvIdentDCOP::setReturnMessage(const TQString &identity, const TQString& message)
{
    const Identity *i = Preferences::identityByName(identity);
    const_cast<Identity *>(i)->setReturnMessage(message);
    static_cast<KonversationApplication *>(kapp)->saveOptions(true);
}

TQString KonvIdentDCOP::getReturnMessage(const TQString &identity)
{
    return Preferences::identityByName(identity)->getReturnMessage();
}

#include "konvdcop.moc"
