/***************************************************************************
 *   Copyright (C) 1999-2001 by Bernd Gehrmann                             *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "grepviewpart.h"

#include <tqpopupmenu.h>
#include <tqvbox.h>
#include <tqwhatsthis.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <tdeaction.h>
#include <kdialogbase.h>
#include <kiconloader.h>
#include <tdeprocess.h>
#include <kstringhandler.h>
#include <tdetexteditor/document.h>

#include "kdevcore.h"
#include "kdevpartcontroller.h"
#include "kdevmainwindow.h"
#include "kdevplugininfo.h"
#include "kdeveditorutil.h"
#include "grepviewwidget.h"

static const KDevPluginInfo data("kdevgrepview");

K_EXPORT_COMPONENT_FACTORY(libkdevgrepview, GrepViewFactory(data))

GrepViewPart::GrepViewPart( TQObject *parent, const char *name, const TQStringList & )
    : KDevPlugin( &data, parent, name ? name : "GrepViewPart" )
{
    setInstance(GrepViewFactory::instance());

    setXMLFile("kdevgrepview.rc");

    connect( core(), TQ_SIGNAL(stopButtonClicked(KDevPlugin*)),
             this, TQ_SLOT(stopButtonClicked(KDevPlugin*)) );
    connect( core(), TQ_SIGNAL(projectOpened()), this, TQ_SLOT(projectOpened()) );
    connect( core(), TQ_SIGNAL(projectClosed()), this, TQ_SLOT(projectClosed()) );
    connect( core(), TQ_SIGNAL(contextMenu(TQPopupMenu *, const Context *)),
             this, TQ_SLOT(contextMenu(TQPopupMenu *, const Context *)) );

    m_widget = new GrepViewWidget(this);
    m_widget->setIcon(SmallIcon("grep"));
    m_widget->setCaption(i18n("Grep Output"));
    TQWhatsThis::add(m_widget, i18n("<b>Find in files</b><p>"
                                   "This window contains the output of a grep "
                                   "command. Clicking on an item in the list "
                                   "will automatically open the corresponding "
                                   "source file and set the cursor to the line "
                                   "with the match."));

    mainWindow()->embedOutputView(m_widget, i18n("Find in Files"), i18n("Output of the grep command"));

    TDEAction *action;

    action = new TDEAction(i18n("Find in Fi&les..."), "grep", CTRL+ALT+Key_F,
                         this, TQ_SLOT(slotGrep()),
                         actionCollection(), "edit_grep");
    action->setToolTip( i18n("Search for expressions over several files") );
    action->setWhatsThis( i18n("<b>Find in files</b><p>"
                               "Opens the 'Find in files' dialog. There you "
                               "can enter a regular expression which is then "
                               "searched for within all files in the directories "
                               "you specify. Matches will be displayed, you "
                               "can switch to a match directly.") );
}


GrepViewPart::~GrepViewPart()
{
	if ( m_widget )
		mainWindow()->removeView( m_widget );
    delete m_widget;
}


void GrepViewPart::stopButtonClicked(KDevPlugin* which)
{
    if ( which != 0 && which != this )
        return;
    kdDebug(9001) << "GrepViewPart::stopButtonClicked()" << endl;
    m_widget->killJob( SIGHUP );
}


void GrepViewPart::projectOpened()
{
    kdDebug(9001) << "GrepViewPart::projectOpened()" << endl;
    m_widget->projectChanged(project());
}


void GrepViewPart::projectClosed()
{
    m_widget->projectChanged(0);
}


void GrepViewPart::contextMenu(TQPopupMenu *popup, const Context *context)
{
    kdDebug(9001) << "context in grepview" << endl;
    if (!context->hasType( Context::EditorContext ))
        return;

    const EditorContext *econtext = static_cast<const EditorContext*>(context);
    TQString ident = econtext->currentWord();
    if (!ident.isEmpty()) {
        m_popupstr = ident;
        TQString squeezed = KStringHandler::csqueeze(ident, 30);
        int id = popup->insertItem( i18n("Grep: %1").arg(squeezed),
                           this, TQ_SLOT(slotContextGrep()) );
        popup->setWhatsThis(id, i18n("<b>Grep</b><p>Opens the find in files dialog "
                               "and sets the pattern to the text under the cursor."));
        popup->insertSeparator();
    }
}


void GrepViewPart::slotGrep()
{
	if ( !m_widget->isRunning() )
	{
		TQString contextString = KDevEditorUtil::currentSelection( dynamic_cast<KTextEditor::Document*>( partController()->activePart() ) );
		if ( contextString.isEmpty() )
		{
			contextString = KDevEditorUtil::currentWord( dynamic_cast<KTextEditor::Document*>( partController()->activePart() ) );
		}
		m_widget->showDialogWithPattern( contextString );
	}
}


void GrepViewPart::slotContextGrep()
{
	if ( !m_widget->isRunning() )
	{
		m_widget->showDialogWithPattern(m_popupstr);
	}
}

#include "grepviewpart.moc"
