/*
    This file is part of Akregator.

    Copyright (C) 2005 Frank Osterfeld <frank.osterfeld at kdemail.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#ifndef AKREGATOR_TAGSET_H
#define AKREGATOR_TAGSET_H

#include <tqobject.h>

class TQDomDocument;
template <class K,class T> class TQMap;
class TQString;
class TQStringList;

namespace Akregator {

class Tag;

/** \brief represents a set of tags (see Tag)
    In an application, there is usually one central tag set that is used.
    The tag set is not necessarily managed by the application itself, it might also be part of 
    a desktop-wide framework (like Tenor) managing a common tag set for all applications 
    
    @author Frank Osterfeld
*/

class TagSet : public TQObject
{
    friend class Tag;
    TQ_OBJECT
  
    public:

        TagSet(TQObject* parent=0);
        virtual ~TagSet();

        /** adds a tag to the tag set. The tag set will emit signalTagAdded */
        void insert(const Tag& tag);

        /** removes a tag from the tag set. The tag set will emit signalTagRemoved */
        void remove(const Tag& tag);

        /** returns the tag set as map ((id, Tag) pairs) */
        TQMap<TQString,Tag> toMap() const;

        /** returns @c true if this set contains @c tag */
        bool contains(const Tag& tag) const;

        /** returns @c true if this set contains a tag with a given ID */
        bool containsID(const TQString& id) const;

        /** returns the tag with the given ID if the tag is element of the set, or a null tag if not */
        Tag findByID(const TQString& id) const;

        /** reads tag set from XML
            see toXML() for an explanation of the format */
        void readFromXML(const TQDomDocument& doc);

        /** returns an XML representation of the tag set. 
            The root element is @c <tagSet>, a tag  ("someID", "someName") is represented as
            \code <tag id="someID">someName</tag> \endcode
            Full example: 
            \code
            <?xml version="1.0" encoding="UTF-8"?>
            <tagSet version="0.1" >
            <tag id="http://akregator.sf.net/tags/Interesting" >Interesting</tag>
            <tag id="AFs3SdaD" >Pretty boring</tag>
            </tagSet>
            \endcode
         */
        TQDomDocument toXML() const;

    signals:
        /** emitted when a tag was added to this tag set */
        void signalTagAdded(const Tag&);
        /** emitted when a tag was removed from this set */
        void signalTagRemoved(const Tag&);
        /** emitted when a tag in this set was changed (e.g. renamed) */
        void signalTagUpdated(const Tag&);

    protected:
        /** called by the tag (Tag is friend class) after a change */
        void tagUpdated(const Tag& tag);
        
    private:
        class TagSetPrivate;
        TagSetPrivate* d;
};

} // namespace Akregator

#endif // AKREGATOR_TAGSET_H
