/***************************************************************************
 *   Copyright (C) 2006 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef COMPILE_CONFIG_H
#define COMPILE_CONFIG_H

#include "common/global/purl.h"
#include "devices/base/hex_buffer.h"
#include "tools/base/tool_group.h"
class Project;

//----------------------------------------------------------------------------
namespace Compile
{
class Config
{
public:
  Config(Project *project) : _group(0), _project(project) {}

  bool hasCustomArguments(Tool::Category category) const { return boolValue(category, "has_custom_arguments", false); }
  void setHasCustomArguments(Tool::Category category, bool custom) { setValue(category, "has_custom_arguments", custom); }
  TQStringList customArguments(Tool::Category category) const { return listValues(category, "custom_arguments", TQStringList()); }
  void setCustomArguments(Tool::Category category, const TQStringList &args) { return setListValues(category, "custom_arguments", args); }
  TQStringList rawIncludeDirs(Tool::Category category) const { return listValues(category, "includes", "$(SRCPATH)"); }
  TQStringList includeDirs(Tool::Category category, const TQString &prefix, const TQString &suffix = TQString(), const TQString &separator = TQString()) const;
  void setRawIncludeDirs(Tool::Category category, const TQStringList &dirs) { setListValues(category, "includes", dirs); }
  TQString rawCustomOptions(Tool::Category category) const { return value(category, "custom_options", TQString()); }
  TQStringList customOptions(Tool::Category category) const;
  void setRawCustomOptions(Tool::Category category, const TQString &value) { setValue(category, "custom_options", value); }
  TQString rawCustomLibraries(Tool::Category category) const { return value(category, "custom_libraries", TQString()); }
  TQStringList customLibraries(Tool::Category category) const;
  void setRawCustomLibraries(Tool::Category category, const TQString &value) { setValue(category, "custom_libraries", value); }
  uint warningLevel(Tool::Category category) const { return uintValue(category, "warning_level", 0); }
  void setWarningLevel(Tool::Category category, uint level) { setValue(category, "warning_level", TQString::number(level)); }
  HexBuffer::Format hexFormat() const;
  void setHexFormat(HexBuffer::Format format);

  static const Tool::Group &toolGroup(const Project *project);
  static void setToolGroup(Project *project, const Tool::Group &group) { setGlobalValue(project, "tool", group.name()); }
  static TQString device(const Project *project);
  static void setDevice(Project *project, const TQString &device) { setGlobalValue(project, "device", device); }
  static TQStringList customCommands(Project *project) { return globalListValues(project, "custom_shell_commands", TQStringList()); }
  static void setCustomCommands(Project *project, const TQStringList &commands) { setGlobalListValues(project, "custom_shell_commands", commands); }
  static TQString globalValue(const Project *project, const TQString &key, const TQString &defaultValue);
  static void setGlobalValue(Project *project, const TQString &key, const TQString &value);
  static TQStringList globalListValues(const Project *project, const TQString &key, const TQStringList &defaultValues);
  static void setGlobalListValues(Project *project, const TQString &key, const TQStringList &value);

  static PURL::Directory directory(const Tool::Group &group, DirectoryType type);
  static void setDirectory(const Tool::Group &group, DirectoryType type, const PURL::Directory &dir);
  static bool withWine(const Tool::Group &group);
  static void setWithWine(const Tool::Group &group, bool withWine);
  static Tool::OutputExecutableType outputExecutableType(const Tool::Group &group);
  static void setOutputExecutableType(const Tool::Group &group, Tool::OutputExecutableType type);

protected:
  const Tool::Group *_group;
  Project *_project;

  void setValue(Tool::Category category, const TQString &key, const TQString &value);
  TQString value(Tool::Category category, const TQString &key, const TQString &defaultValue) const;
  void setListValues(Tool::Category category, const TQString &key, const TQStringList &values);
  TQStringList listValues(Tool::Category category, const TQString &key, const TQStringList &defaultValues) const;
  void setValue(Tool::Category category, const TQString &key, bool value) { setValue(category, key, TQString(value ? "true" : "false")); }
  bool boolValue(Tool::Category category, const TQString &key, bool defaultValue) const;
  void setValue(Tool::Category category, const TQString &key, uint value) { setValue(category, key, TQString::number(value)); }
  uint uintValue(Tool::Category category, const TQString &key, uint defaultValue) const;

  static TQString value(const TQString &group, const TQString &subGroup, const TQString &key, const TQString &defaultValue);
  static void setValue(const TQString &group, const TQString &subGroup, const TQString &key, const TQString &value);
  static TQStringList listValues(const TQString &group, const TQString &subGroup, const TQString &key, const TQStringList &defaultValues);
  static void setListValues(const TQString &group, const TQString &subGroup, const TQString &key, const TQStringList &values);

  friend class Tool::Group;
};

} // namespace

#endif
