/***************************************************************************
 *   Copyright (C) 2006 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "purl_gui.h"

#include <tqlayout.h>
#include <kiconloader.h>
#include <kpushbutton.h>
#include <krun.h>
#include <tdefiledialog.h>
#include <kdirselectdialog.h>

#include "misc_gui.h"

//-----------------------------------------------------------------------------
PURL::Url PURL::getOpenUrl(const TQString &startDir, const TQString &filter,
                           TQWidget *widget, const TQString &caption)
{
  return KFileDialog::getOpenURL(startDir, filter, widget, caption);
}

PURL::UrlList PURL::getOpenUrls(const TQString &startDir, const TQString &filter,
                                TQWidget *widget, const TQString &caption)
{
  return KFileDialog::getOpenURLs(startDir, filter, widget, caption);
}

PURL::Url PURL::getSaveUrl(const TQString &startDir, const TQString &filter,
                           TQWidget *widget, const TQString &caption,
                           SaveAction action)
{
  Url url = KFileDialog::getSaveURL(startDir, filter, widget, caption);
  if ( url.isEmpty() ) return Url();
  switch (action) {
    case NoSaveAction: break;
    case AskOverwrite:
      if ( url.exists() ) {
        if ( !MessageBox::askContinue(i18n("File \"%1\" already exists. Overwrite ?").arg(url.pretty())) ) return Url();
      }
      break;
    case CancelIfExists:
      if ( url.exists() ) return Url();
      break;
  }
  return url;
}

PURL::Directory PURL::getExistingDirectory(const TQString &startDir, TQWidget *widget,
                                           const TQString &caption)
{
  KURL kurl = KDirSelectDialog::selectDirectory(startDir, false, widget, caption);
  if ( kurl.isEmpty() ) return Directory();
  return Directory(kurl.path(1));
}

TQPixmap PURL::icon(FileType type)
{
  if (type.data().xpm_icon) return TQPixmap(type.data().xpm_icon);
  if ( hasMimetype(type) ) return KMimeType::mimeType(type.data().mimetype)->pixmap(TDEIcon::Small);
  return TQPixmap();
}

bool PURL::hasMimetype(FileType type)
{
  if ( type.data().mimetype==0 ) return false;
  KMimeType::Ptr ptr = KMimeType::mimeType(type.data().mimetype);
  return ( ptr!=KMimeType::defaultMimeTypePtr() );
}

//-----------------------------------------------------------------------------
PURL::Label::Label(const TQString &url, const TQString &text,
                   TQWidget *parent, const char *name)
  : KURLLabel(url, text, parent, name)
{
  connect(this, TQ_SIGNAL(leftClickedURL()), TQ_SLOT(urlClickedSlot()));
}

void PURL::Label::urlClickedSlot()
{
  (void)new KRun(url());
}

//-----------------------------------------------------------------------------
PURL::BaseWidget::BaseWidget(TQWidget *parent, const char *name)
  : TQWidget(parent, name)
{
  init();
}

PURL::BaseWidget::BaseWidget(const TQString &defaultDir, TQWidget *parent, const char *name)
  : TQWidget(parent, name), _defaultDir(defaultDir)
{
  init();
}

void PURL::BaseWidget::init()
{
  TQHBoxLayout *top = new TQHBoxLayout(this, 0, 10);

  _edit = new KLineEdit(this);
  connect(_edit, TQ_SIGNAL(textChanged(const TQString &)), TQ_SIGNAL(changed()));
  top->addWidget(_edit);
  TDEIconLoader loader;
  TQIconSet iconset = loader.loadIcon("document-open", TDEIcon::Toolbar);
  TQPushButton *button = new  KPushButton(iconset, TQString(), this);
  connect(button, TQ_SIGNAL(clicked()), TQ_SLOT(buttonClicked()));
  top->addWidget(button);
}

//----------------------------------------------------------------------------
void PURL::DirectoryWidget::buttonClicked()
{
  Directory dir = getExistingDirectory(_defaultDir, this, i18n("Select Directory"));
  if ( dir.isEmpty() ) return;
  _edit->setText(dir.path());
  emit changed();
}

//----------------------------------------------------------------------------
PURL::DirectoriesWidget::DirectoriesWidget(const TQString &title, TQWidget *parent, const char *name)
  : TQVGroupBox(title, parent, name)
{
  init(TQString());
}

PURL::DirectoriesWidget::DirectoriesWidget(const TQString &title, const TQString &defaultDir, TQWidget *parent, const char *name)
  : TQVGroupBox(title, parent, name)
{
  init(defaultDir);
}

void PURL::DirectoriesWidget::init(const TQString &defaultDir)
{
  DirectoryWidget *edit = new DirectoryWidget(defaultDir);
  _editListBox = new EditListBox(1, edit, edit->lineEdit(), this, "directories_editlistbox");
  connect(_editListBox, TQ_SIGNAL(changed()), TQ_SIGNAL(changed()));
}

//----------------------------------------------------------------------------
void PURL::UrlWidget::buttonClicked()
{
  Url url = getOpenUrl(_defaultDir, _filter, this, i18n("Select File"));
  if ( url.isEmpty() ) return;
  _edit->setText(url.filepath());
  emit changed();
}

#include "purl_gui.moc"
