/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  addressbook.cpp  - This class contains functions that interact with kaddressbook.
  begin:     Fri 2004-07-23
  copyright: (C) 2004 by John Tapsell
  email:     john@geola.co.uk
*/

#ifndef ADDRESSBOOKBASE_H
#define ADDRESSBOOKBASE_H

#include "../images.h"
#include "../nickinfo.h"
#include "../channelnick.h"

#include <kstaticdeleter.h>
#include <tqobject.h>
#include <tqregexp.h>
#include <config.h>


#include <tdeabc/addressbook.h>
#include <tdeabc/stdaddressbook.h>


namespace Konversation
{
    class AddressbookBase : public TQObject, public KIMIface
    {
        TQ_OBJECT
  
            public:
            virtual ~AddressbookBase();           // This needs to be public so it can be deleted by our static pointer
            TDEABC::Addressee getKABCAddresseeFromNick(const TQString &ircnick, const TQString &servername, const TQString &servergroup);
            TDEABC::Addressee getKABCAddresseeFromNick(const TQString &nick_server);
            bool hasNick(const TDEABC::Addressee &addressee, const TQString &ircnick, const TQString &servername, const TQString &servergroup);
            bool hasNick(const TDEABC::Addressee &addressee, const TQString &nick_server);

            /** For a given contact, remove the ircnick if they have it. If you
             *  pass an addressBook, the contact is inserted if it has changed.
             */
            void unassociateNick(TDEABC::Addressee &addressee, const TQString &ircnick, const TQString &servername, const TQString &servergroup);
            void associateNick(TDEABC::Addressee &addressee, const TQString &ircnick, const TQString &servername, const TQString &servergroup);
            bool associateNickAndUnassociateFromEveryoneElse(TDEABC::Addressee &addressee, const TQString &ircnick, const TQString &servername, const TQString &servergroup);
            /** If this user is online, return one of the nicks that they are
             * using.  Otherwise return the first nick listed.
             * If there are multiple matches, it will prefer ones that are not set to away.
             * @return online nick, first nick, or TQString() if they aren't known at all.
             */
            TQString getBestNick(const TDEABC::Addressee &addressee);
            bool hasAnyNicks(const TDEABC::Addressee &addresse);
            int presenceStatusByAddressee(const TDEABC::Addressee &addressee);
            bool isOnline(TDEABC::Addressee &addressee);
            bool getAndCheckTicket();
            bool saveTicket();
            void releaseTicket();
            bool saveAddressee(TDEABC::Addressee &addressee);
            bool saveAddressbook();

            TQStringList allContactsNicksForServer(const TQString &servername, const TQString &servergroup);
            TQStringList getNicks(const TDEABC::Addressee &addressee, const TQString &servername, const TQString &servergroup);

            TDEABC::AddressBook *getAddressBook();

            /**  Return an online NickInfo for this addressee.
             *  If there are multiple matches, it tries to pick one that is not away.
             *  Note: No NickInfo is returned if the addressee is offline.
             *  NickInfo's are for online and away nicks only.
             *  @param addressee The addressee to get a nickInfo for
             *  @return A nickInfo.  It tries hard to return a nickInfo that is not away if one exists.
             */
            static NickInfoPtr getNickInfo(const TDEABC::Addressee &addressee);
            /**
             * Lets outsiders tell us to emit presenceChanged signal.
             */
            virtual void emitContactPresenceChanged(const TQString &uid, int presence) = 0;
            /**
             * Lets outsiders tell us to emit presenceChanged signal.
             */
            virtual void emitContactPresenceChanged(const TQString &uid) = 0;

            /**
             *  Run kmail (or whatever the users email client is)
             *  to create a single email addressed to all of the nicks passed in.
             *  Gives an error dialog to the user if any of the contacts don't have an
             *  email address associated, and gives the user the option to continue
             *  with the contacts that did have email addresses.
             */
            bool sendEmail(const ChannelNickList &nicklist);
            /**
             *  Run kmail (or whatever the users email client is)
             *  to create a single email addressed to the addressee passed in.
             *  Gives an error dialog to the user if the addressee doesn't have an email address associated.
             */
            bool sendEmail(const TDEABC::Addressee &addressee);
            /**
             *  Run kaddressbook to edit the addressee passed in.
             */
            bool editAddressee(const TQString &uid);
            /**
             *  Run the users email program (e.g. kmail) passing "mailto:" + mailtoaddress.
             *  Note that mailto:  will be prepended for you.
             *  @param mailtoaddress A comma delimited set of email address topass as "mailto:"
             *  @return True if there were no problems running the email program.  An error will be shown to the user if there was.
             */
            bool runEmailProgram(const TQString &mailtoaddress);

            /** Just calls KonversationMainWindow::focusAndShowErrorMessage(const TQString *errorMsg)
             *
             *  @see KonversationMainWindow::focusAndShowErrorMessage(const TQString *errorMsg)
             */
            void focusAndShowErrorMessage(const TQString &errorMsg);
            signals:
            void addresseesChanged();

        protected:
            AddressbookBase();
            TDEABC::AddressBook* addressBook;
            TDEABC::Ticket *m_ticket;
    };

}                                                 //NAMESPACE
#endif
