/***************************************************************************
 *   Copyright (C) 2005 by Joe Ferris                                      *
 *   jferris@optimistictech.com                                            *
 *                                                                         *
 *   This document is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This document is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this document; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include <kservicegroup.h>
#include <tdesycocaentry.h>
#include <tdesycocatype.h>
#include <tdeapplication.h>
#include <knuminput.h>
#include <kcombobox.h>

#include <tqcheckbox.h>

#include <actionregistry.h>

#include "settings.h"
#include "documentcatalog.h"
#include "document.h"
#include "directory.h"
#include <status.h>
#include <match.h>
#include "actionopendocument.h"

K_EXPORT_COMPONENT_FACTORY( katapult_documentcatalog,
                            KGenericFactory<DocumentCatalog>( "katapult_documentcatalog" ) )

DocumentCatalog::DocumentCatalog(TQObject *, const char *, const TQStringList&)
 : KatapultCatalog()
{
	_minQueryLen = 1;
	currentPath = "";
	currentDirDoc = 0;
	queryMatched = 0;
	filesListed = false;
	files.setAutoDelete(true);
	ActionRegistry::self()->registerAction(new ActionOpenDocument());
}

DocumentCatalog::~DocumentCatalog()
{
}

void DocumentCatalog::initialize()
{
	dir = TQDir::home();
	refreshFolders();
	refreshFiles();
}

void DocumentCatalog::refreshFolders()
{
	folders.clear();
	files.clear();
	currentDirDoc = 0;
	filesListed = false;
	TQStringList folderNames = dir.entryList(TQDir::Dirs);
	for(TQStringList::Iterator it = folderNames.begin(); it != folderNames.end(); ++it) {
		TQString dirname = *it;
 		if(!dirname.startsWith("."))
			folders.append(dirname);
	}
}

void DocumentCatalog::refreshFiles()
{
	const TQFileInfoList *fileList = dir.entryInfoList();
	Directory *item = new Directory(currentPath, dir.absPath());
	currentDirDoc = item;
	files.append(item);

	for(TQPtrListStdIterator<TQFileInfo> it = fileList->begin(); it != fileList->end(); ++it) {
		const TQFileInfo *file = *it;
		if(!file->fileName().startsWith(".")) {
			if(file->isDir())
				files.append(new Directory(currentPath+file->fileName(), file->absFilePath()));
			else
				files.append(new Document(currentPath+file->fileName(), file->absFilePath(), _showPreview));
		}
	}
	filesListed = true;
}

void DocumentCatalog::queryChanged()
{
	int newStatus = 0;
	if(query() == "")
	{
		// reset query
		dir = TQDir::home();
		currentPath = "";
		queryMatched = 0;
		refreshFolders();
	} else {
		if(query().length() >= minQueryLen())
		{
			TQString path = query().lower().remove(0, queryMatched);
			
			int index;
			while((index = path.find('/')) != -1) {
				TQString folderQuery = path.left(index);
				TQString guess = TQString();
				
				for(TQStringList::Iterator it = folders.begin(); it != folders.end(); ++it) {
					TQString folderName = *it;
					if(folderName.lower().startsWith(folderQuery) && (guess.isNull() || folderName.length() < guess.length()))
						guess = folderName;
				}
				
				if(guess == TQString()) {
					path = TQString();
					break;
				}
				
				if(!dir.cd(guess)) {
					path = TQString();
					break;
				}
				refreshFolders();
				
				queryMatched += folderQuery.length() + 1;
				currentPath += guess + "/";
				path = path.remove(0, index+1);
			}
			
			Match newBestMatch;
			
			if(path.isNull()) {
				files.clear();
			} else {
				if(!filesListed) {
					refreshFiles();
				}
				if(!path.isEmpty()) {
					if(currentDirDoc != 0) {
						files.removeRef(currentDirDoc);
						currentDirDoc = 0;
					}
					TQPtrListIterator<Document> it(files);
					Document *document;
					while((document = it.current()) != 0) {
						++it;
						if(document->name().lower().startsWith(path)) {
							int rank = 100*query().length()/document->text().length();
							if(newBestMatch.isNull() || rank > newBestMatch.rank())
								newBestMatch = Match(document, rank, currentPath.length() + path.length());
						} else {
							files.removeRef(document);
						}
					}
				}
			}
			
			if(currentDirDoc != 0 && path.isEmpty())
				newBestMatch = Match(currentDirDoc, 100, currentPath.length());

			newStatus |= S_Active;
			if(files.count() > 0)
			{
				newStatus |= S_HasResults;
				if(files.count() > 1 || ( strcmp(files.at(0)->className(), "Directory") == 0 ) )
					newStatus |= S_Multiple;
			} else
				newStatus |= S_NoResults;
				
 			setBestMatch(newBestMatch);
		} else {
			setBestMatch(Match());
		}
	}
	setStatus(newStatus);
}

unsigned int DocumentCatalog::minQueryLen() const
{
	return _minQueryLen;
}

void DocumentCatalog::readSettings(TDEConfigBase *config)
{
	_minQueryLen = config->readUnsignedNumEntry("MinQueryLen", 1);
	_showPreview = config->readBoolEntry("showPreview", false);
}

void DocumentCatalog::writeSettings(TDEConfigBase *config)
{
	config->writeEntry("MinQueryLen", _minQueryLen);
	config->writeEntry("showPreview", _showPreview);
}

TQWidget * DocumentCatalog::configure()
{
	DocumentCatalogSettings *settings = new DocumentCatalogSettings();
	
	settings->minQueryLen->setValue(_minQueryLen);
	connect(settings->minQueryLen, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(minQueryLenChanged(int)));

	settings->showPreview->setChecked(_showPreview);
	connect(settings->showPreview, TQ_SIGNAL(toggled(bool)), this, TQ_SLOT(toggleshowPreview(bool)));
	
	return settings;
}

void DocumentCatalog::minQueryLenChanged(int _minQueryLen)
{
	this->_minQueryLen = _minQueryLen;
}

void DocumentCatalog::toggleshowPreview(bool _showPreview)
{
	this->_showPreview = _showPreview;

}
#include "documentcatalog.moc"
