/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-07-11
 * Description : digiKam image editor Color Balance tool.
 *
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqcheckbox.h>
#include <tqcolor.h>
#include <tqcombobox.h>
#include <tqframe.h>
#include <tqgroupbox.h>
#include <tqhbuttongroup.h>
#include <tqhgroupbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqpushbutton.h>
#include <tqslider.h>
#include <tqtooltip.h>
#include <tqvbox.h>
#include <tqvgroupbox.h>
#include <tqwhatsthis.h>

// KDE includes.

#include <tdeapplication.h>
#include <tdeconfig.h>
#include <kcursor.h>
#include <kiconloader.h>
#include <tdelocale.h>
#include <tdestandarddirs.h>

// LibKDcraw includes.

#include <libkdcraw/rnuminput.h>

// Digikam includes.

#include "colorgradientwidget.h"
#include "colormodifier.h"
#include "dimg.h"
#include "editortoolsettings.h"
#include "histogramwidget.h"
#include "imageiface.h"
#include "imagewidget.h"

// Local includes.

#include "rgbtool.h"
#include "rgbtool.moc"

using namespace KDcrawIface;
using namespace Digikam;

namespace DigikamImagesPluginCore
{

RGBTool::RGBTool(TQObject* parent)
       : EditorTool(parent)
{
    setName("colorbalance");
    setToolName(i18n("Color Balance"));
    setToolIcon(SmallIcon("adjustrgb"));

    m_destinationPreviewData = 0;

    m_previewWidget = new ImageWidget("colorbalance Tool", 0,
                                      i18n("<p>Here you can see the image "
                                           "color-balance adjustments preview. "
                                           "You can pick color on image "
                                           "to see the color level corresponding on histogram."));
    setToolView(m_previewWidget);

    // -------------------------------------------------------------

    m_gboxSettings = new EditorToolSettings(EditorToolSettings::Default|
                                            EditorToolSettings::Ok|
                                            EditorToolSettings::Cancel);

    TQGridLayout* gridSettings = new TQGridLayout(m_gboxSettings->plainPage(), 7, 4);

    TQLabel *label1 = new TQLabel(i18n("Channel:"), m_gboxSettings->plainPage());
    label1->setAlignment(TQt::AlignRight | TQt::AlignVCenter);
    m_channelCB = new TQComboBox(false, m_gboxSettings->plainPage());
    m_channelCB->insertItem(i18n("Luminosity"));
    m_channelCB->insertItem(i18n("Red"));
    m_channelCB->insertItem(i18n("Green"));
    m_channelCB->insertItem(i18n("Blue"));
    TQWhatsThis::add( m_channelCB, i18n("<p>Select the histogram channel to display here:<p>"
                                       "<b>Luminosity</b>: display the image's luminosity values.<p>"
                                       "<b>Red</b>: display the red image-channel values.<p>"
                                       "<b>Green</b>: display the green image-channel values.<p>"
                                       "<b>Blue</b>: display the blue image-channel values.<p>"));

    m_scaleBG = new TQHButtonGroup(m_gboxSettings->plainPage());
    m_scaleBG->setExclusive(true);
    m_scaleBG->setFrameShape(TQFrame::NoFrame);
    m_scaleBG->setInsideMargin(0);
    TQWhatsThis::add( m_scaleBG, i18n("<p>Select the histogram scale here.<p>"
                                     "If the image's maximal counts are small, you can use the linear scale.<p>"
                                     "Logarithmic scale can be used when the maximal counts are big; "
                                     "if it is used, all values (small and large) will be visible on the graph."));

    TQPushButton *linHistoButton = new TQPushButton(m_scaleBG);
    TQToolTip::add(linHistoButton, i18n("<p>Linear"));
    m_scaleBG->insert(linHistoButton, HistogramWidget::LinScaleHistogram);
    TDEGlobal::dirs()->addResourceType("histogram-lin", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
    TQString directory = TDEGlobal::dirs()->findResourceDir("histogram-lin", "histogram-lin.png");
    linHistoButton->setPixmap(TQPixmap(directory + "histogram-lin.png"));
    linHistoButton->setToggleButton(true);

    TQPushButton *logHistoButton = new TQPushButton(m_scaleBG);
    TQToolTip::add(logHistoButton, i18n("<p>Logarithmic"));
    m_scaleBG->insert(logHistoButton, HistogramWidget::LogScaleHistogram);
    TDEGlobal::dirs()->addResourceType("histogram-log", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
    directory = TDEGlobal::dirs()->findResourceDir("histogram-log", "histogram-log.png");
    logHistoButton->setPixmap(TQPixmap(directory + "histogram-log.png"));
    logHistoButton->setToggleButton(true);

    TQHBoxLayout* l1 = new TQHBoxLayout();
    l1->addWidget(label1);
    l1->addWidget(m_channelCB);
    l1->addStretch(10);
    l1->addWidget(m_scaleBG);

    gridSettings->addMultiCellLayout(l1, 0, 0, 0, 4);

    // -------------------------------------------------------------

    TQVBox *histoBox   = new TQVBox(m_gboxSettings->plainPage());
    m_histogramWidget = new HistogramWidget(256, 140, histoBox, false, true, true);
    TQWhatsThis::add( m_histogramWidget, i18n("<p>Here you can see the target preview image histogram drawing "
                                             "of the selected image channel. This one is re-computed at any "
                                             "settings changes."));
    TQLabel *space = new TQLabel(histoBox);
    space->setFixedHeight(1);
    m_hGradient = new ColorGradientWidget(ColorGradientWidget::Horizontal, 10, histoBox);
    m_hGradient->setColors(TQColor("black"), TQColor("white"));

    gridSettings->addMultiCellWidget(histoBox, 1, 2, 0, 4);

    // -------------------------------------------------------------

    TQLabel *labelLeft = new TQLabel(i18n("Cyan"), m_gboxSettings->plainPage());
    labelLeft->setAlignment ( TQt::AlignRight | TQt::AlignVCenter );
    m_rSlider = new TQSlider(-100, 100, 1, 0, TQt::Horizontal, m_gboxSettings->plainPage(), "m_rSlider");
    m_rSlider->setTickmarks(TQSlider::Below);
    m_rSlider->setTickInterval(20);
    TQWhatsThis::add( m_rSlider, i18n("<p>Set here the cyan/red color adjustment of the image."));
    TQLabel *labelRight = new TQLabel(i18n("Red"), m_gboxSettings->plainPage());
    labelRight->setAlignment ( TQt::AlignLeft | TQt::AlignVCenter );
    m_rInput = new RIntNumInput(m_gboxSettings->plainPage());
    m_rInput->setDefaultValue(0);
    m_rInput->input()->setRange(-100, 100, 1, false);

    gridSettings->addMultiCellWidget(labelLeft,     3, 3, 0, 0);
    gridSettings->addMultiCellWidget(m_rSlider,     3, 3, 1, 1);
    gridSettings->addMultiCellWidget(labelRight,    3, 3, 2, 2);
    gridSettings->addMultiCellWidget(m_rInput,      3, 3, 3, 3);

    // -------------------------------------------------------------

    labelLeft = new TQLabel(i18n("Magenta"), m_gboxSettings->plainPage());
    labelLeft->setAlignment(TQt::AlignRight | TQt::AlignVCenter);
    m_gSlider = new TQSlider(-100, 100, 1, 0, TQt::Horizontal, m_gboxSettings->plainPage(), "m_gSlider");
    m_gSlider->setTickmarks(TQSlider::Below);
    m_gSlider->setTickInterval(20);
    TQWhatsThis::add( m_gSlider, i18n("<p>Set here the magenta/green color adjustment of the image."));
    labelRight = new TQLabel(i18n("Green"), m_gboxSettings->plainPage());
    labelRight->setAlignment(TQt::AlignLeft | TQt::AlignVCenter);
    m_gInput = new RIntNumInput(m_gboxSettings->plainPage());
    m_gInput->setDefaultValue(0);
    m_gInput->input()->setRange(-100, 100, 1, false);

    gridSettings->addMultiCellWidget(labelLeft,     4, 4, 0, 0);
    gridSettings->addMultiCellWidget(m_gSlider,     4, 4, 1, 1);
    gridSettings->addMultiCellWidget(labelRight,    4, 4, 2, 2);
    gridSettings->addMultiCellWidget(m_gInput,      4, 4, 3, 3);

    // -------------------------------------------------------------

    labelLeft = new TQLabel(i18n("Yellow"), m_gboxSettings->plainPage());
    labelLeft->setAlignment ( TQt::AlignRight | TQt::AlignVCenter );
    m_bSlider = new TQSlider(-100, 100, 1, 0, TQt::Horizontal, m_gboxSettings->plainPage(), "m_bSlider");
    m_bSlider->setTickmarks(TQSlider::Below);
    m_bSlider->setTickInterval(20);
    TQWhatsThis::add( m_bSlider, i18n("<p>Set here the yellow/blue color adjustment of the image."));
    labelRight = new TQLabel(i18n("Blue"), m_gboxSettings->plainPage());
    labelRight->setAlignment(TQt::AlignLeft | TQt::AlignVCenter);
    m_bInput = new RIntNumInput(m_gboxSettings->plainPage());
    m_bInput->setDefaultValue(0);
    m_bInput->input()->setRange(-100, 100, 1, false);

    gridSettings->addMultiCellWidget(labelLeft,     5, 5, 0, 0);
    gridSettings->addMultiCellWidget(m_bSlider,     5, 5, 1, 1);
    gridSettings->addMultiCellWidget(labelRight,    5, 5, 2, 2);
    gridSettings->addMultiCellWidget(m_bInput,      5, 5, 3, 3);

    m_rInput->setValue(0);
    m_gInput->setValue(0);
    m_bInput->setValue(0);

    gridSettings->setRowStretch(6, 10);
    setToolSettings(m_gboxSettings);
    init();

    // -------------------------------------------------------------

    connect(m_channelCB, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotChannelChanged(int)));

    connect(m_scaleBG, TQ_SIGNAL(released(int)),
            this, TQ_SLOT(slotScaleChanged(int)));

    connect(m_previewWidget, TQ_SIGNAL(spotPositionChangedFromTarget( const Digikam::DColor &, const TQPoint & )),
            this, TQ_SLOT(slotColorSelectedFromTarget( const Digikam::DColor & )));

    connect(m_rSlider, TQ_SIGNAL(valueChanged(int)),
            m_rInput, TQ_SLOT(setValue(int)));
    connect(m_rInput, TQ_SIGNAL(valueChanged (int)),
            m_rSlider, TQ_SLOT(setValue(int)));
    connect(m_rInput, TQ_SIGNAL(valueChanged (int)),
            this, TQ_SLOT(slotTimer()));

    connect(m_gSlider, TQ_SIGNAL(valueChanged(int)),
            m_gInput, TQ_SLOT(setValue(int)));
    connect(m_gInput, TQ_SIGNAL(valueChanged (int)),
            m_gSlider, TQ_SLOT(setValue(int)));
    connect(m_gInput, TQ_SIGNAL(valueChanged (int)),
            this, TQ_SLOT(slotTimer()));

    connect(m_bSlider, TQ_SIGNAL(valueChanged(int)),
            m_bInput, TQ_SLOT(setValue(int)));
    connect(m_bInput, TQ_SIGNAL(valueChanged (int)),
            m_bSlider, TQ_SLOT(setValue(int)));
    connect(m_bInput, TQ_SIGNAL(valueChanged (int)),
            this, TQ_SLOT(slotTimer()));

    connect(m_previewWidget, TQ_SIGNAL(signalResized()),
            this, TQ_SLOT(slotEffect()));

    // -------------------------------------------------------------

    m_gboxSettings->enableButton(EditorToolSettings::Ok, false);
}

RGBTool::~RGBTool()
{
    if (m_destinationPreviewData)
       delete [] m_destinationPreviewData;
}

void RGBTool::slotChannelChanged(int channel)
{
    switch (channel)
    {
        case LuminosityChannel:
            m_histogramWidget->m_channelType = HistogramWidget::ValueHistogram;
            m_hGradient->setColors(TQColor("black"), TQColor("white"));
            break;

        case RedChannel:
            m_histogramWidget->m_channelType = HistogramWidget::RedChannelHistogram;
            m_hGradient->setColors(TQColor("black"), TQColor("red"));
            break;

        case GreenChannel:
            m_histogramWidget->m_channelType = HistogramWidget::GreenChannelHistogram;
            m_hGradient->setColors(TQColor("black"), TQColor("green"));
            break;

        case BlueChannel:
            m_histogramWidget->m_channelType = HistogramWidget::BlueChannelHistogram;
            m_hGradient->setColors(TQColor("black"), TQColor("blue"));
            break;
    }

    m_histogramWidget->repaint(false);
}

void RGBTool::slotScaleChanged(int scale)
{
    m_histogramWidget->m_scaleType = scale;
    m_histogramWidget->repaint(false);
}

void RGBTool::slotColorSelectedFromTarget(const DColor &color)
{
    m_histogramWidget->setHistogramGuideByColor(color);
}

void RGBTool::readSettings()
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("colorbalance Tool");
    m_channelCB->setCurrentItem(config->readNumEntry("Histogram Channel", 0));    // Luminosity.
    m_scaleBG->setButton(config->readNumEntry("Histogram Scale", HistogramWidget::LogScaleHistogram));
    int r = config->readNumEntry("RedAjustment", m_rInput->defaultValue());
    int g = config->readNumEntry("GreenAjustment", m_gInput->defaultValue());
    int b = config->readNumEntry("BlueAjustment", m_bInput->defaultValue());
    adjustSliders(r, g, b);
    m_histogramWidget->reset();
    slotChannelChanged(m_channelCB->currentItem());
    slotScaleChanged(m_scaleBG->selectedId());
}

void RGBTool::writeSettings()
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("colorbalance Tool");
    config->writeEntry("Histogram Channel", m_channelCB->currentItem());
    config->writeEntry("Histogram Scale", m_scaleBG->selectedId());
    config->writeEntry("RedAjustment", m_rSlider->value());
    config->writeEntry("GreenAjustment", m_gInput->value());
    config->writeEntry("BlueAjustment", m_bInput->value());
    m_previewWidget->writeSettings();
    config->sync();
}

void RGBTool::slotResetSettings()
{
    int r = m_rInput->defaultValue();
    int g = m_gInput->defaultValue();
    int b = m_bInput->defaultValue();

    adjustSliders(r, g, b);
}

void RGBTool::adjustSliders(int r, int g, int b)
{
    m_rSlider->blockSignals(true);
    m_gSlider->blockSignals(true);
    m_bSlider->blockSignals(true);
    m_rInput->blockSignals(true);
    m_gInput->blockSignals(true);
    m_bInput->blockSignals(true);

    m_rSlider->setValue(r);
    m_gSlider->setValue(g);
    m_bSlider->setValue(b);
    m_rInput->setValue(r);
    m_gInput->setValue(g);
    m_bInput->setValue(b);

    m_rSlider->blockSignals(false);
    m_gSlider->blockSignals(false);
    m_bSlider->blockSignals(false);
    m_rInput->blockSignals(false);
    m_gInput->blockSignals(false);
    m_bInput->blockSignals(false);

    slotEffect();
}

void RGBTool::slotEffect()
{
    tdeApp->setOverrideCursor( KCursor::waitCursor() );

    m_gboxSettings->enableButton(EditorToolSettings::Ok,
                                 (m_rInput->value() != 0 ||
                                  m_gInput->value() != 0 ||
                                  m_bInput->value() != 0));

    m_histogramWidget->stopHistogramComputation();

    if (m_destinationPreviewData)
       delete [] m_destinationPreviewData;

    ImageIface* iface = m_previewWidget->imageIface();
    m_destinationPreviewData   = iface->getPreviewImage();
    int w                      = iface->previewWidth();
    int h                      = iface->previewHeight();
    bool alpha                 = iface->previewHasAlpha();
    bool sixteenBit            = iface->previewSixteenBit();

    double r = ((double) m_rInput->value() + 100.0) / 100.0;
    double g = ((double) m_gInput->value() + 100.0) / 100.0;
    double b = ((double) m_bInput->value() + 100.0) / 100.0;
    double a = 1.0;

    DImg preview(w, h, sixteenBit, alpha, m_destinationPreviewData);
    ColorModifier cmod;
    cmod.applyColorModifier(preview, r, g, b, a);
    iface->putPreviewImage(preview.bits());

    m_previewWidget->updatePreview();

    // Update histogram.

    memcpy(m_destinationPreviewData, preview.bits(), preview.numBytes());
    m_histogramWidget->updateData(m_destinationPreviewData, w, h, sixteenBit, 0, 0, 0, false);

    tdeApp->restoreOverrideCursor();
}

void RGBTool::finalRendering()
{
    tdeApp->setOverrideCursor( KCursor::waitCursor() );

    double r = ((double) m_rInput->value() + 100.0) / 100.0;
    double g = ((double) m_gInput->value() + 100.0) / 100.0;
    double b = ((double) m_bInput->value() + 100.0) / 100.0;
    double a = 1.0;

    ImageIface* iface = m_previewWidget->imageIface();
    uchar *data                = iface->getOriginalImage();
    int w                      = iface->originalWidth();
    int h                      = iface->originalHeight();
    bool alpha                 = iface->originalHasAlpha();
    bool sixteenBit            = iface->originalSixteenBit();
    DImg original(w, h, sixteenBit, alpha, data);
    delete [] data;

    ColorModifier cmod;
    cmod.applyColorModifier(original, r, g, b, a);

    iface->putOriginalImage(i18n("Color Balance"), original.bits());
    tdeApp->restoreOverrideCursor();
}

}  // NameSpace DigikamImagesPluginCore

