/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-06-26
 * Description : Drag object info container
 * 
 * Copyright (C) 2004 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2004 by Joern Ahrens <joern.ahrens@kdemail.net>
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// Local includes.

#include "ddebug.h"
#include "dragobjects.h"

namespace Digikam
{

ItemDrag::ItemDrag(const KURL::List &urls,
                   const KURL::List &kioURLs,
                   const TQValueList<int>& albumIDs,
                   const TQValueList<int>& imageIDs,
                   TQWidget* dragSource, const char* name)
    : KURLDrag(urls, dragSource, name),
      m_kioURLs(kioURLs),
      m_albumIDs(albumIDs), m_imageIDs(imageIDs)
{
}

bool ItemDrag::canDecode(const TQMimeSource* e)
{
    return e->provides("digikam/item-ids")  || 
           e->provides("digikam/album-ids") ||
           e->provides("digikam/image-ids") ||
           e->provides("digikam/digikamalbums");
}

bool ItemDrag::decode(const TQMimeSource* e, KURL::List &urls, KURL::List &kioURLs,
                      TQValueList<int>& albumIDs, TQValueList<int>& imageIDs)
{
    urls.clear();
    kioURLs.clear();
    albumIDs.clear();
    imageIDs.clear();

    if (KURLDrag::decode(e, urls))
    {
        TQByteArray albumarray = e->encodedData("digikam/album-ids");
        TQByteArray imagearray = e->encodedData("digikam/image-ids");
        TQByteArray kioarray = e->encodedData("digikam/digikamalbums");

        if (albumarray.size() && imagearray.size() && kioarray.size())
        {
            int id;

            TQDataStream dsAlbums(albumarray, IO_ReadOnly);
            while (!dsAlbums.atEnd())
            {
                dsAlbums >> id;
                albumIDs.append(id);
            }

            TQDataStream dsImages(imagearray, IO_ReadOnly);
            while (!dsImages.atEnd())
            {
                dsImages >> id;
                imageIDs.append(id);
            }

            KURL u;
            TQDataStream dsKio(kioarray, IO_ReadOnly);
            while (!dsKio.atEnd())
            {
                dsKio >> u;
                kioURLs.append(u);
            }

            return true;
        }
    }

    return false;
}

TQByteArray ItemDrag::encodedData(const char* mime) const
{
    TQCString mimetype(mime);

    if (mimetype == "digikam/album-ids")
    {
        TQByteArray byteArray;
        TQDataStream ds(byteArray, IO_WriteOnly);

        TQValueList<int>::const_iterator it;
        for (it = m_albumIDs.begin(); it != m_albumIDs.end(); ++it)
        {
            ds << (*it);
        }

        return byteArray;
    }
    else if (mimetype == "digikam/image-ids")
    {
        TQByteArray byteArray;
        TQDataStream ds(byteArray, IO_WriteOnly);

        TQValueList<int>::const_iterator it;
        for (it = m_imageIDs.begin(); it != m_imageIDs.end(); ++it)
        {
            ds << (*it);
        }

        return byteArray;
    }
    else if (mimetype == "digikam/digikamalbums")
    {
        TQByteArray byteArray;
        TQDataStream ds(byteArray, IO_WriteOnly);

        KURL::List::const_iterator it;
        for (it = m_kioURLs.begin(); it != m_kioURLs.end(); ++it)
        {
            ds << (*it);
        }

        return byteArray;
    }
    else
    {
        return KURLDrag::encodedData(mime);
    }
}

const char* ItemDrag::format(int i) const
{
    if (i == 0)
        return "text/uri-list";
    else if (i == 1)
        return "digikam/item-ids";
    else if (i == 2)
        return "digikam/album-ids";
    else if (i == 3)
        return "digikam/image-ids";
    else if (i == 4)
        return "digikam/digikamalbums";
    else
        return 0;
}

// ------------------------------------------------------------------------

TagDrag::TagDrag( int albumid, TQWidget *dragSource, 
                  const char *name ) :
    TQDragObject( dragSource, name )
{
    mAlbumID = albumid;
}

bool TagDrag::canDecode( const TQMimeSource* e )
{
    return e->provides("digikam/tag-id");
}

const char* TagDrag::format( int i ) const
{
    if ( i == 0 )
        return "digikam/tag-id";

    return 0;
}

TQByteArray TagDrag::encodedData( const char* ) const
{
    TQByteArray ba;
    TQDataStream ds(ba, IO_WriteOnly);
    ds << mAlbumID;
    return ba;
}

// ------------------------------------------------------------------------

AlbumDrag::AlbumDrag(const KURL &url, int albumid, 
                     TQWidget *dragSource, 
                     const char *name) :
    KURLDrag(url, dragSource, name)
{
    mAlbumID = albumid;
}

bool AlbumDrag::canDecode( const TQMimeSource* e )
{
    return e->provides("digikam/album-id");
}

const char* AlbumDrag::format( int i ) const
{
    if (i == 0)
        return "text/uri-list";
    else if ( i == 1 )
        return "digikam/album-id";

    return 0;
}

TQByteArray AlbumDrag::encodedData(const char *mime) const
{
    TQCString mimetype( mime );
    if(mimetype == "digikam/album-id")
    {
        TQByteArray ba;
        TQDataStream ds(ba, IO_WriteOnly);
        ds << mAlbumID;
        return ba;
    }
    else
    {
        return KURLDrag::encodedData(mime);
    }
}

bool AlbumDrag::decode(const TQMimeSource* e, KURL::List &urls, 
                       int &albumID)
{
    urls.clear();
    albumID = -1;

    if(KURLDrag::decode(e, urls))
    {
        TQByteArray ba = e->encodedData("digikam/album-id");
        if (ba.size())
        {
            TQDataStream ds(ba, IO_ReadOnly);
            if(!ds.atEnd())
            {
                ds >> albumID;
            }
            return true;
        }
    }

    return false;
}

// ------------------------------------------------------------------------

TagListDrag::TagListDrag(const TQValueList<int>& tagIDs, TQWidget *dragSource,
                         const char *name)
           : TQDragObject( dragSource, name )
{
    m_tagIDs = tagIDs;
}

bool TagListDrag::canDecode(const TQMimeSource* e)
{
    return e->provides("digikam/taglist");
}

TQByteArray TagListDrag::encodedData(const char*) const
{
    TQByteArray ba;
    TQDataStream ds(ba, IO_WriteOnly);
    ds << m_tagIDs;
    return ba;
}

const char* TagListDrag::format(int i) const
{
    if ( i == 0 )
        return "digikam/taglist";

    return 0;
}

// ------------------------------------------------------------------------

CameraItemListDrag::CameraItemListDrag(const TQStringList& cameraItemPaths,
                                       TQWidget *dragSource,
                                       const char *name)
                  : TQDragObject( dragSource, name )
{
    m_cameraItemPaths = cameraItemPaths;
}

bool CameraItemListDrag::canDecode(const TQMimeSource* e)
{
    return e->provides("digikam/cameraItemlist");
}

TQByteArray CameraItemListDrag::encodedData(const char*) const
{
    TQByteArray ba;
    TQDataStream ds(ba, IO_WriteOnly);
    ds << m_cameraItemPaths;
    return ba;
}

const char* CameraItemListDrag::format(int i) const
{
    if ( i == 0 )
        return "digikam/cameraItemlist";

    return 0;
}

// ------------------------------------------------------------------------

CameraDragObject::CameraDragObject(const CameraType& ctype, TQWidget *dragSource)
                : TQStoredDrag("camera/unknown", dragSource)
{
    setCameraType(ctype);
}

CameraDragObject::~CameraDragObject()
{
}

void CameraDragObject::setCameraType(const CameraType& ctype)
{
    TQByteArray byteArray;
    TQDataStream ds(byteArray, IO_WriteOnly);

    ds << ctype.title();
    ds << ctype.model();
    ds << ctype.port();
    ds << ctype.path();
    ds << ctype.lastAccess();

    setEncodedData(byteArray);
}

bool CameraDragObject::canDecode(const TQMimeSource* e)
{
    return e->provides("camera/unknown");
}

bool CameraDragObject::decode(const TQMimeSource* e, CameraType& ctype)
{
    TQByteArray payload = e->encodedData("camera/unknown");
    if (payload.size()) 
    {
        TQString   title, model, port, path;
        TQDateTime lastAccess;

        TQDataStream ds(payload, IO_ReadOnly);
        ds >> title;
        ds >> model;
        ds >> port;
        ds >> path;
        ds >> lastAccess;

        ctype = CameraType(title, model, port, path, lastAccess);

        return true;
    }
    else
        return false;
}

}  // namespace Digikam
