/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef NOTEFACTORY_H
#define NOTEFACTORY_H

#include <tqevent.h>
#include <kurl.h>
#include <tqstringlist.h>

class TQString;
class TQPixmap;
class TQColor;

class Basket;
class Note;

/** Factory class to create (new, drop, past) or load BasketIem, and eventuelly save them (?)
  * @author Sbastien Laot
  */
namespace NoteFactory
{
	/** Functions to create a new note from a content.
	  * Content, if any, is saved to file but the note is not insterted in the basket, and the basket is not saved.
	  * Return 0 if the note has not been successfully created.
	  * In some cases, the returned note can be a group containing several notes or the first note of a chained list.
	  * The method Basket::TODO() can insert several grouped or chained notes without problem.
	  */
	Note* createNoteText(     const TQString &text,     Basket *parent, bool reallyPlainText = false);
	Note* createNoteHtml(     const TQString &html,     Basket *parent);
	Note* createNoteLink(     const KURL    &url,      Basket *parent);
	Note* createNoteLink(     const KURL    &url,      const TQString &title, Basket *parent);
	Note* createNoteImage(    const TQPixmap &image,    Basket *parent);
	Note* createNoteColor(    const TQColor  &color,    Basket *parent);
	Note* createNoteFromText( const TQString &content,  Basket *parent); // Find automatically the type from the text meaning  // TODO: Return Note::List?
	Note* createNoteLauncher( const KURL    &url,      Basket *parent);
	Note* createNoteLauncher( const TQString &command,  const TQString &name, const TQString &icon, Basket *parent);
	Note* createNoteUnknown(  TQMimeSource *source,     Basket *parent);
	/** Functions to create derived notes from a content */
	Note* createNoteLinkOrLauncher( const KURL &url,   Basket *parent);
	Note* copyFileAndLoad(    const KURL &url,         Basket *parent);
	Note* moveFileAndLoad(    const KURL &url,         Basket *parent);
	Note* loadFile(           const TQString &fileName, Basket *parent); /// << Determine the content of the file (the file SHOULD exists) and return a note of the good type.
	Note* loadFile(           const TQString &fileName, NoteType::Id type, Basket *parent ); /// <<  Create a note of type @p type. The file is not obliged to exist.
	/** Functions to create a new note from a drop or past event */
	Note* dropNote(TQMimeSource *source, Basket *parent,
	               bool fromDrop = false, TQDropEvent::Action action = TQDropEvent::Copy, Note *noteSource = 0);
	bool movingNotesInTheSameBasket(TQMimeSource *source, Basket *parent, TQDropEvent::Action action);
	Note* dropURLs(KURL::List urls, Basket *parent, TQDropEvent::Action action, bool fromDrop);
	Note* decodeContent(TQDataStream &stream, NoteType::Id type, Basket *parent); /// << Decode the @p stream to a note or return 0 if a general loadFile() is sufficient.
	void consumeContent(TQDataStream &stream, NoteType::Id type); /// << Decode the @p stream to a note or return 0 if a general loadFile() is sufficient.
	/** Functions to create a note file but not load it in a note object */
	TQString createNoteLauncherFile(const TQString &command, const TQString &name, const TQString &icon, Basket *parent);
	/** Other useful functions */
	NoteType::Id typeForURL(const KURL &url, Basket *parent);
	bool         maybeText(const KURL &url);
	bool         maybeHtml(const KURL &url);
	bool         maybeImageOrAnimation(const KURL &url);
	bool         maybeAnimation(const KURL &url);
	bool         maybeSound(const KURL &url);
	bool         maybeLauncher(const KURL &url);
	TQString      fileNameForNewNote(Basket *parent, const TQString &wantedName);
	TQString      createFileForNewNote(Basket *parent, const TQString &extension, const TQString &wantedName = "");
	KURL         filteredURL(const KURL &url);
	TQString      titleForURL(const KURL &url);
	TQString      iconForURL(const KURL &url);
	TQString      iconForCommand(const TQString &command);
	bool         isIconExist(const TQString &icon);
	TQStringList  textToURLList(const TQString &text); // @Return { url1, title1, url2, title2, url3, title3... }
	/** Insert GUI menu */
	Note* createEmptyNote(  NoteType::Id type, Basket *parent ); // Insert empty if of type Note::Type
	Note* importKMenuLauncher(Basket *parent);
	Note* importIcon(Basket *parent);
	Note* importFileContent(Basket *parent);
}

#endif // NOTEFACTORY_H
