//
// Copyright 2025 Pixar
//
// Licensed under the terms set forth in the LICENSE.txt file available at
// https://openusd.org/license.
//
#ifndef PXR_EXEC_EXEC_COMPUTATION_BUILDERS_H
#define PXR_EXEC_EXEC_COMPUTATION_BUILDERS_H

/// \file
///
/// This is a public header, but many of the symbols have private names because
/// they are not intended for direct use by client code. The public API here is
/// accessed by client code via the 'self' parameter generated by the
/// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA macro. The documentation is set up to highlight all
/// relevant public details.
///

#include "pxr/pxr.h"

#include "pxr/exec/exec/api.h"
#include "pxr/exec/exec/builtinComputations.h"
#include "pxr/exec/exec/providerResolution.h"
#include "pxr/exec/exec/typeRegistry.h"
#include "pxr/exec/exec/types.h"

#include "pxr/base/tf/token.h"
#include "pxr/base/tf/type.h"
#include "pxr/base/vt/traits.h"
#include "pxr/exec/vdf/context.h"
#include "pxr/usd/sdf/path.h"

#include <memory>
#include <type_traits>
#include <utility>

PXR_NAMESPACE_OPEN_SCOPE

struct Exec_InputKey;


/// \defgroup group_Exec_ComputationDefinitionLanguage Computation Definition Language
///
/// Plugin computations are defined using the domain-specific **Computation
/// Definition Language**.
///
/// Each plugin computation is registered for a particular schema, either typed
/// or applied. When a computation is requested on a provider prim or attribute,
/// if the requested computation name is not a [builtin
/// computation](#group_Exec_Builtin_Computations) name, exec compilation
/// considers the computations registered for the typed schema for the prim, the
/// ancestor schema types, and API schemas applied to the prim, and looks for a
/// computation of the requested name.
///
/// To define computations for a schema, the plugin code must invoke the
/// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA() macro. The macro invocation is
/// immediately followed by a block of code that uses [computation
/// registrations](#group_Exec_ComputationRegistrations) that registers the
/// associated plugin computations. Most of the language is dedicated to
/// expressing [input registrations](#group_Exec_InputRegistrations), which
/// provide exec compilation with the information it needs to compile the input
/// connections that supply input values when the network is evaluated.
///
/// # Example
///
/// The following cpp file could be used in a plugin library to define the
/// `computeMyAttributeValue` prim computation for the `MySchemaType` schema:
///
/// ```{.cpp}
/// #include "pxr/exec/exec/registerSchema.h"
/// #include "pxr/exec/vdf/context.h"
///
/// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
/// {
///     // Register a prim computation that returns the computed value of an
///     // attribute.
///     self.PrimComputation(_tokens->computeMyAttributeValue)
///         .Callback<double>(+[](const VdfContext &ctx) {
///             ctx->SetOutput(ctx.GetInputValue<double>(_tokens->myAttribute));
///         })
///         .Inputs(
///             AttributeValue<double>(_tokens->myAttribute).Required());
/// }
/// ```
///
/// The library's `plugInfo.json` must contain the following data in the `Info`
/// block in order for the execution system to load the library when
/// computations are requested on a prim that uses `MySchemaType`:
///
/// ```
///     "Info": {
///         "Exec": {
///             "Schemas": {
///                 "MyComputationalSchema1": {
///                     "allowsPluginComputations": true
///                 },
///                 "MyComputationalSchema2": {
///                 },
///                 "MyNonComputationalSchema": {
///                     "allowsPluginComputations": false
///                 }
///             }
///         }
///     }
/// ```
///
/// The boolean `allowsPluginComputations` is used to declare schemas for which
/// computations _cannot_ be registered. If `allowsPluginComputations` isn't
/// present in the plugInfo, its value defaults to true. I.e., schemas that
/// appear in the Exec/Schemas plugInfo allow plugin computations by default.

/// \defgroup group_Exec_ComputationRegistrations Computation Registrations
///
/// Computation registrations initiate the process of defining computations. The
/// object returned by a computation registration has methods that are used to
/// specify the callback that implements the computation and the inputs that are
/// provided to the callback at evaluation time.
/// 
/// \ingroup group_Exec_ComputationDefinitionLanguage

/// \defgroup group_Exec_InputRegistrations Input Registrations
///
/// An **input registration** is a specification of how an input value will be
/// provided to a computation callback at evaluation time.
///
/// An input registration is a sequence of:
/// - zero or more [object accessors](#group_Exec_Accessors), which provide
///   access to one or more scene objects that act as computation providers, and
///   which _must_ be followed by:
/// - exactly one [value specifier](#group_Exec_ValueSpecifiers), which request
///   a value from the computation provider(s), and which _may_ be followed
///   by:
/// - zero or more [input options](#group_Exec_InputOptions), which modify the
///   behavior of the resulting input registration.
/// 
/// For convenience, certain object accessor/value specifier/input option
/// sequences may be replaced by an [alias](#group_Exec_Aliases), which
/// compactly represents a compound input registration.
///
/// \ingroup group_Exec_ComputationDefinitionLanguage

/// \defgroup group_Exec_Accessors Object Accessors
///
/// **Object accessors** provide access to computation providers, the scene
/// objects from which input values are requested. An [input
/// registration](#group_Exec_InputRegistrations) starts with a sequence of zero
/// or more accessors. If no accessor is present, the origin object, the object
/// that owns the consuming computation, is the provider. Otherwise, starting
/// from that object, the sequence of accessors describes hops through namespace
/// that end at the computation provider.
///
/// A sequence of object accessors does not fully specify an input, however. The
/// sequence _must_ be followed by exactly one [value
/// specifier](#group_Exec_ValueSpecifiers) to fully specify an input
/// registration.
///
/// \ingroup group_Exec_InputRegistrations

/// \defgroup group_Exec_ValueSpecifiers Value Specifiers
/// 
/// A **value specifier** is an element of an [input
/// registration](#group_Exec_InputRegistrations) that identifies the value that
/// is requested from a given computation provider.
///
/// Each computation input registration must contain exactly one value
/// specifier. A value specifier comes after a sequence of zero or more [object
/// accessors](#group_Exec_Accessors), which determine the provider. A value
/// specifier may be followed by one or more [input
/// options](#group_Exec_InputOptions).
///
/// \ingroup group_Exec_InputRegistrations

/// \defgroup group_Exec_InputOptions Input Options
///
/// An **input option** is an element of an [input
/// registration](#group_Exec_InputRegistrations) that applies to a [value
/// specifier](#group_Exec_ValueSpecifiers), modifying its behavior.
///
/// A value specifier may be followed by zero or more input options.
///
/// \ingroup group_Exec_InputRegistrations

/// \defgroup group_Exec_Aliases Aliases
///
/// Aliases are compact representations of compound [input
/// registrations](#group_Exec_InputRegistrations), combining one or more
/// [object accessors](#group_Exec_Accessors) with a [value
/// specifier](#group_Exec_ValueSpecifiers) into a single input registration.
///
/// \ingroup group_Exec_InputRegistrations


/// An enum that is used as a template parameter to specify which kinds of
/// providers a given input registration is allowed to be used on.
/// 
enum class Exec_ComputationBuilderProviderTypes: unsigned char
{
    None = 0,
    Prim = 1 << 0,
    Attribute = 1 << 1,
    Relationship = 1 << 2,
    Property = Attribute | Relationship,
    Any = 0xff
};

constexpr bool operator&(
    const Exec_ComputationBuilderProviderTypes a,
    const Exec_ComputationBuilderProviderTypes b)
{
    return static_cast<unsigned char>(a) & static_cast<unsigned char>(b);
}


/// Untemplated value specifier base class.
///
/// This class builds up an Exec_InputKey that specifies how to source an input
/// value at exec compilation time.
///
class Exec_ComputationBuilderValueSpecifierBase
{
public:
    EXEC_API
    Exec_ComputationBuilderValueSpecifierBase(
        const TfToken &computationName,
        TfType resultType,
        ExecProviderResolution &&providerResolution,
        const TfToken &inputName,
        bool fallsBackToDispatched);

    EXEC_API
    Exec_ComputationBuilderValueSpecifierBase(
        const Exec_ComputationBuilderValueSpecifierBase&);

    EXEC_API
    ~Exec_ComputationBuilderValueSpecifierBase();

protected:
    EXEC_API
    void _SetInputName(const TfToken &inputName);

    EXEC_API
    void _SetOptional (const bool optional);

    EXEC_API
    void _SetFallsBackToDispatched(bool fallsBackToDispatched);

private:
    // Only computation builders can get the input key.
    friend class Exec_PrimComputationBuilder;

    EXEC_API
    void _GetInputKey(Exec_InputKey *inputKey) const;

private:
    // We PIMPL the data for this class to avoid exposing more private details
    // in this public header.
    struct _Data;
    const std::unique_ptr<_Data> _data;
};

/// A value specifier that requests the value of a computation.
///
/// The template parameter determines which types of providers the input
/// registration is allowed to be used on.
/// 
template <Exec_ComputationBuilderProviderTypes allowed>
struct Exec_ComputationBuilderComputationValueSpecifier
    : public Exec_ComputationBuilderValueSpecifierBase
{
    Exec_ComputationBuilderComputationValueSpecifier(
        const TfToken &computationName,
        const TfType resultType,
        ExecProviderResolution &&providerResolution,
        const bool fallsBackToDispatched = false)
        : Exec_ComputationBuilderValueSpecifierBase(
            computationName, resultType,
            std::move(providerResolution),
            computationName /* inputName */,
            fallsBackToDispatched)
    {
    }
    
    using This = Exec_ComputationBuilderComputationValueSpecifier<allowed>;

    static constexpr Exec_ComputationBuilderProviderTypes
        allowedProviders = allowed;

    /// \addtogroup group_Exec_InputOptions
    /// @{

    /// Overrides the default input name, setting it to \p inputName.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that returns the value of another
    ///     // prim computation, using a non-default input name.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<double>(+[](const VdfContext &ctx) {
    ///             const double *const valuePtr =
    ///                 ctx.GetInputValuePtr<double>(_tokens->myInputName);
    ///             return valuePtr ? *valuePtr : 0.0;
    ///         })
    ///         .Inputs(
    ///             Computation<double>(_tokens->anotherComputation)
    ///                 .InputName(_tokens->myInputName));
    /// }
    /// ```
    ///
    This&
    InputName(const TfToken &inputName)
    { 
        _SetInputName(inputName);
        return *this;
    }

    /// Declares the input is required, i.e., that the computation expects an
    /// input value always to be provided at evaluation time.
    ///
    /// If exec compilation is unable to compile input connections for a
    /// required input, an error will be emitted.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that returns the value of another
    ///     // prim computation, using a non-default input name.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<int>(+[](const VdfContext &ctx) {
    ///             return ctx.GetInputValue<int>(_tokens->myInputName);
    ///         })
    ///         .Inputs(
    ///             Computation<int>(_tokens->anotherComputation).Required());
    /// }
    /// ```
    ///
    This&
    Required()
    {
        _SetOptional(false);
        return *this;
    }

    /// Declares the input can find dispatched computations *if* the requested
    /// computation name doesn't match a local computation on the provider.
    ///
    /// \see [DispatchedPrimComputation](#Exec_ComputationBuilder::DispatchedPrimComputation)
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a dispatched prim computation.
    ///     self.DispatchedPrimComputation(_tokens->myDispatchedComputation)
    ///         .Callback<double>(+[](const VdfContext &) { return 11.0; })
    ///
    ///     // Register a prim computation that requests the above dispatched
    ///     // computation via uses relationship targets.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<double>(+[](const VdfContext &ctx) {
    ///             const double *const valuePtr =
    ///                 ctx.GetInputValuePtr<double>(
    ///                     _tokens->myDispatchedComputation);
    ///             return valuePtr ? *valuePtr : -1.0;
    ///         })
    ///         .Inputs(
    ///             Relationship(_tokens->myRelationship)
    ///                 .TargetedObjects<double>(
    ///                     _tokens->myDispatchedComputation)
    ///                 .FallsBackToDispatched())
    /// }
    /// ```
    ///
    This&
    FallsBackToDispatched()
    {
        _SetFallsBackToDispatched(true);
        return *this;
    }

    /// @}
};


/// Untemplated object accessor base class.
struct Exec_ComputationBuilderAccessorBase {
    Exec_ComputationBuilderAccessorBase(const SdfPath &localTraversal)
        : _localTraversal(localTraversal)
    {
    }

protected:
    const SdfPath &_GetLocalTraversal() const {
        return _localTraversal;
    }

private:
    // The relative path used for the first phase of provider resolution.
    SdfPath _localTraversal;
};

/// Accessor common to all scene object types that support computing
/// computations on the object.
///
/// This class is templated in order to classify accessors that are allowed as
/// inputs for prim computations vs attribute computations.
///
template <Exec_ComputationBuilderProviderTypes allowed>
struct Exec_ComputationBuilderAccessor
    : public Exec_ComputationBuilderAccessorBase
{
    Exec_ComputationBuilderAccessor(const SdfPath &localTraversal)
        : Exec_ComputationBuilderAccessorBase(localTraversal)
    {
    }

    using ValueSpecifier =
        Exec_ComputationBuilderComputationValueSpecifier<allowed>;

    /// \addtogroup group_Exec_ValueSpecifiers
    /// @{

    /// Requests an input value from the computation \p computationName of type
    /// \p ResultType.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that returns the value of another
    ///     // prim computation.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<double>(+[](const VdfContext &ctx) {
    ///             const double *const valuePtr =
    ///                 ctx.GetInputValuePtr<double>(_tokens->sourceComputation);
    ///             return valuePtr ? *valuePtr : 0.0;
    ///         })
    ///         .Inputs(
    ///             Computation<double>(_tokens->sourceComputation)
    /// }
    /// ```
    ///
    template <typename ResultType>
    ValueSpecifier
    Computation(const TfToken &computationName)
    {
        static_assert(!VtIsArray<ResultType>::value,
                      "VtArray is not a supported result type");

        return ValueSpecifier(
            computationName,
            ExecTypeRegistry::GetInstance().CheckForRegistration<ResultType>(),
            {_GetLocalTraversal(),
             ExecProviderResolution::DynamicTraversal::Local});
    }

    // XXX:TODO
    // Also: Metadata

    /// @}
};

/// Property accessor
template <Exec_ComputationBuilderProviderTypes allowed>
struct Exec_ComputationBuilderPropertyAccessor
    : public Exec_ComputationBuilderAccessor<allowed>
{
    Exec_ComputationBuilderPropertyAccessor(const SdfPath &localTraversal)
        : Exec_ComputationBuilderAccessor<allowed>(localTraversal)
    {
    }
};

/// Attribute accessor
template <Exec_ComputationBuilderProviderTypes allowed>
struct Exec_ComputationBuilderAttributeAccessor
    : public Exec_ComputationBuilderPropertyAccessor<allowed>
{
    Exec_ComputationBuilderAttributeAccessor(const SdfPath &localTraversal)
    : Exec_ComputationBuilderPropertyAccessor<allowed>(localTraversal)
    {
    }

    // XXX:TODO
    // Accessors for AnimSpline, Connections, IncomingConnections
};

/// Relationship accessor
template <Exec_ComputationBuilderProviderTypes allowed>
struct Exec_ComputationBuilderRelationshipAccessor
    : public Exec_ComputationBuilderPropertyAccessor<allowed>
{
    Exec_ComputationBuilderRelationshipAccessor(const SdfPath &localTraversal)
    : Exec_ComputationBuilderPropertyAccessor<allowed>(localTraversal)
    {
    }

    using ValueSpecifier =
        Exec_ComputationBuilderComputationValueSpecifier<allowed>;

    /// \addtogroup group_Exec_ValueSpecifiers
    /// @{

    /// After a [Relationship()](#exec_registration::Relationship::Relationship)
    /// accessor, requests input values from the computation \p computationName
    /// of type \p ResultType on the objects targeted by the relationship.
    ///
    /// Relationship forwarding is applied, so if the relationship targets
    /// another relationship, the targets are transitively expanded, resulting
    /// in the ultimately targeted, non-relationship objects.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that looks for the computation
    ///     // 'sourceComputation' on all targeted objects of the relationship
    ///     // 'myRel' and returns the number of matching targets.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<int>(+[](const VdfContext &ctx) {
    ///             VdfReadIterator<int> it(_tokens->sourceComputation);
    ///             return static_cast<int>(it.ComputeSize());
    ///         })
    ///         .Inputs(
    ///             Relationship(_tokens->myRel)
    ///             .TargetedObjects<int>(_tokens->sourceComputation));
    /// }
    /// ```
    ///
    template <typename ResultType>
    ValueSpecifier
    TargetedObjects(const TfToken &computationName)
    {
        return ValueSpecifier(
            computationName,
            ExecTypeRegistry::GetInstance().CheckForRegistration<ResultType>(),
            {Exec_ComputationBuilderAccessorBase::_GetLocalTraversal(),
             ExecProviderResolution::DynamicTraversal::
                 RelationshipTargetedObjects});
    }

    /// @}
};


// The following registrations are in the exec_registration namespace so that
// the registration macro can make them available (without the namespace) as
// arguments to registrations methods (i.e., Inputs()).
namespace exec_registration {


/// Attribute accessor, valid on a prim.
struct Attribute
    : public Exec_ComputationBuilderAttributeAccessor<
        Exec_ComputationBuilderProviderTypes::Prim>
{
    /// \addtogroup group_Exec_Accessors
    /// @{

    /// On a prim computation, provides access to the attribute named
    /// \p attributeName.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that returns the value of an
    ///     // attribute owned by the prim.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<double>(+[](const VdfContext &ctx) {
    ///             return ctx.GetInputValue<double>(
    ///                 ExecBuiltinComputations->computeValue));
    ///         })
    ///         .Inputs(
    ///             Attribute(_tokens->doubleAttribute)
    ///                 .Computation<double>(
    ///                     ExecBuiltinComputations->computeValue).Required());
    /// }
    /// ```
    ///
    Attribute(const TfToken &attributeName)
        : Exec_ComputationBuilderAttributeAccessor<
            Exec_ComputationBuilderProviderTypes::Prim>(
                SdfPath::ReflexiveRelativePath().AppendProperty(attributeName))
    {
    }

    /// @} // Accessors
};


/// Relationship accessor, valid on a prim.
struct Relationship
    : public Exec_ComputationBuilderRelationshipAccessor<
        Exec_ComputationBuilderProviderTypes::Prim>
{
    /// \addtogroup group_Exec_Accessors
    /// @{

    /// On a prim computation, provides access to the relationship named
    /// \p relationshipName.
    ///
    /// See
    /// [TargetedObjects()](#Exec_ComputationBuilderRelationshipAccessor::TargetedObjects)
    ///
    Relationship(const TfToken &relationshipName)
        : Exec_ComputationBuilderRelationshipAccessor<
            Exec_ComputationBuilderProviderTypes::Prim>(
                SdfPath::ReflexiveRelativePath().AppendProperty(
                    relationshipName))
    {
    }

    /// @} // Accessors
};


/// Provides access to the stage.
struct Stage
    : public Exec_ComputationBuilderAccessor<
        Exec_ComputationBuilderProviderTypes::Any>
{
    /// \addtogroup group_Exec_Accessors
    /// @{

    /// On any computation, provides access to the stage. This accessor can be
    /// used to access stage-level builtin computations.
    ///
    /// > **Note:**
    /// > The Stage() accessor must be the sole accessor in any input
    /// > registration in which it appears.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that returns the current time.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<EfTime>(+[](const VdfContext &ctx) {
    ///             return ctx.GetInputValue<EfTime>(
    ///                 ExecBuiltinComputations->computeTime));
    ///         })
    ///         .Inputs(
    ///             Stage().Computation<EfTime>(
    ///                 ExecBuiltinComputations->computeTime).Required());
    /// }
    /// ```
    ///
    Stage()
        : Exec_ComputationBuilderAccessor<
            Exec_ComputationBuilderProviderTypes::Any>(
                SdfPath::AbsoluteRootPath())
    {
    }

    /// @} // Accessors
};

// XXX:TODO
// Property, Relationship, Prim, NamespaceParent, NamespaceChildren, etc.


/// Computation value specifier, valid on a prim or attribute computation
template <typename ValueType>
struct Computation
    : public Exec_ComputationBuilderComputationValueSpecifier<
        Exec_ComputationBuilderProviderTypes::Any>
{
    /// \addtogroup group_Exec_ValueSpecifiers
    /// @{

    /// Requests an input value from the computation \p computationName of
    /// type \p ResultType.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that returns the value of an
    ///     // attribute owned by the prim.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<double>(+[](const VdfContext &ctx) {
    ///             return ctx.GetInputValue<double>(
    ///                 ExecBuiltinComputations->computeValue));
    ///         })
    ///         .Inputs(
    ///             Attribute(_tokens->doubleAttribute)
    ///                 .Computation<double>(
    ///                     ExecBuiltinComputations->computeValue).Required());
    /// }
    /// ```
    ///
    Computation(const TfToken &computationName)
        : Exec_ComputationBuilderComputationValueSpecifier<
            Exec_ComputationBuilderProviderTypes::Any>(
                computationName, 
                ExecTypeRegistry::GetInstance().
                    CheckForRegistration<ValueType>(),
                {SdfPath::ReflexiveRelativePath(),
                 ExecProviderResolution::DynamicTraversal::Local})
    {
    }

    /// @}
};

// XXX:TODO
// This should be implemented as an alias for an accessor that takes a predicate
// plus .Compute(), but that requires implementing predicates plus having a way
// to express the computation name and result type as computation parameters.
// Therefore, for now, this is implemented as a value specifier.
template <typename ValueType>
struct NamespaceAncestor
    : public Exec_ComputationBuilderComputationValueSpecifier<
        Exec_ComputationBuilderProviderTypes::Prim>
{
    /// \addtogroup group_Exec_ValueSpecifiers
    /// @{

    /// On a prim computation, requests an input value from the computation
    /// \p computationName of type \p ResultType on the nearest namespace
    /// ancestor prim.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a prim computation that finds the nearest namespace
    ///     // ancestor that defines a computation 'sourceComputation' with
    ///     // an `int` result type. If found, the result is the value of the
    ///     // ancestor compuation; otherwise, returns 0.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<int>(+[](const VdfContext &ctx) {
    ///             const int *const valuePtr =
    ///                 ctx.GetInputValuePtr<int>(_tokens->sourceComputation);
    ///             return valuePtr ? *valuePtr : 0;
    ///         })
    ///         .Inputs(
    ///             NamespaceAncestor<int>(_tokens->sourceComputation));
    /// }
    /// ```
    ///
    NamespaceAncestor(const TfToken &computationName)
        : Exec_ComputationBuilderComputationValueSpecifier<
            Exec_ComputationBuilderProviderTypes::Prim>(
                computationName,
                ExecTypeRegistry::GetInstance().
                    CheckForRegistration<ValueType>(),
                {SdfPath::ReflexiveRelativePath(),
                 ExecProviderResolution::DynamicTraversal::NamespaceAncestor})
    {
    }

    /// @}
};

// XXX:TODO
// Metadata, AnimSpline


/// \addtogroup group_Exec_Aliases
/// @{

// Note:
// Aliases are implemented as generator functions, rather than as structs,
// because that way they can simply be expressed as registrations.

/// Input alias that yields the value of the named attribute.
/// 
/// This registration must follow a
/// [PrimComputation](#Exec_ComputationBuilder::PrimComputation) registration.
///
/// > **Note:**  
/// > ```{.cpp}
/// > AttributeValue<T>(attrToken)
/// > ```
/// >
/// > is equivalent to:
/// >
/// > ```{.cpp}
/// > Attribute(attrToken)
/// >     .Compute<T>(ExecBuiltinComputations->computeValue)
/// >     .InputName(attrToken)
/// > ```
///
/// # Example
///
/// ```{.cpp}
/// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
/// {
///     // Register a prim computation that returns the computed value of an
///     // attribute.
///     self.PrimComputation(_tokens->eleven)
///         .Callback<double>(+[](const VdfContext &ctx) {
///             ctx->SetOutput(ctx.GetInputValue<double>(_tokens->myAttribute));
///         })
///         .Inputs(
///             AttributeValue<double>(_tokens->myAttribute).Required());
/// }
/// ```
///
template <typename ValueType>
auto
AttributeValue(const TfToken &attributeName)
{
    return Attribute(attributeName)
        .Computation<ValueType>(ExecBuiltinComputations->computeValue)
        .InputName(attributeName);
}

/// @} // Aliases


} // namespace exec_registration


// We forward declare this just so the generated documentation for
// PrimComputation() comes before the Callback() and Inputs() docs.
// 
class Exec_PrimComputationBuilder;

/// The top-level builder object (aka, the 'self' variable generated by
/// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA()).
/// 
class Exec_ComputationBuilder
{
    EXEC_API
    Exec_ComputationBuilder(TfType schemaType);

public:
    EXEC_API
    ~Exec_ComputationBuilder();

    /// Allows access to the constructor.
    ///
    /// Only schema computation registration functions should create computation
    /// builders.
    struct ConstructionAccess {
        static Exec_ComputationBuilder
        Construct(TfType schemaType) {
            return Exec_ComputationBuilder(schemaType);
        }
    };

    /// \addtogroup group_Exec_ComputationRegistrations
    ///  @{

    /// Registers a prim computation named \p computationName.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a trivial prim computation.
    ///     self.PrimComputation(_tokens->eleven)
    ///         .Callback<double>(+[](const VdfContext &) { return 11.0; })
    /// }
    /// ```
    ///
    EXEC_API
    Exec_PrimComputationBuilder 
    PrimComputation(const TfToken &computationName);

    /// Registers a dispatched prim computation named \p computationName.
    ///
    /// A dispatched prim computation is only visible to computations on the
    /// prim that does the dispatching. I.e., the computation registrations for
    /// a schema can include dispatched computations and inputs to computations
    /// registered on the same schema can request the dispatched computations,
    /// using the input option FallsBackToDispatched(), from *other provider
    /// prims* and find them there. *Other schema computation registrations*
    /// will not be able to find the dispatched computations, however.
    ///
    /// Dispatched computations can be restricted as to which prims they can
    /// dispatch onto, based on the typed and applied schemas of a given target
    /// prim. The second parameter to the DispatchedPrimComputation registration
    /// function can be used to specify zero or more schema types (as
    /// TfType%s). If any types are given, the dispatched computation will only
    /// be found on a target prim if that prim's typed schema type (or one of
    /// its base type) is among the given schema types or if the fully expanded
    /// list of API schemas applied to the prim includes a schema that is among
    /// the given schema types.
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a dispatched prim computation that can be found on
    ///     // scopes.
    ///     const TfType scopeType = TfType::FindByName("UsdGeomScope");
    ///     self.DispatchedPrimComputation(_tokens->eleven, scopeType)
    ///         .Callback<double>(+[](const VdfContext &) { return 11.0; })
    ///
    ///     // Register a prim computation that requests the above dispatched
    ///     // computation via uses relationship targets.
    ///     self.PrimComputation(_tokens->myComputation)
    ///         .Callback<double>(+[](const VdfContext &ctx) {
    ///             const double *const valuePtr =
    ///                 ctx.GetInputValuePtr<double>(_tokens->eleven);
    ///             return valuePtr ? *valuePtr : -1.0;
    ///         })
    ///         .Inputs(
    ///
    ///             // This input opts-in to finding dispatched computations.
    ///             Relationship(_tokens->myRelationship)
    ///                 .TargetedObjects<double>(_tokens->eleven)
    ///                 .FallsBackToDispatched())
    /// }
    /// ```
    ///
    EXEC_API
    Exec_PrimComputationBuilder 
    DispatchedPrimComputation(
        const TfToken &computationName,
        ExecDispatchesOntoSchemas &&ontoSchemas);

    // Convenience variadic template overload that takes zero or more schema
    // types.
    template <class... DispatchedOntoSchemaTypes>
    Exec_PrimComputationBuilder
    DispatchedPrimComputation(
        const TfToken &computationName,
        DispatchedOntoSchemaTypes &&...schemaTypes);

    // XXX:TODO
    // - AttributeComputation
    // - DispatchedAttributeComputation

    ///  @}

private:
    // The type of the schema for which this builder defines computations.
    TfType _schemaType;
};


/// Class used to build prim computation definitions.
class Exec_PrimComputationBuilder
{
    // Only Exec_ComputationBuilder can create instances.
    friend class Exec_ComputationBuilder;

    EXEC_API
    Exec_PrimComputationBuilder(
        TfType schemaType,
        const TfToken &computationName,
        bool dispatched = false,
        ExecDispatchesOntoSchemas &&dispatchesOntoSchemas = {});

public:
    EXEC_API
    ~Exec_PrimComputationBuilder();

    /// \ingroup group_Exec_ComputationRegistrations
    ///
    /// Registers a callback function that implements the evaluation time logic
    /// for a computation.
    /// 
    /// This registration must follow a [computation
    /// registration](#group_Exec_ComputationRegistrations).
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a trivial  prim computation that returns a constant
    ///     // value.
    ///     self.PrimComputation(_tokens->eleven)
    ///         .Callback<double>(+[](const VdfContext &) { return 11.0; })
    /// }
    /// ```
    ///
    template <typename ResultType>
    Exec_PrimComputationBuilder&
    Callback(ExecCallbackFn &&callback);

    /// \overload
    template <typename ResultType>
    std::enable_if_t<
        !std::is_void_v<ResultType>,
        Exec_PrimComputationBuilder&>
    Callback(ResultType (*callback)(const VdfContext &));

    /// \ingroup group_Exec_ComputationRegistrations
    ///
    /// Takes one or more [input registrations](#group_Exec_InputRegistrations)
    /// that specify how to source input values for a computation.
    /// 
    /// This registration must follow a [computation
    /// registration](#group_Exec_ComputationRegistrations).
    ///
    /// # Example
    ///
    /// ```{.cpp}
    /// EXEC_REGISTER_COMPUTATIONS_FOR_SCHEMA(MySchemaType)
    /// {
    ///     // Register a  prim computation that reads from two inputs.
    ///     self.PrimComputation(_tokens->myPrimComputation)
    ///         .Callback<int>(&_MyCallbackFn)
    ///         .Inputs(
    ///             AttributeValue<int>(_tokens->myAttribute),
    ///             NamespaceAncestor<double>(_tokens->anotherPrimComputation));
    /// }
    /// ```
    ///
    template <typename... Args>
    Exec_PrimComputationBuilder&
    Inputs(Args && ... args);

private:
    // Adds the callback with result type.
    EXEC_API
    void _AddCallback(ExecCallbackFn &&calback, TfType resultType);

    // Validates that all input are of types types that are allowed on a prim
    // computation.
    // 
    template <Exec_ComputationBuilderProviderTypes allowed, typename T>
    static void _ValidateInputs();

    // Adds an input key from the given value specifier.
    //
    // This extra level of indirection helps keep Exec_InputKey out of the
    // header so that type can remain private.
    // 
    EXEC_API
    void _AddInputKey(
        const Exec_ComputationBuilderValueSpecifierBase *valueSpecifier);

private:
    // We PIMPL the data for this class to avoid exposing more private details
    // in this public header.
    struct _Data;
    const std::unique_ptr<_Data> _data;
};

template <typename ResultType>
Exec_PrimComputationBuilder&
Exec_PrimComputationBuilder::Callback(
    ExecCallbackFn &&callback)
{
    static_assert(!VtIsArray<ResultType>::value,
                  "VtArray is not a supported result type");

    _AddCallback(
        std::move(callback),
        ExecTypeRegistry::GetInstance().CheckForRegistration<ResultType>());

    return *this;
}

template <typename ResultType>
std::enable_if_t<
    !std::is_void_v<ResultType>,
    Exec_PrimComputationBuilder&>
Exec_PrimComputationBuilder::Callback(
    ResultType (*callback)(const VdfContext&))
{
    static_assert(!std::is_reference_v<ResultType>,
                  "Callback functions must return by value");
    static_assert(!VtIsArray<ResultType>::value,
                  "VtArray is not a supported result type");

    _AddCallback(
        [callback](const VdfContext& ctx) { ctx.SetOutput(callback(ctx)); },
        ExecTypeRegistry::GetInstance().CheckForRegistration<ResultType>());

    return *this;
}

template <typename... Args>
Exec_PrimComputationBuilder&
Exec_PrimComputationBuilder::Inputs(
    Args && ... args)
{
    // Validate inputs
    (_ValidateInputs<Exec_ComputationBuilderProviderTypes::Prim, Args>(), ...);

    // Add inputs
    (_AddInputKey(&args), ...);

    return *this;
}

template <Exec_ComputationBuilderProviderTypes allowed, typename T>
void
Exec_PrimComputationBuilder::_ValidateInputs() {
    using regType = std::decay_t<T>;
    static_assert(
        !std::is_base_of_v<Exec_ComputationBuilderAccessorBase, regType>,
        "Accessor can't provide an input value.");
    static_assert(
        std::is_base_of_v<Exec_ComputationBuilderValueSpecifierBase, regType>,
        "Invalid type used as an input registration.");
    static_assert(
        regType::allowedProviders & allowed,
        "Input is not allowed on a provider of this type.");
}

template <class... DispatchedOntoSchemaTypes>
Exec_PrimComputationBuilder
Exec_ComputationBuilder::DispatchedPrimComputation(
    const TfToken &computationName,
    DispatchedOntoSchemaTypes &&...schemaTypes)
{
    static_assert(
        (std::is_same_v<
             std::decay_t<DispatchedOntoSchemaTypes>, TfType> && ...));

    return DispatchedPrimComputation(
        computationName,
        {std::forward<DispatchedOntoSchemaTypes>(schemaTypes)...});
}

PXR_NAMESPACE_CLOSE_SCOPE

#endif
