def name_to_id(name)
  name.gsub('_', '')
end

class ResourceTypeSpecParser
  def initialize(filename)
    @file = File.open(filename)
    @state = :nothing
    @typedef = {}
  end

  def parse_types(&block)
    @file.each_line { |line| parse_types_line(line, &block) }
  end

  private

  def parse_types_line(line, &block)
    case @state
      when :nothing
        if /<h2 id="(\w+)">(\w+)<\/h2>/ =~ line
          return if ($1 != name_to_id($2))
          @typedef[:name] = $2
          @typedef[:params] = {}
          @state = :waiting1
        end
      when :waiting1
        if /<h3 id="#{@typedef[:name]}-description">/ =~ line
          @typedef[:description] = ""
          @state = :description
        elsif /<h3 id="#{@typedef[:name]}-attributes">/ =~ line
          @state = :waiting2
        end
      when :description
        if /<h3 id="#{@typedef[:name]}-attributes">/ =~ line
          @state = :waiting2
        else
          @typedef[:description] += line
        end
      when :waiting2
        if /<pre><code>/ =~ line
          @state = :parsing
        end
      when :parsing
        if /<a href="\S+">(\w+)<\/a>.*<em>(.+)<\/em>/ =~ line
          @typedef[:params][$1] = {:comment => ""}
        end
        if /<\/code><\/pre>/ =~ line
          @state = :waiting_params
        end
      when :waiting_params
        if /<h4 id="#{@typedef[:name]}-attribute-(\w+)">\1<\/h4>/ =~ line
          @cur_param = $1
          @state = :parsing_param
        end
      when :parsing_param
        if /<h2 id="(\w+)">(\w+)<\/h2>/ =~ line
          return if ($1 != name_to_id($2))
          block.yield(@typedef)
          @typedef = {}
          @typedef[:name] = $2
          @typedef[:params] = {}
          @state = :waiting1
        elsif /<h4 id="#{@typedef[:name]}-attribute-(\w+)">\1<\/h4>/ =~ line
          @cur_param = $1
        elsif /<p>\(<a href="##{@typedef[:name]}-attributes">/ !~ line
          @typedef[:params][@cur_param][:comment] += line + "\n"
        end
       
      else
        # type code here
    end
  end

end

class MetaparametersSpecParser
  def initialize(filename)
    @file = File.open(filename)
    @state = :nothing
    @params = {}
  end

  def parse_params(&block)
    @file.each_line do |line|
      parse_toc(line)
      parse_params_line(line) 
    end
    block.yield(@params)
  end

  private
  def parse_toc(line)
    if /<li class="toc-lv3"><a href="#(\w+)/ =~ line
      @params[$1] = ""
    end
  end

  def parse_params_line(line) 
    case @state
    when :nothing
      if /<h3 id="(\w+)">\1<\/h3>/ =~ line and @params.key?($1)
        @cur_param = $1
        @state = :parsing
      end
    when :parsing
      if /<h([1-4]).*<\/h\1>/ !~ line and not_end(line)
        @params[@cur_param] += line
      elsif /<h3 id="(\w+)">\1<\/h3>/ =~ line and @params.key?($1)
        @cur_param = $1
      else
        @state = :nothing
      end
    end
  end

  def not_end(line)
    /<hr/ !~ line and /This page autogenerated/ !~ line
  end

end

class FactsSpecParser
  def initialize(filename)
    @file = File.open(filename)
    @state = :nothing
    @params = {}
  end

  def parse_facts(&block)
    @file.each_line do |line|
      parse_toc(line)
      parse_facts_line(line) 
    end
    block.yield(@params)
  end

  private
  def parse_toc(line)
    if /<li class="toc-lv2"><a href="#(\w+)">(\w+)<\/a><\/li>/ =~ line
      return if ($1 != name_to_id($2))
      @params[$2] = ""
    end
  end

  def parse_facts_line(line) 
    case @state
    when :nothing
      if /<h2 id="(\w+)"><code>(\w+)<\/code><\/h2>/ =~ line and @params.key?($2)
        return if ($1 != name_to_id($2))
        @cur_param = $2
        @state = :parsing
      end
    when :parsing
      if /<p>\(<a href="#page-nav"/ !~ line
        @params[@cur_param] += line
      else
        @state = :nothing
      end
    end
  end

end

class FunctionSpecParser
  def initialize(filename)
    @file = File.open(filename)
    @state = :nothing
    @params = {}
  end

  def parse_functions(&block)
    @file.each_line do |line|
      parse_toc(line)
      parse_function_line(line) 
    end
    block.yield(@params)
  end

  private
  def parse_toc(line)
    if /<li class="toc-lv2"><a href="#(\w+)">\1<\/a><\/li>/ =~ line
      @params[$1] = ""
    end
  end

  def parse_function_line(line) 
    case @state
    when :nothing
      if /<h2 id="(\w+)"/ =~ line and @params.key?($1)
        @cur_param = $1
        @state = :parsing
      end
    when :parsing
      if /<h2 id="(\w+)"/ =~ line and @params.key?($1)
        @cur_param = $1
      elsif /<hr \/>/ !~ line
        @params[@cur_param] += line
      else
        @state = :nothing
      end
    end
  end

end

class StubPrinter

  def print_comment(text)
    text.split("\n").each do |line|
      if !line.empty?
        puts "#   #{line}"
      end
    end
  end

  def print_resource_definition_stub(obj)
    puts "# <h2>#{obj[:name]}</h2>"
    print_comment obj[:description]

    puts "define #{obj[:name]} ("
    obj[:params].each do |name, param|
      puts "# <h2>#{name}</h2>"

      print_comment param[:comment]
      puts "  $#{name},"
    end
    puts ") {\n  # Stub\n}\n\n"
  end

  def print_metaparams_stub(obj)
    puts "define stub__metaparams__ ("
    obj.each do |name, comment|
      puts "# <h2>#{name}</h2>"

      print_comment comment
      puts "  $#{name},"
    end
    puts ") {\n  # Stub\n}\n\n"
  end

  def print_facts(obj)
    obj.each do |name, comment|
      puts "# <h2>#{name}</h2>"
      print_comment comment
      puts "  $#{name} = $#{name}\n\n"
    end
  end

  def print_ruby_functions(obj)
    puts "module Puppet::Parser::Functions"
    obj.each do |name, comment|
      puts "  newfunction(:#{name}) do |args|"
      puts "    desc <<EOT\n#{comment}\nEOT"
      puts "  end\n\n"
    end
    puts "end"
  end
end

