### Copyright (C) 1995, 1996, 1997 Jeppe Buk (buk@imada.sdu.dk)
### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; either version 2 of the License, or
### (at your option) any later version.
###
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
###
### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

##################################################
# The parser program called from TDG

proc fvwm2parser { outfile pages } {
  global p__fname p__format p__ok p__winname
  set p__fname "~/.fvwm2rc"
  set p__format 2
  set p__ok 0

  ##################################################
  # Widgets
  set win [makeTempWindow "Import"]
  set p__winname $win
  label $win.header -text "Enter information about the dotfile:" \
      -font "-adobe-helvetica-bold-r-*-*-*-140-*-*-*-*-*-*"
  pack $win.header
  # line
  frame $win.line -height 0.1c -relief sunken -bd 1
  pack $win.line -fill x -expand 1
  frame $win.entryFrame
  label $win.entryFrame.label -text "File name"
  entry $win.entryFrame.fname -textvariable p__fname
  pack $win.entryFrame
  pack $win.entryFrame.label $win.entryFrame.fname -side left -anchor w
  radiobutton $win.format1 -variable p__format -value 1 -text "Fvwm-1 format"
  radiobutton $win.format2 -variable p__format -value 2 -text "Fvwm-2 format"
  pack $win.format1 $win.format2
  label $win.note -text "Note:\n\nThis parser doesn't handle module settings (i.e. lines beginning with an asterisk). Along with other lines in the dotfile not handled by this parser they will be saved to the file ~/.dotfile/Fvwm2parser.unknown for further investigation by you.\n\nThe filter used to convert from Fvwm-1 syntax is based on fvwmrc_convert created by Martin Kraemer (Martin.Kraemer@mch.sni.de) and Grant McDorman (grant@isgtec.com)." \
      -font "-adobe-helvetica-medium-r-*-*-*-120-*-*-*-*-*-*" \
      -justify left -wraplength 350 -relief sunken
  pack $win.note -ipadx 2
  button $win.ok -text OK -command {set p__ok 1; destroy $p__winname}
  button $win.cancel -text Cancel -command {set p__ok 0; destroy $p__winname}
  pack $win.cancel $win.ok -side right

  ##################################################
  # Wait
  tkwait window $win
  if {$p__ok == 1} {
    if {$p__format == 1} {
      global argv env
      puts "Converting to version 2 format ..."
      catch {exec [lindex $argv 1]/fvwmrc_convert [glob $p__fname] $env(HOME)/.dotfile/tmp.fvwmconvert} output
      puts $output
      p_parser "$env(HOME)/.dotfile/tmp.fvwmconvert" $outfile
      catch {exec rm $env(HOME)/.dotfile/tmp.fvwmconvert}
    } else {
      p_parser $p__fname $outfile
    }
  } else {
    # Create empty file
    close [open $outfile "w"]
  }
}

proc p_log { text } {
  global p_logfile
  puts $p_logfile $text
}

proc p_debug { text } {
  puts "PARSER NOTE: $text"
}

proc p_info { text } {
#  puts "INFO: $text"
}

proc p_first { text } {
  return [string trim [lindex $text 0]]
}

proc p_rest { text } {
  return [string trim [lrange $text 1 end]]
}

proc p_escape { text } {
  set result ""
  foreach char [split $text ""] {
    if {$char == "\""} {
      append result "\\\""
    } else {
      append result $char
    }
  }
  return $result
}

proc p_context2list { text } {
  set result {}
  foreach char [split $text ""] {
    switch $char {
      "A" { lappend result "Any" }
      "R" { lappend result "Root" }
      "W" { lappend result "Window" }
      "T" { lappend result "Title bar" }
      "S" { lappend result "Sides" }
      "F" { lappend result "Frame" }
      "I" { lappend result "Icon" }
      "0" { lappend result "0 button" }
      "1" { lappend result "1 button" }
      "2" { lappend result "2 button" }
      "3" { lappend result "3 button" }
      "4" { lappend result "4 button" }
      "5" { lappend result "5 button" }
      "6" { lappend result "6 button" }
      "7" { lappend result "7 button" }
      "8" { lappend result "8 button" }
      "9" { lappend result "9 button" }
      default {
	p_debug "Unknown context \"$char\""
      }
    }
  }
  return $result
}

proc p_modifier2list { text } {
  set result {}
  foreach char [split $text ""] {
    switch $char {
      "A" { lappend result "Any" }
      "C" { lappend result "Control" }
      "S" { lappend result "Shift" }
      "M" { lappend result "Meta" }
      "1" { lappend result "1mod" }
      "2" { lappend result "2mod" }
      "3" { lappend result "3mod" }
      "4" { lappend result "4mod" }
      "5" { lappend result "5mod" }
      "N" { }
      default {
	p_debug "Unknown modifier \"$char\""
      }
    }
  }
  return $result
}

proc p_parser { infile outfile } {

  ##################################################
  # Open log file
  global p_logfile
  set p_logfile [open ~/.dotfile/Fvwm2import.unknown "w"]

  set err [catch {open $infile} rc]
  set last ""

  ##################################################
  # Does the infile exist?
  if { $err } {
    tk_dialog .p_err "File not found" "$infile not found!\n\nNo settings will be read." error 0 OK
    close [open $outfile "w"]
    return
  }

  ##################################################
  # Initialize parser export, functions and menus
  foreach elm {
    iconpath modulepath desktopsize edgeresistance edgescroll buttonstyle
    menustyle hilightcolor windowfont titlestyle iconfont clicktime
    colormapfocus opaquemovesize xorvalue key mouse style } {
    set p_export($elm) ""
  }
  set lastFM ""
  set lastType ""

  while { ![eof $rc]} {
    set line [gets $rc]
    ##################################################
    # handle backslash - newline
    if {[string index $line [expr [string length $line] -1]] == "\\"} {
      set last "$last[string range $line 0 [expr [string length $line] -2]]"
      continue
    } else {
      set line "$last$line"
      set last ""
    }

    ##################################################'
    # Remove comments
    set checkLine $line
    set commIdx [string first "\#" $checkLine]
    set lastIdx -1
    while {$commIdx != -1} {
      if {$commIdx == 0} {
	set line [string range $line 0 $lastIdx]
	break
      } else {
	# Don't delete escaped hashes and colors
	# (How does one repeat a regexp 6 times!?!)
	if {[string index $checkLine [expr $commIdx - 1]] != "\\" && [regexp \[0-9a-fA-F\]\[0-9a-fA-F\]\[0-9a-fA-F\]\[0-9a-fA-F\]\[0-9a-fA-F\]\[0-9a-fA-F\] [string range $checkLine [expr $commIdx + 1] end]] != 1} {
	  set line [string range $line 0 [expr $lastIdx + $commIdx]]
	  break
	}
      }
      set lastIdx $commIdx
      set checkLine [string range $checkLine [expr $commIdx + 1] end]
      set commIdx [string first "\#" $checkLine]
    }
    
    set line [string trim $line]

    ##################################################
    # Skip empty lines
    if {[string length $line] == 0} {
      continue
    }

    ##################################################
    # Ignore module settings (for now)
    if {[string index $line 0] == "*"} {
      p_log $line
      continue
    }

    ##################################################
    # Save the line for logging
    set saveLine $line

    ##################################################
    # Get the token (try to catch special cases that
    #                produce Tcl errors)
    set err1 [catch {p_first $line} token]
    set err2 [catch {p_rest $line} line]
    if {$err1 != 0 || $err2 != 0} {
      p_debug "Catched error in line \"$saveLine\""
      p_log $saveLine
      continue
    }

    switch [string tolower $token] {

      "iconpath" {
        set p_export(iconpath) \
            [concat $p_export(iconpath) [split $line ":"]]
      }
      "pixmappath" {
        # Alias for icon path
        set p_export(iconpath) \
            [concat $p_export(iconpath) [split $line ":"]]
      }
      "modulepath" {
        set p_export(modulepath) \
            [concat $p_export(modulepath) [split $line ":"]]
      }
      "desktopsize" {
        set p_export(desktopsize) [split $line "x"]
      }
      "edgeresistance" {
        set p_export(edgeresistance) $line
      }
      "edgescroll" {
        set p_export(edgescroll) $line
      }
      "buttonstyle" {
        set thisbutton [p_first $line]
        lappend thisbutton [p_rest $line]
        lappend p_export(buttonstyle) $thisbutton
      }
      "menustyle" {
	set p_export(menustyle) $line
      }
      "hilightcolor" {
	set p_export(hilightcolor) $line
      }
      "windowfont" {
	set p_export(windowfont) $line
      }
      "titlestyle" {
	set p_export(titlestyle) $line
      }
      "iconfont" {
	set p_export(iconfont) $line
      }
      "clicktime" {
	set p_export(clicktime) $line
      }
      "colormapfocus" {
	set p_export(colormapfocus) $line
      }
      "opaquemovesize" {
	set p_export(opaquemovesize) $line
      }
      "xorvalue" {
	set p_export(xorvalue) $line
      }
      "addtofunc" {
	set func [p_first $line]
	set line [p_rest $line]
	set lastFM $func
	set lastType "functions"
	if {![info exist p_functions($func)]} {
	  set p_functions($func) {}
	}
	if {$line != ""} {
	  lappend p_functions($func) $line
	}
      }
      "addtomenu" {
	set menu [p_first $line]
	set line [p_rest $line]
	set lastFM $menu
	set lastType "menus"
	if {![info exist p_menus($menu)]} {
	  set p_menus($menu) {}
	}
	if {$line != ""} {
	  lappend p_menus($menu) $line
	}
      }
      "+" {
	if {[info exist p_${lastType}($lastFM)]} {
	  lappend p_${lastType}($lastFM) $line
	} else {
	  p_debug "\"+\" without former AddTo..."
	}
      }
      "key" {
	lappend p_export(key) $line
      }
      "mouse" {
	lappend p_export(mouse) $line
      }
      "style" {
	lappend p_export(style) $line
      }
      default {
	p_debug "Unknown line: $saveLine"
        p_log $saveLine
      }
    }
  }

  ##################################################
  # Write export file
  set exp [open $outfile w]

  ##################################################
  # Paths
  puts $exp "PAGE paths"

  if {$p_export(iconpath) != ""} {
    p_info "Exporting IconPath"
    puts $exp "EXTENTRY icp"
    foreach iconp $p_export(iconpath) {
      puts $exp "\t{"
      puts $exp "\t\tENTRY icpB \"[p_escape $iconp]\""
      puts $exp "\t\tCOMMAND com_icpB \"[p_escape $iconp]\""
      puts $exp "\t}"
    }
  }

  if {$p_export(modulepath) != ""} {
    p_info "Exporting ModulePath"
    puts $exp "EXTENTRY mdp"
    foreach modp $p_export(modulepath) {
      puts $exp "\t{"
      puts $exp "\t\tENTRY mdpB \"[p_escape $modp]\""
      puts $exp "\t\tCOMMAND com_mdpB \"[p_escape $modp]\""
      puts $exp "\t}"
    }
  }

  ##################################################
  # Virtual desktop
  puts $exp "PAGE virtual"

  if {$p_export(desktopsize) != ""} {
    p_info "Exporting DesktopSize"
    puts $exp "CHECKBOX desktopSize 1"
    puts $exp "INT desktopSizeX \"[lindex $p_export(desktopsize) 0]\""
    puts $exp "INT desktopSizeY \"[lindex $p_export(desktopsize) 1]\""
  }
  if {$p_export(edgeresistance) != ""} {
    p_info "Exporting EdgeResistance"
    puts $exp "CHECKBOX edgeRes 1"
    puts $exp "INT edgeRes1 \"[lindex $p_export(edgeresistance) 0]\""
    puts $exp "INT edgeRes2 \"[lindex $p_export(edgeresistance) 1]\""
  }
  if {$p_export(edgescroll) != ""} {
    p_info "Exporting EdgeScroll"
    puts $exp "CHECKBOX edgeScroll 1"
    puts $exp "INT xScroll \"[lindex $p_export(edgescroll) 0]\""
    puts $exp "INT yScroll \"[lindex $p_export(edgescroll) 1]\""
  }

  ##################################################
  # Button styles
  puts $exp "PAGE buttonsAndCursors"

  if {$p_export(buttonstyle) != ""} {
    p_info "Exporting ButtonStyle"
    puts $exp "EXTENTRY but"
    foreach but $p_export(buttonstyle) {
      p_info " - button [lindex $but 0]"
      puts $exp "\t{"
      puts $exp "\t\tCHECKBOX but[lindex $but 0] 1"
      set thisstyle [lindex $but 1]
      if {[string tolower [lindex $thisstyle 0]] == "pixmap" ||
          [string tolower [lindex $thisstyle 0]] == "fullpixmap"} {
        puts $exp "\t\tCHECKBOX pixB 1"
        puts $exp "\t\tENTRY pixEup \"[p_escape [lindex $thisstyle 1]]\""
        puts $exp "\t\tENTRY pixEdown \"[p_escape [lindex $thisstyle 2]]\""
        if {[string tolower [lindex $thisstyle 0]] == "fullpixmap"} {
          puts $exp "\t\tRADIO pixRbut no"
        }
      } else {
        puts $exp "\t\tCHECKBOX butB 1"
        puts $exp "\t\tEXTENTRY coord"
        # ignore the number of coordinates (index from 1)
        foreach coord [lrange $thisstyle 1 end] {
          puts $exp "\t\t\t{"
          puts $exp "\t\t\t\tINT coordX \"[string range $coord 0 [expr [string first "x" $coord] -1]]\""
          puts $exp "\t\t\t\tINT coordY \"[string range $coord [expr [string first "x" $coord] + 1] [expr [string first "@" $coord] - 1]]\""
          if {[string index $coord [expr [string length $coord] -1]] == 1} {
            puts $exp "\t\t\t\tRADIO shade highlight"
          }
          puts $exp "\t\t\t}"
        }
      }
      puts $exp "\t}"
    }
  }

  ##################################################
  # Colors and fonts
  puts $exp "PAGE colorsfonts"
  
  if {$p_export(menustyle) != ""} {
    p_info "Exporting MenuStyle"
    puts $exp "CHECKBOX menuB 1"
    puts $exp "COMMAND foreColor \"[p_escape [lindex $p_export(menustyle) 0]] 0\""
    puts $exp "COMMAND backColor \"[p_escape [lindex $p_export(menustyle) 1]] 0\""
    puts $exp "COMMAND shadeColor \"[p_escape [lindex $p_export(menustyle) 2]] 0\""
    puts $exp "COMMAND menuFont \"[p_escape [lindex $p_export(menustyle) 3]]\""
    if {[string tolower [lindex $p_export(menustyle) 4]] == "mwm"} {
      puts $exp "RADIO menuStyle MWM"
    }
  }

  if {$p_export(hilightcolor) != ""} {
    p_info "Exporting HilightColor"
    puts $exp "CHECKBOX wColB 1"
    puts $exp "COMMAND wForeColor \"[p_escape [lindex $p_export(hilightcolor) 0]] 0\""
    puts $exp "COMMAND wBackColor \"[p_escape [lindex $p_export(hilightcolor) 1]] 0\""
  }

  if {$p_export(windowfont) != ""} {
    p_info "Exporting WindowFont"
    puts $exp "CHECKBOX wFonB 1"
    puts $exp "COMMAND winFont \"[p_escape $p_export(windowfont)]\""
  }

  if {$p_export(titlestyle) != ""} {
    p_info "Exporting TitleStyle"
    puts $exp "CHECKBOX tStyleB 1"
    switch [string tolower [lindex $p_export(titlestyle) 0]] {
      "leftjustified" {
	puts $exp "RADIO tStyleAlign left"
      }
      "centered" {
	puts $exp "RADIO tStyleAlign centered"
      }
      "rightjustified" {
	puts $exp "RADIO tStyleAlign right"
      }
    }
    switch [string tolower [lindex $p_export(titlestyle) 1]] {
      "raised" {
	puts $exp "RADIO tStyleRelief raised"
      }
      "flat" {
	puts $exp "RADIO tStyleRelief flat"
      }
      "sunk" {
	puts $exp "RADIO tStyleRelief sunken"
      }
    }
  }

  if {$p_export(iconfont) != ""} {
    p_info "Exporting IconFont"
    puts $exp "CHECKBOX iFonB 1"
    puts $exp "COMMAND icoFont \"[p_escape $p_export(iconfont)]\""
  }    

  ##################################################
  # Miscellaneous
  puts $exp "PAGE miscellaneous"

  if {$p_export(clicktime) != ""} {
    p_info "Exporting ClickTime"
    puts $exp "CHECKBOX clickTime 1"
    puts $exp "INT clickDelay \"$p_export(clicktime)\""
  }

  if {$p_export(colormapfocus) != ""} {
    p_info "Exporting ColormapFocus"
    if {[string tolower $p_export(colormapfocus)] == "followfocus"} {
      puts $exp "CHECKBOX cmf 1"
    }
  }

  if {$p_export(opaquemovesize) != ""} {
    p_info "Exporting OpaqueMoveSize"
    puts $exp "CHECKBOX opmB 1"
    puts $exp "INT opmE \"$p_export(opaquemovesize)\""
  }

  if {$p_export(xorvalue) != ""} {
    p_info "Exporting XORvalue"
    puts $exp "CHECKBOX xorB 1"
    puts $exp "INT xorV \"$p_export(xorvalue)\""
  }

  ##################################################
  # Functions
  puts $exp "PAGE functions"

  if {[array names p_functions] != {}} {
    p_info "Exporting Functions"
    puts $exp "EXTENTRY function"
    foreach func [lsort [array names p_functions]] {
      p_info " - $func"
      puts $exp "\t{"
      puts $exp "\t\tENTRY functionName \"[p_escape $func]\""
      puts $exp "\t\tEXTENTRY functionEntry"
      foreach fentry [set p_functions($func)] {
	set events [p_first $fentry]
	set fentry [p_rest $fentry]
	set builtin [p_first $fentry]
	set args [p_rest $fentry]
	puts $exp "\t\t\t{"
	puts $exp "\t\t\t\tCOMBOBOX builtinCall \"[p_escape $builtin]\""
	switch [string tolower [lindex [split $events ""] 0]] {
	  "i" { }
	  "m" { puts $exp "\t\t\t\tMENU events Motion" }
	  "c" { puts $exp "\t\t\t\tMENU events Click" }
	  "d" { puts $exp "\t\t\t\tMENU events DoubleClick" }
	  default { p_debug "Unknown event $events in function $func" }
	}
	puts $exp "\t\t\t\tENTRY arguments \"[p_escape $args]\""
	puts $exp "\t\t\t}"
      }
      puts $exp "\t}"
    }
  }

  ##################################################
  # Menus
  puts $exp "PAGE menus"

  if {[array names p_menus] != {}} {
    p_info "Exporting Menus"
    puts $exp "EXTENTRY popup"
    foreach menu [lsort [array names p_menus]] {
      p_info " - $menu"
      puts $exp "\t{"
      puts $exp "\t\tENTRY popupName \"[p_escape $menu]\""
      puts $exp "\t\tEXTENTRY popupEntry"
      foreach mentry [set p_menus($menu)] {
	set title [p_first $mentry]
	if {[regexp {(.*)\*([^*]*)\*(.*)} $title dummy fi pix re]} {
	  set pixmap $pix
	  set title "$fi$re"
	  set place "T"
	} elseif {[regexp {(.*)%([^*]*)%(.*)} $title dummy fi pix re]} {
	  set pixmap $pix
	  set title "$fi$re"
	  set place "L"
	} else {
	  set pixmap {}
	  set place "L"
	}
	set mentry [p_rest $mentry]
	set builtin [p_first $mentry]
	set args [p_rest $mentry]
	puts $exp "\t\t\t{"
	puts $exp "\t\t\t\tRADIO place \"$place\""
	puts $exp "\t\t\t\tENTRY pixmap \"[p_escape $pixmap]\""
	puts $exp "\t\t\t\tENTRY name \"[p_escape $title]\""
	puts $exp "\t\t\t\tCOMBOBOX builtinCall \"[p_escape $builtin]\""
	puts $exp "\t\t\t\tENTRY arguments \"[p_escape $args]\""
	puts $exp "\t\t\t}"
      }
      puts $exp "\t}"
    }
  }

  ##################################################
  # Key Bindings
  puts $exp "PAGE keys"

  if {$p_export(key) != ""} {
    p_info "Exporting Key"
    puts $exp "EXTENTRY keys"
    foreach key $p_export(key) {
      set keysym [p_first $key]
      set key [p_rest $key]
      set context [p_first $key]
      set key [p_rest $key]
      set mods [p_first $key]
      set key [p_rest $key]
      set builtin [p_first $key]
      set args [p_rest $key]
      puts $exp "\t{"
      puts $exp "\t\tLABEL index \"Key: $mods $keysym, Context: $context\""
      puts $exp "\t\tCOMBOBOX keysym \"[p_escape $keysym]\""
      puts $exp "\t\tLISTBOX keyContext \"[p_context2list $context]\""
      puts $exp "\t\tLISTBOX keyModifier \"[p_modifier2list $mods]\""
      puts $exp "\t\tCOMBOBOX builtinCall \"[p_escape $builtin]\""
      puts $exp "\t\tENTRY keyArgs \"[p_escape $args]\""
      puts $exp "\t}"
    }
  }

  ##################################################
  # Mouse Bindings
  puts $exp "PAGE mouse"

  if {$p_export(mouse) != ""} {
    p_info "Exporting Mouse"
    puts $exp "EXTENTRY mouse"
    foreach mouse $p_export(mouse) {
      set button [p_first $mouse]
      set mouse [p_rest $mouse]
      if {$button == 0} {
	set button "Any"
      }
      set context [p_first $mouse]
      set mouse [p_rest $mouse]
      set mods [p_first $mouse]
      set mouse [p_rest $mouse]
      set builtin [p_first $mouse]
      set args [p_rest $mouse]
      puts $exp "\t{"
      puts $exp "\t\tLABEL index \"Mouse: $mods $button, Context: $context\""
      puts $exp "\t\tMENU button \"[p_escape $button]\""
      puts $exp "\t\tLISTBOX mouseContext \"[p_context2list $context]\""
      puts $exp "\t\tLISTBOX mouseModifier \"[p_modifier2list $mods]\""
      puts $exp "\t\tCOMBOBOX builtinCall \"[p_escape $builtin]\""
      puts $exp "\t\tENTRY mouseArgs \"[p_escape $args]\""
      puts $exp "\t}"
    }
  }

  ##################################################
  # Style
  puts $exp "PAGE style"

  if {$p_export(style) != ""} {
    p_info "Collapsing styles"
    set styles {}
    set remain $p_export(style)
    while { $remain != {} } {
      set matchName [p_first [lindex $remain 0]]
      set match {}
      set nomatch {}
      foreach styleLine $remain {
	set curName [p_first $styleLine]
	if { $matchName == $curName } {
	  if { $match == {} } {
	    set match $styleLine
	  } else {
	    append match "," [p_rest $styleLine]
	  }
	} else {
	  lappend nomatch $styleLine
	}
      }
      lappend styles $match
      set remain $nomatch
    }

    p_info "Exporting Style"
    puts $exp "EXTENTRY style"
    foreach style $styles {
      set winName [p_first $style]
      set style [split [p_rest $style] ","]
      p_info " - $winName"
      puts $exp "\t{"
      puts $exp "\t\tENTRY windowName \"[p_escape $winName]\""
      foreach flag $style {
	set tag [p_first $flag]
	set arg [p_rest $flag]
	switch [string tolower $tag] {
	  "usestyle" { puts $exp "\t\tENTRY useStyle \"[p_escape $arg]\"" }
	  "color" {
	    set cols [split $arg "/"]
	    set fore [p_first $cols]
	    set back [p_first [p_rest $cols]]
	    puts $exp "\t\tCOMMAND foreColor \"[p_escape "{$fore} 0"]\""
	    puts $exp "\t\tCOMMAND backColor \"[p_escape "{$back} 0"]\""
	    puts $exp "\t\tCHECKBOX colBox 1"
	  }
	  "forecolor" {
	    puts $exp "\t\tCOMMAND foreColor \"[p_escape [p_first $arg]] 0\""
	    puts $exp "\t\tCHECKBOX colBox 1"
	  }
	  "backcolor" {
	    puts $exp "\t\tCOMMAND backColor \"[p_escape [p_first $arg]] 0\""
	    puts $exp "\t\tCHECKBOX colBox 1"
	  }
	  "nohandles" { puts $exp "\t\tCHECKBOX noHandles 1"  }
	  "handles"   { puts $exp "\t\tCHECKBOX useHandles 1" }
	  "borderwidth" {
	    puts $exp "\t\tCHECKBOX borBox 1"
	    puts $exp "\t\tINT borderWidth \"$arg\""
	  }
	  "handlewidth" {
	    puts $exp "\t\tCHECKBOX hanBox 1"
	    puts $exp "\t\tINT handleWidth \"$arg\""
	  }
	  "notitle" { puts $exp "\t\tCHECKBOX noTitle 1" }
	  "title" {
	    puts $exp "\t\tCHECKBOX useTitle 1"
	    puts $exp "\t\tENTRY title \"[p_escape [p_first $arg]]\""
	  }
	  "nakedtransient"    { puts $exp "\t\tCHECKBOX noTrans 1"     }
	  "decoratetransient" { puts $exp "\t\tCHECKBOX forceTrans 1"  }
	  "starticonic"       { puts $exp "\t\tCHECKBOX startIconic 1" }
	  "startnormal"       { puts $exp "\t\tCHECKBOX startNormal 1" }
	  "startsondesk" {
	    puts $exp "\t\tCHECKBOX startsOnDesk 1"
	    puts $exp "\t\tINT startsOnDeskNo \"[p_first $arg]\""
	  }
	  "startsanywhere"  { puts $exp "\t\tCHECKBOX startsAnywhere 1"  }
	  "smartplacement"  { puts $exp "\t\tCHECKBOX smartPlacement 1"  }
	  "dumbplacement"   { puts $exp "\t\tCHECKBOX dumbPlacement 1"   }
	  "randomplacement" { puts $exp "\t\tCHECKBOX randomPlacement 1" }
	  "activeplacement" { puts $exp "\t\tCHECKBOX activePlacement 1" }
	  "skipmapping"     { puts $exp "\t\tCHECKBOX skipMapping 1"     }
	  "showmapping"     { puts $exp "\t\tCHECKBOX showMapping 1"     }
	  "nopposition"     { puts $exp "\t\tCHECKBOX noPPosition 1"     }
	  "usepposition"    { puts $exp "\t\tCHECKBOX usePPosition 1"    }
	  "windowlistskip"  { puts $exp "\t\tCHECKBOX windowListSkip 1"  }
	  "windowlisthit"   { puts $exp "\t\tCHECKBOX windowListHit 1"   }
	  "circulateskip"   { puts $exp "\t\tCHECKBOX circulateSkip 1"   }
	  "circulatehit"    { puts $exp "\t\tCHECKBOX circulateHit 1"    }
	  "staysontop"      { puts $exp "\t\tCHECKBOX staysOnTop 1"      }
	  "staysput"        { puts $exp "\t\tCHECKBOX staysPut 1"        }
	  "sticky"          { puts $exp "\t\tCHECKBOX sticky 1"          }
	  "slippery"        { puts $exp "\t\tCHECKBOX slippery 1"        }
	  "clicktofocus"    { puts $exp "\t\tCHECKBOX clickToFocus 1"    }
	  "sloppyfocus"     { puts $exp "\t\tCHECKBOX sloppyFocus 1"     }
	  "mousefocus"      { puts $exp "\t\tCHECKBOX mouseFocus 1"      }
	  "oldecor"         { puts $exp "\t\tCHECKBOX olDecor 1"         }
	  "nooldecor"       { puts $exp "\t\tCHECKBOX noOlDecor 1"       }
	  "lenience"        { puts $exp "\t\tCHECKBOX lenience 1"        }
	  "nolenience"      { puts $exp "\t\tCHECKBOX noLenience 1"      }
	  "icon" {
	    puts $exp "\t\tCHECKBOX useIcon 1"
	    set icon [p_first $arg]
	    if { $icon != "" } {
	      puts $exp "\t\tCHECKBOX icoBox 1"
	      puts $exp "\t\tENTRY icon \"[p_escape $icon]\""
	    }
	  }
	  "noicon"      { puts $exp "\t\tCHECKBOX noIcon 1" }
	  "icontitle"   { puts $exp "\t\tCHECKBOX useIct 1" }
	  "noicontitle" { puts $exp "\t\tCHECKBOX noIct 1"  }
	  "iconbox" {
	    puts $exp "\t\tCHECKBOX icbC 1"
	    puts $exp "\t\tINT icbLeft \"[lindex $arg 0]\""
	    puts $exp "\t\tINT icbTop \"[lindex $arg 1]\""
	    puts $exp "\t\tINT icbRight \"[lindex $arg 2]\""
	    puts $exp "\t\tINT icbBottom \"[lindex $arg 3]\""
	  }
	  "stickyicon"      { puts $exp "\t\tCHECKBOX stickyIcon 1"         }
	  "slipperyicon"    { puts $exp "\t\tCHECKBOX slipperyIcon 1"       }
	  "mwmbuttons"      { puts $exp "\t\tCHECKBOX mwmBut 1"             }
	  "fvwmbuttons"     { puts $exp "\t\tCHECKBOX fvwmBut 1"            }
	  "mwmborder"       { puts $exp "\t\tCHECKBOX mwmBor 1"             }
	  "fvwmborder"      { puts $exp "\t\tCHECKBOX fvwmBor 1"            }
	  "mwmdecor"        { puts $exp "\t\tCHECKBOX mwmDec 1"             }
	  "nodecorhints"    { puts $exp "\t\tCHECKBOX noDec 1"              }
	  "mwmfunctions"    { puts $exp "\t\tCHECKBOX mwmFunc 1"            }
	  "nofunctionhints" { puts $exp "\t\tCHECKBOX noFunc 1"             }
	  "hintoverride"    { puts $exp "\t\tCHECKBOX mwmHint 1"            }
	  "nooverride"      { puts $exp "\t\tCHECKBOX noHint 1"             }
	  "button"          { puts $exp "\t\tCHECKBOX ibut[p_first $arg] 1" }
	  "nobutton"        { puts $exp "\t\tCHECKBOX ebut[p_first $arg] 1" }
	  default {
	    p_debug "Unknown style flag $tag in style of $winName"
	  }
	}
      }
      puts $exp "\t}"
    }
  }
  
  ##################################################
  # Close the export and log files
  close $exp
  close $p_logfile
}
