/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.deprecation;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexTemplateMetaData;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.common.compress.CompressedXContent;
import org.elasticsearch.common.joda.JodaDeprecationPatterns;
import org.elasticsearch.discovery.DiscoverySettings;
import org.elasticsearch.ingest.IngestService;
import org.elasticsearch.ingest.PipelineConfiguration;
import org.elasticsearch.search.SearchModule;
import org.elasticsearch.xpack.core.deprecation.DeprecationIssue;
import org.elasticsearch.xpack.deprecation.IndexDeprecationChecks;

public class ClusterDeprecationChecks {
    private static final Logger logger = LogManager.getLogger(ClusterDeprecationChecks.class);

    static DeprecationIssue checkShardLimit(ClusterState state) {
        int shardsPerNode = (Integer)MetaData.SETTING_CLUSTER_MAX_SHARDS_PER_NODE.get(state.metaData().settings());
        int nodeCount = state.getNodes().getDataNodes().size();
        int maxShardsInCluster = shardsPerNode * nodeCount;
        int currentOpenShards = state.getMetaData().getTotalOpenIndexShards();
        if (nodeCount > 0 && currentOpenShards >= maxShardsInCluster) {
            return new DeprecationIssue(DeprecationIssue.Level.WARNING, "Number of open shards exceeds cluster soft limit", "https://www.elastic.co/guide/en/elasticsearch/reference/7.0/breaking-changes-7.0.html#_cluster_wide_shard_soft_limit", "There are [" + currentOpenShards + "] open shards in this cluster, but the cluster is limited to [" + shardsPerNode + "] per data node, for [" + maxShardsInCluster + "] maximum.");
        }
        return null;
    }

    static DeprecationIssue checkNoMasterBlock(ClusterState state) {
        if (state.metaData().settings().hasValue(DiscoverySettings.NO_MASTER_BLOCK_SETTING.getKey())) {
            return new DeprecationIssue(DeprecationIssue.Level.WARNING, "Master block setting will be renamed", "https://www.elastic.co/guide/en/elasticsearch/reference/7.0/breaking-changes-7.0.html#_new_name_for_literal_no_master_block_literal_setting", "The setting discovery.zen.no_master_block will be renamed to cluster.no_master_block in 7.0. Please unset discovery.zen.no_master_block and set cluster.no_master_block after upgrading to 7.0.");
        }
        return null;
    }

    static DeprecationIssue checkClusterName(ClusterState state) {
        String clusterName = state.getClusterName().value();
        if (clusterName.contains(":")) {
            return new DeprecationIssue(DeprecationIssue.Level.CRITICAL, "Cluster name cannot contain ':'", "https://www.elastic.co/guide/en/elasticsearch/reference/7.0/breaking-changes-7.0.html#_literal_literal_is_no_longer_allowed_in_cluster_name", "This cluster is named [" + clusterName + "], which contains the illegal character ':'.");
        }
        return null;
    }

    static DeprecationIssue checkUserAgentPipelines(ClusterState state) {
        List pipelines = IngestService.getPipelines((ClusterState)state, (String[])new String[0]);
        List pipelinesWithDeprecatedEcsConfig = pipelines.stream().filter(Objects::nonNull).filter(pipeline -> {
            Map pipelineConfig = pipeline.getConfigAsMap();
            List processors = (List)pipelineConfig.get("processors");
            return processors.stream().filter(Objects::nonNull).filter(processor -> processor.containsKey("user_agent")).map(processor -> (Map)processor.get("user_agent")).anyMatch(processorConfig -> !processorConfig.containsKey("ecs"));
        }).map(PipelineConfiguration::getId).sorted().collect(Collectors.toList());
        if (!pipelinesWithDeprecatedEcsConfig.isEmpty()) {
            return new DeprecationIssue(DeprecationIssue.Level.WARNING, "User-Agent ingest plugin will use ECS-formatted output", "https://www.elastic.co/guide/en/elasticsearch/reference/7.0/breaking-changes-7.0.html#ingest-user-agent-ecs-always", "Ingest pipelines " + pipelinesWithDeprecatedEcsConfig + " will change to using ECS output format by default in 7.0");
        }
        return null;
    }

    static DeprecationIssue checkFormatOnPipeline(ClusterState state) {
        List pipelines = IngestService.getPipelines((ClusterState)state, (String[])new String[0]);
        StringJoiner pipelinesMessages = new StringJoiner("\n");
        for (PipelineConfiguration pipeline : pipelines) {
            Map pipelineConfig = pipeline.getConfigAsMap();
            List processors = (List)pipelineConfig.get("processors");
            ArrayList<String> fields = new ArrayList<String>();
            for (Map processor : processors) {
                Map date_index_name;
                String index_name_format;
                if (processor.containsKey("date")) {
                    Map date = (Map)processor.get("date");
                    List formats = (List)date.get("formats");
                    for (String format : formats) {
                        if (!JodaDeprecationPatterns.isDeprecatedPattern((String)format)) continue;
                        fields.add(ClusterDeprecationChecks.formatPipelineField("date", format));
                    }
                }
                if (processor.containsKey("date_index_name") && JodaDeprecationPatterns.isDeprecatedPattern((String)(index_name_format = (String)(date_index_name = (Map)processor.get("date_index_name")).get("index_name_format")))) {
                    fields.add(ClusterDeprecationChecks.formatPipelineField("index_name_format", index_name_format));
                }
                if (fields.isEmpty()) continue;
                pipelinesMessages.add("pipelineId: " + pipeline.getId() + ", fields: " + fields + ".");
            }
        }
        if (pipelinesMessages.length() != 0) {
            return new DeprecationIssue(DeprecationIssue.Level.WARNING, "Pipelines contain date fields with deprecated format", "https://www.elastic.co/guide/en/elasticsearch/reference/7.0/breaking-changes-7.0.html#breaking_70_java_time_changes", "Ingest pipelines which contain deprecated date formats: " + pipelinesMessages);
        }
        return null;
    }

    private static String formatPipelineField(String field, String format) {
        return "[field: " + field + ", format: " + format + ", suggestion: " + JodaDeprecationPatterns.formatSuggestion((String)format) + "]";
    }

    static DeprecationIssue checkTemplatesWithTooManyFields(ClusterState state) {
        Integer maxClauseCount = (Integer)SearchModule.INDICES_MAX_CLAUSE_COUNT_SETTING.get(state.getMetaData().settings());
        ArrayList templatesOverLimit = new ArrayList();
        state.getMetaData().getTemplates().forEach(templateCursor -> {
            AtomicInteger maxFields = new AtomicInteger(0);
            String templateName = (String)templateCursor.key;
            boolean defaultFieldSet = ((IndexTemplateMetaData)templateCursor.value).getSettings().get("index.query.default_field") != null;
            ((IndexTemplateMetaData)templateCursor.value).getMappings().forEach(mappingCursor -> {
                String mappingTypeName = (String)mappingCursor.key;
                MappingMetaData mappingMetaData = null;
                try {
                    mappingMetaData = new MappingMetaData((CompressedXContent)mappingCursor.value);
                }
                catch (IOException e) {
                    logger.error("failed to parse mapping for type {}: {}", (Object)mappingTypeName, (Object)e);
                }
                if (mappingMetaData != null && !defaultFieldSet) {
                    maxFields.set(IndexDeprecationChecks.countFieldsRecursively(mappingMetaData.type(), mappingMetaData.sourceAsMap()));
                }
                if (maxFields.get() > maxClauseCount) {
                    templatesOverLimit.add(templateName);
                }
            });
        });
        if (!templatesOverLimit.isEmpty()) {
            return new DeprecationIssue(DeprecationIssue.Level.WARNING, "Fields in index template exceed automatic field expansion limit", "https://www.elastic.co/guide/en/elasticsearch/reference/7.0/breaking-changes-7.0.html#_limiting_the_number_of_auto_expanded_fields", "Index templates " + templatesOverLimit + " have a number of fields which exceeds the automatic field expansion limit of [" + maxClauseCount + "] and does not have [" + "index.query.default_field" + "] set, which may cause queries which use automatic field expansion, such as query_string, simple_query_string, and multi_match to fail if fields are not explicitly specified in the query.");
        }
        return null;
    }
}

