/*
* last edit before me 19.dec.'05
* MyGoGrinder - a program to practice Go problems
* Copyright (c) 2004-2006 Tim Kington
*   timkington@users.sourceforge.net
* Portions Copyright (C) Ruediger Klehn (2014)
*   RuediRf@users.sourceforge.net
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*
*/
// Compiler message: "GoGrinder\WGFController.java uses unchecked or unsafe operations."

package GoGrinder;

import java.io.*;
import java.awt.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;

import GoGrinder.sgf.*;
import GoGrinder.tests.Test;
import GoGrinder.ui.WGFFrame;
import GoGrinder.ui.SelectionDialog;
import GoGrinder.command.*;
import GoGrinder.editor.EditTool;
import GoGrinder.editor.FakeTool;
import GoGrinder.editor.LabelTool;
import GoGrinder.editor.LetterTool;
import GoGrinder.editor.LocalMoveTool;
import GoGrinder.editor.MarkTool;
import GoGrinder.editor.MoveTool;
import GoGrinder.editor.NumberTool;
import GoGrinder.editor.RealTool;
import GoGrinder.editor.TerritoryTool;

/**
 *
 * @author  tkington
 */
public class WGFController extends Controller implements HyperlinkListener {
  public static final int TEACH_MODE = 0; // is here thought of a 3rd mode "DEMO_MODE" / "VIEW_MODE"?
  public static final int EDIT_MODE = 1;  // else it would be enough to use boolean

  public static final int TEXTMODE_VIEWHTML = 0;
  public static final int TEXTMODE_EDITHTML = 1;
  public static final int TEXTMODE_EDITWGF = 2;

  public static EditTool TOOL_REAL;
  public static EditTool TOOL_FAKE;
  public static EditTool TOOL_MOVE;
  public static EditTool TOOL_LOCALMOVE;
  public static EditTool TOOL_TRI;
  public static EditTool TOOL_CIR;
  public static EditTool TOOL_SQU;
  public static EditTool TOOL_X;
  public static EditTool TOOL_LETTER;
  public static EditTool TOOL_NUMBER;
  public static EditTool TOOL_LABEL;
  public static EditTool TOOL_TERRITORY;
  public static String thisWGFFileStr = "";
  private WGFFrame mainFrame;
    private int mode;
    private File curWgfFile;
    private WGFNode curNode;
    private ArrayList nodes;
    private JEditorPane textPane;
    private JButton stepButton;
    private HashMap nodeMap;
    private Stack undoStack;
    private ArrayList globalReasons;

    private ArrayList stepCommands;
    private int stepNum;
    
    private boolean isDirty;
    private EditTool editTool;
    private int textPaneMode;
    
    public WGFController(WGFFrame mainFrame) {
      this.mainFrame = mainFrame;
      isDirty = false;
      mode = TEACH_MODE;
        undoStack = new Stack();

        TOOL_FAKE = new FakeTool(this);
        TOOL_REAL = new RealTool(this);
        TOOL_MOVE = new MoveTool(this);
        TOOL_LOCALMOVE = new LocalMoveTool(this);
        TOOL_TRI = new MarkTool(this, NodeMark.TRI);
        TOOL_CIR = new MarkTool(this, NodeMark.CIR);
        TOOL_SQU = new MarkTool(this, NodeMark.SQU);
        TOOL_X = new MarkTool(this, NodeMark.X);
        TOOL_LETTER = new LetterTool(this);
        TOOL_NUMBER = new NumberTool(this);
        TOOL_LABEL = new LabelTool(this);
        TOOL_TERRITORY = new TerritoryTool(this);
        
        editTool = TOOL_FAKE;
    }
    
    public void setControls(JEditorPane textPane, JButton stepButton) {
        this.textPane = textPane;
        this.stepButton = stepButton;
    }
    
    public void onNewFile() {
      if(!saveChanges())
        return;

      String str = JOptionPane.showInputDialog(mainFrame, Messages.getString("WGFController.BoardSize")); //$NON-NLS-1$
      int size = 0;
      try {
        size = Integer.parseInt(str);
      }
      catch(NumberFormatException e) {
        JOptionPane.showMessageDialog(mainFrame, "Not enough number for me \n"  // :-) //$NON-NLS-1$
                                                 + "Setting boardsize to default = 9"); //$NON-NLS-1$
        size = 9;  // this is better - and no return!
      }
      if(size < 5 || size > 19) { // WAS: WGFController.LegalSizes5To19
        JOptionPane.showMessageDialog(mainFrame, Messages.getString("WGFController.AcceptedSizes5To19") + "\n" //$NON-NLS-1$ //$NON-NLS-2$
                                                 + "Setting boardsize to default = 9"); //$NON-NLS-1$ 
        size = 9; // this is better - and no return!                 // WGFController.DefaultSize
      }
 //SGFParser.wgfSZ = size;
      setFile(null);
//      setTextPaneMode(TEXTMODE_VIEWHTML);
      nodes = new ArrayList();
      
      nodeMap = new HashMap();
      undoStack = new Stack();
//d.b.g(size);
      // the size needs to be handed over to sgfParser.wgfSZ + szIsSet!
      Node.szIsSet = false;
      try {
        curNode = new WGFNode("SZ[" + size //$NON-NLS-1$
                  + "]C[<html></html>]", null); //$NON-NLS-1$ 
                      // ... C[<html></html>] - would it be enough to set this when beginning to edit?
      }
      catch(SGFParseException e) { Main.log(e); }
      
      nodes.add(curNode);
      globalReasons = null;
      
      curNode.registerNode(nodeMap);
      
      createBoard(-1);
      
      curNode.execute(board);
      fillTextPane();
      
      panel.hideGhost();
      
      setDirty(false);
    }
    
    public void openFile(File wgfFileOpen) throws SGFParseException, IOException {
        if(!saveChanges())
            return;
        setFile(wgfFileOpen);
//        setTextPaneMode(TEXTMODE_VIEWHTML);
        thisWGFFileStr = wgfFileOpen.toString();
        nodes = WGFParser.parse(wgfFileOpen);
        
        nodeMap = new HashMap();
        undoStack = new Stack();
        
        curNode = (WGFNode)nodes.get(0);
        globalReasons = curNode.getReasons();

        for(int i = 0; i < nodes.size(); i++)
            ((WGFNode)nodes.get(i)).registerNode(nodeMap);
        
        createBoard(-1);
        
        curNode.execute(board);
        fillTextPane();
        
        panel.hideGhost();
        
        setDirty(false);
    }
    
    public boolean saveChanges() {
      //boolean success = false;
      boolean curWgfFileNameIsValid = true;
      if(textPaneMode != TEXTMODE_VIEWHTML) {
        if(!setTextPaneMode(TEXTMODE_VIEWHTML)) {  // why not simply a combined query (&&)? tk thought of more here?
          JOptionPane.showMessageDialog(mainFrame, Messages.getString("WGFController.SaveFailed")); //$NON-NLS-1$
          return false;
        }
      }
      
      if(isDirty) {
      // just switching to "edit html"/"edit wgf" sets dirty to true - should check, if html/wgf is edited
        String filename = Messages.getString("WGFController.TheCurrentFile"); //$NON-NLS-1$
        if(curWgfFile != null)
          filename = curWgfFile.getName();
        else
          curWgfFileNameIsValid = false;
        int choice;
        while(true){ // true instead of success
          choice = JOptionPane.showConfirmDialog(mainFrame, " " + filename 
              + Messages.getString("WGFController.HasChangedSaveChanges"), Messages.getString("WGFController.ConfirmSave"), //$NON-NLS-1$ //$NON-NLS-2$
            JOptionPane.YES_NO_CANCEL_OPTION);
            // here was an abort[esc]-problem: it's overwritten!  seemingly DONE
            // here is an [esc]-problem: if dirty, we're not thrown back
          if(choice == JOptionPane.CANCEL_OPTION || choice == JOptionPane.CLOSED_OPTION) // 2 || -1 ([ESC]/[x])
          // NO_OPTION = 1
            return false;
          else if(choice == JOptionPane.YES_OPTION) { // 0
            if(!curWgfFileNameIsValid){
              if (!saveFileAs())
                continue;
              else{
                setDirty(true);
                break;
              }
            }
            if(curWgfFileNameIsValid && !saveFile(curWgfFile))
              return false;
          }
          else break;
        } // while
        setDirty(false);
      }
      
      return true;
    }
    
    private void createBoard(int oldSize) {
// d.b.g(SGFParser.wgfSZ, "wgfcontr, createBoard");
        int size = curNode.getSize(); // curNode is WGFNode
        if(size == 0) {
            if(oldSize != -1)
                size = oldSize;
            else size = 19; // SGFParser.wgfSZ 
                            // noSiZe -> SiZe = 19 is this setting size to default if SZ[##] is not set? YES 
                            // seems like unclean solution without looking for the real size
                            // not sure: does this just default to 19 is sz is not set? (then this is o.k.)
        }

        setBoard(new Board(size));
    }
    
    public void setBoard(Board b) {
        board = b;
        panel.setBoard(board);
        board.setPanel(panel);

        int size = board.getSize();
        panel.setSize(new Rectangle(size, size), null);
    }
    
    public void mouseClicked(int x, int y, int modifiers) {
      switch(mode) {
      case TEACH_MODE: normalMouseClicked(x, y, modifiers); break;
      case EDIT_MODE: editMouseClicked(x, y, modifiers); break;
      }
    }
    
    public void mouseWheelMoved(int numClicks) { /* */ }
    
    private void normalMouseClicked(int x, int y, int modifiers) {
        Test t = curNode.getTest();
        if(t == null) {
            Toolkit.getDefaultToolkit().beep();
            return;
        }
        
        t.handleClick(board, curNode, globalReasons, x, y, modifiers);
    }
    
    private void editMouseClicked(int x, int y, int modifiers) {
      if(editTool.mouseClicked(x, y, modifiers))
        setDirty(true);
      else Toolkit.getDefaultToolkit().beep();
    }
    
    public void home() {
      if(!updateNodeFromTextPane())
        return;
      
        undoStack = new Stack();
        
        curNode = (WGFNode)nodes.get(0);
        
        createBoard(-1);
        
        curNode.execute(board);
        fillTextPane();
    }
    
    public void prev() {
      if(!updateNodeFromTextPane())
        return;
      
        if(undoStack.isEmpty()) {
            Toolkit.getDefaultToolkit().beep();
            return;
        }
        
        UndoEntry e = (UndoEntry)undoStack.pop();
        if(e.board == null) {
            for(int i = 0; i < e.numUndos; i++) {
                curNode = (WGFNode)curNode.getParent();
                board.undoLast();
            }
            
            curNode.updateState(board);
            fillTextPane();
        }
        else {
            board = e.board;
            setBoard(board);
            curNode = e.prevNode;
            curNode.updateState(board);
            fillTextPane();
        }
        
        enableStep();
    }
    
    public void next() {
      if(!updateNodeFromTextPane())
        return;
      
        //  If we need to show the answer to a test before
        //  moving on, do so
        if(curNode.hasAnswer()) {
          curNode.showAnswer(board);
            return;
        }
        
        String nextLink = curNode.getNextLink();
        if(nextLink == null) {
            Toolkit.getDefaultToolkit().beep();
            return;
        }
        
        WGFNode next;
        if(nextLink.equals("NEXTNODE")) //$NON-NLS-1$
            next = (WGFNode)curNode.getFirstChild();
        else next = (WGFNode)nodeMap.get(nextLink);
        
        goToNode(next, true);
    }
    
    public void backToParent() {
        UndoEntry e = (UndoEntry)undoStack.peek();
        
        curNode = (WGFNode)curNode.getParent();
        board.undoLast();

        e.numUndos--;
        if(e.numUndos == 0)
            undoStack.pop();

        curNode.updateState(board);
        fillTextPane();
    }
    
    public void goToNode(WGFNode next, boolean updateState) {
      if(next == null) {
            Toolkit.getDefaultToolkit().beep();
            return;
        }
        
        if(next == curNode.getFirstChild()) {
            undoStack.push(new UndoEntry(1, null, null, null));
            curNode = next;
            curNode.execute(board);
            if(updateState)
              fillTextPane();
        }
        else if(isDescendant(curNode, next)) {
            int num = 0;
            while(curNode != next) {
                curNode = (WGFNode)curNode.getFirstChild();
                curNode.execute(board);
                if(updateState)
                  fillTextPane();
                num++;
            }

            undoStack.push(new UndoEntry(num, null, null, null));
        }
        else if(isDescendant(next, curNode)) {
            while(curNode != next)
                backToParent();
        }
        else {
            //  Next node is unrelated
            undoStack.push(new UndoEntry(-1, curNode, next, board));
            curNode = next;
            createBoard(board.getSize());
            curNode.execute(board);
            if(updateState)
              fillTextPane();
        }
        
        enableStep();
    }
    
    public void hyperlinkUpdate(HyperlinkEvent e) {
        if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
            String desc = e.getDescription();
            
            if(desc.equals("NEXTNODE")) { //$NON-NLS-1$
                next();
                return;
            }
            
            WGFNode next = null;
            
            String [] t = desc.split("\\:"); //$NON-NLS-1$
            switch(t.length) {
                case 1:
                    next = (WGFNode)nodeMap.get(t[0]);
                    break;
                case 2:
                    next = (WGFNode)nodeMap.get(t[1]);
                    break;
                default:
                    Toolkit.getDefaultToolkit().beep();
                    return;
            }
            
            if(!updateNodeFromTextPane())
              return;
            goToNode(next, true);
        }
    }
    
    public void enableStep() {
        stepCommands = curNode.getStepCommands(board);
        
        if(stepCommands != null && stepCommands.size() > 1) {
            stepNum = stepCommands.size();
            if(mode != EDIT_MODE)
              stepButton.setEnabled(true);
        }
        else {
            stepNum = -1;
            stepCommands = null;
            stepButton.setEnabled(false);
        }
    }
    
    public void step() {try{ // ############  needs to be handled correctly!
        if(stepNum == stepCommands.size()) {
            for(int i = stepNum - 1; i > 0; i--)
                board.executeTempCommand((Command)stepCommands.get(i));
            stepNum = 1;
        }
        else {
            board.undoTempCommand();
            stepNum++;
        }
        }
        catch(Exception e){Main.logSilent(e);} // ######################### needs to be handled correctly!
    }
    
    public void saveHistory() { // what is the sense of the history? (the program remembers only one?)
        LinkedList hist = new LinkedList<String>();  // ########### "new LinkedList<String ??>()" 
        while(!undoStack.isEmpty()) {
            UndoEntry e = (UndoEntry)undoStack.pop();
            if(e.numUndos != -1)
                hist.addFirst(new Integer(e.numUndos));
            else hist.addFirst(e.nextNode.getName());
        }
        
        hist.addFirst(curWgfFile);
        
        GS.setHistory(hist);
    }
    
    public void loadHistory() {
        String loadingDefault = "Loading default file: \n"
                              + "\"" + Main.pathToSettings + Main.SLASH + "default.wgf\"";
        LinkedList hist = GS.getHistory(); // ##############  "new LinkedList<String ??>()"
        // without the default file we get a nullPointer when aborting selection of a new file (is still very wormy)
        File defaultWGF = new File(Main.pathToDefaultWGFFile);
        File wgfFileFromHist = null;
        if(hist == null){ // fresh installation  // ###### Main.warningsStartup
          if(Main.warningsStartup)
            JOptionPane.showMessageDialog(null, "No wgf file history found. \n" + loadingDefault);
          wgfFileFromHist = defaultWGF;
          hist = new LinkedList<String>();
          hist.addFirst(wgfFileFromHist.getName());
        }
        else
          wgfFileFromHist = (File)hist.removeFirst();
          
        if(wgfFileFromHist == null){ // exited program with no name for the current wgf
          JOptionPane.showMessageDialog(null, "No wgf file found in file history. \n" + loadingDefault);
          wgfFileFromHist = defaultWGF;
        }
        
        try {
            if ( (!wgfFileFromHist.exists() || !wgfFileFromHist.isFile())) { // file from history moved or deleted, or is now e.g. a folder name
              if (!Main.PORTABLE ) {
                JOptionPane.showMessageDialog(mainFrame, "File from saved history \n\"" 
                                                       + wgfFileFromHist.getAbsolutePath()
                                                       + "\"\n not found! \n\n"
                                                       + loadingDefault);
              }
           // JOptionPane.  ... Buttons retry, default, ([x] = default)
              wgfFileFromHist = defaultWGF;  // there is still enough, what can go wrong
            }


            if ( !wgfFileFromHist.canRead() ) { // which restrictions? - file system error?
              JOptionPane.showMessageDialog(mainFrame, "File from saved history \n\"" 
                                                     + wgfFileFromHist.getAbsolutePath()
                                                     + "\"\n cannot be read! \n\n"
                                                     + loadingDefault);
              wgfFileFromHist = defaultWGF;
            }
            openFile(wgfFileFromHist); 
        }
        catch(Exception e) { // I don't know, what happened
            JOptionPane.showMessageDialog(mainFrame, Messages.getString("WGFController.ErrWhileReadingFile") //$NON-NLS-1$
                                                + "\"" + wgfFileFromHist.getAbsolutePath() + "\"\n"
                                                //+ " Please check the file with a text editor or rename \n"
                                                //+ "it" 
                                                + Messages.getString("WGFController.SeeGrindLog")
                                                + " We will try to load the default file."); //$NON-NLS-1$ // how could we come here?
                                                // WGFController.SeeGrindLog begins with ". " 
            Main.logSilent(e);
            try{openFile(defaultWGF);}catch(Exception e3){/*shit!*/;}
        }  // if we have no valid file up to here - we have a problem... 

        while(!hist.isEmpty()) {
            Object o = hist.removeFirst();
            if(o instanceof Integer) {
                int n = ((Integer)o).intValue();
                WGFNode node = curNode;
                for(int i = 0; i < n; i++) {
                    node = (WGFNode)node.getFirstChild();
                }
                goToNode(node, false);
            }
            else {
                String name = (String)o;
                WGFNode node = (WGFNode)nodeMap.get(name);
                goToNode(node, false);
            }
        }
        curNode.updateState(board);
        fillTextPane();
        enableStep();
    }
    
    private boolean isDescendant(WGFNode parent, WGFNode d) {
        while(d != null && d != parent)
            d = (WGFNode)d.getParent();

        return d == parent;
    }
    
    public boolean saveFile() {
      if(curWgfFile == null || !curWgfFile.canWrite()) 
        return saveFileAs();
      return saveFile(curWgfFile);
    }
    
    public boolean saveFileAs() {
      String filename = FileWorks.selectFile(mainFrame, FileFilters.wgfFilter,
                                             Messages.getString("WGFFrame.SaveFile"),
                                             true, Messages.getString("WGFFrame.Save"));
         //              Main.chooseFile(mainFrame, FileFilters.wgfFilter, 
         //Messages.getString("WGFFrame.SaveFile"), //$NON-NLS-1$
         //true, Messages.getString("WGFFrame.Save")); //$NON-NLS-1$

      if (filename == null )
        return false;
 /*NEW->*/ 
      //File filenameF = new File(filename);
      //if (filenameF.exists()) {
      //  if (!SelectionDialog.OKCancelMsg("Warning! File exists - overwrite?", "Overwrite file?")){
      //    return false;
      //  }
      //}
 /*<-NEW*/ 
      curWgfFile = new File(filename); //filenameF;
      return saveFile(curWgfFile);
    }

    private boolean saveFile(File wgfFileSave) {
      if(textPaneMode != TEXTMODE_VIEWHTML) {
        if(!setTextPaneMode(TEXTMODE_VIEWHTML)) {
          JOptionPane.showMessageDialog(mainFrame, Messages.getString("WGFController.SaveFailed")); //$NON-NLS-1$
          return false;
        }
      }
      try { // if write file not possible...?
          PrintWriter out = new PrintWriter(new FileOutputStream(wgfFileSave));
          for(int i = 0; i < nodes.size(); i++) {
            out.println("("); //$NON-NLS-1$
            WGFNode node = (WGFNode)nodes.get(i);
            StringBuffer s = new StringBuffer();
            node.toFileFormat(s);
            out.print(s);
            out.println(")"); //$NON-NLS-1$
          }
        
          out.close();
          
          setFile(wgfFileSave);
          setDirty(false);
          
          return true;
      }
      catch(Exception e) {
          JOptionPane.showMessageDialog(mainFrame,
              Messages.getString("WGFFrame.ErrSavingSeeGrindLog")); //$NON-NLS-1$
          Main.logSilent(e);
          return false;
      }
    }
    
    public int getMode() { return mode; }
    
    public void setEditMode(boolean m) {
      mode = m ? EDIT_MODE : TEACH_MODE;
      if(m)
        board.undoTempCommands();
      else enableStep();
    }
    
    public void insertNode() {
      WGFNode node = null;
      try {
        node = new WGFNode("", null); //$NON-NLS-1$
      }
      catch(SGFParseException e) {
        Main.log(e);
      }
      insertNode(node);
    }
    
    public void insertNodeCopy() {
      insertNode(curNode.createCopy());
    }
    
    private void insertNode(WGFNode node) {
      if(!updateNodeFromTextPane())
        return;
      curNode.insertNodeAfter(node);
      goToNode(node, true);
      setDirty(true);
    }
    
    public boolean hasAddCommand(int x, int y, int color) {
      return curNode.hasAddCommand(x, y, color);
    }
    
    public void addAddCommand(int x, int y, int color) {
      board.undoLast();
      curNode.addAddCommand(x, y, color);
      curNode.execute(board);
    }
    
    public void removeAddCommand(int x, int y, int color) {
      board.undoLast();
      curNode.removeAddCommand(x, y, color);
      curNode.execute(board);
    }
    
    public boolean hasCommand(Command c) {
      return curNode.hasCommand(c);
    }
    
    public void addCommand(Command c) {
      board.undoLast();
      curNode.addCommand(c);
      curNode.execute(board);
      enableStep();
    }
    
    public void removeCommand(Command c) {
      board.undoLast();
      curNode.removeCommand(c);
      curNode.execute(board);
      enableStep();
    }
    
    public SimpleMark getSimpleMark(int x, int y, int type) {
      return curNode.getSimpleMark(x, y, type);
    }
    
    public boolean hasStoneAt(int x, int y){
// try{
      if(board.getAt(x, y) != 0)
        return true;
      
      return curNode.hasStoneMarkAt(x, y);
 //     }
 //catch(SGFParseException e){Main.logSilent(e, curWgfFile.getName());return false;}
}
    
    public void addMark(NodeMark m) {
      curNode.addMark(m);
      board.addMark(m);
      setDirty(true);
      enableStep();
    }
    
    public void removeMark(NodeMark m) {
      curNode.removeMark(m);
      board.removeMark(m);
      setDirty(true);
      enableStep();
    }
    
    public String getNextLabel() {
      return curNode.getNextLabel();
    }
    
    public String getNextNumberLabel() {
      return curNode.getNextNumberLabel();
    }
    
    public boolean setTextPaneMode(int mode) {
      if(!updateNodeFromTextPane())
        return false;
      
      textPaneMode = mode;
      if(mode != TEXTMODE_VIEWHTML)
        setDirty(true);
      
      String contentType = null;
      switch(mode) {
      case TEXTMODE_VIEWHTML:
        contentType = "text/html"; //$NON-NLS-1$
        break;
      case TEXTMODE_EDITHTML:
        contentType = "text/plain"; //$NON-NLS-1$
        break;
      case TEXTMODE_EDITWGF:
        contentType = "text/plain"; //$NON-NLS-1$
        break;
      }
      
      textPane.setContentType(contentType);
      textPane.setEditable(mode != TEXTMODE_VIEWHTML);
      
      fillTextPane();
      return true;
    }
    
    private void fillTextPane() {
      String text = null;
      
      switch(textPaneMode) {
      case TEXTMODE_VIEWHTML:
      case TEXTMODE_EDITHTML:
        text = curNode.getComment();
        break;
      case TEXTMODE_EDITWGF:
        StringBuffer wgf = new StringBuffer();
        curNode.getLocalWGF(wgf);
        text = wgf.toString();
        break;
      }
      
      textPane.setText(text);
      textPane.setCaretPosition(0);
    }
    
    private boolean updateNodeFromTextPane() {
    switch(textPaneMode) {
    case TEXTMODE_EDITHTML:
      curNode.setComment(textPane.getText());
      break;
    case TEXTMODE_EDITWGF:
      //	Remove node from map in case name changes
      String name = curNode.getName();
      if(name != null)
        nodeMap.remove(name.toUpperCase());
      
        board.undoLast();
        
      boolean badWGF = false;
        try {
          curNode.parse(textPane.getText());
        }
        catch(SGFParseException e) {
          JOptionPane.showMessageDialog(mainFrame,
              Messages.getString("WGFController.ErrorParsingNodeSeeGrindLog")); //$NON-NLS-1$
          Main.logSilent(e);
          badWGF = true;
        }
        
        name = curNode.getName();
        if(name != null)
          nodeMap.put(name.toUpperCase(), curNode);
        
        curNode.execute(board);
        enableStep();
        
        if(badWGF)
          return false;
      break;
    }
    
    return true;
  }
    
    private void setDirty(boolean dirty) {
      isDirty = dirty;
      setTitle();
    }
    
    private void setFile(File wgfFileSet) {
      curWgfFile = wgfFileSet;
      setTitle();
    }
    
    private void setTitle() { // This doesn't belong to WGFFrame?
      String title = (Main.TEST) ? "TEST_WGF_Editor_TEST - " : Messages.getString("WGFController.GoGrinderDash") + " "; //$NON-NLS-1$
      
      if(curWgfFile != null)
        title += curWgfFile.getName();
      else title += Messages.getString("WGFController.Untitled"); //$NON-NLS-1$
      
      if(isDirty)
        title += "*"; //$NON-NLS-1$
      title += "       -       Testing the WGF editor";
      mainFrame.setTitle(title);
    }
    
    public void setEditTool(EditTool editTool) { this.editTool = editTool; }
    public JEditorPane getTextPane() { return textPane; }
  
  static class UndoEntry {
        //  If the user went forward, num contains the number of moves forward,
        //  and board and node are null.  If not, the user followed a hyperlink.
        Board board;
        WGFNode prevNode;
        WGFNode nextNode;
        int numUndos;
        
        UndoEntry(int num, WGFNode prevNode, WGFNode nextNode, Board b) {
            numUndos = num;
            this.prevNode = prevNode;
            this.nextNode = nextNode;
            board = b;
        }
    }
}
