//
// $Id: ObjectOutStream.h,v 1.13 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(OBJECTOUTSTREAM_OL_GUARD)
#define OBJECTOUTSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredOutStream.h>

@class OLPointerRefTable;
@class OLObjectReplaceTable;

/**
 * @class OLObjectOutStream ObjectOutStream.h ObjectiveLib/ObjectOutStream.h
 *
 * A stream that is capable of writing objects and classes. In addition to having
 * the ability to write objects and classes, OLObjectOutStream tags every type
 * written to the stream, which is then verified when OLObjectInStream reads the
 * value. If a certain type is expected but another type is found, an exception
 * is raised to indicate the inconsistency. Also, objects and classes are tracked
 * as they are written, and when an object or class is written that already appears
 * in the stream, a handle is written that refers to the instance already written.
 * This improves performance and reduces the size of the resulting data.
 *
 * @sa OLObjectInStream
 *
 * @ingroup Streams
 */
@interface OLObjectOutStream : OLLayeredOutStream
{
@protected
    /**
     * The map of pointers to object, selectors and classes that assoicates
     * them with handles
     */
    OLPointerRefTable*      pointerMap;

    /**
     * The map of objects and their replacements
     */
    OLObjectReplaceTable*   replacementMap;
}

/**
 * Create and return a new stream. The @a underStream becomes the next lower
 * stream in the stream stack.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param underStream the stream under the new one in the stream stack
 * @return a reference to the newly created stream
 */
+ (id) streamWithOutStream: (OLOutStream*)underStream;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. @a underStream becomes the next lower stream in the
 * stream stack.
 *
 * @param underStream the stream under the new one in the stream stack
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream;

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif
/**
 * Replace all instances of an object. After receiving this message the stream
 * will always replace any occurrence of @a obj with @a rep.
 *
 * @param obj the object to replace
 * @param rep the replacement object
 */
- (void) replaceObject: (id)obj withObject: (id)rep;
- (void) writeBool: (BOOL)value;

/**
 * Write a class. The class will be written unless this class has already been
 * written to the stream, in which case a reference to it will be written.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param cls the class to write
 */
- (void) writeClass: (Class)cls;
- (void) writeDouble: (double)value;
- (void) writeFloat: (float)value;

/**
 * Write the object stream header. This method can be overridden to write a custom
 * header to the object stream. It will be called in due course, so it is
 * @b never necessary to call this method.
 *
 * @note Subclasses that override this method should call the superclass' method
 * before writing any additional data.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 */
- (void) writeHeader;
- (void) writeInt: (unsigned)value;
- (void) writeInt16: (uint16_t)value;
- (void) writeInt32: (uint32_t)value;
- (void) writeInt64: (uint64_t)value;

/**
 * Write an object. The object will be written unless this object has already been
 * written to the stream, in which case a reference to it will be written.
 *
 * @pre @a object must support either the OLStreamable protocol or the NSCoding
 * protocol. If both protocols are found, then preference is given to the methods
 * from OLStreamable.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param object the object to write
 */
- (void) writeObject: (id)object;
#if !defined(OL_NO_OPENSTEP)
/**
 * @name NSCoder Support
 *
 * Support the streaming of objects that adopt the NSCoding protocol
 */
/* @{ */
- (void) encodeBytes: (const void*)address length: (unsigned)numBytes;
- (void) encodeDataObject: (NSData*)data;

/**
 * Encode an object. The object will be written unless this object has already been
 * written to the stream, in which case a reference to it will be written. This
 * method forwards @a object to #writeObject:, so there is nothing to be gained by
 * using this method instead of that one.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param object the object to write
 */
- (void) encodeObject: (id)object;

/**
 * Encode a value of a specified Objective-C type. Write a value of an arbitrary type to
 * the stream. Note that in order to preserve platform-independence in the resulting
 * stream, constraints are placed on how the data are written. Types that otherwise have
 * indeterminate sizes will always be normalized to sizes chosen by @b ObjectiveLib. The
 * sizes written are as follows:
 * <table border=0>
 * <tr><td><b>Type</b></td><td><b>Size</b></td></tr>
 * <tr><td>char</td><td>1</td></tr>
 * <tr><td>short</td><td>2</td></tr>
 * <tr><td>int</td><td>4</td></tr>
 * <tr><td>long</td><td>4</td></tr>
 * <tr><td>long long</td><td>8</td></tr>
 * </table>
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param valueType the type of the value at @a address
 * @param address the location of the value
 */
- (void) encodeValueOfObjCType: (const char*)valueType at: (const void*)address;

/**
 * Return the current system version. The version refers to the version of the
 * @b ObjectiveLib streaming system.
 *
 * @return the system version
 */
- (unsigned) systemVersion;

/**
 * Return the current version of a class. The name of the class is used to look up
 * its version in the Objective-C runtime system. This version is returned.
 *
 * @note If no class by the given name can be found in the runtime, then the value
 * NSNotFound is returned.
 *
 * @param className the name of the class
 * @return the version of the class
 */
- (unsigned) versionForClassName: (NSString*)className;
/* @} */
#endif

@end

#endif
