#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "xtux.h"
#include "server.h"
#include "entity.h"
#include "clients.h"
#include "sv_net.h"
#include "weapon.h"
#include "sv_map.h"
#include "hitscan.h"

extern map_t *map;
extern entity *root;
extern entity *tail;
extern server_t server;
extern byte num_entity_types;

byte num_weapon_types;

static void muzzle_flash(point P, byte dir);

void weapon_init(void)
{
    ent_type_t *et;
    int weapon;
    int i;

    num_weapon_types = weapon_type_init();

    /* Match weapon names to weapon types */
    for( i=0 ; i < num_entity_types ; i++) {
	et = entity_type(i);

	if( et->weapon_name[0] == '\0' )
	    continue; /* Empty, skip */

	if( et->class == ITEM && et->item_action == GIVEWEAPON ) {
	    /* The Weapon a GIVEWEAPON item gives you */
	    if( (weapon = match_weapon_type( et->weapon_name )) >= 0 )
		et->item_type = weapon;
	    else {
		printf("Couldn't match weapon type for item %s\n", et->name);
		et->item_type = 0;
	    }
	} else if( et->weapon_name ) {
	    /* Default weapons for entities */
	    if( (weapon = match_weapon_type( et->weapon_name )) >= 0 )
		et->weapon = weapon;
	    else {
		printf("Couldn't match weapon type for entity %s\n", et->name);
		et->weapon = 0;
	    }
	}

    }

}


void weapon_update(void)
{
    entity *ent;
    msec_t now;
    int weap, i;

    for( ent = root ; ent != NULL ; ent = ent->next ) {
	if( ent->trigger )
	    weapon_fire(ent);

	if( ent->weapon_switch ) {
	    now = gettime();
	    if( now - ent->last_switch >= WEAPON_SWITCH_TIME ) {

		weap = ent->weapon;
		i = 0; /* Amount of weapons we have tried */
		do {
		    if( i++ >= num_weapon_types ) {
			printf("entity has no weapons!\n");
			weap = WT_NONE;
			break;
		    }
		    
		    /* Increment within range */
		    weap += ent->weapon_switch;
		    if( weap >= num_weapon_types )
			weap = 1;
		    else if( weap < 1 )
			weap = num_weapon_types - 1;

		} while( ent->has_weapon[ weap ] == 0 );

		ent->weapon = weap;
		ent->last_switch = now;
		ent->weapon_switch = 0;

	    }
	}

    }

}


extern float sin_lookup[DEGREES];
extern float cos_lookup[DEGREES];

void weapon_fire(entity *ent)
{
    weap_type_t *wt;
    ent_type_t *et, *pt;
    entity *proj;
    msec_t now, reload_time;
    int i;
    short *ammo, ammo_needed;
    float x, y;
    float x_v, y_v;
    point start;
    byte dir, d;

    now = gettime();
    if( (wt = weapon_type(ent->weapon)) == NULL ) {
	printf("Error firing weapon %d for entity %d (%s)\n", 
	       ent->weapon, ent->id, entity_type(ent->type)->name);
	return;
    }

    reload_time = wt->reload_time;
    if( ent->powerup & (1<<PU_DOUBLE_FIRE_RATE) )
	reload_time /= 2;

    if( now - ent->last_fired < reload_time ) {
	/* Send client "click" sound? */
	return;
    }

    if( wt->ammo_type != AMMO_INFINITE ) {
	ammo = &ent->ammo[ wt->ammo_type ];
	ammo_needed = wt->ammo_usage;

	if( ent->powerup & (1<<PU_HALF_AMMO_USAGE) ) {
	    if( (rand()%2) == 0 ) /* Avg = 50% usage */
		ammo_needed = 0;
	}

	if( wt->class == WT_BEAM ) {
	    ammo_needed = wt->ammo_usage / (float)server.fps + 0.5; /* Round */
	}

	if( *ammo == 0 ) {
	    ent->weapon_switch = 1;
	    return;
	} else {
	    *ammo -= ammo_needed;
	    if( *ammo < 0 )
		*ammo = 0; /* Might be less with beam */
	}

    }
	
    et = entity_type(ent->type);
    ent->last_fired = now;

    start.x = ent->x + et->width/2;
    start.y = ent->y + et->height - (et->height/5 * 2);
    /* adjust y to top of ground clip box (see world.c) */

    d = ent->dir + 16;
    d /= 32;
    d *= 32;

    /* Get it outside shooters body */
    start.x += sin_lookup[d] * (et->width/2);
    start.y += -cos_lookup[d] * (et->height/2);

    if( wt->muzzle_flash )
	muzzle_flash(start, d);

    for( i=0 ; i < wt->number ; i++ ) {
	dir = ent->dir;

	if( wt->spread )
	    dir += rand()%(wt->spread * 2) - wt->spread;

	switch( wt->class ) {
	case WT_NONE:
	    break;
	case WT_PROJECTILE:
	    /* Only 1/2 relative motion, but it makes it easier to aim */
	    x_v = ent->x_v / 2;
	    y_v = ent->y_v / 2;
	    x_v += sin_lookup[ent->dir] * wt->speed;
	    y_v -= cos_lookup[ent->dir] * wt->speed;

	    pt = entity_type(wt->projectile);
	    x = start.x - pt->width/2;
	    y = start.y - pt->height + pt->height/5 * 2;
	    proj = entity_new( wt->projectile, x, y, x_v, y_v);
	    proj->dir = ent->dir;
	    proj->pid = ent->id; /* Set parent id to the projectiles shooter */
	    /* If the projectile hits something, the projectile's weapon value
	       (which is the weapon that caused the projectile to be fired)
	       will be the one used to call weapon_hit() */
	    proj->weapon = ent->weapon;
	    break;
	default:
	    hitscan_fire( ent, ent->weapon, start, dir );
	}
    }

}


/* Shooter hit victim at point P with weapon of type weapon */
void weapon_hit(entity *shooter, entity *victim, point P, byte weaptype)
{
    ent_type_t *et;
    weap_type_t *wt;
    int damage;
    netmsg msg;

    wt = weapon_type(weaptype);

    if( victim == NULL ) {
	printf("VICTIM = NULL!\n");
	return;
    }

    et = entity_type(victim->type);

    /* Work out if we should be here first */
    if( victim->class == ITEM || victim->health <= 0 )
	return;
    
    /* Apply damage */
    damage = wt->damage;

    /* Damage is inflicted over a second */
    if( wt->class == WT_BEAM )
	damage /= server.fps;

    /* Resistance reduces damage by 50% */
    if( victim->powerup & (1<<PU_RESISTANCE) )
	damage /= 2;

    victim->health -= damage;
    
    if( victim->health <= 0 )
	entity_killed(shooter, victim, P, weaptype);

    /* Spawn blood particles if victim bleeds */
    if( et->bleeder ) {
	msg.type = NETMSG_PARTICLES;
	msg.particles.effect = P_BLOOD;
	msg.particles.dir = shooter? shooter->dir : 0;
	msg.particles.length = damage;
	msg.particles.color1 = COL_RED;
	msg.particles.color2 = COL_RED;
	msg.particles.x = P.x;
	msg.particles.y = P.y;
	sv_net_send_to_all(msg);
    }

}


void weapon_explode(entity *explosion, weap_type_t *wt)
{
    netmsg msg;
    entity *victim, *shooter;
    point P;
    int d1, dist, radius, damage;

    /* Where the explosion starts */
    P.x = explosion->x + explosion->width/2;
    P.y = explosion->y + explosion->height/2;

    /* Draw explosion */
    msg.type = NETMSG_PARTICLES;
    msg.particles.effect = P_EXPLOSION;
    msg.particles.dir = 0;
    msg.particles.length = wt->explosion * 40;
    msg.particles.color1 = COL_ORANGE;
    msg.particles.color2 = COL_RED;
    msg.particles.x = P.x;
    msg.particles.y = P.y;
    sv_net_send_to_all(msg);

    /* Find the person who shot the exploding weapon */
    if( (shooter = findent(explosion->pid)) == NULL )
	return; /* Shooter is no longer in the game */

    /* Do explosion damage */
    for( victim = root ; victim != NULL ; victim = victim->next ) {
	if( victim->class == ITEM || victim->health <= 0 )
	    continue;
	radius = wt->explosion + d1;
	dist = entity_dist(explosion, victim);

	/* Account for entity's width in factoring distance from explosion */
	d1 = MIN( victim->width, victim->height ) / 2;
	if( dist > d1 )
	    dist -= d1;

	/* Apply damage if entity is within blast range */
	if( dist < wt->explosion ) {
	    damage = wt->splashdamage * (wt->explosion - dist) / wt->explosion;
	    if( victim->powerup & (1<<PU_RESISTANCE) )
		damage /= 2;

	    victim->health -= damage;
	    if( victim->health <= 0 )
		entity_killed(shooter, victim, P, explosion->weapon);
	}
    }

}


void weapon_reset(entity *ent)
{
    ent_type_t *et;
    weap_type_t *wt;

    et = entity_type(ent->type);
    wt = weapon_type(et->weapon);
    memset(ent->has_weapon, 0, num_weapon_types);
    ent->has_weapon[et->weapon] = 1; /* Has default weapon */
    ent->ammo[wt->ammo_type] = wt->initial_ammo;
    ent->weapon = et->weapon;

}

static void muzzle_flash(point P, byte dir)
{
    netmsg msg;

    msg.type = NETMSG_PARTICLES;
    msg.particles.effect = P_MFLASH;
    msg.particles.dir = dir;
    msg.particles.length = 50;
    msg.particles.color1 = COL_YELLOW;
    msg.particles.color2 = COL_RED;
    msg.particles.x = P.x;
    msg.particles.y = P.y;
    sv_net_send_to_all(msg);

}
