/*-
 * Copyright (c) 1999-2006 Thomas Runge  (coto@core.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*
 * routines for drawing the clockface and the hands are based on TolleUhr for
 * Amiga, made by Matthias Fleischer (fleischr@izfm.uni-stuttgart.de) and
 * Gunther Nikl (gnikl@informatik.uni-rostock.de)
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>
#include "XCC.h"
#include "xturc.h"

#undef USE_PIXMAPS

#ifdef USE_IMLIB
#define USE_PIXMAPS
#undef USE_XPM
#include <Imlib.h>
#endif

#ifdef USE_XPM
#define USE_PIXMAPS
#include <X11/xpm.h>
#endif

#include "TolleUhr.h"

static char const id[] = "$Id: xtu Version 1.3  by Thomas Runge (coto@core.de) $";

#define _RCFILE_   ".xturc"
#define ShadowSize 2
#define program    "XTu"

static int      ignoreXError;

struct settings_struct st;
struct win_struct ws;

const short     sinus[] =
{
    0, 13, 26, 39, 52, 64, 75, 85, 94, 103, 110, 116, 121, 124, 126,
    127, 126, 124, 121, 116, 110, 103, 94, 85, 75, 64, 52, 39, 26, 13,
    0, -13, -26, -39, -52, -64, -75, -85, -94, -103, -110, -116, -121, -124, -126,
    -127, -126, -124, -121, -116, -110, -103, -94, -85, -75, -64, -52, -39, -26, -13
};
const short     cosinus[] =
{
    127, 126, 124, 121, 116, 110, 103, 94, 85, 75, 64, 52, 39, 26, 13,
    0, -13, -26, -39, -52, -64, -75, -85, -94, -103, -110, -116, -121, -124, -126,
    -127, -126, -124, -121, -116, -110, -103, -94, -85, -75, -64, -52, -39, -26, -13,
    0, 13, 26, 39, 52, 64, 75, 85, 94, 103, 110, 116, 121, 124, 126
};
const short     srect[] =
{
    0, 13, 27, 41, 57, 73, 92, 111, 124, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 124, 111, 92, 73, 57, 41, 27, 13,
    0, -13, -27, -41, -57, -73, -92, -111, -124, -127, -127, -127, -127, -127, -127,
    -127, -127, -127, -127, -127, -127, -127, -124, -111, -92, -73, -57, -41, -27, -13
};
const short     crect[] =
{
    127, 127, 127, 127, 127, 127, 127, 124, 111, 92, 73, 57, 41, 27, 13,
    0, -13, -27, -41, -57, -73, -92, -111, -124, -127, -127, -127, -127, -127, -127,
    -127, -127, -127, -127, -127, -127, -127, -124, -111, -92, -73, -57, -41, -27, -13,
    0, 13, 27, 41, 57, 73, 92, 111, 124, 127, 127, 127, 127, 127, 127
};
const short     dx1[] = {-2, 2, 1, -2, 1, 2, -2, -2, -1, -2, -1, -2};
const short     dy1[] = {-4, -1, -2, -2, 2, 1, -2, 1, 2, -2, -2, -1};
const short     dx2[] = {4, -2, 1, 4, 1, -2, 4, 2, -1, 4, -1, 2};
const short     dy2[] = {0, 3, 2, 0, -2, -3, 0, -3, -2, 0, 2, 3};
const short     dx3[] = {0, -2, -3, 0, -3, -2, 0, 2, 3, 0, 3, 2};
const short     dy3[] = {8, -1, 2, 4, -2, 1, 4, 1, -2, 4, 2, -1};
const short     dx4[] = {-4, 2, -1, -4, -1, 2, -4, -2, 1, -4, 1, -2};
const short     dy4[] = {0, -3, -2, 0, 2, 3, 0, 3, 2, 0, -2, -3};
const int       HandWidth[] = {4000, 3250, 2500, 1750, 1000};
const int       pent[] = {CO_TWELFE, CO_HOUR, CO_HOUR, CO_QUARTER, CO_HOUR,
							CO_HOUR, CO_QUARTER, CO_HOUR, CO_HOUR, CO_QUARTER,
							CO_HOUR, CO_HOUR};

int
main(int argc, char **argv)
{
	XEvent          event;
	int             i, x = 0, y = 0, rsx = 0, rsy = 0, resizer_size;
	char            text[10];
	KeySym          key;
	Atom            wm_protocols, wm_delete_window;

	ws.resizing = False;
	ws.moving = False;
	ws.display_name = NULL;
	ws.progname = argv[0];

	XSetErrorHandler(HandleXError);
	SetDefaults();

	ParseCmdLine(argc, argv, True);

	ws.dsp = XOpenDisplay(ws.display_name);
	if(ws.dsp == NULL)
	{
		fprintf(stderr, "%s: Cannot connect to X server %s\n", ws.progname,
				XDisplayName(ws.display_name));
		exit(EXIT_FAILURE);
	}

#ifdef USE_IMLIB
	ws.id = Imlib_init(ws.dsp);
#endif

	ParseXdefaults();
	ParseRcFile();
	ParseCmdLine(argc, argv, False);

	ws.screen = DefaultScreen(ws.dsp);
	ws.rwin = DefaultRootWindow(ws.dsp);
#ifdef USE_IMLIB
	ws.visual = ws.id->x.visual;
	ws.depth = ws.id->x.depth;
#else
	ws.visual = DefaultVisual(ws.dsp, ws.screen);
	ws.depth = DefaultDepth(ws.dsp, ws.screen);
#endif

	for(i = 0; i < CO_LAST; i++)
		ws.gc[i] = XCreateGC(ws.dsp, ws.rwin, 0, 0);

	ws.fg = WhitePixel(ws.dsp, ws.screen);
	ws.bg = BlackPixel(ws.dsp, ws.screen);

	if(st.geometry)
		ReallyCheckGeometry(st.geometry);

	ws.pixmap = -1;
	resizer_size = 20;

	ws.xcc = XCCCreate(ws.dsp, ws.visual, False, False, XA_COLORMAP, &(ws.cmap));

	SetColorsAndDC();
	MakeWindow();

	StartTimer();

	wm_protocols = XInternAtom(ws.dsp, "WM_PROTOCOLS", False);
	wm_delete_window = XInternAtom(ws.dsp, "WM_DELETE_WINDOW", False);
	XSetWMProtocols(ws.dsp, ws.win, &wm_delete_window, 1);

	ws.done = 0;
	while(ws.done == 0)
	{
		XNextEvent(ws.dsp, &event);

		switch(event.type)
		{
			case Expose:
					if(event.xexpose.count == 0)
						Redraw();
				break;
			case ClientMessage:
					if(event.xclient.message_type == wm_protocols &&
							*event.xclient.data.l == wm_delete_window)
						ws.done = 1;
				break;
			case ConfigureNotify:
					if(ws.width != event.xconfigure.width ||
							ws.height != event.xconfigure.height)
					{
						ws.width = event.xconfigure.width;
						ws.height = event.xconfigure.height;
						UpdatePixmapSize();
						UpdateBackground();
					}
					else
						if(st.wm && (ws.winx != event.xconfigure.x ||
							ws.winy != event.xconfigure.y))
						{
							ws.winx = event.xconfigure.x;
							ws.winy = event.xconfigure.y;
							if(!st.pixmaps[AR_BG])
								UpdateBackground();
						}
				break;
			case MappingNotify:
					XRefreshKeyboardMapping((XMappingEvent *) & event);
				break;
		    case ButtonPress:
					if(st.immovable)
						break;

					x = event.xbutton.x_root;
					y = event.xbutton.y_root;
					rsx = event.xbutton.x;
					rsy = event.xbutton.y;

					XRaiseWindow(ws.dsp, ws.win);

					if(rsx > ws.width - resizer_size && rsx < ws.width &&
							rsy > ws.height - resizer_size && rsy < ws.height)
						StartResizing();
					else
					{
						x -= ws.winx;
						y -= ws.winy;
						if(!st.shext)
							StartMoving();
					}
				break;
			case ButtonRelease:
					if(st.immovable)
						break;

					if(ws.resizing)
					{
						EndResizing();
						XResizeWindow(ws.dsp, ws.win,
										ws.width + event.xbutton.x - rsx,
										ws.height + event.xbutton.y - rsy);
					}
					else
					{
						if(!st.shext)
							EndMoving();

						ws.winx = event.xbutton.x_root - x;
						ws.winy = event.xbutton.y_root - y;

						if(!st.pixmaps[AR_BG])
							UpdateBackground();
						else
							XMoveWindow(ws.dsp, ws.win, ws.winx, ws.winy);
						Redraw();
						XFlush(ws.dsp);
					}
				break;
			case MotionNotify:
					if(st.immovable)
						break;

					if(!ws.resizing)
					{
						if(!st.shext)
							MoveTo(event.xbutton.x_root - x,
									event.xbutton.y_root - y);
						else
						{
							ws.winx = event.xbutton.x_root - x;
							ws.winy = event.xbutton.y_root - y;
							XMoveWindow(ws.dsp, ws.win, ws.winx, ws.winy);
						}
					}
					else
					{
						if(event.xbutton.x > 0 && event.xbutton.y > 0 &&
						(ws.width + event.xbutton.x - rsx > 2 * st.bw + 20) &&
						(ws.height + event.xbutton.y - rsy > 2 * st.bw + 20))
					ResizeTo(ws.winx + ws.width + event.xbutton.x - rsx,
								ws.winy + ws.height + event.xbutton.y - rsy);
					}
				break;
			case EnterNotify:
					if(event.xcrossing.mode == NotifyNormal)
						XGrabKeyboard(ws.dsp, ws.win, False, GrabModeAsync,
										GrabModeAsync, CurrentTime);
				break;
			case LeaveNotify:
					if(event.xcrossing.mode == NotifyNormal)
						XUngrabKeyboard(ws.dsp, CurrentTime);
				break;
			case KeyPress:
					i = XLookupString((XKeyEvent *) & event, text, 10, &key, 0);
					HandleKeypress(i, text);
				break;
			}
	}

	FreeColorsAndPixmaps();
	XCCFree(ws.xcc);
	for(i = 0; i < CO_LAST; i++)
		XFreeGC(ws.dsp, ws.gc[i]);
	XDestroyWindow(ws.dsp, ws.win);
	XCloseDisplay(ws.dsp);

	if(st.display)
		free(st.display);

	if(st.configfile)
		free(st.configfile);

	if(st.geometry)
		free(st.geometry);

    return(EXIT_SUCCESS);
}

void
FreeColorsAndPixmaps()
{
	int i;

	for(i = 0; i < AR_LAST; i++)
	{
		if(st.pixmaps[i] != NULL)
		{
			free(st.pixmaps[i]);
			st.pixmaps[i] = NULL;
#ifdef USE_XPM
			XFreePixmap(ws.dsp, ws.pixmaps[i]);
#endif
#ifdef USE_IMLIB
			Imlib_free_pixmap(ws.id, ws.pixmaps[i]);
#endif
		}
	}

	for(i = 0; i < CO_LAST; i++)
		if(st.color[i] != NULL)
		{
			free(st.color[i]);
			st.color[i] = NULL;
		}
}

void
MakeWindow()
{
	int             i;
	XSetWindowAttributes attr;
	unsigned long   value_mask;
	Cursor          cursor;

	ws.winx = st.x;
	ws.winy = st.y;
	ws.width = st.w;
	ws.height = st.h;

#ifdef USE_PIXMAPS
	for(i = 0; i < AR_LAST; i++)
		ws.pmok[i] = False;
#endif

	UpdatePixmapSize();
	GetPixmap();

#ifdef USE_PIXMAPS
	if(st.use_pm_size)
		ResizeToPmSize();
#endif

	cursor = XCreateFontCursor(ws.dsp, XC_top_left_arrow);

	attr.event_mask = KeyPressMask | ExposureMask | StructureNotifyMask |
						EnterWindowMask | LeaveWindowMask;
	attr.override_redirect = !st.wm;
	attr.save_under = True;
	value_mask = CWOverrideRedirect | CWEventMask | CWSaveUnder;
	if(!st.immovable)
	{
		attr.event_mask |= ButtonPressMask | ButtonReleaseMask |
							Button1MotionMask;
		attr.cursor = cursor;
		value_mask |= CWCursor;
	}
	ws.win = XCreateWindow(ws.dsp, ws.rwin, ws.winx, ws.winy,
							ws.width, ws.height, 0, ws.depth,
							InputOutput, ws.visual, value_mask, &attr);

	XStoreName(ws.dsp, ws.win, "xtolleuhr");
	XRaiseWindow(ws.dsp, ws.win);
	XMapWindow(ws.dsp, ws.win);
}

void
DestroyWindow()
{
    XDestroyWindow(ws.dsp, ws.win);
}

void
StartTimer()
{
	struct sigaction act;
	struct itimerval ival;
	struct tm       tm;
	struct timeval  tp;
	struct timezone tzp;
	time_t          theTime;

	act.sa_handler = TimerHandler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if(sigaction(SIGALRM, &act, NULL) == -1)
	{
		perror("sigaction ALRM");
		exit(EXIT_FAILURE);
	}
	act.sa_handler = HupHandler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if(sigaction(SIGHUP, &act, NULL) == -1)
	{
		perror("sigaction HUP");
		exit(EXIT_FAILURE);
	}
	act.sa_handler = KillHandler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if(sigaction(SIGTERM, &act, NULL) == -1)
	{
		perror("sigaction TERM");
		exit(EXIT_FAILURE);
	}
	act.sa_handler = KillHandler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if(sigaction(SIGINT, &act, NULL) == -1)
	{
		perror("sigaction INT");
		exit(EXIT_FAILURE);
	}
	if((theTime = time(NULL)) == -1)
	{
		perror("time()");
		exit(EXIT_FAILURE);
	}
	tm = *localtime(&theTime);

	if(gettimeofday(&tp, &tzp) == -1)
	{
		perror("gettimeofday()");
		exit(EXIT_FAILURE);
	}
	if(st.seconds)
	{
		ival.it_interval.tv_sec = 1;
		ival.it_interval.tv_usec = 0;
		ival.it_value.tv_sec = 0;
		ival.it_value.tv_usec = 1000050 - tp.tv_usec;
	}
	else
	{
		ival.it_interval.tv_sec = 60;
		ival.it_interval.tv_usec = 0;
		ival.it_value.tv_sec = 60 - tm.tm_sec;
		ival.it_value.tv_usec = 1000050 - tp.tv_usec;
	}

	if(setitimer(ITIMER_REAL, &ival, NULL) == -1)
	{
		perror("setitimer");
		exit(EXIT_FAILURE);
	}
}

void
KillHandler()
{
    exit(EXIT_SUCCESS);
}

void
HupHandler()
{
    ws.done = True;

    FreeColorsAndPixmaps();
    SetDefaults();
    ParseXdefaults();
    ParseRcFile();
    DestroyWindow();
    SetColorsAndDC();
    MakeWindow();

    ws.done = False;
}

void
TimerHandler()
{
    if (!ws.done)
    {
	Redraw();
	XFlush(ws.dsp);
    }
}

void
HandleKeypress(int i, char *text)
{
	if(i == 1)
	switch(text[0])
	{
		case 'q':
				ws.done = 1;
			break;
		case 'w':
				st.shadow = !st.shadow;
				Redraw();
			break;
		case 'u':
				UpdateBackground();
			break;
		case 'h':
				kill(getpid(), SIGHUP);
			break;
		case 's':
				st.seconds = !st.seconds;
				StartTimer();
				Redraw();
				XFlush(ws.dsp);
			break;
#ifdef USE_PIXMAPS
		case 'r':
				if (st.pixmaps[AR_BG])
				{
					ResizeToPmSize();
					XResizeWindow(ws.dsp, ws.win, ws.width, ws.height);
				}
			break;
#endif
		case 'i':
				printf("x: %d  y: %d  w: %d  h: %d\n", ws.winx, ws.winy,
						ws.width, ws.height);
			break;
	}
}

void
GetPixmap()
{
	if(!st.shext)
	{
#ifdef USE_PIXMAPS
		if(st.pixmaps[AR_BG] != NULL)
		{
			if(!ws.pmok[AR_BG])
				ws.pixmaps[AR_BG] = ReallyGetPixmap(st.pixmaps[AR_BG]);
			if(ws.pixmaps[AR_BG] == NULL)
			{
				if(st.pixmaps[AR_BG] != NULL)
					free(st.pixmaps[AR_BG]);
				st.pixmaps[AR_BG] = NULL;
				UpdatePixmapSize();
				GetPixmap();
				return;
			}
			else
			{
				ResizePixmap(&(ws.pixmaps[AR_BG]), ws.width, ws.height);
				XSetTile(ws.dsp, ws.gc[CO_COMMON], ws.pixmaps[AR_BG]);
				XSetFillStyle(ws.dsp, ws.gc[CO_COMMON], FillTiled);
				XFillRectangle(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
								0, 0, ws.width, ws.height);
				XSetFillStyle(ws.dsp, ws.gc[CO_COMMON], FillSolid);
				ws.pmok[AR_BG] = True;
			}
		}
		else
#endif /* USE_PIXMAPS */
			XCopyArea(ws.dsp, ws.rwin, ws.pixmap, ws.gc[CO_COMMON],
						ws.winx, ws.winy, ws.width, ws.height, 0, 0);
	}
	else
	{
		XSetForeground(ws.dsp, ws.gc[CO_COMMON], WhitePixel(ws.dsp, ws.screen));
		XFillRectangle(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
						0, 0, ws.width, ws.height);
	}

	DrawBorder();
	DrawClockFace();
}

#ifdef USE_PIXMAPS
Pixmap
ReallyGetPixmap(char *filename)
{
	Pixmap tmp_pixmap = NULL;
#ifdef USE_XPM
	int    err;

	if((err = XpmReadFileToPixmap(ws.dsp, ws.rwin,
		filename, &tmp_pixmap, NULL, NULL)) < XpmSuccess)
	{
		fprintf(stderr, "could'nt create pixmap from xpm file (%s): ", filename);
		switch(err)
		{
			case XpmOpenFailed:
					fprintf(stderr, "could not open file\n");
				break;
			case XpmFileInvalid:
					fprintf(stderr, "file invalid\n");
				break;
			case XpmNoMemory:
					fprintf(stderr, "not enough memory\n");
				break;
			case XpmColorFailed:
					fprintf(stderr, "not enough free colors\n");
				break;
			default:
					fprintf(stderr, "unknown error ?!?\n");
		}
		return((Pixmap) NULL);
	}
#endif /* USE_XPM */
#ifdef USE_IMLIB
	int w, h;
	ws.im = Imlib_load_image(ws.id, filename);
	if(ws.im == NULL)
	{
		fprintf(stderr, "could not open file: %s\n", filename);
		return((Pixmap) NULL);
	}
	w = ws.im->rgb_width;
	h = ws.im->rgb_height;
	Imlib_render(ws.id, ws.im, w, h);
	tmp_pixmap = Imlib_move_image(ws.id, ws.im);
#endif
	return(tmp_pixmap);
}
#endif /* USE_PIXMAPS */

void
DrawBorder()
{
    switch (st.border)
    {
	    case CB_SINGLE:
	    XSetForeground(ws.dsp, ws.gc[CO_COMMON], WhitePixel(ws.dsp, ws.screen));
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON], 0, 0, ws.width - 1, 0);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON], 0, 0, 0, ws.height - 1);
	    XSetForeground(ws.dsp, ws.gc[CO_COMMON], BlackPixel(ws.dsp, ws.screen));
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
		      0, ws.height - 1, ws.width - 1, ws.height - 1);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
		      ws.width - 1, ws.height - 1, ws.width - 1, 0);
	    break;
	case CB_DOUBLE:
	    XSetForeground(ws.dsp, ws.gc[CO_COMMON], WhitePixel(ws.dsp, ws.screen));
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON], 0, 0, ws.width - 1, 0);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON], 0, 0, 0, ws.height - 1);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
		      st.bw, ws.height - (st.bw + 1), ws.width - (st.bw + 1), ws.height - (st.bw + 1));
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
		      ws.width - (st.bw + 1), ws.height - (st.bw + 1), ws.width - (st.bw + 1), st.bw);

	    XSetForeground(ws.dsp, ws.gc[CO_COMMON], BlackPixel(ws.dsp, ws.screen));
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
		      0, ws.height - 1, ws.width - 1, ws.height - 1);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON],
		      ws.width - 1, ws.height - 1, ws.width - 1, 0);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON], st.bw, st.bw,
		      ws.width - (st.bw + 1), st.bw);
	    XDrawLine(ws.dsp, ws.pixmap, ws.gc[CO_COMMON], st.bw, st.bw,
		      st.bw, ws.height - (st.bw + 1));
	    break;
	case CB_NONE:
	default:
	    break;
    }
}

void
DrawClockFace()
{
    int             i;
    int             w = (int) ws.width - 2 * st.bw;
    int             h = (int) ws.height - 2 * st.bw;
    int             r = (w > h ? w : h) / 40;

    if (r < 1)
	r = 1;

    if (st.clockface_digits == CF_NONE)
	return;

    for (i = 0; i < 60; i += st.clockface_digits)
    {
	int             x = w / 2 + st.bw + ((st.clockface_shape == CF_OVAL ?
					    sinus[i] : srect[i]) * w / 300);
	int             y = h / 2 + st.bw - ((st.clockface_shape == CF_OVAL ?
					  cosinus[i] : crect[i]) * h / 300);

	if (i % 5)
	{
	    XPoint          points[5];

	    if (st.shapes == CF_ROUND)
	    {
		XFillArc(ws.dsp, ws.pixmap, ws.gc[CO_MINUTES], x - r / 2, y - r / 2, r, r,
			 0, 360 * 64);
	    }
	    else
	    {			/* CF_ORIGINAL */
		points[0].x = x + w / 100;
		points[0].y = y;
		points[1].x = x;
		points[1].y = y + h / 100;
		points[2].x = x - w / 100;
		points[2].y = y;
		points[3].x = x;
		points[3].y = y - h / 100;
		points[4] = points[0];
		XFillPolygon(ws.dsp, ws.pixmap, ws.gc[CO_MINUTES], points, 5,
			     Convex, CoordModeOrigin);
	    }
	}
	else
	{
	    int             b = i / 5;
	    XPoint          points[5];

	    if (st.shapes == CF_ROUND)
	    {
		if (i == 0)
		    XFillArc(ws.dsp, ws.pixmap, ws.gc[pent[b]],
			     x - r, y - r, 2 * r, 2 * r, 0, 360 * 64);
		else
		    XFillArc(ws.dsp, ws.pixmap, ws.gc[pent[b]],
		    x - r * .75, y - r * .75, 1.5 * r, 1.5 * r, 0, 360 * 64);
	    }
	    else
	    {			/* CF_ORIGINAL */
		points[0].x = x += w * dx1[b] / 100;
		points[0].y = y += h * dy1[b] / 100;
		points[1].x = x += w * dx2[b] / 100;
		points[1].y = y += h * dy2[b] / 100;
		points[2].x = x += w * dx3[b] / 100;
		points[2].y = y += h * dy3[b] / 100;
		points[3].x = x += w * dx4[b] / 100;
		points[3].y = y += h * dy4[b] / 100;
		points[4] = points[0];
		XFillPolygon(ws.dsp, ws.pixmap, ws.gc[pent[b]], points, 5,
			     Convex, CoordModeOrigin);
	    }
	}
    }
}

void
Redraw()
{
    struct tm       now;
    int             std;

    if ((ws.localTime = time(NULL)) == -1)
    {
	perror("time()");
	exit(EXIT_FAILURE);
    }
    now = *localtime(&ws.localTime);

    std = now.tm_hour % 12 * 5 + now.tm_min / 12;

    /*
     * printf("Time: %2d:%2d:%2d  (std = %d)\n", now.tm_hour, now.tm_min,
     * now.tm_sec, std);
     */

    /* blit background pixmap into hidden workpixmap */
	XCopyArea(ws.dsp, ws.pixmap, ws.hidden_pm, ws.gc[CO_COMMON],
		  0, 0, ws.width, ws.height, 0, 0);

    if (st.shadow)
    {
	DrawHands(std, 440, ShadowSize, CO_SHADOW, CO_HOURAREA);
	DrawHands(now.tm_min, 300, ShadowSize, CO_SHADOW, CO_MINAREA);
    }
    DrawHands(std, 440, 0, CO_HOURAREA, -1);
    DrawHands(now.tm_min, 300, 0, CO_MINAREA, -1);

    if (st.seconds)
    {
	int             x1, y1, x2, y2, w, h;

	w = (int) ws.width - 2 * st.bw;
	h = (int) ws.height - 2 * st.bw;

	x1 = w / 2 + st.bw;
	x2 = x1 + sinus[now.tm_sec] * w / 300;
	y1 = h / 2 + st.bw;
	y2 = y1 - cosinus[now.tm_sec] * h / 300;
	if (st.shadow)
	{
	    XDrawLine(ws.dsp, ws.hidden_pm, ws.gc[CO_SHADOW],
		      x1 + ShadowSize, y1 + ShadowSize, x2 + ShadowSize, y2 + ShadowSize);
	}
	XDrawLine(ws.dsp, ws.hidden_pm, ws.gc[CO_SECONDS], x1, y1, x2, y2);
    }
    /* blit hidden workpixmap into window */
	XCopyArea(ws.dsp, ws.hidden_pm, ws.win, ws.gc[CO_COMMON],
		  0, 0, ws.width, ws.height, 0, 0);

	if(st.shext)
	{
		CreateBitmapFromPixmap();
		XShapeCombineMask(ws.dsp, ws.win, ShapeBounding, 0, 0, ws.shape_pm, ShapeSet);
	}
    if (st.chime == CH_QUARTERS &&
	!((now.tm_min + 15) % 15) && !now.tm_sec)
	XBell(ws.dsp, 50);

    if (st.chime != CH_NONE && !now.tm_min && !now.tm_sec)
	XBell(ws.dsp, 70);
}

void
DrawHands(int angle, int lfactor, int offset, int what, int helper)
{
    int             x0, y0, x1, y1, x2, y2, w, h;
    Bool            fill = False;
    XPoint          points[5];

    w = (int) ws.width - 2 * st.bw;
    h = (int) ws.height - 2 * st.bw;

    x0 = w / 2 + st.bw + offset;
    y0 = h / 2 + st.bw + offset;
    x1 = cosinus[angle] * w / HandWidth[st.hands_width];
    y1 = sinus[angle] * h / HandWidth[st.hands_width];
    x2 = sinus[angle] * w / lfactor;
    y2 = cosinus[angle] * h / lfactor;

    if ((helper == -1 && (st.color[what] || st.pixmaps[what])) ||
	(helper != -1 && (st.color[helper] || st.pixmaps[helper])))
	fill = True;

    switch (st.hands_type)
    {
	case HS_LINE:
	    XDrawLine(ws.dsp, ws.hidden_pm, ws.gc[what + 2], x0, y0, x0 + x2, y0 - y2);
	    break;
	case HS_TRIANGLE:
	    points[0].x = x0 + x2;
	    points[0].y = y0 - y2;
	    points[1].x = x0 + x1 - x2 / 4;
	    points[1].y = y0 + y1 + y2 / 4;
	    points[2].x = x0 - x1 - x2 / 4;
	    points[2].y = y0 - y1 + y2 / 4;
	    points[3] = points[0];
	    if (fill)
		XFillPolygon(ws.dsp, ws.hidden_pm, ws.gc[what],
			     points, 4, Convex, CoordModeOrigin);
	    else
		XDrawLines(ws.dsp, ws.hidden_pm, ws.gc[what],
			   points, 4, CoordModeOrigin);
	    if (what != CO_SHADOW)
		XDrawLines(ws.dsp, ws.hidden_pm, ws.gc[what + 2], points, 4,
			   CoordModeOrigin);
	    break;
	case HS_RHOMBUS:
	    points[0].x = x0 + x2;
	    points[0].y = y0 - y2;
	    points[1].x = x0 + x1;
	    points[1].y = y0 + y1;
	    points[2].x = x0 - x2 / 4;
	    points[2].y = y0 + y2 / 4;
	    points[3].x = x0 - x1;
	    points[3].y = y0 - y1;
	    points[4] = points[0];
	    if (fill)
		XFillPolygon(ws.dsp, ws.hidden_pm, ws.gc[what],
			     points, 5, Convex, CoordModeOrigin);
	    else
		XDrawLines(ws.dsp, ws.hidden_pm, ws.gc[what],
			   points, 5, CoordModeOrigin);
	    if (what != CO_SHADOW)
		XDrawLines(ws.dsp, ws.hidden_pm, ws.gc[what + 2], points, 5,
			   CoordModeOrigin);
	    break;
	case HS_RECTANGLE:
	    points[0].x = x0 += x2 - x1 / 2;
	    points[0].y = y0 -= y2 + y1 / 2;
	    points[1].x = x0 += x1;
	    points[1].y = y0 += y1;
	    points[2].x = x0 -= x2 * 5 / 4;
	    points[2].y = y0 += y2 * 5 / 4;
	    points[3].x = x0 -= x1;
	    points[3].y = y0 -= y1;
	    points[4] = points[0];
	    if (fill)
		XFillPolygon(ws.dsp, ws.hidden_pm, ws.gc[what],
			     points, 5, Convex, CoordModeOrigin);
	    else
		XDrawLines(ws.dsp, ws.hidden_pm, ws.gc[what],
			   points, 5, CoordModeOrigin);
	    if (what != CO_SHADOW)
		XDrawLines(ws.dsp, ws.hidden_pm, ws.gc[what + 2], points, 5,
			   CoordModeOrigin);
	    break;
    }
}

void
my_usleep(unsigned t)
{
    struct timeval  tv;

    tv.tv_sec = 0;
    tv.tv_usec = t;
    select(0, NULL, NULL, NULL, &tv);
}

void
UpdateBackground()
{
	XUnmapWindow(ws.dsp, ws.win);
	XFlush(ws.dsp);
	my_usleep(st.sleep);
	GetPixmap();

	XSetWindowBackgroundPixmap(ws.dsp, ws.win, ws.pixmap);
	if(st.wm)
	{
		XSizeHints      hints;
		hints.width = ws.width;
		hints.height = ws.height;
		hints.flags = USPosition | USSize;
		XSetWMNormalHints(ws.dsp, ws.win, &hints);
	}
	else
		XMoveWindow(ws.dsp, ws.win, ws.winx, ws.winy);
	XMapWindow(ws.dsp, ws.win);
}

void
UpdatePixmapSize()
{
	if(ws.pixmap != -1)
	{
		XFreePixmap(ws.dsp, ws.pixmap);
		XFreePixmap(ws.dsp, ws.hidden_pm);
		XFreePixmap(ws.dsp, ws.shape_pm);
	}
	ws.pixmap = XCreatePixmap(ws.dsp, ws.rwin,
									ws.width, ws.height, ws.depth);
	ws.hidden_pm = XCreatePixmap(ws.dsp, ws.rwin,
									ws.width, ws.height, ws.depth);
	ws.shape_pm = XCreatePixmap(ws.dsp, ws.rwin,
									ws.width, ws.height, 1);
}

void
ResizePixmap(Pixmap *pm, int width, int height)
{
#ifdef USE_XPM
	/* TODO: resizing pixmap without imlib */
	return;
#endif
#ifdef USE_IMLIB
	if(st.use_pm_size == False)
	{
		Imlib_render(ws.id, ws.im, width, height);
		Imlib_free_pixmap(ws.id, *pm);
		*pm = Imlib_move_image(ws.id, ws.im);
	}
#endif
}

void
StartResizing()
{
	ws.resizing = True;
	ws.first_resizer = True;
}

void
ResizeTo(int x, int y)
{
	struct xpoints  p;

	p.p[0].x = ws.winx;
	p.p[0].y = ws.winy;
	p.p[1].x = x;
	p.p[1].y = ws.winy;
	p.p[2].x = x;
	p.p[2].y = y;
	p.p[3].x = ws.winx;
	p.p[3].y = y;
	p.p[4] = p.p[0];

	DrawResizerLines(p);

	if(!ws.first_resizer)
		DrawResizerLines(ws.oldpoints);
	else
		ws.first_resizer = False;

	ws.oldpoints = p;
}

void
EndResizing()
{
	ws.resizing = False;
	if(!ws.first_resizer)
		DrawResizerLines(ws.oldpoints);
	st.use_pm_size = False;
}

void
DrawResizerLines(struct xpoints p)
{
	XDrawLines(ws.dsp, ws.rwin, ws.gc[CO_RESIZE], p.p, 5, CoordModeOrigin);
}

void
StartMoving()
{
    ws.moving = True;
    ws.first_resizer = True;
}

void
MoveTo(int x, int y)
{
    struct xpoints  p;

    p.p[0].x = x;
    p.p[0].y = y;
    p.p[1].x = x + ws.width;
    p.p[1].y = y;
    p.p[2].x = x + ws.width;
    p.p[2].y = y + ws.height;
    p.p[3].x = x;
    p.p[3].y = y + ws.height;
    p.p[4] = p.p[0];

    DrawResizerLines(p);

    if (!ws.first_resizer)
	DrawResizerLines(ws.oldpoints);
    else
	ws.first_resizer = False;

    ws.oldpoints = p;
}

void
EndMoving()
{
    ws.moving = False;
    if (!ws.first_resizer)
	DrawResizerLines(ws.oldpoints);
}

int
CreateBitmapFromPixmap()
{
    GC              gc;
    int             i;

    gc = XCreateGC(ws.dsp, ws.shape_pm, 0, NULL);
    XSetForeground(ws.dsp, gc, 0);
    XSetBackground(ws.dsp, gc, 1);
    XFillRectangle(ws.dsp, ws.shape_pm, gc, 0, 0, ws.width, ws.height);
    XSetFunction(ws.dsp, gc, GXor);

    XSync(ws.dsp, False);
    ignoreXError = True;
    for (i = 0; i < ws.depth; i++)
    {
	XCopyPlane(ws.dsp, ws.hidden_pm, ws.shape_pm, gc, 0, 0, ws.width, ws.height,
		   0, 0, i);
    }
    XSync(ws.dsp, True);
    ignoreXError = False;

    XFreeGC(ws.dsp, gc);

    return True;
}

#ifdef USE_PIXMAPS
void
ResizeToPmSize()
{
	Window       root;
	int          x, y;
	unsigned int w, h;
	unsigned int bw, depth;

	if(st.pixmaps[AR_BG] == NULL)
		return;

	XGetGeometry(ws.dsp, ws.pixmaps[AR_BG], &root, &x, &y, &w, &h, &bw, &depth);

	ws.width = w;
	ws.height = h;

	UpdatePixmapSize();
	GetPixmap();
}
#endif

void
Usage()
{
    printf("%s\n", id);
    printf("usage: %s\n", ws.progname);
    printf("-help\n");
	printf("    this help text\n");
    printf("-display <display>\n");
	printf("    which display to use\n");
    printf("-geometry <geometry>\n");
	printf("    position and size\n");
    printf("-seconds <true | false>\n");
	printf("    show seconds\n");
    printf("-clockface <oval (def.) | rectangle>\n");
    printf("    oval or rectangular clock face\n");
    printf("-digits <minutes (def.) | hour | quarter | twelfe>\n");
    printf("    which digits to show\n");
    printf("-handsform <triangle (def.) | line | rhombus | rectangle>\n");
    printf("    form of hands\n");
    printf("-handwidth <normal (def.) | vthin | thin | thick | vthick>\n");
    printf("    width of hands\n");
    printf("-shadow <true (def.) | false>\n");
    printf("    draw a shadow around hands\n");
    printf("-bordertype <none (def.) | single | double>\n");
    printf("    draw a border around clock\n");
    printf("-borderwidth <pixel_count (def. is 4)>\n");
    printf("    width of \"double\" border\n");
    printf("-chime <none (def.) | hours | quarters>\n");
    printf("    ring a bell\n");
    printf("-immovable <false (def.) | true>\n");
    printf("    window is not movable\n");
    printf("-faceform <original (def.) | round>\n");
    printf("    form of clock face\n");
    printf("-pixmap <none (def.) | filename>\n");
    printf("    use this image as background\n");
#ifndef USE_PIXMAPS
    printf(" (feature not compiled into xtu)\n");
#endif
    printf("-resize_to_pm <true (def.) | false>\n");
    printf("    resize to pixmap size and ignore -geometry\n");
    printf("-wm <false (def.) | true>\n");
    printf("    use windowmanager\n");
    printf("-shext <false (def.) | true>\n");
    printf("    use shape extension\n");

    printf("\n");
    printf("Following options require a color name:\n");
    printf("-seconds_color\n");
    printf("-min_area_color\n");
    printf("-min_outline\n");
    printf("-hour_area_color\n");
    printf("-hour_outline\n");
    printf("-shadow_color\n");
    printf("-twelfe_color\n");
    printf("-quarter_color\n");
    printf("-hour_color\n");
    printf("-minutes_color\n");

    printf("\n");
    printf("Following options require a pixmap filename:\n");
    printf("-hour_area_pixmap\n");
    printf("-min_area_pixmap\n");
#ifndef USE_PIXMAPS
    printf(" (feature not compiled into xtu)\n");
#endif
}

void
SetDefaults()
{
	int             i;

	if(st.display != NULL)
		free(st.display);
	if(st.configfile != NULL)
		free(st.configfile);

	st.display = NULL;
	st.configfile = NULL;
	st.seconds = True;
	st.clockface_shape = CF_OVAL;
	st.clockface_digits = CF_MINUTES;
	st.hands_type = HS_TRIANGLE;
	st.hands_width = HS_NORMAL;
	st.shadow = True;
	st.border = CB_NONE;
	st.chime = CH_NONE;
	st.immovable = False;
	st.bw = 4;
	st.shapes = CF_ORIGINAL;
	st.use_pm_size = True;
	st.x = 0;
	st.y = 0;
	st.w = 200;
	st.h = 200;
	st.wm = False;
	st.shext = False;

	for(i = 0; i < CO_LAST; i++)
		st.color[i] = NULL;

	for(i = 0; i < AR_LAST; i++)
		st.pixmaps[i] = NULL;

	return;
}

void InstallRcFile(char *filename)
{
	FILE *fp;
	if((fp = fopen(filename, "w")) != NULL)
	{
		fwrite(xturc_string, strlen(xturc_string), 1, fp);
		fclose(fp);
		printf("Wrote sample config file as %s\n", filename);
		printf("Edit to change the way xtu looks.\n");

#ifndef USE_IMLIB
		printf("You can compile xtu with imlib support.\n");
		printf("This way all imlib supported fileformats can be used.\n");
		printf("(This binary is limited to xpm pixmaps)\n");
#endif
	}
}
  
static const struct
{
    char           *option;
    int             (*func) (const char *);
}               turcTable[] =

{
    {
	"display", CheckDisplay
    },
    {
	"configfile", CheckConfigFile
    },
    {
	"geometry", CheckGeometry
    },
    {
	"seconds", CheckSeconds
    },
    {
	"clockface", CheckClockFace
    },
    {
	"digits", CheckDigits
    },
    {
	"handsform", CheckHands
    },
    {
	"handwidth", CheckHandsWidth
    },
    {
	"shadow", CheckShadow
    },
    {
	"bordertype", CheckBorder
    },
    {
	"borderwidth", CheckBorderWidth
    },
    {
	"chime", CheckChime
    },
    {
	"immovable", CheckImmovable
    },
    {
	"faceform", CheckForm
    },
    {
	"pixmap", CheckBgPixmap
    },
    {
	"sleep", CheckSleepTime
    },
    {
	"resize_to_pm", CheckResizeToPmSize
    },
    {
	"seconds_color", CheckSecondsColor
    },
    {
	"min_area_color", CheckMinAreaColor
    },
    {
	"min_area_pixmap", CheckMinAreaPixmap
    },
    {
	"min_outline", CheckMinOutlineColor
    },
    {
	"hour_area_color", CheckHourAreaColor
    },
    {
	"hour_area_pixmap", CheckHourAreaPixmap
    },
    {
	"hour_outline", CheckHourOutlineColor
    },
    {
	"shadow_color", CheckShadowColor
    },
    {
	"twelfe_color", CheckTwelfeColor
    },
    {
	"quarter_color", CheckQuarterColor
    },
    {
	"hour_color", CheckHourColor
    },
    {
	"minutes_color", CheckMinutesColor
    },
    {
	"wm", CheckWm
    },
    {
	"shext", CheckShext
    }
};

void
ParseCmdLine(int argc, char **argv, int only_watch_for_display)
{
    int             i;
    char           *op, *ap;

    if ((argc - 1) % 2)
    {
	Usage();
	if((argc > 1) && (!strcmp(argv[1], "-h") || !strcmp(argv[1], "-help")))
		exit(EXIT_SUCCESS);
	else
	{
		fprintf(stderr, "wrong count of parameters.\n");
		exit(EXIT_FAILURE);
	}
    }
    for (i = 1; i < argc; i += 2)
    {
	if (*argv[i] != '-')
	{
	    fprintf(stderr, "wrong parameterlist. ignoring \"%s\".\n", argv[i]);
	    continue;
	}
	else
	{
	    op = argv[i];
	    op++;
	    ap = argv[i + 1];

	    if (only_watch_for_display)
	    {
		if (!strcmp(op, "configfile"))
		    ParseOptions(op, ap);
		if (!strcmp(op, "display"))
		    ParseOptions(op, ap);
	    }
	    else
		ParseOptions(op, ap);
	}
    }
    parseColors();
}

void
ParseXdefaults()
{
    int             i, j;
    char           *opt, *arg;

    j = sizeof(turcTable) / sizeof(turcTable[0]);

    for (i = 0; i < j; i++)
    {
	opt = turcTable[i].option;
	arg = XGetDefault(ws.dsp, program, opt);
	if (arg)
	{
	    turcTable[i].func(arg);
	}
    }
}

void
ParseRcFile()
{
    FILE *fp;
    char *pt, *op, *ap, buf[80];

	if(st.configfile == NULL)
	{
		char *home;
    	if ((home = getenv("HOME")) == NULL)
	    {
			home = ".";
		}
		st.configfile = (char *) malloc(strlen(home) + strlen(_RCFILE_) + 2);

		sprintf(st.configfile, "%s/%s", home, _RCFILE_);
	}

	if((fp = fopen(st.configfile, "r")) == NULL)
	{
		InstallRcFile(st.configfile);
		if((fp = fopen(st.configfile, "r")) == NULL)
			return;
	}

    while (!feof(fp))
    {
		if (fgets(buf, 80, fp) != NULL)
		{
		    pt = buf;

		    /* Leerzeichen und Tabs ueberspringen */
		    while ((*pt) && ((*pt == ' ') || (*pt == '\t')))
			pt++;

		    /* # und ! kennzeichnen Kommentare */
		    if (!(*pt) || (*pt == '#') || (*pt == '!') || (*pt == '\n'))
			continue;

		    /* option gefunden */
		    op = pt;

		    /*
		     * : mit 0 ersetzen, dadurch bleibt nur argument uebrig
		     */
		    while ((*pt) && (*pt != ':'))
				pt++;
		    *pt = 0;
		    pt++;

		    /* Leerzeichen und Tabs ueberspringen */
		    while ((*pt) && ((*pt == ' ') || (*pt == '\t')))
				pt++;
		    ap = pt;

		    /* am Ende der Option 0 einfuegen */
		    while ((*pt) && (*pt != ' ') && (*pt != '\t') && (*pt != '\n'))
				pt++;
		    *pt = 0;

		    ParseOptions(op, ap);
		}
    }

    fclose(fp);
    parseColors();
}

void
ParseOptions(char *opt, char *arg)
{
    int             i, j, k;

    k = False;
    j = sizeof(turcTable) / sizeof(turcTable[0]);

    for (i = 0; i < j && k == False; i++)
    {
	if (!strcmp(opt, turcTable[i].option))
	{
	    turcTable[i].func(arg);
	    k = True;
	}
    }
    if (k == False)
    {
	printf("unknown option \"%s\". ignoring.\n", opt);
    }
}

void
parseColors()
{
    int             i;

    for (i = 0; i < CO_LAST; i++)
    {
	if (st.color[i] && (strlen(st.color[i]) == 4) &&
	    !strcasecmp(st.color[i], "none"))
	{
	    free(st.color[i]);
	    st.color[i] = NULL;
	}
    }

    for (i = 0; i < AR_LAST; i++)
    {
	if (st.pixmaps[i] && (strlen(st.pixmaps[i]) == 4) &&
	    !strcasecmp(st.pixmaps[i], "none"))
	{
	    free(st.pixmaps[i]);
	    st.pixmaps[i] = NULL;
	}
    }
}

int
CheckDisplay(const char *string)
{
    if (ws.display_name)
    {
	fprintf(stderr, "Display already in use.\nIt is not useful to set the display in your %s or in the app-defaults.\nIgnoring option.\n", _RCFILE_);
	return (False);
    }
    ws.display_name = strdup(string);
    return (True);
}

int
CheckConfigFile(const char *string)
{
	if((string != NULL) && (strlen(string) != 0))
	{
		st.configfile = strdup(string);
		printf("using config file: %s\n", st.configfile);
    	return (True);
	}
	else
	    return (False);
}

int
CheckGeometry(const char *string)
{
    if (st.geometry)
	free(st.geometry);

    st.geometry = strdup(string);
    return (True);
}

int
ReallyCheckGeometry(const char *string)
{
    char            defaultstring[32];

    sprintf(defaultstring, "%dx%d+%d+%d", st.w, st.h, st.x, st.y);

    XGeometry(ws.dsp, ws.screen, string, defaultstring, 0, 1, 1, 0, 0,
	      &st.x, &st.y, &st.w, &st.h);

    return (True);
}

int
Truecheck(const char *string)
{
    size_t          len;

    len = strlen(string);
    if (len < 4 || len > 5)
	return (-1);

    if (!strcasecmp(string, "true"))
	return (True);
    else if (!strcasecmp(string, "false"))
	return (False);
    else
	return (-1);
}

int
CheckSeconds(const char *string)
{
    int             a;

    if ((a = Truecheck(string)) == -1)
	return (False);

    st.seconds = a;
    return (True);
}

int
CheckClockFace(const char *string)
{
    if (!strcmp(string, "oval") || !strcmp(string, "o"))
    {
	st.clockface_shape = CF_OVAL;
    }
    else if (!strcmp(string, "rectangle") || !strcmp(string, "r"))
    {
	st.clockface_shape = CF_RECT;
    }
    else
	return (False);

    return (True);
}

int
CheckDigits(const char *string)
{
    if (!strcmp(string, "none") || !strcmp(string, "n"))
    {
	st.clockface_digits = CF_NONE;
    }
    else if (!strcmp(string, "minutes") || !strcmp(string, "m"))
    {
	st.clockface_digits = CF_MINUTES;
    }
    else if (!strcmp(string, "hour") || !strcmp(string, "h"))
    {
	st.clockface_digits = CF_HOUR;
    }
    else if (!strcmp(string, "quarter") || !strcmp(string, "q"))
    {
	st.clockface_digits = CF_QUARTER;
    }
    else if (!strcmp(string, "twelfe") || !strcmp(string, "t"))
    {
	st.clockface_digits = CF_TWELFE;
    }
    else
	return (False);

    return (True);
}

int
CheckHands(const char *string)
{
    if (!strcmp(string, "line") || !strcmp(string, "l"))
    {
	st.hands_type = HS_LINE;
    }
    else if (!strcmp(string, "triangle") || !strcmp(string, "t"))
    {
	st.hands_type = HS_TRIANGLE;
    }
    else if (!strcmp(string, "rhombus") || !strcmp(string, "rh"))
    {
	st.hands_type = HS_RHOMBUS;
    }
    else if (!strcmp(string, "rectangle") || !strcmp(string, "re"))
    {
	st.hands_type = HS_RECTANGLE;
    }
    else
	return (False);

    return (True);
}

int
CheckHandsWidth(const char *string)
{
    if (!strcmp(string, "vthin") || !strcmp(string, "1"))
    {
	st.hands_width = HS_VERY_THIN;
    }
    else if (!strcmp(string, "thin") || !strcmp(string, "2"))
    {
	st.hands_width = HS_THIN;
    }
    else if (!strcmp(string, "normal") || !strcmp(string, "3"))
    {
	st.hands_width = HS_NORMAL;
    }
    else if (!strcmp(string, "thick") || !strcmp(string, "4"))
    {
	st.hands_width = HS_THICK;
    }
    else if (!strcmp(string, "vthick") || !strcmp(string, "5"))
    {
	st.hands_width = HS_VERY_THICK;
    }
    else
	return (False);

    return (True);
}

int
CheckShadow(const char *string)
{
    int             a;

    if ((a = Truecheck(string)) == -1)
	return (False);

    st.shadow = a;
    return (True);
}

int
CheckBorderWidth(const char *string)
{
    st.bw = atoi(string);
    if (st.bw < 0)
	st.bw = 0;
    return (True);
}

int
CheckBorder(const char *string)
{
    if (!strcmp(string, "none") || !strcmp(string, "n"))
    {
	st.border = CB_NONE;
    }
    else if (!strcmp(string, "single") || !strcmp(string, "s"))
    {
	st.border = CB_SINGLE;
    }
    else if (!strcmp(string, "double") || !strcmp(string, "d"))
    {
	st.border = CB_DOUBLE;
    }
    else
	return (False);

    return (True);
}

int
CheckChime(const char *string)
{
    if (!strcmp(string, "none") || !strcmp(string, "n"))
    {
	st.chime = CH_NONE;
    }
    else if (!strcmp(string, "hours") || !strcmp(string, "h"))
    {
	st.chime = CH_HOURS;
    }
    else if (!strcmp(string, "quarters") || !strcmp(string, "q"))
    {
	st.chime = CH_QUARTERS;
    }
    else
	return (False);

    return (True);
}

int
CheckImmovable(const char *string)
{
    int             a;

    if ((a = Truecheck(string)) == -1)
	return (False);

    st.immovable = a;
    return (True);
}

int
CheckForm(const char *string)
{
    if (!strcmp(string, "original") || !strcmp(string, "o"))
	st.shapes = CF_ORIGINAL;
    else if (!strcmp(string, "round") || !strcmp(string, "r"))
	st.shapes = CF_ROUND;
    else
	return (False);

    return (True);
}

int
CheckBgPixmap(const char *string)
{
#ifdef USE_PIXMAPS
	if(strcmp(string, "none"))
	{
		if(st.pixmaps[AR_BG])
			free(st.pixmaps[AR_BG]);

		st.pixmaps[AR_BG] = strdup(string);
	}
#else
	printf("pixmap support disabled, no background pixmap possible\n");
#endif
	return(True);
}

int
CheckSleepTime(const char *string)
{
    st.sleep = atoi(string);
    if (st.sleep < 0)
	st.sleep = 0;
    return (True);
}

int
CheckResizeToPmSize(const char *string)
{
#ifdef USE_PIXMAPS
	int a;

	if((a = Truecheck(string)) == -1)
		return(False);

	st.use_pm_size = a;
	return(True);
#else
	printf("pixmap support disabled, no background pixmap possible\n");
	return(True);
#endif
}

int
CheckSecondsColor(const char *string)
{
    if (st.color[CO_SECONDS])
	free(st.color[CO_SECONDS]);

    st.color[CO_SECONDS] = strdup(string);
    return (True);
}

int
CheckMinAreaColor(const char *string)
{
    if (st.color[CO_MINAREA])
	free(st.color[CO_MINAREA]);

    st.color[CO_MINAREA] = strdup(string);
    return (True);
}

int
CheckMinAreaPixmap(const char *string)
{
    if (st.pixmaps[AR_MIN])
	free(st.pixmaps[AR_MIN]);

    st.pixmaps[AR_MIN] = strdup(string);
    return (True);
}

int
CheckMinOutlineColor(const char *string)
{
    if (st.color[CO_MINOUTLINE])
	free(st.color[CO_MINOUTLINE]);

    st.color[CO_MINOUTLINE] = strdup(string);
    return (True);
}

int
CheckHourAreaColor(const char *string)
{
    if (st.color[CO_HOURAREA])
	free(st.color[CO_HOURAREA]);

    st.color[CO_HOURAREA] = strdup(string);
    return (True);
}

int
CheckHourAreaPixmap(const char *string)
{
    if (st.pixmaps[AR_HOUR])
	free(st.pixmaps[AR_HOUR]);

    st.pixmaps[AR_HOUR] = strdup(string);
    return (True);
}

int
CheckHourOutlineColor(const char *string)
{
    if (st.color[CO_HOUROUTLINE])
	free(st.color[CO_HOUROUTLINE]);

    st.color[CO_HOUROUTLINE] = strdup(string);
    return (True);
}

int
CheckShadowColor(const char *string)
{
    if (st.color[CO_SHADOW])
	free(st.color[CO_SHADOW]);

    st.color[CO_SHADOW] = strdup(string);
    return (True);
}

int
CheckTwelfeColor(const char *string)
{
    if (st.color[CO_TWELFE])
	free(st.color[CO_TWELFE]);

    st.color[CO_TWELFE] = strdup(string);
    return (True);
}

int
CheckQuarterColor(const char *string)
{
    if (st.color[CO_QUARTER])
	free(st.color[CO_QUARTER]);

    st.color[CO_QUARTER] = strdup(string);
    return (True);
}

int
CheckHourColor(const char *string)
{
    if (st.color[CO_HOUR])
	free(st.color[CO_HOUR]);

    st.color[CO_HOUR] = strdup(string);
    return (True);
}

int
CheckMinutesColor(const char *string)
{
    if (st.color[CO_MINUTES])
	free(st.color[CO_MINUTES]);

    st.color[CO_MINUTES] = strdup(string);
    return (True);
}

int
CheckWm(const char *string)
{
    int             a;

    if ((a = Truecheck(string)) == -1)
	return (False);

    st.wm = a;
    st.immovable = st.wm;
    return (True);
}

int
CheckShext(const char *string)
{
    int             a;

    if ((a = Truecheck(string)) == -1)
	return (False);

    st.shext = a;

    if (a)
    {
	int             event_basep, error_basep;

	if (XShapeQueryExtension(ws.dsp, &event_basep, &error_basep))
	{
	}
	else
	{
	    st.shext = False;
	    fprintf(stderr, "no shape extension available.\n");
	}
    }
    return (True);
}

void
SetColorsAndDC()
{
	int    i;
	XColor color[CO_LAST];
	Status status;

	for(i = 0; i < CO_LAST; i++)
	{
		if(st.color[i] == NULL)
			ws.pixel[i] = (i == CO_SHADOW) ? ws.bg : ws.fg;
		else
		{
			status = XParseColor(ws.dsp, ws.cmap, st.color[i], &color[i]);
			if(!status)
			{
				ws.pixel[i] = ws.fg;
				fprintf(stderr, "could'nt get color #%d.\n", i);
			}
			else
			{
				ws.pixel[i] = XCCGetPixel(ws.xcc,
								color[i].red, color[i].green, color[i].blue);
			}
		}
		XSetForeground(ws.dsp, ws.gc[i], ws.pixel[i]);
		XSetBackground(ws.dsp, ws.gc[i], ws.bg);
	}
	XSetForeground(ws.dsp, ws.gc[CO_COMMON], ws.fg);
	XSetBackground(ws.dsp, ws.gc[CO_COMMON], ws.bg);
	XSetSubwindowMode(ws.dsp, ws.gc[CO_COMMON], IncludeInferiors);

	XSetForeground(ws.dsp, ws.gc[CO_RESIZE], WhitePixel(ws.dsp, ws.screen));
	XSetFunction(ws.dsp, ws.gc[CO_RESIZE], GXxor);
	XSetSubwindowMode(ws.dsp, ws.gc[CO_RESIZE], IncludeInferiors);

	XSetFillStyle(ws.dsp, ws.gc[CO_HOURAREA], FillSolid);
	XSetFillStyle(ws.dsp, ws.gc[CO_MINAREA], FillSolid);

#ifdef USE_PIXMAPS
	if(st.pixmaps[AR_HOUR] != NULL)
	{
		if(!ws.pmok[AR_HOUR])
			ws.pixmaps[AR_HOUR] = ReallyGetPixmap(st.pixmaps[AR_HOUR]);
		if(ws.pixmaps[AR_HOUR] == NULL)
		{
			if(st.pixmaps[AR_HOUR] != NULL)
				free(st.pixmaps[AR_HOUR]);
			st.pixmaps[AR_HOUR] = NULL;
		}
		else
		{
			XSetTile(ws.dsp, ws.gc[CO_HOURAREA], ws.pixmaps[AR_HOUR]);
			XSetFillStyle(ws.dsp, ws.gc[CO_HOURAREA], FillTiled);
			ws.pmok[AR_HOUR] = True;
		}
	}
	if(st.pixmaps[AR_MIN] != NULL)
	{
		if(!ws.pmok[AR_MIN])
			ws.pixmaps[AR_MIN] = ReallyGetPixmap(st.pixmaps[AR_MIN]);
		if(ws.pixmaps[AR_MIN] == NULL)
		{
			if(st.pixmaps[AR_MIN] != NULL)
				free(st.pixmaps[AR_MIN]);
			st.pixmaps[AR_MIN] = NULL;
		}
		else
		{
			XSetTile(ws.dsp, ws.gc[CO_MINAREA], ws.pixmaps[AR_MIN]);
			XSetFillStyle(ws.dsp, ws.gc[CO_MINAREA], FillTiled);
			ws.pmok[AR_MIN] = True;
		}
	}
#endif /* USE_PIXMAPS */
}

int
HandleXError(Display * dsp, XErrorEvent * event)
{
    char            msg[80];

    if (!ignoreXError)
    {
	XGetErrorText(dsp, event->error_code, msg, 80);
	fprintf(stderr, "\nAn X error occured:\n");
	fprintf(stderr, " Error code %s\n", msg);
	fprintf(stderr, "gna, I'll try to continue...\n");
    }
    return 0;
}

