/*
 * Copyright (c) 1997, 1999, 2000, 2001, Mark Buser.
 * Copyright (c) 2001, 2002, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/opttick.c,v 1.41 2004/12/27 08:38:56 danny Exp $
 */
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <Xm/XmAll.h>

#include "optdetail.h"
#include "parse.h"
#include "pixmap.h"
#include "server.h"
#include "status.h"
#include "tape.h"
#include "xquote.h"
#include "xutil.h"
#include "session.h"

#ifndef XQUOTE
#include "xinvest.h"
#endif

typedef struct tick_struct {
  char		*name;
  char		*nickName;
  int		type;
  int		server;
  QUERY_STRUCT	*detail;
  char		*low, *high;
  int		lowcheck, highcheck;
} TICK_STRUCT;

/* Globals */
static TICK_STRUCT **tickTable;
static int numTick = 0;
static int curType = 0;
static int curServer = 0;

static TICK_STRUCT **newTickTable;
static int newNumTick = 0;

#define TICKFIELDS 2
static Widget textField[TICKFIELDS];
static Widget optServ, optServType;


/* Forward declarations */
void tickMakeTypeMenu (Widget, int);
extern void tickCheckHighLow(int i);


/*
** Ticker Symbol Support
*/

/* 
** Session save and restore functions 
*/

/* Return value must checked for NULL */
char *tickGetItem(int pos)
{
	int	size;
	char	*item;
	char	*serv;
	char	*type;

	if (pos >= numTick)
		return (NULL);

	serv = getServerTitle(tickTable[pos]->server);
	type = getServerType(tickTable[pos]->server, tickTable[pos]->type);

	size = strlen(tickTable[pos]->name) + strlen(serv) + strlen(type) + 8;
	if (tickTable[pos]->nickName) 
		size += strlen(tickTable[pos]->nickName);
	if (tickTable[pos]->low)
		size += strlen(tickTable[pos]->low) + 1;
	if (tickTable[pos]->high)
		size += strlen(tickTable[pos]->high) + 1;
	item = XtCalloc (size, sizeof(char));

	if (item == NULL || serv == NULL || type == NULL) {
		sprintf(errmsg,
			"Can not save ticker settings: \n" "item (0x%x) server (%s), type (%s).",
			(unsigned int)item, serv, type);
		write_status(errmsg, WARN);
		return NULL;
	}

	sprintf(item, "%s,%s,%s,%s,%s,%s",
			tickTable[pos]->name,
			type,
			serv,
			tickTable[pos]->nickName ? tickTable[pos]->nickName : "",
			tickTable[pos]->low ? tickTable[pos]->low : "",
			tickTable[pos]->high ? tickTable[pos]->high : ""); 

	return item;   
}

void tickAddItemAt(char *item, int at)
{
	char	*next, *stype;
	int	serv, type;

#ifdef	DEBUG
	fprintf(stderr, "tickAddItemAt(%s) @ %d\n", item, at);
#endif
	/* Reallocate the array with one more entry */
	tickTable = (TICK_STRUCT **) XtRealloc ((char *)tickTable,
			(numTick+1) * sizeof(TICK_STRUCT **));

	if (0 <= at && at <= numTick) {
		/* Move the existing entries */
		int	i;

		for (i=numTick; at < i; i--) {
			tickTable[i] = tickTable[i-1];
		}
	} else {
		at = numTick;
	}
	tickTable[at] = (TICK_STRUCT *)XtCalloc(1, sizeof(TICK_STRUCT));

	if ((next = strtok(item, ","))) {
		tickTable[at]->name = XtNewString(next);
	} else {
		/* In case we get passed an empty string, make sure we have a valid name */
		tickTable[at]->name = XtNewString("");
	}

	/* Save for comparison after getting server */
	if ((next = strtok(NULL, ","))) {
		stype = next;
	}

	serv=0;	/* Precaution against not entering the if */
	if ((next = strtok(NULL, ","))) {
		for (serv=0; serv < numServer(); serv++)
			if (getServerTitle(serv) && strcmp (next, getServerTitle(serv)) == 0) {
				tickTable[at]->server = serv;
				break;
			}
	}
	type=0;	/* Precaution against not entering the if */
	if (serv != numServer() && stype != NULL) {
		setCurServer(serv);
		for (type=0; type < numCurServerType(); type++)
			if (getServerType(serv, type)
					&& strcmp (stype, getServerType(serv, type)) == 0) {
				tickTable[at]->type = type;
				break;
			}
	}

	if ((next = strtok (NULL, ",")))
		tickTable[at]->nickName = XtNewString (next);
	tickTable[at]->detail = (QUERY_STRUCT *)NULL;

	if (tickTable[at]->name == NULL ||
			serv == numServer() || type == numCurServerType()) {
		sprintf(errmsg, "Can not restore ticker settings: \n"
				"Ticker '%s'\nServer '%s'\nType '%s'",
				tickTable[at]->name, getServerTitle(serv),
				getServerType(serv,type));
		write_status(errmsg, WARN);
#if 0
		/* What's this ??? */
		tickTable = (TICK_STRUCT **) XtRealloc ((char *)tickTable,
				(numTick+1) * sizeof(TICK_STRUCT **));
#endif
	}
	tickTable[at]->low = tickTable[at]->high = NULL;
	if ((next = strtok (NULL, ",")))
		tickTable[at]->low = XtNewString(next);
	if ((next = strtok (NULL, ",")))
		tickTable[at]->high = XtNewString(next);
#ifdef DEBUG
	fprintf(stderr, "\tlo %s hi %s\n",
  			tickTable[at]->low, tickTable[at]->high);
#endif
	numTick++;
}

extern char *tickGetLowTrigger(int which)
{
	return tickTable[which]->low;
}

extern char *tickGetHighTrigger(int which)
{
	return tickTable[which]->high;
}

extern void tickSetHighTrigger(int which, char *val)
{
	if (tickTable[which]->high)
		XtFree(tickTable[which]->high);
	tickTable[which]->high = val;
	}

extern void tickSetLowTrigger(int which, char *val)
{
	if (tickTable[which]->low)
		XtFree(tickTable[which]->low);
	tickTable[which]->low = val;
}

void tickAddItem(char *item)
{
	tickAddItemAt(item, -1);
}

int tickGetNum ()
{
	return (numTick);
}

/* Get saved ticker server */
int tickGetServer (int which)
{
	if (which < numTick)
		return ( tickTable[which]->server );
	else
		return (0);
}

/* The equivalent - Set saved ticker server */
void tickSetServer(int which, int srv)
{
	if (which < numTick)
		tickTable[which]->server = srv;
}

/* Get saved ticker type */
int tickGetType ( int which )
{
	if (which < numTick)
		return ( tickTable[which]->type );
	else
		return (0);
}

/* Set ticker type */
extern void tickSetType(int tick, int tp)
{
	if (tick < numTick)
		tickTable[tick]->type = tp;
}

/* Get saved ticker name */
char *tickGetName ( int which )
{
	if (which < numTick) {
		if (tickTable[which]->name) {
			return ( tickTable[which]->name );
		}
	}
	return ((char *)NULL);
}

/* Set symbol to use in the HTTP query */
tickSetName(int which, char *name)
{
	if (which < numTick) {
		if (tickTable[which]->name) {
			free(tickTable[which]->name);
		}
		tickTable[which]->name = strdup(name);
	}
#ifdef	DEBUG
	{ int i;
	fprintf(stderr, "tickSetName(%d,%s) -> [%s", which, name, tickTable[0]->name);
	for (i=1; i<numTick; i++)
		fprintf(stderr, ",%s", tickTable[i]->name);
	fprintf(stderr, "]\n");
	}
#endif
}

/* Get saved ticker display string */
char *tickGetNickName ( int which )
{
	if (which >= numTick)
		return NULL;
	if (tickTable[which]->nickName)
		return tickTable[which]->nickName;
	if (tickTable[which]->detail && tickTable[which]->detail->values)
		return tickTable[which]->detail->values[DETAIL_NAME];
	return NULL;
}

/* Get saved data retrieved from net for displaying */
QUERY_STRUCT *tickGetDetail ( int which )
{
	if (which < numTick)
		return ( tickTable[which]->detail );
	else
		return ((QUERY_STRUCT *)NULL);
}

/* Save data retrieved from net for displaying */
void tickSetDetail(int which, QUERY_STRUCT *data)
{
	/* Update the data structure */
	if (which < numTick) {
		/* Replace existing */
		freePattern(tickTable[which]->detail);
		tickTable[which]->detail = data;
	} 

	/* Update the main window display */
        detailUpdateMainWindow(DETAIL_UPDATE);

	/* Check whether we've gone out of our boundaries */
	tickCheckHighLow(which);
}

void tickSetSymbol(int which, char *key)
{
	fprintf(stderr, "tickSetSymbol(%d,%s)\n", which, key);
	if (which < numTick && tickTable[which]->detail) {
		QUERY_STRUCT	*data = tickTable[which]->detail;
		data->values[DETAIL_SYMBOL] = key;
	}
}

/* Add everything from the tick editor list to the main window list */
void tickAddMainWindow ()
{
	detailUpdateMainWindow (DETAIL_EDIT);
	makeTextTape();
}


/* If symbol is in list, return its position, else return 0 */
static int tickIsDuplicate ( Widget list, char *symbol )
{
  XmStringTable ticklist;
  char *item;
  char name[16];
  int i, count;

  XtVaGetValues ( list, XmNitemCount, &count,
		        XmNitems,     &ticklist,
	          NULL);
  for ( i=0; i < count; i++) {
    XmStringGetLtoR ( ticklist[i], XmFONTLIST_DEFAULT_TAG, &item);
    sscanf ( item, "%15s", name );
    if (strcasecmp ( name, symbol) == 0 ) {
      XtFree(item);
      return (i+1);
    }
    XtFree(item);
  }
  return 0;  
}

/* Callback for ticker symbol dialog */
/* ARGSUSED */
static void procTick (Widget w, int client_data, XtPointer call_data)
{
  static Widget list = (Widget) NULL;
  char *symbol[TICKFIELDS];

  XmString orig;
  int origCount;

  if (list == (Widget) NULL) {
    list = XtNameToWidget (GetTopShell(w), 
                 "StockPane.TickFrame.TickForm.StockListSW.StockList");
  }

  switch (client_data) {
 
    case 0: /* Ok */
         XtPopdown (GetTopShell(w));

         /* Destroy old list */
	 for (origCount=0; origCount < numTick; origCount++) {
           freePattern (tickTable[origCount]->detail);
           XtFree (tickTable[origCount]->name);
           XtFree (tickTable[origCount]->nickName);
           XtFree ((char *)tickTable[origCount]);
         }
         XtFree ((char *)tickTable);
 
         /* Preserve new list */
         tickTable = newTickTable;
         numTick = newNumTick;

         newTickTable = NULL;
         newNumTick = 0;

         /* Update main window table */
         tickAddMainWindow ();

         break;

    case 1: /* Cancel */
         XtPopdown (GetTopShell(w));

         /* Destroy new list */
	 for (origCount=0; origCount < newNumTick; origCount++) {
           freePattern (newTickTable[origCount]->detail);
           XtFree (newTickTable[origCount]->name);
           XtFree (newTickTable[origCount]->nickName);
           XtFree ((char *)newTickTable[origCount]);
         }
         XtFree ((char *)newTickTable);
         newTickTable = NULL;

	 /* Restore previous list */
	 XmListDeleteAllItems (list);
	 for (origCount=0; origCount < numTick; origCount++) {
           orig = XmStringCreateLocalized (tickTable[origCount]->name);
	   XmListAddItem (list, orig, 0);
           XmStringFree (orig);
         }
	 XmListSetBottomPos (list, 0);
         break;

    case 2: /* Add */
         {
           XmString listItem;
           char line[80];

           /* Get ticker symbol */
           symbol[0] = XmTextFieldGetString (textField[0]);

           /* If not blank or duplicate, add to list */
           if (symbol[0] == NULL || strlen(symbol[0]) == 0) {
	     write_status ("Symbol is empty.", ERR);
	   } else {
	     if (!tickIsDuplicate (list, symbol[0])) {
	       symbol[1] = XmTextFieldGetString (textField[1]);
               if (symbol[1] == NULL || strlen(symbol[1]) == 0) {
	         sprintf (errmsg, 
                          "Symbol text is empty. Reuse\n"
                          "'%s' again here or supply\n"
                          "any other text to display\n"
                          "for this ticker symbol.",
		          symbol[0]);
	         write_status (errmsg, ERR);
               } else {

                 /* Add new symbol */
                 newTickTable = (TICK_STRUCT **) 
                                   XtRealloc ((char *)newTickTable, 
                                       (newNumTick+1) * sizeof(TICK_STRUCT **));
                 newTickTable[newNumTick] = 
                             (TICK_STRUCT *) XtCalloc ( 1, sizeof(TICK_STRUCT));

                 newTickTable[newNumTick]->name = XtNewString (symbol[0]);
                 newTickTable[newNumTick]->type = curType;
                 newTickTable[newNumTick]->server = curServer;
                 newTickTable[newNumTick]->nickName = XtNewString(symbol[1]);
                 newTickTable[newNumTick++]->detail = (QUERY_STRUCT *)NULL;

                 sprintf (line, "%-15.15s => %-60s", symbol[0], symbol[1]);
                 listItem = XmStringCreateLocalized (line);
                 XmListAddItem (list, listItem, 0);
		 XmListSetBottomPos (list, 0);

                 XmStringFree (listItem);
               }
               XtFree (symbol[1]);
             } else {
	       /* Duplicate */
	       sprintf (errmsg, "Symbol '%s' is already in the list.", 
		        symbol[0]);
	       write_status (errmsg, ERR);
	     }
	   }
           XtFree (symbol[0]);
         }
         break;

    case 3: /* Remove */
         {
           char line[80];
	   int pos, i;

           /* Get ticker symbol */
           symbol[0] = XmTextFieldGetString (textField[0]);

           /* If not blank and exists, remove from list */
           if (strlen(symbol[0]) && (pos = tickIsDuplicate (list, symbol[0]))) {
	     XmListDeletePos (list, pos--); 
             XtFree (newTickTable[pos]->name);
             XtFree (newTickTable[pos]->nickName);
             freePattern (newTickTable[pos]->detail);
             XtFree ((char *)newTickTable[pos]);

             if (pos < newNumTick-1) 
               for (i=pos; i<newNumTick-1; i++) { 
                 newTickTable[i] = newTickTable[i+1];
                 newTickTable[i]->name = newTickTable[i+1]->name;
                 newTickTable[i]->nickName = newTickTable[i+1]->nickName;
                 newTickTable[i]->server = newTickTable[i+1]->server;
                 newTickTable[i]->type = newTickTable[i+1]->type;
                 newTickTable[i]->detail = newTickTable[i+1]->detail;
               }
             newTickTable = (TICK_STRUCT **) 
                             XtRealloc ((char *)newTickTable, 
                                       (--newNumTick) * sizeof(TICK_STRUCT **));
	   } else {
	     sprintf (line, "Symbol '%s' is not in the list.", symbol[0]);
	     write_status (line, ERR);
	   }
           XtFree (symbol[0]);
         }
         break;

    default: break;
  }

}

/* ARGSUSED */
static void textTick (Widget w, XtPointer client_data, XtPointer call_data)
{
  int otherW = (int)client_data;
  char *foo;
 
  /* Simulate add button */
  if (otherW == 0)
    procTick (w, 2, (XtPointer) NULL);

  /* if field is empty don't change focus */
  foo = XmTextFieldGetString (w);
  if (foo == NULL || strlen(foo) == 0) {
    XtFree (foo);
    return;
  }

  /* Change focus to name/nickname as appropriate */
  XmProcessTraversal (w, (otherW==0)?XmTRAVERSE_UP:XmTRAVERSE_DOWN);

  /* Highlight text for replacement */
  XmTextFieldSetSelection (textField[otherW], 0,
       XmTextGetLastPosition(textField[otherW]), CurrentTime);
}

/* Read list params back into text fields when list item clicked on */
/* ARGSUSED */
static void loadTick (Widget w, XtPointer client_data, 
                      XmListCallbackStruct *call_data)
{
  int pos = call_data->item_position-1;

  /* Assume that the ticker symbol is never NULL or blanks */
  if (textField[0]) {
    /* Restore ticker */
    XmTextFieldSetString (textField[0], newTickTable[pos]->name);

    /* Restore server */
    if (newTickTable[pos]->server <= numServer()) {
      setCurServer (newTickTable[pos]->server);
      setOptionMenu (optServ, newTickTable[pos]->server);
    }

    /* Restore server type */
    if (newTickTable[pos]->type <= numCurServerType()) {
      setCurServerType (newTickTable[pos]->type);
      setOptionMenu (optServType, newTickTable[pos]->type);
    }

    /* Restore nickname */
    XmTextFieldSetString (textField[1], newTickTable[pos]->nickName);
  }
}

/* ARGSUSED */
static void tickTypeCB ( Widget w, XtPointer client_data, XtPointer call_data)
{
  if ((int)client_data != curType) {
    curType = (int)client_data;
    setCurServerType (curType);
  }
}


/* ARGSUSED */
static void tickServerCB ( Widget w, XtPointer client_data, XtPointer call_data)
{
  if ((int)client_data != curServer) {
    Widget othermenu;
    XtVaGetValues (XtParent(w), XmNuserData, &othermenu, NULL);
    curServer = (int)client_data;
    setCurServer (curType);
    tickMakeTypeMenu(othermenu, 0);
  }
}

void tickMakeTypeMenu(Widget menu, int type)
{
  WidgetList buttons;
  int num;
  Widget button, othermenu;
  int i;

  if (menu == (Widget) NULL)
    return;

  /* Out with the old... */
  XtVaGetValues ( menu,
		  XmNchildren, &buttons,
		  XmNnumChildren, &num,
		  XmNuserData, &othermenu,
		  NULL );
  for (i=0; i < num; i++)
    XtDestroyWidget ( buttons[i] );

  /* ...in with the new. */
  setCurServer(curServer);
  if (type == 0 ) {       /* Server types */
    for (i = 0; i < numCurServerType(); i++) {
      setCurServerType(i);
      button = XtVaCreateManagedWidget (getServer(DETAIL_TITLE),
		                        xmPushButtonGadgetClass, menu, NULL);
      XtAddCallback (button, XmNactivateCallback, tickTypeCB, (XtPointer)i);
    }
    setCurServerType(0);
  } else if (type == 1) { /* Quote server */
    for (i = 0; i < numServer(); i++) {
      button = XtVaCreateManagedWidget (getServerTitle(i),
		     xmPushButtonGadgetClass, menu, NULL);
      XtAddCallback (button, XmNactivateCallback, tickServerCB, (XtPointer)i);
    }
    /* When we change servers, we need to refresh type list */
    if (getCurServer() != 0) {
      setCurServer(0);
      tickMakeTypeMenu(othermenu, 0);
    }
  }
}

void createTickDialog ()
{
  static Widget dialog = (Widget) NULL, list; 
  Widget form, frame, pane, row, menu[2], button;

  Dimension width, height, border;

  int num;
  Arg args[10];

  char *labels[]  = { "label_0", "label_1", "label_2", "label_3"};
  char *texts[]   = { "text_0", "text_1"};
  char *buttons[] = { "button_0","button_1","button_2","button_3"};
  char *menus[]   = { "TickServerMenu", "TickTypeMenu"};
  char *menupulldown[] = { "TickServerMenu_pulldown", "TickTypeMenu_pulldown"};

  /* Create if not done yet */
  if (dialog == (Widget)NULL) {
    dialog = XtVaCreatePopupShell ("OptionTick", 
                            xmDialogShellWidgetClass,
                            GetTopShell (per->Toplevel),
                            NULL);

    pane = XtVaCreateWidget ("StockPane", xmPanedWindowWidgetClass, 
                            dialog,
                            XmNsashWidth, 1,
                            XmNsashHeight, 1,
                            NULL);

    frame =   XtVaCreateManagedWidget( "TickFrame",
                          xmFrameWidgetClass, pane,
                          NULL);
    XtVaCreateManagedWidget ("TickFrameLabel",
                           xmLabelGadgetClass, frame,
                           XmNchildType, XmFRAME_TITLE_CHILD,
                           XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
                           NULL);
    /* form to hold rowcolumn form, scrolled list */
    form = XtVaCreateWidget ("TickForm", xmFormWidgetClass, frame, NULL);


    row = XtVaCreateWidget ("TickRow", xmRowColumnWidgetClass, form,
                           XmNnavigationType, XmNONE,
                           XmNtopAttachment, XmATTACH_FORM,
                           XmNleftAttachment, XmATTACH_FORM,
                           XmNnumColumns, XtNumber(labels),
                           XmNorientation, XmHORIZONTAL,
                           XmNpacking, XmPACK_COLUMN,
                            NULL);
                
    for (num=0; num < XtNumber(labels); num++) {
      /* Label describing entry */
      XtVaCreateManagedWidget (labels[num], xmLabelGadgetClass, row, NULL);

      switch (num) {
      case 0: /* Server */
              menu[0] = XmCreatePulldownMenu (row, menupulldown[0], args, 0);
              XtSetArg (args[0], XmNsubMenuId, menu[0]);
              optServ = XmCreateOptionMenu (row, menus[0], args, 1);
              tickMakeTypeMenu (menu[0], 1);
              XtManageChild (optServ);
              break;

      case 1: /* Server Type */
              menu[1] = XmCreatePulldownMenu (row, menupulldown[1], args, 0);
              XtSetArg (args[0], XmNsubMenuId, menu[1]);
              optServType = XmCreateOptionMenu (row, menus[1], args, 1);
              tickMakeTypeMenu (menu[1], 0);
              /* Save type menu in server menu, so it can be recreated */
              /* on server change                                      */
              XtVaSetValues (menu[0], XmNuserData, menu[1], NULL); 
              XtManageChild (optServType);
              break;

      case 2: /* Ticker/Code Symbol */
              textField[0] = XtVaCreateManagedWidget (texts[0],
                            xmTextFieldWidgetClass, row,
                            XmNnavigationType, XmNONE,
                            NULL);
              XtAddCallback (textField[0], XmNactivateCallback,
                   (XtCallbackProc) textTick, (XtPointer)1);
              break;

      case 3:
              textField[1] = XtVaCreateManagedWidget (texts[1],
                            xmTextFieldWidgetClass, row,
                            XmNnavigationType, XmNONE,
                            NULL);
              XtAddCallback (textField[1], XmNactivateCallback,
                   (XtCallbackProc) textTick, (XtPointer)0);
              break;
      }
    }
    XtManageChild (row);

    /* List holding what's been added to date */
    num = 0;
    XtSetArg(args[num], XmNscrollBarDisplayPolicy, XmSTATIC); num++;
    XtSetArg(args[num], XmNvisibleItemCount, 5); num++;
    XtSetArg(args[num], XmNselectionPolicy,  XmSINGLE_SELECT); num++;
    XtSetArg(args[num], XmNtopAttachment,    XmATTACH_WIDGET); num++;
    XtSetArg(args[num], XmNtopWidget,        row); num++;
    XtSetArg(args[num], XmNleftAttachment,   XmATTACH_FORM); num++;
    XtSetArg(args[num], XmNrightAttachment,  XmATTACH_FORM); num++;
    list = XmCreateScrolledList (form, "StockList", args, num);

    XtAddCallback (list, XmNdefaultActionCallback,
                        (XtCallbackProc) loadTick, NULL);
    XtManageChild (list);
    XtManageChild (form);

    /* Buttons to add delete cancel exit */
    form = XtVaCreateWidget ("ButForm", xmFormWidgetClass, pane,
                            XmNfractionBase, XtNumber(buttons),
                            NULL);

    for (num=0; num < XtNumber(buttons); num++) {
      button = XtVaCreateManagedWidget (buttons[num], 
                            xmPushButtonWidgetClass, form,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            XmNleftAttachment, XmATTACH_POSITION,
                            XmNleftPosition, num,
                            XmNrightAttachment, XmATTACH_POSITION,
                            XmNrightPosition, num+1,
                            XmNshowAsDefault, (num==0)?True:False,
                            XmNdefaultButtonShadowThickness, 1,
                            NULL);
      XtAddCallback (button, XmNactivateCallback,
                            (XtCallbackProc) procTick, (XtPointer)num);
    }
    XtManageChild (form);
    XtManageChild (pane);

    /* Prevent pane from changing size */
    XtVaGetValues (dialog, 
                 XmNwidth, &width,
                 XmNheight, &height,
                 XmNborderWidth, &border,
                 NULL);

    XtVaSetValues (dialog,
                 XmNminWidth,  width +  border,
                 XmNmaxWidth,  width +  border,
                 XmNminHeight, height + border,
                 XmNmaxHeight, height + border,
                 NULL);
  }

  /* Get a working copy of ticker contents to use while dialog posted */
  {
    int count;
    char line[80];
    XmString xline;

    XmListDeleteAllItems (list);
    newTickTable = (TICK_STRUCT **) XtRealloc ((char *)newTickTable, 
                                       (numTick) * sizeof(TICK_STRUCT **));
    for (count=0; count < numTick; count++) {
      newTickTable[count] = 
                             (TICK_STRUCT *) XtCalloc (1, sizeof(TICK_STRUCT));
      newTickTable[count]->name = XtNewString (tickTable[count]->name);
      newTickTable[count]->type = tickTable[count]->type;
      newTickTable[count]->server = tickTable[count]->server;
      newTickTable[count]->nickName = XtNewString(tickTable[count]->nickName);
      newTickTable[count]->detail = (QUERY_STRUCT *)NULL;

      if (newTickTable[count]->nickName)
        sprintf (line, "%-15.15s => %-60s", 
                 tickTable[count]->name, tickTable[count]->nickName);
      else
        sprintf (line, "%-15.15s => ", tickTable[count]->name);
      xline = XmStringCreateLocalized (line);
      XmListAddItem (list, xline, 0);
      XmStringFree (xline);
    }
    newNumTick = numTick;
  }
  XmListSetBottomPos (list, 0);

  XtManageChild (dialog);
  XtPopup (dialog, XtGrabNone);

  return;
}

extern void tickSwap(int i, int j)
{
	TICK_STRUCT	*p;

	if (i < 0 || j < 0 || i >= numTick || j >= numTick)
		return;
	p = tickTable[i];
	tickTable[i] = tickTable[j];
	tickTable[j] = p;
}

static void tickCheckCommand(int tick)
{
	char	*cmd, *s, *p, *q;
	int	len;
	int	i;

	cmd = prefGetMailCommand();
	if (! cmd)
		return;

	len = strlen(cmd) * 3 + 100;
	s = XtMalloc(len);

	/*
	 * Do search and replace for :
	 * %n	-> tickTable[tick]->detail->values[DETAIL_NAME]
	 * %l	-> tickTable[tick]->low
	 * %h	-> tickTable[tick]->high
	 * %v	-> tickTable[tick]->detail->values[DETAIL_PRICE]
	 * %%	-> %
	 */
	*s = '\0';
	for (i=0, p=s; cmd[i]; i++) {
		if (cmd[i] != '%') {
			/* Simple copy */
			*(p++) = cmd[i];
			*p = '\0';
			continue;
		}

		/* Replace */
		switch (cmd[i+1]) {
			case 'n':
				q = tickTable[tick]->detail->values[DETAIL_NAME];
				break;
			case 'l':
				q = tickTable[tick]->low;
				break;
			case 'v':
				q = tickTable[tick]->detail->values[DETAIL_PRICE];
				break;
			case 'h':
				q = tickTable[tick]->high;
				break;
			case '%':
			default:
				q = NULL;
				break;
		}
		if (q) {
			for (; *q; )
				*(p++) = *(q++);
			*p = '\0';
			i++;
		} else {
			*(p++) = cmd[i++];
			*(p++) = cmd[i];
			*p = '\0';
		}
	}
	*p = '\0';
#if 0
	fprintf(stderr, "Command[%s]\n", s);
#endif
	(void)system(s);
	XtFree(s);
}

extern void tickCheckHighLow(int i)
{
	float	val, low, high;

	if (tickTable[i]->detail->values[DETAIL_PRICE] != NULL &&
			sscanf(tickTable[i]->detail->values[DETAIL_PRICE], "%f", &val) != 1)
		return;

	if (tickTable[i]->low && tickTable[i]->lowcheck == 0) {
		if (tickTable[i]->low != NULL && sscanf(tickTable[i]->low, "%f", &low) == 1) {
			if (val < low) {
				tickCheckCommand(i);
#if 0
				fprintf(stderr, "tickCheckHighLow(LOW, %d, %s, %s, %s, %s)\n",
						i,
						tickTable[i]->detail->values[DETAIL_NAME],
						tickTable[i]->detail->values[DETAIL_PRICE],
						tickTable[i]->low,
						tickTable[i]->high);
#endif
				tickTable[i]->lowcheck = 1;
			}
		}
	}

	if (tickTable[i]->high && tickTable[i]->highcheck == 0) {
		if (tickTable[i]->high != NULL && sscanf(tickTable[i]->high, "%f", &high) == 1) {
			if (high < val) {
				tickCheckCommand(i);
#if 0
				fprintf(stderr, "tickCheckHighLow(HIGH, %d, %s, %s, %s, %s)\n",
						i,
						tickTable[i]->detail->values[DETAIL_NAME],
						tickTable[i]->detail->values[DETAIL_PRICE],
						tickTable[i]->low,
						tickTable[i]->high);
#endif
				tickTable[i]->highcheck = 1;
			}
		}
	}
}
