/*----------------------------------------------------------------------------
--
--  Module:           xtmRemove
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Removes obsolete appointments in the diary older than n days. By 
--    default, all entries without a tag are removed. If one or more tags
--    are specified, entries with these tags are also removed.
--
--    Tags are case sensitive; if tag 'Imp' is to be removed, entries with
--    the tag 'imp' will not be removed.
--
--  Filename:         xtmRemove.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-07-30
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmRemove.c, Version: 1.1, Date: 95/02/18 16:10:19";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <ctype.h>

#include "System.h"
#include "LstLinked.h"
#include "TimDate.h"

#include "xtmDbTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdremove"

/* Max tags to check for deletion. */
#define MAX_DEL_TAG  50


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Application data record. */
typedef struct {

  Boolean  confirm;
  Boolean  log_id;
  char     database_dir[ 150 ];
  int      days;
  int      del_tag_index;
  char     *del_tag[ MAX_DEL_TAG ];

} APPL_BASE_DATA_DEF, *APPL_BASE_DATA_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static  char  *program_name;

/* Name of module. */
static char  *module_name = "xtmRemoveObsolete";

/* Global flags. */
static Boolean  wipe_all;
static char     today_date_buffer[ 50 ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  deleteEntries( APPL_BASE_DATA_REF      appl_data_ref,
                 XTM_DB_ENTRY_DATABASES  *database_ref,
                 TIM_TIME_REF            date_limit );

static void
  deleteEntry( APPL_BASE_DATA_REF      appl_data_ref,
               XTM_DB_ENTRY_DATABASES  *database_ref,
               XTM_DB_ALL_ENTRY_DEF    *entry_ref );

static void
  deleteRepeatedEntries( APPL_BASE_DATA_REF      appl_data_ref,
                         XTM_DB_ENTRY_DATABASES  *database_ref,
                         TIM_TIME_REF            date_limit );

static void
  displayUsage();

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

int
main( int argc, char *argv[] )
{

  /* Variables. */
  Boolean                 do_file_lock = True;
  int                     arg_ref;
  char                    buffer[ 80 ];
  APPL_BASE_DATA_DEF      appl_data;
  TIM_STATUS_TYPE         tstatus;
  TIM_TIME_REF            date_limit;
  TIM_TIME_REF            now;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;


  /* Code. */

  /* Defaults. */
  today_date_buffer[ 0 ] = '\0';

  /* Name of program. */
  program_name = PROGRAM_NAME;

  
  /* Initialization. */
  SysInitializeEnvironment();

  
  /* Default values. */
  appl_data.log_id            = False;
  appl_data.confirm           = True;
  appl_data.database_dir[ 0 ] ='\0';
  appl_data.days              = 0;
  appl_data.del_tag_index     = 0;

  /* Fetch flags. */
  arg_ref = 1;
  while( arg_ref < argc && argv[ arg_ref ][ 0 ] == '-' ) {

    if( noCaseStrcmp( argv[ arg_ref ], "-todayis" ) == 0 ) {
      arg_ref++;
      if( arg_ref >= argc ) {
        fprintf( stderr, "%s: -todayis requires parameter.\n", program_name );
        displayUsage();
        exit( 0 );
      }

      strcpy( today_date_buffer, argv[ arg_ref ] );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-noconfirm" ) == 0 ) {
      appl_data.confirm = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-nofilelock" ) == 0 ) {
      do_file_lock = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-log" ) == 0 ) {
      appl_data.log_id = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-usage" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-wipe" ) == 0 ) {
      wipe_all = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-help" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-h" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-version" ) == 0 ) {
      printf( "%s: Version: %s\n", program_name, VERSION_ID );
      exit( 0 );

    } if( noCaseStrcmp( argv[ arg_ref ], "-tag" ) == 0 ) {
      arg_ref++;

      if( arg_ref >= argc ) {
        fprintf( stderr, "%s: The -tag parameter requires a tag.\n",
                 program_name );
        displayUsage();
        exit( 0 );
      }

      if( appl_data.del_tag_index >= MAX_DEL_TAG ) {
        fprintf( stderr, "%s: To many -tag flags.\n",
                 program_name );
        displayUsage();
        exit( 0 );
      }

      /* Save the tag to delete. */
      appl_data.del_tag[ appl_data.del_tag_index ] = 
        SysNewString( argv[ arg_ref ] );

      appl_data.del_tag_index++;

    } /* if */

    arg_ref++;

  } /* while */


  /* Required parameters. */
  if( argc - arg_ref < 2 ) {
    fprintf( stderr, "%s: The database dir and/or days are missing.\n",
             program_name );
    displayUsage();
    exit( 0 );
  }


  /* Save the database dir and the 'delete older than days' parameters. */
  strcpy( appl_data.database_dir, argv[ arg_ref ] );
  appl_data.days = atoi( argv[ arg_ref + 1 ] );

  if( appl_data.days <= 0 ) {
    fprintf( stderr, "%s: The number of days must be >0.\n",
             program_name );
    exit( 1 );
  }

  /* Delete entries older than today - n days. */
  if( strlen( today_date_buffer ) > 0 ) {
    tstatus = TimMakeDateFromString( &date_limit, today_date_buffer );

    if( tstatus != TIM_OK ) {
      fprintf( stderr, "%s: Invalid date %s.\n", 
               program_name, today_date_buffer );
      exit( 1 );
    }

  /* Use today's date? */
  } else {
    now = TimLocalTime( TimMakeTimeNow() );

    date_limit = TimMakeTime( TimIndexOfYear(  now ),
                              TimIndexOfMonth( now ),
                              TimIndexOfDay(   now ),
                              0, 0, 0 );
  }

  TimAddDays( &date_limit, -1 * appl_data.days );


  /* Confirm the action. */
  if( appl_data.confirm ) {
    TimFormatStrTime( date_limit, "%B, %d %Y",
                      buffer, sizeof( buffer ) );

    printf( "\n"
            "Entries older than %d days (before %s) will be removed\n"
            "from the diary database in directory %s.\n"
            "\n",
            appl_data.days, buffer, appl_data.database_dir );
    printf( "Is this OK? (Y/N) [N]: " );

    gets( buffer );
    if( toupper( buffer[ 0 ] ) != 'Y' )
      exit( 0 );
  }


  /* Use file locking? */
  xtmDbUseFileLock( do_file_lock );


  /* Open the entry database. */
  open_request.name       = "";
  open_request.directory  = appl_data.database_dir;
  open_request.operations = XTM_DB_FLAG_MODE_WRITE;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, &database );
  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot open diary database in %s.\n",
             program_name, appl_data.database_dir );
    exit( 1 );
  }


  /* Delete normal entries. */
  deleteEntries( &appl_data, &database, date_limit );

  /* Delete repeated entries. */
  deleteRepeatedEntries( &appl_data, &database, date_limit );


  /* Close the database and exit. */
  xtmDbCloseEntryDb( &database );


  exit( 0 );

} /* main */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  /* Code. */

  printf( 
    "\n"
    "%s (%s): Removes obsolete appointments from the diary database.\n"
    "\n"
    "Removes obsolete appointments in the diary older than n days. Only\n"
    "entries with the specified tag(s) are removed.\n"
    "\n"
    "Usage:\n"
    "  %s [flags] database days\n"
    "\n"
    "where:\n"
    "  database  : Diary database to use (directory name).\n"
    "  days      : Remove entries older than <days> days.\n"
    "Flags:\n"
    "  -help           : Displays this help.\n"
    "  -h              : See above.\n"
    "  -log            : Displays the id of the entries deleted.\n"
    "  -noConfirm      : Does not ask for confirmation.\n"
    "  -noFileLock     : Doesn't use any file locking.\n"
    "  -repeated       : Removes repeated entries also.\n"
    "  -tag <tag>      : Removes entries with the tag <tag>. Up to\n"
    "                    %d tags can be given. If you give /all/ as tag,\n"
    "                    all entries (with or without tags) will be removed.\n"
    "  -todayis <date> : Instead of using today as starting date, use\n"
    "                    <date>.\n"
    "  -us             : Uses US date for the -todayis date.\n"
    "  -usage          : Displays a short help.\n"
    "  -version        : Displays the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name, MAX_DEL_TAG );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void
  deleteEntry( APPL_BASE_DATA_REF      appl_data_ref,
               XTM_DB_ENTRY_DATABASES  *database_ref,
               XTM_DB_ALL_ENTRY_DEF    *entry_ref )
{

  /* Variables. */
  Boolean        delete_entry;
  int            count;
  char           *tag_ref;
  XTM_DB_STATUS  db_status;


  /* Code. */


  /* Strip spaces from the tag. */
  tag_ref = entry_ref -> entry.tag;
  while( isspace( *tag_ref ) )
    tag_ref++;

  delete_entry = False;

  /* Clean the house? */
  if( wipe_all ) {
    delete_entry = True;

  /* Check the tag. */
  } else {
    for( count = 0; count < appl_data_ref -> del_tag_index; count++ ) {

      if( strcmp( appl_data_ref -> del_tag[ count ], "/all/" ) == 0 ||
          strcmp( appl_data_ref -> del_tag[ count ], tag_ref ) == 0 ) {
        delete_entry = True;
        break;
      }

    } /* loop */

  } /* if */

  /* Delete the entry? */
  if( delete_entry ) {
    db_status = xtmDbDeleteEntry( database_ref, entry_ref -> entry.id );

    if( db_status != XTM_DB_OK ) {
      fprintf( stderr, "%s: Cannot delete entry %d from database.\n",
               program_name, entry_ref -> entry.id );

    } else if( appl_data_ref -> log_id ) {
      printf( "  Entry %d deleted.\n" , entry_ref -> entry.id );

    } /* if */

  } /* if */


  return;

} /* deleteEntry */


/*----------------------------------------------------------------------*/

static void
  deleteEntries( APPL_BASE_DATA_REF      appl_data_ref,
                 XTM_DB_ENTRY_DATABASES  *database_ref,
                 TIM_TIME_REF            date_limit )
{

  /* Variables. */
  int                   index;
  LST_DESC_TYPE         dates_list;
  LST_DESC_TYPE         list_ref[ 2 ];
  LST_STATUS            lst_status;
  TIM_TIME_REF          check_date;
  XTM_DB_ALL_ENTRY_DEF  entry_record;
  XTM_DB_DATE_DEF       date_record;
  XTM_DB_STATUS         db_status;


  /* Code. */

  /* Fetch all dates with entries. */
  db_status = xtmDbFetchDates( database_ref, &dates_list );


  /* Check all dates and look for dates earlier than date_limit. */
  lst_status = LstLinkCurrentFirst( dates_list );
  while( lst_status == LST_OK ) {

    lst_status = LstLinkGetCurrent( dates_list, &date_record );

    /* Check entries this date? */
    check_date = (TIM_TIME_REF) date_record.date;

    if( check_date >= date_limit ) {
      lst_status = LstLinkCurrentNext( dates_list );
      continue;
    }

    /* Fetch the entries for the current day (only normal entries). */
    db_status = xtmDbFetchEntriesInDay( database_ref, date_record.date, 
                                        0,
                                        &list_ref[ 1 ], &list_ref[ 0 ] );

    if( db_status != XTM_DB_OK ) {
      fprintf( stderr, "%s: Cannot fetch entries from database.\n",
               program_name );
      exit( 1 );
    }


    /* Check each entry. */
    for( index = 0; index < 2; index++ ) {

      lst_status = LstLinkCurrentFirst( list_ref[ index ] );
      while( lst_status == LST_OK ) {

        lst_status = LstLinkGetCurrent( list_ref[ index ], &entry_record );

        /* Delete the entry? */
        deleteEntry( appl_data_ref, database_ref, &entry_record );

        lst_status = LstLinkCurrentNext( list_ref[ index ] );

      } /* while */
    
      /* Delete the list. */
      LstLinkClear( list_ref[ index ] );

    } /* loop */


    /* Next date. */
    lst_status = LstLinkCurrentNext( dates_list );

  } /* while */

  /* Delete the list. */
  LstLinkClear( dates_list );


  return;

} /* deleteEntries */


/*----------------------------------------------------------------------*/

static void
  deleteRepeatedEntries( APPL_BASE_DATA_REF      appl_data_ref,
                         XTM_DB_ENTRY_DATABASES  *database_ref,
                         TIM_TIME_REF            date_limit )
{

  /* Variables. */
  int                   index;
  LST_DESC_TYPE         list_ref[ 2 ];
  LST_STATUS            lst_status;
  TIM_TIME_REF          check_date;
  XTM_DB_ALL_ENTRY_DEF  entry_record;
  XTM_DB_STATUS         db_status;


  /* Code. */

  /* Fetch all repeated entries. */
  db_status = xtmDbFetchStandEntries( database_ref,
                                      &list_ref[ 1 ], &list_ref[ 0 ] );
  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot fetch repeated entries from database.\n",
             program_name );
    return;
  }
  

  /* Check each entry. */
  for( index = 0; index < 2; index++ ) {

    lst_status = LstLinkCurrentFirst( list_ref[ index ] );
    while( lst_status == LST_OK ) {

      lst_status = LstLinkGetCurrent( list_ref[ index ], &entry_record );

      check_date = (TIM_TIME_REF) entry_record.stand_entry.to;

      /* Within date limits? */
      if( wipe_all )
        deleteEntry( appl_data_ref, database_ref, &entry_record );

      else if( check_date != 0 && check_date < date_limit )
        deleteEntry( appl_data_ref, database_ref, &entry_record );

      lst_status = LstLinkCurrentNext( list_ref[ index ] );

    } /* while */
    
    /* Delete the list. */
    LstLinkClear( list_ref[ index ] );

  } /* loop */


  return;

} /* deleteRepeatedEntries */


/*----------------------------------------------------------------------*/

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 )
{

  /* Variables. */
  char  *char_ref1;
  char  *char_ref2;


  /* Code. */

  if( strlen( buffer1 ) != strlen( buffer2 ) )
    return( strcmp( buffer1, buffer2 ) );

  char_ref1 = buffer1;
  char_ref2 = buffer2;

  while( *char_ref1 != '\0' ) {
    if( tolower( *char_ref1 ) < tolower( *char_ref2 ) )
      return( -1 );
    else if( tolower( *char_ref1 ) > tolower( *char_ref2 ) )
      return( 1 );

    char_ref1++;
    char_ref2++;
  }

  return( 0 );

} /* noCaseStrcmp */


