/*----------------------------------------------------------------------------
--
--  Module:           LstLinked
--
--  Project:          Tools - General C objects.
--  System:           Lst - Linked lists.
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Manages a double linked list. Operations for create, insert, 
--    delete etc.
--
--  Filename:         LstLinked.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-12-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: LstLinked.c, Version: 1.1, Date: 95/02/18 14:32:28";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <malloc.h>
#include <memory.h>
#include <stdio.h>

#include "System.h"
#include "LstLinked.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

LST_DESC_TYPE  
  LstLinkNew( int               record_size,
              EQUALS_FUNC_TYPE  equals_func )
{

  /* Variables. */
  LST_DESC_TYPE  list_desc;


  /* Code. */

  list_desc = SysNew( LST_DESC_RECORD );

  list_desc -> head    = NULL;
  list_desc -> tail    = NULL;
  list_desc -> current = NULL;

  list_desc -> elements    = 0;
  list_desc -> record_size = record_size;
  list_desc -> equals_func = equals_func;


  return( list_desc );

} /* LstLinkNew */


/*----------------------------------------------------------------------*/

void
  LstLinkClear( LST_DESC_TYPE  list_desc )
{

  /* Variables. */
  LST_RECORD_TYPE  *node_ref;
  LST_RECORD_TYPE  *tmp_ref;


  /* Code. */

  if( list_desc == NULL )
    return;

  node_ref = list_desc -> head;

  while( node_ref != NULL ) {

    tmp_ref  = node_ref;
    node_ref = node_ref -> next;

    SysFree( tmp_ref -> record );
    SysFree( tmp_ref );

  } /* while */

  list_desc -> head    = NULL;
  list_desc -> tail    = NULL;
  list_desc -> current = NULL;

  list_desc -> elements = 0;


  return;

} /* LstLinkClear */


/*----------------------------------------------------------------------*/

void
  LstLinkClearDataAndList( LST_DESC_TYPE         list_desc,
                           CLEAR_DATA_FUNC_TYPE  clear_func )
{

  /* Variables. */
  LST_RECORD_TYPE  *node_ref;
  LST_RECORD_TYPE  *tmp_ref;


  /* Code. */

  if( list_desc == NULL )
    return;

  node_ref = list_desc -> head;

  while( node_ref != NULL ) {

    tmp_ref  = node_ref;
    node_ref = node_ref -> next;

    /* Call function to clear allocated data. */
    ( *clear_func )( (void *)tmp_ref -> record );

    SysFree( tmp_ref -> record );
    SysFree( tmp_ref );

  } /* while */

  list_desc -> head    = NULL;
  list_desc -> tail    = NULL;
  list_desc -> current = NULL;

  list_desc -> elements = 0;


  return;

} /* LstLinkClearDataAndList */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkInsertFirst( LST_DESC_TYPE  list_desc,
                      void           *record )
{

  /* Variables. */
  char             *new_record;
  LST_RECORD_TYPE  *new_node;
  

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  /* Create the new node and copy the record. */
  new_record = SysMalloc( list_desc -> record_size );
  memcpy( new_record, record, list_desc -> record_size );

  new_node = SysNew( LST_RECORD_TYPE );
  new_node -> record = new_record;

  /* Link the new record into the list. */
  if( list_desc -> head == NULL ) {

    new_node -> next     = NULL;
    new_node -> previous = NULL;

    list_desc -> head    = new_node;
    list_desc -> tail    = new_node;
    list_desc -> current = new_node;

  } else {

    new_node -> next     = list_desc -> head;
    new_node -> previous = NULL;

    list_desc -> head -> previous = new_node;
    list_desc -> head             = new_node;

  } /* if */

  list_desc -> elements++;


  return( LST_OK );

} /* LstLinkInsertFirst */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkInsertLast( LST_DESC_TYPE  list_desc,
                     void           *record )
{

  /* Variables. */
  char             *new_record;
  LST_RECORD_TYPE  *new_node;


  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  /* Create the new node and copy the record. */
  new_record = SysMalloc( list_desc -> record_size );
  memcpy( new_record, record, list_desc -> record_size );

  new_node = SysNew( LST_RECORD_TYPE );
  new_node -> record = new_record;

  /* Link the new record into the list. */
  if( list_desc -> head == NULL ) {

    new_node -> next     = NULL;
    new_node -> previous = NULL;

    list_desc -> head    = new_node;
    list_desc -> tail    = new_node;
    list_desc -> current = new_node;

  } else {

    list_desc -> tail -> next = new_node;

    new_node -> next     = NULL;
    new_node -> previous = list_desc -> tail;

    list_desc -> tail = new_node;

  } /* if */

  list_desc -> elements++;


  return( LST_OK );

} /* LstLinkInsertLast */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkInsertCurrent( LST_DESC_TYPE  list_desc,
                        void           *record )
{

  /* Variables. */
  char             *new_record;
  LST_RECORD_TYPE  *new_node;


  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  /* If the current pointer is not defined, insert at the beginning. */
  if( (list_desc -> current == NULL) || (list_desc -> head == NULL) )
    return( LstLinkInsertLast( list_desc, record ) );

  /* Create the new node and copy the record. */
  new_record = SysMalloc( list_desc -> record_size );
  memcpy( new_record, record, list_desc -> record_size );

  new_node = SysNew( LST_RECORD_TYPE );
  new_node -> record = new_record;

  /* Link the new record into the list. */
  if( list_desc -> current -> previous == NULL ) {

    new_node -> next     = list_desc -> current;
    new_node -> previous = NULL;

    list_desc -> current -> previous = new_node;
    list_desc -> head                = new_node;
    list_desc -> current             = new_node;

  } else {

    new_node -> next     = list_desc -> current;
    new_node -> previous = list_desc -> current -> previous;

    list_desc -> current -> previous -> next = new_node;
    list_desc -> current -> previous         = new_node;

    list_desc -> current = new_node;

  } /* if */

  list_desc -> elements++;


  return( LST_OK );

} /* LstLinkInsertCurrent */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkCurrentFirst( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  list_desc -> current = NULL;

  if( list_desc -> head == NULL )
    return( LST_STUCK );

  list_desc -> current = list_desc -> head;


  return( LST_OK );

} /* LstLinkCurrentFirst */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkCurrentLast( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  list_desc -> current = NULL;

  if( list_desc -> tail == NULL )
    return( LST_STUCK );

  list_desc -> current = list_desc -> tail;


  return( LST_OK );

} /* LstLinkCurrentLast */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkCurrentPosition( LST_DESC_TYPE  list_desc,
                          int            position )
{

  /* Variables. */
  int               counter = 1;
  LST_RECORD_TYPE   *ref;


  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  list_desc -> current = NULL;

  if( list_desc -> head == NULL )
    return( LST_STUCK );

  ref = list_desc -> head;

  /* Try to move to the position. */
  while( ref != NULL && counter < position ) {
    ref = ref -> next;
    counter++;
  }

  /* Did we reach the end? */
  if( ref == NULL )
    return( LST_STUCK );

  list_desc -> current = ref;


  return( LST_OK );

} /* LstLinkCurrentPosition */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkCurrentNext( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> current == NULL )
    return( LST_STUCK );

  if( list_desc -> current -> next == NULL )
    return( LST_STUCK );

  list_desc -> current = list_desc -> current -> next;


  return( LST_OK );

} /* LstLinkCurrentNext */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkCurrentPrevious( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> current == NULL )
    return( LST_STUCK );

  if( list_desc -> current -> previous == NULL )
    return( LST_STUCK );

  list_desc -> current = list_desc -> current -> previous;


  return( LST_OK );

} /* LstLinkCurrentPrevious */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkDeleteFirst( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> head == NULL )
    return( LST_ERROR );

  if( list_desc -> head -> next == NULL ) {

    SysFree( list_desc -> head -> record );
    SysFree( list_desc -> head );

    list_desc -> head    = NULL;
    list_desc -> tail    = NULL;
    list_desc -> current = NULL;

  } else {

    /* Remove the data for the record. */
    SysFree( list_desc -> head -> record );

    if( list_desc -> current == list_desc -> head )
      list_desc -> current = list_desc -> head -> next;

    list_desc -> head = list_desc -> head -> next;

    /* Remove the record. */
    SysFree( list_desc -> head -> previous );

    list_desc -> head -> previous = NULL;

  } /* if */

  list_desc -> elements--;


  return( LST_OK );

} /* LstLinkDeleteFirst */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkDeleteLast( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> tail == NULL )
    return( LST_ERROR );

  if( list_desc -> tail -> previous == NULL ) {

    SysFree( list_desc -> tail -> record );
    SysFree( list_desc -> tail );

    list_desc -> head    = NULL;
    list_desc -> tail    = NULL;
    list_desc -> current = NULL;

  } else {

    /* Remove the data for the record. */
    SysFree( list_desc -> tail -> record );

    if( list_desc -> current == list_desc -> tail )
      list_desc -> current = list_desc -> tail -> previous;

    list_desc -> tail = list_desc -> tail -> previous;

    /* Remove the record. */
    SysFree( list_desc -> tail -> next );

    list_desc -> tail -> next = NULL;

  } /* if */

  list_desc -> elements--;


  return( LST_OK );

} /* LstLinkDeleteLast */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkDeleteCurrent( LST_DESC_TYPE  list_desc )
{

  /* Variables. */
  LST_RECORD_TYPE  *temp_ref;


  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> current == NULL )
    return( LST_ERROR );

  /* Is this the first element? */
  if( list_desc -> current -> previous == NULL )
    return( LstLinkDeleteFirst( list_desc ) );

  /* Is this the last element? */
  if( list_desc -> current -> next == NULL )
    return( LstLinkDeleteLast( list_desc ) );


  /* Remove the data for the record. */
  SysFree( list_desc -> current -> record );

  list_desc -> current -> next -> previous = 
    list_desc -> current -> previous;

  list_desc -> current -> previous -> next = 
    list_desc -> current -> next;

  /* Before the node can be deleted, keep a reference. */
  temp_ref = list_desc -> current;

  list_desc -> current = list_desc -> current -> next;

  SysFree( temp_ref );

  list_desc -> elements--;


  return( LST_OK );

} /* LstLinkDeleteCurrent */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkGetCurrent( LST_DESC_TYPE  list_desc,
                     void           *record )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> current == NULL )
    return( LST_ERROR );

  memcpy( record, list_desc -> current -> record, list_desc -> record_size );


  return( LST_OK );

} /* LstLinkGetCurrent */


/*----------------------------------------------------------------------*/

void
  *LstLinkGetCurrentRef( LST_DESC_TYPE  list_desc )
{

  /* Code. */

  if( list_desc == NULL )
    return( (void *)NULL );

  if( list_desc -> current == NULL )
    return( (void *)NULL );


  return( (void *) list_desc -> current -> record );

} /* LstLinkGetCurrentRef */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkGetFirst( LST_DESC_TYPE  list_desc,
                   void           *record )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> head == NULL )
    return( LST_ERROR );

  memcpy( record, list_desc -> head -> record, list_desc -> record_size );


  return( LST_OK );

} /* LstLinkGetFirst */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkGetLast( LST_DESC_TYPE  list_desc,
                  void           *record )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> tail == NULL )
    return( LST_ERROR );

  memcpy( record, list_desc -> tail -> record, list_desc -> record_size );


  return( LST_OK );

} /* LstLinkGetLast */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkSetCurrent( LST_DESC_TYPE  list_desc,
                     void           *record )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> current == NULL )
    return( LST_ERROR );

  memcpy( list_desc -> current -> record, record, list_desc -> record_size );


  return( LST_OK );

} /* LstLinkSetCurrent */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkSetFirst( LST_DESC_TYPE  list_desc,
                   void           *record )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> head == NULL )
    return( LST_ERROR );

  memcpy( list_desc -> head -> record, record, list_desc -> record_size );


  return( LST_OK );

} /* LstLinkSetFirst */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkSetLast( LST_DESC_TYPE  list_desc,
                  void           *record )
{

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> tail == NULL )
    return( LST_ERROR );

  memcpy( list_desc -> tail -> record, record, list_desc -> record_size );


  return( LST_OK );

} /* LstLinkSetLast */


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkSearchFirst( LST_DESC_TYPE     list_desc,
                      void              *element,
                      EQUALS_FUNC_TYPE  equals_func )
{

  /* Variables. */
  EQUALS_FUNC_TYPE  compare_func;
  LST_RECORD_TYPE   *node_ref;
  

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> head == NULL )
    return( LST_NOT_FOUND );

  /* Use the supplied compare function? */
  if( equals_func != NULL )
    compare_func = equals_func;
  else if( list_desc -> equals_func != NULL )
    compare_func = list_desc -> equals_func;
  else
    return( LST_NOT_FOUND );

  node_ref = list_desc -> head;

  /* Search the element in the list. */
  while( node_ref != NULL ) {

    if( ( *compare_func )( node_ref -> record, element ) == LST_EQUAL ) {
      list_desc -> current = node_ref;
      return( LST_OK );
    }

    node_ref = node_ref -> next;

  } /* while */


  return( LST_NOT_FOUND );

} /* LstLinkSearchFirst */  


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkSearchLast( LST_DESC_TYPE     list_desc,
                     void              *element,
                     EQUALS_FUNC_TYPE  equals_func )
{

  /* Variables. */
  EQUALS_FUNC_TYPE  compare_func;
  LST_RECORD_TYPE   *node_ref;
  

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> tail == NULL )
    return( LST_NOT_FOUND );

  /* Use the supplied compare function? */
  if( equals_func != NULL )
    compare_func = equals_func;
  else if( list_desc -> equals_func != NULL )
    compare_func = list_desc -> equals_func;
  else
    return( LST_NOT_FOUND );

  node_ref = list_desc -> tail;

  /* Search the element in the list. */
  while( node_ref != NULL ) {

    if( ( *compare_func )( node_ref -> record, element ) == LST_EQUAL ) {
      list_desc -> current = node_ref;
      return( LST_OK );
    }

    node_ref = node_ref -> previous;

  } /* while */


  return( LST_NOT_FOUND );

} /* LstLinkSearchLast */  


/*----------------------------------------------------------------------*/

LST_STATUS
  LstLinkSearchCurrent( LST_DESC_TYPE     list_desc,
                        void              *element,
                        LST_DIRECTION     direction,
                        EQUALS_FUNC_TYPE  equals_func )
{

  /* Variables. */
  EQUALS_FUNC_TYPE  compare_func;
  LST_RECORD_TYPE   *node_ref;
  

  /* Code. */

  if( list_desc == NULL )
    return( LST_ERROR );

  if( list_desc -> current == NULL )
    return( LST_NOT_FOUND );

  /* Use the supplied compare function? */
  if( equals_func != NULL )
    compare_func = equals_func;
  else if( list_desc -> equals_func != NULL )
    compare_func = list_desc -> equals_func;
  else
    return( LST_NOT_FOUND );

  node_ref = list_desc -> current;

  /* Search the element in the list. */
  while( node_ref != NULL ) {

    if( ( *compare_func )( node_ref -> record, element ) == LST_EQUAL ) {
      list_desc -> current = node_ref;
      return( LST_OK );
    }

    if( direction == LST_FORWARD )
      node_ref = node_ref -> next;
    else
      node_ref = node_ref -> previous;

  } /* while */


  return( LST_NOT_FOUND );

} /* LstLinkSearchCurrent */  


/*----------------------------------------------------------------------*/

int
  LstLinkElements( LST_DESC_TYPE list_desc )
{

  /* Code. */

  return( list_desc -> elements );

} /* LstLinkElements */


