/*----------------------------------------------------------------------------
--
--  Module:           xtmSetup
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Check that the current XDiary setup is correct and if no, try to
--    create a correct environment.
--
--  Filename:         xtmSetup.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmSetup.c, Version: 1.1, Date: 95/02/18 15:52:49";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <errno.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmIcons.h"
#include "xitError.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xtmSetup.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* The different setup actions. */
typedef enum {
  NONE, DEFAULT_DB_MISSING, CANNOT_READ_DEFAULT_DB
} SETUP_ACTION;


/* Local data for the setup. */
typedef struct {

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  /* Actions to do. */
  SETUP_ACTION  action;

  /* Toplevel widget. */
  Widget  toplevel;

} SETUP_REC, *SETUP_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmSetup";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Boolean
  createDbDir( Widget                  baseW,
               XTM_GL_CUSTOM_DATA_REF  custom_data_ref,
               char                    *directory );

static Boolean
  createDb( Widget  baseW,
            char    *directory );

static void 
  exitCB( void  *user_data );

static void 
  okCB( void  *user_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmStCheckSetup( XTM_GL_CUSTOM_DATA_REF  custom_data_ref,
                   Widget                  parentW,
                   XtAppContext            context )
{

  /* Variables. */
  Boolean          ok;
  char             explain_buffer[ 500 ];
  Widget           toplevelW;
  SETUP_ACTION     action = NONE;
  SETUP_REC_REF    setup_ref;
  XTM_CD_CAL_INFO  default_db;

  static Pixmap  logo_pixmap = XmUNSPECIFIED_PIXMAP;


  /* Code. */

  toplevelW = xitGetToplevelWidget( parentW );


  /* Do we have a default database? */
  ok = xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                            &default_db, NULL );
  if( ! ok )
    action = DEFAULT_DB_MISSING;


  /* Can we read our default database? */
  if( action == NONE && 
      ! flagIsSet( default_db.operations, XTM_DB_FLAG_MODE_READ ) )
    action = CANNOT_READ_DEFAULT_DB;


  /* Is the environment correct? If yes, just return. */
  if( action == NONE )
    return;


  /* Our local setup data. */
  setup_ref = SysNew( SETUP_REC );

  setup_ref -> custom_data_ref = custom_data_ref;
  setup_ref -> action          = action;
  setup_ref -> toplevel        = toplevelW;


  /* What is the problem? */
  switch( action ) {
    case DEFAULT_DB_MISSING:
      sprintf( explain_buffer, msgGetText( MXDI_SETUP_DB_MISSING ) );
      break;

    case CANNOT_READ_DEFAULT_DB:
      sprintf( explain_buffer, msgGetText( MXDI_SETUP_CANNOT_READ_DB ), 
               default_db.short_name, default_db.directory );
      break;
  }


  /* Create the logo Pixmap? */
  if( logo_pixmap == XmUNSPECIFIED_PIXMAP ) {

    Widget  shellW;
    Widget  tempW;

    shellW = xitCreateToplevelDialog( toplevelW, "TempTl", 
                                      0, 0, NULL, 0 );

    tempW = XtNameToWidget( shellW, "TempTlBase.TempTlFo" );

    logo_pixmap = xtmIcFetchSimplePixmap( tempW, 
                                          XTM_IC_ICON_XDIARY_LOGO, False );
    XtDestroyWidget( shellW );
  }


  /* Display a message about what we want to do, return when done. */
  xitStDisplaySticky( toplevelW,
                      explain_buffer, logo_pixmap,
                      msgGetText( MXDI_OK_BUTTON ),
                      msgGetText( MXDI_CANCEL_BUTTON ),
                      okCB, exitCB, exitCB,
                      (void *) setup_ref );

  SysFree( setup_ref );


  return;

} /* xtmStCheckSetup */


/*----------------------------------------------------------------------*/

static Boolean
  createDbDir( Widget                  baseW,
               XTM_GL_CUSTOM_DATA_REF  custom_data_ref,
               char                    *directory )
{

  /* Variables. */
  int          status;
  char         buffer[ PATH_MAX + 1024 ];
  struct stat  file_stat;


  /* Code. */

  /* Create the directory as user read only (rwx--x--x). */
  status = mkdir( directory, (S_IRWXU | S_IXGRP | S_IXOTH) );

  if( status != 0 && errno != EEXIST ) {
    sprintf( buffer, msgGetText( MXDI_CANNOT_CREATE_DB_DIR ),
             directory);

    xitErMessage( baseW, XIT_ER_ERROR,
                  module_name, "createDbDir",
                  buffer );

    return( False );
  }


  /* Create the message directory. */
  sprintf( buffer, "%s/Message", directory );
  status = mkdir( buffer, (S_IRWXU | S_IXGRP | S_IXOTH) );

  if( status != 0 && errno != EEXIST ) {
    sprintf( buffer, msgGetText( MXDI_CANNOT_CREATE_DB_DIR ),
             directory );

    xitErMessage( baseW, XIT_ER_ERROR,
                  module_name, "createDbDir",
                  buffer );

    return( False );
  }


  /* Create the private directory. */
  sprintf( buffer, "%s/Private", directory );
  status = mkdir( buffer, S_IRWXU );

  if( status != 0 && errno != EEXIST ) {
    sprintf( buffer, msgGetText( MXDI_CANNOT_CREATE_DB_DIR ),
             directory );

    xitErMessage( baseW, XIT_ER_ERROR,
                  module_name, "createDbDir",
                  buffer );

    return( False );
  }


  /* Set AFS ACL (if directory in AFS). */
  status = stat( directory, &file_stat );

#ifdef XD_HAS_AFS
  if( file_stat.st_vfstype == custom_data_ref -> afs_vfs_id ) {

    char           user_acl[ 512 ];
    char           user[ 256 ];
    struct passwd  *password_ref;

    /* Fetch the name of the user. */
    password_ref = getpwuid( getuid() );

    if( password_ref == NULL )
      strcpy( user, "deadbeef" );
    else
      strcpy( user, password_ref -> pw_name );

    sprintf( user_acl, "-acl %s all", user );

    sprintf( buffer, "fs setacl -dir %s -clear %s", 
             directory, user_acl );
    status = system( buffer );

    sprintf( buffer, "fs setacl -dir %s/Message -clear %s", 
             directory, user_acl );
    status = system( buffer );

    sprintf( buffer, "fs setacl -dir %s/Private -clear %s",
             directory, user_acl );
    status = system( buffer );

  } /* if */
#endif


  return( True );

} /* createDbDir */


/*----------------------------------------------------------------------*/

static Boolean
  createDb( Widget  baseW,
            char    *directory )
{

  /* Variables. */
  XTM_DB_CREATE_REQUEST  create_request;
  XTM_DB_STATUS          db_status;


  /* Code. */

  /* Initialize the database (create if not exist). */
  create_request.directory = directory;

  create_request.database = XTM_DB_ENTRY_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( baseW, XIT_ER_ERROR, 
                  module_name, "createDb",
                  msgGetText( MXDI_ERRMSG_CREATE_DB ) );

    return( False );
  }

  create_request.database = XTM_DB_DATE_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( baseW, XIT_ER_ERROR, 
                  module_name, "createDb",
                  msgGetText( MXDI_ERRMSG_CREATE_DB ) );

    return( False );
  }

  create_request.database = XTM_DB_STAND_ENTRY_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( baseW, XIT_ER_ERROR, 
                  module_name, "createDb",
                  msgGetText( MXDI_ERRMSG_CREATE_STAND_DB ) );

    return( False );
  }


  create_request.database = XTM_DB_PRIV_ENTRY_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( baseW, XIT_ER_ERROR, 
                  module_name, "createDb",
                  msgGetText( MXDI_ERRMSG_CREATE_PRIV_DB ) );

    return( False );
  }


  /* Create the identifier file. */
  db_status = xtmDbInitializeAuxFiles( directory );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( baseW, XIT_ER_ERROR, 
                  module_name, "createDb",
                  msgGetText( MXDI_ERRMSG_CREATE_ID_FILE ) );

    return( False );
  }


  return( True );

} /* createDb */


/*----------------------------------------------------------------------*/

static void 
  exitCB( void  *user_data )
{

  /* Code. */

  exit( 1 );

} /* exitCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( void  *user_data )
{

  /* Variables. */
  char           calendar_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  struct passwd  *password_ref;
  SETUP_REC_REF  setup_ref;


  /* Code. */

  setup_ref = (SETUP_REC_REF) user_data;


  /* No default database? */
  if( setup_ref -> action == DEFAULT_DB_MISSING ) {

    Boolean  ok;
    char     directory[ 51 ];
    char     *char_ref;

    /* Construct the default diary directory. */
    char_ref = getenv( "HOME" );

    if( char_ref != NULL )
      sprintf( directory, "%s/%s", char_ref, "Diary" );
    else
      sprintf( directory, "./%s", "Diary" );


    /* Create the database directories. */
    ok = createDbDir( setup_ref -> toplevel, 
                      setup_ref -> custom_data_ref, directory );
    if( ! ok )
      return;


    /* Create the database. */
    ok = createDb( setup_ref -> toplevel, directory );
    if( ! ok )
      return;


    /* Fetch the name of the user. */
    password_ref = getpwuid( getuid() );

    if( password_ref != NULL ) {
      strncpy( calendar_name, password_ref -> pw_name, XTM_GL_MAX_CAL_NAME );
      calendar_name[ XTM_GL_MAX_CAL_NAME ] = '\0';
    } else {
      strcpy( calendar_name, "Default" );
    }


    /* Mark the created database as the default database. */
    xtmCdAddDatabase( setup_ref -> custom_data_ref -> cal_db_handle,
                      calendar_name, directory,
                      (XTM_CD_FLAG_ALARM | XTM_CD_FLAG_DEFAULT_DB) );

    (void) xtmCbPutDataToFile( setup_ref -> custom_data_ref );

  } /* if */


  /* Cannot read default database. */
  if( setup_ref -> action == CANNOT_READ_DEFAULT_DB ) {

    exit( 0 );

  } /* if */


} /* okCB */
